/* Copyright 2006 VMware, Inc.   All rights reserved. -- VMware Confidential */

/**
 * public class WatchDog
 * extends vpx.core.VpxObject
 *
 * The <code>WatchDog</code> monitors an <code>Agent</code> to make sure that
 * its requests are coming back in a timely fashion.  If it detects slowness in
 * the agent's request/response lifecycle (or no response at all), it will open
 * up the update cycle of the agent in an attempt to give the server a breath.
 * It will then attempt tighten the update cycle after a suitable "cooling
 * down" period has elapsed.  If the agent remains unresponsive after several
 * attempts to give the server a break, this will shut down the agent and alert
 * the user of the problem.
 *
 * @version 1.0 (May 10, 2006)
 */

/**
 * Constructs a new <code>WatchDog</code>, to be associated with the given
 * updates agent.
 *
 * @param agent vpx.updates.Agent
 *    The agent to watch over
 */
vpx.updates.WatchDog = function WatchDog(agent) {
   this.agent = agent;
   this.boundFunctions = {
      analyzeAgent : vpx.updates.WatchDog.prototype._analyzeAgentPerformance.bind(this)
   };
};

// WatchDog extends vpx.core.VpxObject
vpx.updates.WatchDog.prototype = new vpx.core.VpxObject(vpx.ABSTRACT_PASS);
vpx.updates.WatchDog.prototype.constructor = vpx.updates.WatchDog;

// Shorthand for brevity's sake
var _c = vpx.updates.WatchDog;        // Class
var _i = _c.prototype;                // Instance
_i._concrete = true;                  // vpx system flag for concrete classes

// Instance variables
_i.agent = null;                      // private vpx.updates.Agent
_i.lastRequest = null;                // private int
_i.lastUpdate = null;                 // private int
_i.lastRoundTrip = null;              // private int
_i.boundFunctions = null;             // private transient Map<String,Function>

// Class constants
_c.INTERVAL = 5000;                   // private int
_c.MAX_WAIT = 120000;                 // public int

/**
 * Starts this watchdog's thread.
 */
_i.start = function() {
   if (this.dying) {
      return;
   }
   if (this.intervalId != null) {
      throw new Error("WatchDog#start(): WatchDog already running");
   }
   this.intervalId = window.setInterval(this.boundFunctions.analyzeAgent,
                                        vpx.updates.WatchDog.INTERVAL);
};

/**
 * Stops the operation of this watchdog.
 */
_i.stop = function() {
   if (this.intervalId == null) {
      throw new Error("WatchDog#stop(): WatchDog not running");
   }
   window.clearInterval(this.intervalId);
   this.intervalId = null;
};

/**
 * Destroys this watchdog, freeing up any resources that have been allocated
 * to this object.
 */
_i.destroy = function() {
   // super.destroy()
   vpx.core.VpxObject.prototype.destroy.call(this);

   if (this.intervalId != null) {
      this.stop();
   }
   delete this.agent;
   delete this.boundFunctions.analyzeAgent;
   delete this.boundFunctions;
};

/**
 * Records the fact that the updates agent associated with this watchdog has
 * made a request to the server.
 */
_i.recordRequest = function() {
   if (this.dying) {
      return;
   }
   var now = (new Date()).getTime();
   this.lastRequest = (new Date()).getTime();
};

/**
 * Records the fact that the updates agent associated with this watchdog has
 * received a response from the server.
 */
_i.recordResponse = function() {
   if (this.dying) {
      return;
   }
   var now = (new Date()).getTime();
   if (this.lastRequest != null) {
      this.lastRoundTrip = now - this.lastRequest;
   }
   this.lastUpdate = now;
   this._analyzeAgentPerformance();
};

/**
 * Records a "no request needed" (or NO-OP) from the updates agent.
 */
_i.recordNoOp = function() {
   if (this.dying) {
      return;
   }
   this.lastRoundTrip = null;
   this.lastRequest = null;
   this.lastUpdate = (new Date()).getTime();
};

/*
 * (non-doc)
 *
 * @see Object#toString()
 */
_i.toString = function() {
   return "[Object vpx.updates.WatchDog]";
};


/*************************************************************************
 * All data and procedures below this point are part of the internal     *
 * implementation, should not be accessed outside of this module, and    *
 * are subject to change.                                                *
 *************************************************************************/


/**
 * Analyzes the performance of the updates agent associated with this watchdog
 * and adjusts the updates interval of the agent as necessary.
 */
_i._analyzeAgentPerformance = function() {
   if (this.dying) {
      return;
   }
   var sinceUpdate = (new Date()).getTime() - this.lastUpdate;
   var frequency = this.agent.getInterval();
   var upr = Math.min(frequency * 2, Math.ceil(vpx.updates.WatchDog.MAX_WAIT / 2));
   var lwr = Math.floor(frequency / 2);

   vpx.log.debug("WatchDog: Checking agent's performance... lag:" + this.lastRoundTrip);
   if (sinceUpdate > vpx.updates.WatchDog.MAX_WAIT) {
      // Abandon request that's timed out, & re-request
      this.recordNoOp();
      this.agent._cancelPoll();
      this.agent._doPoll();
   } else if ((this.lastRoundTrip != null && this.lastRoundTrip > upr) || sinceUpdate > upr) {
      vpx.log.info("WatchDog: raising agent's interval to " + upr);
      this.agent.setInterval(upr);
   } else if (this.lastRoundTrip != null && this.lastRoundTrip < lwr) {
      vpx.log.info("WatchDog: lowering agent's interval to " + lwr);
      this.agent.setInterval(lwr);
   }
   this.lastRoundTrip = null;
};
