/* Copyright 2005 VMware, Inc.   All rights reserved. -- VMware Confidential */

/**
 * public class HtmlEditor
 * extends Object
 * implements vpx.net.event.ResponseListener
 *
 * A simple utility class that listens for responses from the server containing
 * html Strings and replaces the contents of an <code>HTMLElement</code> with
 * the specified String.  It's used when you want to dynamically update the
 * contents of a client-side element, but you need the server to tell you what
 * the value of the element should be.  Such can be the case when you need to
 * convert a numeric value into a localized string.
 *
 * @version 1.0 (Feb 1, 2006)
 */

/**
 * Constructs a new <code>HtmlEditor</code> object and associates it with the
 * given html element.
 *
 * @param el HTMLElement
 *    The element that this editor will control
 */
vpx.util.HtmlEditor = function HtmlEditor(el) {
   this.el = el;
   //if el is null, an own dataListener should be provided
   if (el != null) {
      this.setDataListener(this);
   }
};

// Shorthand for brevity's sake
var _c = vpx.util.HtmlEditor;         // Class
var _i = _c.prototype;                // Instance
_i._concrete = true;                  // vpx system flag for concrete classes

// Instance variables
_i.el = null;                         // protected HTMLElement
_i.dataListener = null;               // dataListener that will be executed when a response is received


/**
 * Default DataListener callback: sets the element's innerHTML
 *
 * @param html the new content to be set in the element
 *    The element that this editor controls
 */
_i.dataReceived = function(html) {
   if (this.el != null) {
      this.el.innerHTML = html;
   }
};

/**
 * Gets the html element associated with this editor.
 *
 * @return HTMLElement
 *    The element that this editor controls
 */
_i.getElement = function() {
   return this.el;
};

/**
 * Registers a data listener that will be executed when a response is received
 *
 * @param l listener
 *    The element that this editor controls
 */

_i.setDataListener = function(l) {
   this.dataListener = l;
};



/**
 * Invoked when the server has fully responded to a client-side request.  This
 * pulls out the value of the html from the response's xml and updates this
 * object's html element to contain that html.
 *
 * @param e vpx.net.event.ResponseEvent
 *    The ResponseEvent object
 */
_i.responseReceived = function(e) {
   var req = e.getRequest();
   var resp = e.getSource();
   var status = resp.getStatus();
   if (status != 200) {
      // Only proceed if HTTP status is "200 OK" (otherwise throw error or return)
      var url = req.getUrl();
      var msg = resp.getStatusMsg();
      switch (status) {
      case 404:
         throw new Error("HtmlEditor#responseReceived(): 404 Not Found: " + url);
      case 500:
         throw new Error("HtmlEditor#responseReceived(): 500 Application Error: " + url);
      default:
         /**
          * All other non-200 status codes represent some (esoteric) failure of the
          * request/response transaction.  In these cases, we know the server didn't
          * successfully process the request in the manner we intended.  If the server
          * didn't process the request, we need to make the request again.
          */
         vpx.log.warn("HtmlEditor#responseReceived(): " + status + " " + msg);
         var pool = vpx.net.HttpRequestPool.getInstance();
         req = pool.getRequest();
         try {
            req.setUrl(url);
            req.addResponseListener(this);
            req.send();
         } finally {
            pool.releaseRequest(req);
         }
         return;
      }
   }

   var contentType = resp.getContentType();
   if (contentType != "text/xml") {
      throw new Error("HtmlEditor#responseReceived(): invalid content-type: " +
                      contentType + " (status " + status + ")");
   }

   vpx.log.trace("HtmlEditor#responseReceived(): Processing data from server");
   var xml = resp.getXml();
   var root = xml.documentElement;
   if (root == null) {
      // XML was not well-formed
      throw new Error("HtmlEditor#responseReceived(): xml was not well-formed " +
                      "while processing " + req.getUrl());
   }
   root.normalize();

   if (root.nodeName == "errors") {
      tle.processErrors(root, false);
      return;
   }

   var nodes = root.getElementsByTagName("html");
   if (nodes.length > 1) {
      throw new Error("HtmlEditor#responseReceived(): expected 1 edit; got " +
                      nodes.length);
   }

   var html = vpx.xua.getInnerContent(nodes[0]);
   if (this.dataListener != null) {
      this.dataListener.dataReceived(html);
   }
};
