/* Copyright 2006 VMware, Inc.   All rights reserved. -- VMware Confidential */

/**
 * public class Properties
 * extends Object
 *
 * Properties objects hold information on a set of properties for a view.
 * Properties spanning multiple views require multiple instances of this class.
 *
 * @version 1.0 (Feb 13, 2006)
 */

/**
 * Constructs a new <code>Properties</code> object associated with the given
 * viewId.
 *
 * @param viewId String
 *    The id of the view whose properties we're representing
 */
vpx.util.view.Properties = function Properties(viewId) {
   this.viewId = viewId;
   this.keys = [];
   this.props = {};
};

// Shorthand for brevity's sake
var _c = vpx.util.view.Properties;        // Class
var _i = _c.prototype;                    // Instance
_i._concrete = true;                      // vpx system flag for concrete classes

// Instance variables
_i.viewId = null;                         // private String
_i.keys = null;                           // private Set<String>
_i.props = null;                          // private Map<String, String>

/**
 * Adds the given property to this view's list of properties.  If the given
 * property is already in the list, nothing happens.
 *
 * @param property String
 *    The property key
 */
_i.addProperty = function(property) {
   if (this.keys.indexOf(property) >= 0) {
      return;
   }
   this.keys.push(property);
   this.props[property] = undefined;
};

/**
 * Removes the given property from this view's list of properties.  If the
 * given property is not in the list, nothing happens.
 *
 * @param property String
 *    The property key
 */
_i.removeProperty = function(property) {
   var i = this.keys.indexOf(property);
   if (i == -1) {
      return;
   }
   this.keys.splice(i, 1);
   delete this.props[property];
};

/**
 * Gets the current value of the given property.  If the given property has not
 * yet been synced with the server, the value will be <code>undefined</code>.
 * If the property is not in this view's list of properties, an error will be
 * thrown.
 *
 * @param property String
 *    The property key
 * @return String
 *    The value of the property
 * @throws Error
 *    Thrown if the property is not in the view's property list
 */
_i.getProperty = function(property) {
   if (this.keys.indexOf(property) == -1) {
      throw new Error("Properties#getProperty(): property not found: " + property);
   }
   return this.props[property];
};

/**
 * Sets the value of a property.  If the property is not in this view's list of
 * properties, an error will be thrown.
 *
 * @param property String
 *    The property key
 * @param value String
 *    The value of the property
 * @throws Error
 *    Thrown if the property is not in the view's property list
 */
_i.setProperty = function(property, value) {
   if (this.keys.indexOf(property) == -1) {
      throw new Error("Properties#setProperty(): property not found: " + property);
   }
   this.props[property] = value;
};

/**
 * Gets the list of properties in this view's property list.
 *
 * @return String[]
 *    An array of property names tracked for this view
 */
_i.getProperties = function() {
   var res = new Array(this.keys.length);
   vpx.core.System.arraycopy(this.keys, 0, res, 0, this.keys.length);
   return res;
};

/**
 * Gets the number of properties in this view's property list.
 *
 * @return int
 *    The number of properties in this view's property list
 */
_i.getPropertyCount = function() {
   return this.keys.length;
};
