/* Copyright 2006 VMware, Inc.   All rights reserved. -- VMware Confidential */

/**
 * public class PropertyCollector
 * extends Object
 * implements vpx.net.event.ResponseListener
 *
 * A simple utility class with whom callers may register interest in properties
 * of views.  This class can then ask the server for the values of these
 * properties, store the values into JavaScript data structures, and notify
 * the interested parties when the data is ready for processing.
 *
 * @version 1.0 (Feb 13, 2006)
 */

/**
 * Constructs a new <code>PropertyCollector</code> object, initially associated
 * with no views / properties.
 */
vpx.util.view.PropertyCollector = function PropertyCollector() {
   this.viewIds = [];
   this.props = {};
   this.xmlSpec = new vpx.net.XmlSpec(vpx.util.view.PropertyCollector.UPDATE_URL);
   this.listenerList = new vpx.core.event.EventListenerList();
};

// Shorthand for brevity's sake
var _c = vpx.util.view.PropertyCollector; // Class
var _i = _c.prototype;                    // Instance
_i._concrete = true;                      // vpx system flag for concrete classes

// Static (class) constants
_c.UPDATE_URL       = "viewProperties.do";
_c.PARAM_VIEW_IDS   = "viewIds";
_c.PARAM_PROPERTIES = "props.";

// Instance variables
_i.viewIds = null;                     // private Set
_i.props = null;                       // private Map<String, vpx.util.view.Properties>
_i.xmlSpec = null;                     // private vpx.net.XmlSpec
_i.listenerList  = null;               // private vpx.core.event.EventListenerList
_i.propertyEvent = null;               // private vpx.util.view.PropertyEvent

/**
 * Adds the given view property to the list of properties that this collector
 * tracks.  If the given property is already being tracked, nothing happens.
 *
 * @param viewId String
 *    The id of the view that contains the property
 * @param property String
 *    The property key
 */
_i.addProperty = function(viewId, property) {
   var c = vpx.util.view.PropertyCollector;
   var p = this.props[viewId];

   if (!p) {
      this.viewIds.push(viewId);
      this.xmlSpec.setAttribute(c.PARAM_VIEW_IDS, this.viewIds.join(","));
      p = this.props[viewId] = new vpx.util.view.Properties(viewId);
   }

   p.addProperty(property);
   this.xmlSpec.setAttribute(c.PARAM_PROPERTIES + viewId, p.getProperties().join(","));
};

/**
 * Removes the given view property from the list of properties that this
 * collector tracks.  If the given property is not currently being tracked,
 * nothing happens.
 *
 * @param viewId String
 *    The id of the view that contains the property
 * @param property String
 *    The property key
 */
_i.removeProperty = function(viewId, property) {
   var c = vpx.util.view.PropertyCollector;
   var p = this.props[viewId];
   if (!p) {
      // Do nothing
      return;
   }

   p.removeProperty(property);
   this.xmlSpec.setAttribute(c.PARAM_PROPERTIES + viewId, p.getProperties().join(","));

   if (p.getPropertyCount() == 0) {
      this.viewIds.splice(this.viewIds.indexOf(viewId), 1);
      this.xmlSpec.setAttribute(c.PARAM_VIEW_IDS, this.viewIds.join(","));
      this.xmlSpec.removeAttribute(c.PARAM_PROPERTIES + viewId);
      delete this.props[viewId];
      delete p;
   }
};

/**
 * Gets the current value of the given view property.  If the given property
 * is being tracked but has not yet been synced with the server, the value will
 * be <code>undefined</code>.  If the property is not being tracked, an error
 * will be thrown.
 *
 * @param viewId String
 *    The id of the view that contains the property
 * @param property String
 *    The property key
 * @return String
 *    The value of the property
 * @throws Error
 *    Thrown if the property is not being tracked by this collector
 */
_i.getProperty = function(viewId, property) {
   var p = this.props[viewId];
   if (!p) {
      throw new Error("PropertyCollector#getProperty(): untracked view: " + viewId);
   }
   return p.getProperty(property);
};

/**
 * Requests up-to-date values from the server for each of the view properties
 * that this collector tracks.  If this collector tracks no properties, nothing
 * happens.
 */
_i.sync = function() {
   if (this.viewIds.length > 0) {
      this._doRequest(this.xmlSpec.toUrl());
   }
};

/**
 * Resets this collector to its initial state (no tracked properties).
 */
_i.reset = function() {
   this.viewIds = [];
   this.props = {};
   this.xmlSpec.clearAttributes();
   var lstnrs = this.getPropertyListeners();
   for (var i = 0; i < lstnrs.length; i++) {
      this.removePropertyListener(lstnrs[i]);
   }
};

/**
 * Adds the specified property listener to receive property events from this
 * collector. If l is null, no exception is thrown, and no action is performed.
 *
 * @param l vpx.util.view.PropertyListener
 *    The property listener
 */
_i.addPropertyListener = function(l) {
   if (l == null) {
      return;
   }
   this.listenerList.add("PropertyListener", l);
};

/**
 * Removes the specified property listener so it no longer receives property
 * events from this collector. If l is null, no exception is thrown, and no
 * action is performed.
 *
 * @param l vpx.util.view.PropertyListener
 *    The property listener
 */
_i.removePropertyListener = function(l) {
   if (l == null) {
      return;
   }
   this.listenerList.remove("PropertyListener", l);
};

/**
 * Returns an array of all the property listeners registered on this collector.
 *
 * @return vpx.util.view.PropertyListener[]
 *    All of this collector's <code>PropertyListener</code>s or an empty array
 *    if no property listeners are currently registered
 */
_i.getPropertyListeners = function() {
   return this.listenerList.getListeners("PropertyListener");
};

/**
 * Notifies all listeners that have registered interest for notification on
 * this event type.  The property event is created lazily.
 */
_i._firePropertiesUpdated = function() {
   // Guaranteed to return a non-null array
   var listeners = this.listenerList.getListenerList();

   // Process listeners last->first, notifying those who're interested in event
   for (var i = listeners.length - 2; i >= 0; i -= 2) {
      if (listeners[i] == "PropertyListener") {
         if (this.propertyEvent == null) {
            this.propertyEvent = new vpx.util.view.PropertyEvent(this);
         }

         // listeners[i + 1] is of type vpx.util.view.PropertyListener
         listeners[i + 1].propertiesUpdated(this.propertyEvent);
      }
   }
};

/**
 * Invoked when the server has fully responded to a client-side request.  This
 * extracts the properties from the xml, prepares the corresponding
 * <code>vpx.util.view.Properties</code> objects, and notifies all listeners
 * that this collector's properties have been updated.
 *
 * @param e vpx.net.event.ResponseEvent
 *    The ResponseEvent object
 */
_i.responseReceived = function(e) {
   var req = e.getRequest();
   var resp = e.getSource();

   var status = resp.getStatus();
   if (status != 200) {
      // Only proceed if HTTP status is "200 OK" (otherwise throw error or return)
      var url = req.getUrl();
      var msg = resp.getStatusMsg();
      switch (status) {
      case 404:
         throw new Error("PropertyCollector#responseReceived(): 404 Not Found: " + url);
      case 500:
         throw new Error("PropertyCollector#responseReceived(): 500 App Error: " + url);
      default:
         /**
          * All other non-200 status codes represent some (esoteric) failure of the
          * request/response transaction.  In these cases, we know the server didn't
          * successfully process the request in the manner we intended.  If the server
          * didn't process the request, we need to make the request again.
          */
         vpx.log.warn("PropertyCollector#responseReceived(): " + status + " " + msg);
         this._scheduleRequest(url);
         return;
      }
   }

   var contentType = resp.getContentType();
   if (contentType != "text/xml") {
      throw new Error("PropertyCollector#responseReceived(): invalid content-type: " +
                      contentType + " (status " + status + ")");
   }

   var xml = resp.getXml();
   var root = xml.documentElement;
   if (root == null) {
      // XML was not well-formed
      throw new Error("PropertyCollector#responseReceived(): xml was not well-formed " +
                      "while processing " + req.getUrl());
   }
   root.normalize();

   if (root.nodeName == "errors") {
      tle.processErrors(root, false);
      return;
   }

   var viewNodes = root.getElementsByTagName("view");
   for (var i = 0; i < viewNodes.length; i++) {
      // Each <view> node pertains to properties of a particular view
      var viewNode = viewNodes[i];
      var viewId = viewNode.getAttribute("id");
      var p = this.props[viewId];

      if (p) {
         var propNodes = viewNode.getElementsByTagName("prop");
         for (var j = 0; j < propNodes.length; j++) {
            var propNode = propNodes[j];
            var key = propNode.getAttribute("name");
            var value = vpx.xua.getInnerContent(propNode);
            p.setProperty(key, value);
         }
      }
   }

   this._firePropertiesUpdated();
};


/*************************************************************************
 * All data and procedures below this point are part of the internal     *
 * implementation, should not be accessed outside of this module, and    *
 * are subject to change.                                                *
 *************************************************************************/


/**
 * Makes an immediate request to the specified url, and listens for the
 * response.
 *
 * @param url String
 *    The url (including query parameters) to which to make the request
 */
_i._doRequest = function(url) {
   var pool = vpx.net.HttpRequestPool.getInstance();
   var req = pool.getRequest();
   if (req == null) {
      // Wait for thread to become available and try again
      this._scheduleRequest(url);
      return;
   }
   try {
      req.setUrl(url);
      req.addResponseListener(this);
      req.send();
   } finally {
      pool.releaseRequest(req);
   }
};

/**
 * Schedules a request to be made to the specified url at some point in the
 * near future.
 *
 * @param url String
 *    The url (including query parameters) to which to make the request
 */
_i._scheduleRequest = function(url) {
   window.setTimeout(this._doRequest.bind(this, url), 500);
};
