/* Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential */

/**
 * The base class for encapsulated viewports. A window takes the form of one or
 * two IFRAMEs. One (the root pane) provides a container for the window
 * decorations; the second (the content pane) serves as the viewport for the
 * window contents. A root pane may also serve as the content pane for a
 * window, but programmers using the resulting object do not need to know this.
 *
 * The the window decorations and contents are controlled by HTML markup
 * written into the IFRAMEs or loaded from URLs. Although the basic API for a
 * window is predetermined, the root pane HTML page is free to override and
 * extend this API as needed.
 */

vpx.win.Frame = function (contentSpec, frameSpec, rootSpec, context) {
   this._ready = false;
   this.content = contentSpec || clone(vpx.win.DEFAULT_CONTENT_SPEC, true);
   this.context = context || self;

   if (! isDefined(rootSpec)) {
      rootSpec = clone(vpx.win.DEFAULT_ROOT_SPEC, true);
   }
   extendObject(this, rootSpec);

   if (! isDefined(frameSpec)) {
      frameSpec = clone(vpx.win.DEFAULT_FRAME_SPEC, false);
   }
   extendObject(this, frameSpec, /*include*/ null, /*exclude*/ ["size"]);

   if (! isDefined(this.element)) {
      this.element = vpx.xua.create.call(this.context, "iframe");

      vpx.xua.setAttribute(this.element,
                           "frameborder", "no",
                           "scrolling",   "no",
                           "src",         vpx.xua.BLANK_FRAME_SRC);

      // XXX:
      var container = vpx.xua.viewport.getPage(this.context).body;
      this.element = vpx.xua.append(container, this.element);

      vpx.xua.setStyle(this.element,
                       "position", "absolute",
                       "overflow", "hidden",
                       "margin",   "0px",
                       "padding",  "0px",
                       "border",   "none");

      if (isDefined(frameSpec.size)) {
         vpx.xua.setDimension(this.element, frameSpec.size);
      }

      vpx.xua.setPosition(this.element,
	 new Position(0, 0).subtract(
	    vpx.xua.getDimension(this.element).toPosition())
      );
   }

   extendObject(this, VPX_WIN_SOURCE_TYPES);
   extendObject(this, VPX_WIN_PLACEMENT_POLICIES);
   extendObject(this, VPX_WIN_RESIZE_POLICIES);

   this.sourceTypeMasks = {
      "example": this.SOURCE_TYPE_URL +
      this.SOURCE_TYPE_STRING_DOCUMENT +
      this.SOURCE_TYPE_DOM_DOCUMENT
   };

   this.callbacks = {
      "ready"  : [],
      "update" : [],
      "resize" : [],
      "okay"   : [],
      "cancel" : []
   };

   this.eventHandlers = [];

   this.init = this.initRoot;
};

vpx.win.Frame.prototype.getBoundary = function () {
   if (isDefined(this.boundary)) {
      return this.boundary;
   }

   var dim = vpx.xua.getDimension(this.context);
   var pos = vpx.xua.viewport.getOffset(vpx.xua.getView(this.context), this.context);

   if (! isView(this.context)) {
      pos = pos.add(vpx.xua.getPosition(this.context));
   }

   return new Limit(pos, pos.add(dim.toPosition()));
};

vpx.win.Frame.prototype.getSourceTypeMaskForMethod = function (method) {
   if (isDefined(this.sourceTypeMasks[method])) {
      return this.sourceTypeMasks[method];
   }

   return 0;
};

vpx.win.Frame.prototype.setSourceTypeMaskForMethod = function (method, mask) {
   return this.sourceTypeMasks[method] = mask;
};

vpx.win.Frame.prototype.listen = function (event, method) {
   this.callbacks[event].push(method);
};

vpx.win.Frame.prototype.ignore = function (event, method) {
   for (var i = 0; i < this.callbacks[event].length; i++) {
      if (this.callbacks[event][i] == method) {
         this.callbacks[event].splice(i, 1);
         break;
      }
   }
};

vpx.win.Frame.prototype.ready = function () {
   this._ready = true;
   
   if (this.resizePolicy & vpx.win.RESIZE_POLICY_SHRINKWRAP) {
      this.shrinkWrap();
   }

   for (var i = 0; i < this.callbacks.ready.length; i++) {
      this.callbacks.ready[i](this);
   }
};

vpx.win.Frame.prototype.update = function () {
   if ((this.resizePolicy & vpx.win.RESIZE_POLICY_SHRINKWRAP) && this._ready) {
      this.shrinkWrap();
   }

   for (var i = 0; i < this.callbacks.update.length; i++) {
      this.callbacks.update[i](this);
   }
};

vpx.win.Frame.prototype.resize = function () {
   for (var i = 0; i < this.callbacks.resize.length; i++) {
      this.callbacks.resize[i](this.query);
   }
};


/**
 * Bind the specified member method as an event handler. Handlers bound in this
 * way will be disposed when this object is destroyed.
 *
 * @param source     any object
 * @param event      the event type to handle--e.g., "load"
 * @param method     execute this instance method 
 *
 * @throws Error
 *    If <code>method</code> is not a member of this object.
 */

vpx.win.Frame.prototype.bindHandler = function(source, event, method) {
   if (! isDefined(this[method])) {
      throw new Error("vpx.xua.win.Frame#bindHandler: Cannot bind event to " + 
	 "undefiend method (" + method + ").");
   }
   
   this.eventHandlers.push(
      vpx.xua.event.listen(source, event, this[method].bindListener(this))
   );
};


vpx.win.Frame.prototype.destroy = function () {
   // PR 76786: When the dialog in question is closed using the OK (but not the
   // X cancel) button, IE erroneously triggers a load event. 
   //
   // PR 95192: FF generates a context resize event when the dialog is
   // destroyed.
   //
   // Having pushed a description of the handler into an array with
   // bindHandler(), we now ignore the event.
   for (var i = 0; i < this.eventHandlers.length; i++) {
      vpx.xua.event.ignore(this.eventHandlers[i]);
   }

   vpx.xua.destroy(this.film, this.ifilm, this.element);

   for (var p in this) {
      delete this[p];
   }
};

vpx.win.Frame.prototype.okay = function () {
   for (var i = 0; i < this.callbacks.okay.length; i++) {
      this.callbacks.okay[i](this.query);
   }
   
   this.destroy();
};

vpx.win.Frame.prototype.cancel = function () {
   for (var i = 0; i < this.callbacks.cancel.length; i++) {
      this.callbacks.cancel[i]();
   }

   this.destroy();
};

vpx.win.Frame.prototype.initRoot = function () {
   var view = this.getRootPane();
   var page = vpx.xua.viewport.getPage(view);

   switch (this.root.sourceType) {
   case this.SOURCE_TYPE_URL:
      // PR 76786: When the dialog in question is closed using the OK (but not
      // the X cancel) button, IE erroneously triggers a load event and
      // executes the following event handler. We push a description of the
      // handler into an array so that the event can be ignored before the
      // Frame object is destroyed.
      this.bindHandler(vpx.xua.viewport.getElement(view), "load", "initRootCb");
      
      try {
         page.location.replace(this.root.source);
      } catch (x) {
         var id = setTimeout(this.initRoot.bind(this), 1);
      }
      break;
   case this.SOURCE_TYPE_STRING_DOCUMENT:
      page.open();
      page.writeln(this.root.source);
      page.close();

      this.initRootCb();
      break;
   case this.SOURCE_TYPE_DOM_DOCUMENT:
      vpx.xua.cut(page.documentElement);
      vpx.xua.append(page, this.root.source);

      this.initRootCb();
      break;
   default:
      throw "xua.win: Cannot set root source. Source type not supported.";
      break;
   }
};

vpx.win.Frame.prototype.initRootCb = function (e) {
   var view = this.getRootPane();
   if (! isDefined(view)) {
      throw "xua.win: rootPane does not exist.";
   }

   var page = vpx.xua.viewport.getPage(view);

   if (isDefined(view.initPane)) {
      view.initPane(this);
   } else {
      throw "xua.win: rootPane does not implement initPane.";
   }
};

vpx.win.Frame.prototype.getRootPane = function () {
   return vpx.xua.viewport.getView(this.element);
};

// XXX: Must adjust for boundaries.
vpx.win.Frame.prototype._shrinkWrap = function () {
   var contentPane = this.getContentPane ? this.getContentPane() : this.getRootPane();

   if (! areStrictlyEqual(contentPane, this.getRootPane())) {
      var contentPage = vpx.xua.viewport.getPage(contentPane);

      // XXX: Assumes that top and bottom offsets are equal.
      var h = vpx.xua.getMinHeight(contentPage.body);
      h += vpx.xua.viewport.getContentOffset(contentPane).y * 2;

      vpx.xua.viewport.setHeight(contentPane, h);
   }

   vpx.xua.setDimension(this.element, this.getContentDimension());
};

vpx.win.Frame.prototype.shrinkWrap = function () {
   // XXX: Prime element. If this step is skipped, the window may not
   // shrink correctly, or it may position itself unexpectedly.
   this._shrinkWrap();
   // Execute _shrinkWrap asynchronously so that the calling function exits and
   // both the root and content pages render before _shrinkWrap executes.
   setTimeout(this._shrinkWrap.bind(this), 0);
};


vpx.win.Frame.prototype.reposition = function () {
   var body = vpx.xua.viewport.getPage(this.context).body;
   var overflow = vpx.xua.getStyle(body, "overflow");
   vpx.xua.setStyle(body, "overflow", "hidden");

   var pos = vpx.xua.getPosition(this.element);
   var min = vpx.xua.viewport.getOffset(this.context,
      vpx.xua.viewport.getView(this.context));
   var max = min.add(vpx.xua.viewport.getDimension(this.context));

   vpx.xua.setPosition(this.element,
      vpx.xua.constrainPosition(
	 vpx.xua.getPosition(this.element),
	 vpx.xua.getDimension(this.element),
	 new Limit(min, max)
      )
   );

   vpx.xua.setStyle(body, "overflow", overflow);
};

vpx.win.Frame.prototype.resizeFilm = function () {
   var body = vpx.xua.viewport.getPage(this.context).body;
   if (vpx.xua.ie || this.context != tle) {
      var overflow = vpx.xua.getStyle(body, "overflow");
      vpx.xua.setStyle(body, "overflow", "hidden");
   }

   vpx.xua.setDimension(this.film,
      vpx.xua.viewport.getDimension(this.context)
   );

   if (isDefined(this.ifilm)) {
      vpx.xua.setDimension(this.ifilm,
	 vpx.xua.viewport.getDimension(this.context)
      );
   }

   if (vpx.xua.ie || this.context != tle) {
      vpx.xua.setStyle(body, "overflow", overflow);
   }
};

vpx.win.Frame.prototype.maximize = function () {
   var body = vpx.xua.viewport.getPage(this.context).body;
   var overflow = vpx.xua.getStyle(body, "overflow");
   vpx.xua.setStyle(body, "overflow", "hidden");

   var absLimit = new Limit(
      new Dimension(0,0),
      vpx.xua.viewport.getDimension(this.context)
   );


   var boundary = this.getBoundary();
   boundary.min = boundary.min.toDimension();
   boundary.max = boundary.max.toDimension();

   vpx.xua.setDimension(this.element,
      vpx.xua.constrainDimension(boundary.max, absLimit)
   );

   vpx.xua.setStyle(body, "overflow", overflow);

   this.resize();
};

vpx.win.Frame.prototype.open = function () {
   var min = vpx.xua.viewport.getOffset(this.context,
      vpx.xua.viewport.getView(this.context));
   var max = min.add(vpx.xua.viewport.getDimension(this.context).toPosition());
   var boundary = new Limit(min, max);
   
   if (this.placementPolicy & vpx.win.PLACEMENT_POLICY_CENTER) {
      vpx.xua.center(this.element, boundary);
   } else {
      vpx.xua.setPosition(this.element,
	 vpx.xua.constrainPosition(
	    vpx.xua.getPosition(this.element),
	    vpx.xua.getDimension(this.element),
	    boundary
	 )
      );
   }
   
   this.bindHandler(this.context, "resize", "reposition");
   
   if (this.modal) {
      var container = vpx.xua.viewport.getPage(this.context).body;
      if (vpx.xua.ie) {
         this.ifilm = vpx.xua.create.call(this.context, "iframe");

         vpx.xua.setAttribute(this.ifilm,
                              "frameborder", "no",
                              "scrolling",   "no",
                              "src",         vpx.xua.BLANK_FRAME_SRC);

         // XXX: Presentation should be defined in a style sheet.
         vpx.xua.setStyle(this.ifilm, "position", "absolute");
         vpx.xua.setOpacity(this.ifilm, 0);

         this.ifilm = vpx.xua.append.call(this.context, container, this.ifilm);
      }

      this.film = vpx.xua.append.call(this.context, container,
	 vpx.xua.create.call(this.context, "div"));

      // XXX: Presentation should be defined in a style sheet.
      vpx.xua.setStyle(this.film,
                       "border",          "solid 1px #69c",
                       "backgroundColor", "#369",
                       "position",        "absolute");

      vpx.xua.setOpacity(this.film, 0.6);

      this.resizeFilm();

      var z = vpx.xua.getZ(this.element);
      z = vpx.xua.setZ(this.element, (isDefined(z) && z != "auto") ? z + 2 : 3);

      if (isDefined(this.ifilm)) {
         vpx.xua.setPosition(this.ifilm, 0, 0, z - 2);
      }

      vpx.xua.setPosition(this.film, 0, 0, z - 1);

      this.bindHandler(this.context, "resize", "resizeFilm");
   }

   if (isDefined(this.setFocus)) {
      this.setFocus();
   }
};


vpx.win.Frame.ROOT_SOURCE_TYPE_MASK =
   vpx.win.SOURCE_TYPE_URL +
   vpx.win.SOURCE_TYPE_STRING_DOCUMENT +
   vpx.win.SOURCE_TYPE_DOM_DOCUMENT;
