/* Copyright 2005 VMware, Inc.	All rights reserved. -- VMware Confidential */

/**
 * Provides methods for dynamic web application development. These methods are
 * spefically intended for use with:
 *
 * <ul>
 *    <li>Mozilla Firefox 1.0/Mozilla 1.7 and higher</li>
 *    <li>Microsoft Internet Explorer 6.0 and higher for Microsoft Windows</li>
 * </ul>
 * <p>
 * No attempt has been made to exclude other browsers, but neither has any
 * attempt been made to include them. Your mileage may vary.
 * <p>
 * Most methods can be reparented via the <code>Function.call</code> method.
 * That is, most methods can be used to inspect the properties of other HTML
 * pages, whether they include vpx.xua.js or not. For example:
 *
 * <blockquote><pre>var myFrame = vpx.xua.viewport.getView($("myFrame"));
 * alert(vpx.xua.getPositions.call(appFrame, "myElement"));</pre></blockquote>
 * <p>
 * The above code will open an alert dialog to present the coordinates of the
 * HTML element with the ID "myElement" in the IFRAME with the ID "myFrame".
 * There is no need for "myFrame" itself to contain any JavaScript code.
 */


vpx.xua = {
   "ELEMENT_NODE":                 1,
   "ATTRIBUTE_NODE":               2,
   "TEXT_NODE":                    3,
   "CDATA_SECTION_NODE":           4,
   "ENTITY_REFERENCE_NODE":        5,
   "ENTITY_NODE":                  6,
   "PROCESSING_INSTRUCTION_NODE":  7,
   "COMMENT_NODE":                 8,
   "DOCUMENT_NODE":                9,
   "DOCUMENT_TYPE_NODE":          10,
   "DOCUMENT_FRAGMENT_NODE":      11,
   "NOTATION_NODE":               12,
   "BLANK_FRAME_SRC":             "javascript:;"
};


vpx.xua.ie = false;

// JScript 3.0 and higher supports conditonal compilation.
// http://msdn.microsoft.com/library/default.asp?url=/library/en-us/script56/html/js56jsoriversioninformation.asp

/*@cc_on
vpx.xua.ie = {};
vpx.xua.BLANK_FRAME_SRC = "javascript:'';";
@if (@_jscript_version >= 5.6)
vpx.xua.ie.version = 6;
@elif (@_jscript_version == 5.5)
vpx.xua.ie.version = 5.5;
@elif (@_jscript_version == 5.1)
vpx.xua.ie.version = 5.01;
@elif (@_jscript_version == 5.0)
vpx.xua.ie.version = 5;
@elif (@_jscript_version == 3.0)
vpx.xua.ie.version = 4;
@end
@*/


/**
 * Given one or more functions, try to execute each in order until one
 * succeeds, ignoring any failures. Doing this is useful if you need to select
 * a browser-specific implementation but do not have an acceptable way to
 * determine which implementation is supported.
 *
 * @param arguments	the functions to try
 * @return		the result of the first successful function; if no
 *			function is executed successfully, then undefined.
 */

vpx.xua.cull = function () {
   for (var i = 0; i < arguments.length; i++) {
      try {
	 return arguments[i]();
      } catch (x) {
	 continue;
      }
   }

   return null;
};


/**
 * Given one or more arguments, try to convert each into a pliable HTML element
 * object.
 *
 * @param arguments	string IDs for HTML elements, or object instances of
 *			the same
 * @return		an HTML element object for a single argument, or an
 *			array of objects for multiple arguments. Note that
 *			arguments in object form will be returned without
 *			modification. No attempt is made to guarantee that a
 *			string identifies a valid element in the calling
 *			context, nor to ensure that object arguments are of the
 *			expected type.
 */

vpx.xua.getElement = function () {
   var view = (isView(this) ? this : self);
   var list = [];

   for (var i = 0; i < arguments.length; i++) {
      var o = arguments[i];

      if (isString(o)) {
	 o = view.document.getElementById(o);
      }

      if (arguments.length == 1) {
	 return o;
      }


      list.push(o);
  }

  return list;
};

// Use $(...) instead of vpx.xua.getElement.call(self, ...).
var $ = vpx.xua.getElement.bind(self);


/**
 * Given an HTML element or a string identifying one, and a node type, return
 * true if the node is of the specified type.
 *
 * @param o     the HTML element
 * @param t     a terse node type (e.g., text instead of TEXT_NODE)
 * @return      true if the element's nodeType matches the supplied type
 */

vpx.xua.isNodeOfType = function (o, t) {
   o = vpx.xua.getElement.call(this, o);

   return o.nodeType == vpx.xua[t.toUpperCase() + "_NODE"];
};


/**
 * Given a DOM node, return a string representation of the node's content
 * (i.e. what innerHTML property would be if the node were an HTMLElement).
 *
 * @param node  The Node object
 * @return      String containing node's inner content
 */

vpx.xua.getInnerContent = function (node) {
   var str = "";
   for (var i = 0; i < node.childNodes.length; i++) {
      var child = node.childNodes[i];
      if (vpx.xua.isNodeOfType(child, "ELEMENT")) {
         str += "<" + child.nodeName;
         for (var j = 0; j < child.attributes.length; j++) {
            var attr = child.attributes[j];
            str += " " + attr.nodeName + "=\"" + attr.nodeValue + "\"";
         }
         if (child.childNodes.length > 0) {
            str += ">";
            str += vpx.xua.getInnerContent(child);
            str += "</" + child.nodeName + ">";
         } else {
            str += "/>";
         }
      } else if (vpx.xua.isNodeOfType(child, "TEXT")) {
         str += child.nodeValue;
      } else if (vpx.xua.isNodeOfType(child, "CDATA_SECTION")) {
         str += child.nodeValue;
      }
   }
   return str;
};


/**
 * Given an HTML element or a string identifying one, get, possibly after
 * setting, one or more CSS properties for the element in question.
 *
 * @param o		the HTML element
 * @param arguments	one or more key-value pairs corresponding to JavaScript
 *			CSS properties (e.g., bacgroundColor instead of
 *			background-color). For each pair, if value is defined,
 *			it will be set.
 * @return		the current value for a single CSS property-value pair,
 *			or a hash of property-value pairs for more than one.
 */

vpx.xua.setStyle = vpx.xua.style = function (o) {
   o = vpx.xua.getElement.call(this, o);

   var result = {};

   for (var i = 1; i < arguments.length; i += 2) {
      var p = arguments[i];
      var v = (i + 1 < arguments.length ? arguments[i + 1] : null);

      if (isDefined(v)) {
         o.style[p] = vpx.xua.css.getValue(p, v);
      }

      if (o.currentStyle) {
         v = o.currentStyle[p];
      } else if (window.getComputedStyle) {
         var style = window.getComputedStyle(o, "");
         if (style != null) {
            var cssP = p.replace(/([A-Z])/g, function(m, m1) {
               return "-" + m1.toLowerCase();
            });
            v = style.getPropertyValue(cssP);
         } else {
            v = "";
         }
      } else {
         v = o.style[p];
      }

      if (arguments.length <= 3) {
         return v;
      }

      result[p] = v;
   }

   return result;
};

vpx.xua.getStyle = function (o) {
   var properties = [o]; // The first argument is not really a property.

   for (var i = properties.length; i < arguments.length; i++) {
      properties.push(arguments[i], null);
   }

   return vpx.xua.style.apply(this, properties);
};


/**
 * Given an HTML element or a string identifying one, add a CSS class name for the
 * element in question.
 *
 * @param o    the HTML element
 * @param classStr the class name to be added to the HTML element's className
 */

vpx.xua.addClass = function (o, classStr) {
   o = vpx.xua.getElement.call(this, o);
   if ( isDefined(o) ) {
      o.className = o.className + ' ' + classStr;
   }
};

/**
 * Given an HTML element or a string identifying one, remove a CSS class name from the
 * element in question.
 *
 * @param o    the HTML element
 * @param classStr the class name to be added to the HTML element's className
 */

vpx.xua.removeClass = function (o, classStr) {
   o = vpx.xua.getElement.call(this, o);
   if ( isDefined(o) && isDefined(o.className) ) {
      var newClassNames = new Array();
      var allClassNames = o.className.split(' ');

      for (i=0; i<allClassNames.length; i++) {
         if (allClassNames[i]!=classStr) {
            newClassNames.push(allClassNames[i]);
         }
      }
      o.className = newClassNames.join(" ");
   }
};


/**
 * Given an HTML element or a string identifying one, look up a whether the given elemnt
 * is member of that class.
 * element in question.
 *
 * @param o    the HTML element
 * @param classStr the class name to be added to the HTML element's className
 * @return boolean
 *    true if element is member of the class
 */

vpx.xua.hasClass = function (o, classStr) {
   o = vpx.xua.getElement.call(this, o);
   if ( isDefined(o) && isDefined(o.className) ) {
      var allClassNames = o.className.split(' ');

      for (i=0; i<allClassNames.length; i++) {
         if (allClassNames[i]==classStr) {
            return true;
         }
      }
   }
   return false;
};


/**
 * Given an HTML element or a string identifying one, get, possibly after
 * setting, the element's position on the x axis relative to the top-left
 * corner of the element's viewport.
 * <p>
 * Note that using this method to set the element's position will have no
 * effect if the element's CSS position is static.
 *
 * @param o	the HTML element
 * @param x	the desired position of the element along the x axis, if any
 * @return	the actual position of the element on the x axis relative to
 *		the top-left corner of the element's viewport.
 */

vpx.xua.getX = vpx.xua.setX = vpx.xua.x = function (o, x) {
   o = vpx.xua.getElement.call(this, o);

   if (isDefined(x)) {
      vpx.xua.style(o, "left", x + "px");
   }

   var ol = o.offsetLeft;
/*jsl:ignore*/
   while (o = o.offsetParent) {
/*jsl:end*/
      ol += o.offsetLeft;
   }

   return ol;
};


/**
 * Given an HTML element or a string identifying one, get, possibly after
 * setting, the element's position on the y axis relative to the top-left
 * corner of the element's viewport.
 * <p>
 * Note that using this method to set the element's position will have no
 * effect if the element's CSS position is static.
 *
 * @param o	the HTML element
 * @param y	the desired position of the element along the y axis, if any
 * @return	the actual position of the element on the y axis relative to
 *		the top-left corner of the element's viewport.
 */

vpx.xua.getY = vpx.xua.setY = vpx.xua.y = function (o, y) {
   o = vpx.xua.getElement.call(this, o);

   if (isDefined(y)) {
      vpx.xua.style(o, "top", y + "px");
   }

   var ot = o.offsetTop;
/*jsl:ignore*/
   while (o = o.offsetParent) {
/*jsl:end*/
      ot += o.offsetTop;
   }

   return ot;
};


/**
 * Given an HTML element or a string identifying one, get, possibly after
 * setting, the element's position on the z axis relative to the element's
 * viewport.
 * <p>
 * Note that using this method to set the element's position will have no
 * effect if the element's CSS position is static.
 *
 * @param o	the HTML element
 * @param z	the desired position of the element along the z axis, if any
 * @return	the actual position of the element on the z axis relative to
 *		the element's viewport.
 */

vpx.xua.getZ = vpx.xua.setZ = vpx.xua.z = function (o, z) {
   o = vpx.xua.getElement.call(this, o);

   if (isDefined(z)) {
      vpx.xua.style(o, "zIndex", z);
   }

   return vpx.xua.style(o, "zIndex");
};


/**
 * Given an HTML element or a string identifying one, get, possibly after
 * setting, the element's position relative to its viewport.
 * <p>
 * Note that using this method to set the element's position will have no
 * effect if the element's CSS position is static.
 *
 * @param o		the HTML element
 * @param arguments	either a single Position object, or x, y and z coordinates
 *			specifying the desired position, if any
 * @return		the actual position of the element relative to its
 *			viewport as a Position object.
 */

vpx.xua.getPosition = vpx.xua.setPosition = vpx.xua.pos = function (o) {
   o = vpx.xua.getElement.call(this, o);

   var x, y, z;

   if (arguments.length > 2) {
      x = arguments[1];
      y = arguments[2];
      z = arguments[3];
   } else if (arguments.length == 2) {
      x = arguments[1].x;
      y = arguments[1].y;
      z = arguments[1].z;
   }

   if (isView(o)) {
      if (areStrictlyEqual(o, o.top)) {
	 return vpx.xua.viewport.pos(o, x, y);
      }
   }

   return new Position(vpx.xua.x(o, x), vpx.xua.y(o, y), vpx.xua.z(o, z));
};


/**
 * Position the supplied object at the center of the specified area.
 *
 * @param o        an HTML element
 * @param area     a Limit object representing the top-left and bottom-right
 *                 coordinates of the boundary
 */

vpx.xua.center = function (o, area) {
   var dim = vpx.xua.dim(o);

   return vpx.xua.pos(o,
      area.min.x + (area.max.x/2) - (dim.w/2),
      area.min.y + (area.max.y/2) - (dim.h/2)
   );
};


/**
 * Given an object's current position and dimension, return a new position that
 * places the object within the area specified.
 *
 * @param pos      the current position
 * @param dim      the current dimensions
 * @param area     a Limit object representing the top-left and bottom-right
 *                 coordinates of the boundary
 */

vpx.xua.constrainPosition = function (pos, dim, area) {
   var x = pos.x;
   var y = pos.y;

   if (pos.x < area.min.x) {
      x = area.min.x;
   }

   if (pos.x + dim.w > area.max.x) {
      x = area.max.x - dim.w;
   }

   if (pos.y < area.min.y) {
      y = area.min.y;
   }

   if (pos.y + dim.h > area.max.y) {
      y = area.max.y - dim.h;
   }

   return new Position(x, y);
};


/**
 * Given any dimension, return a new dimension that is no larger than the area
 * specified.
 *
 * @param dim      the current dimensions
 * @param area     a Limit object representing the top-left and bottom-right
 *                 coordinates of the boundary
 */

vpx.xua.constrainDimension = function (dim, area) {
   var w = dim.w;
   var h = dim.h;

   if (dim.w < area.min.w) {
      w = area.min.w;
   }

   if (dim.w > area.max.w) {
      w = area.max.w;
   }

   if (dim.h < area.min.h) {
      h = area.min.h;
   }

   if (dim.h + dim.h > area.max.h) {
      h = area.max.h;
   }

   return new Dimension(w, h);
};


/**
 * Given an HTML element or a string identifying one, get, possibly after
 * setting, the element's width.
 *
 * @param o	the HTML element
 * @param w	the desired width of the element, if any
 * @return	the actual width of the element
 */

vpx.xua.getWidth = vpx.xua.setWidth = vpx.xua.w = function (o, w) {
   o = vpx.xua.getElement.call(this, o);

   if (isDefined(w)) {
      vpx.xua.style(o, "width", w + "px");

      if (o.offsetWidth > w) {
	 var diff = o.offsetWidth - w;
	 vpx.xua.style(o, "width", (w - diff) + "px");
      }
   }

   return o.offsetWidth;
};


/**
 * Given an HTML element determine its minimum possible width before scrollbars
 * will appear.
 *
 * @param o     an HTML element (usually a document body)
 */

vpx.xua.getMinWidth = function (o) {
   o = vpx.xua.getElement.call(this, o);

   var min  = o.scrollWidth + (o.offsetWidth - o.clientWidth);

   if (min > o.offsetWidth) {
      min = o.offsetWidth;
   }

   return min;
};


/**
 * Given an HTML element or a string identifying one, get, possibly after
 * setting, the element's height.
 *
 * @param o	the HTML element
 * @param h	the desired height of the element, if any
 * @return	the actual height of the element
 */

vpx.xua.getHeight = vpx.xua.setHeight = vpx.xua.h = function (o, h) {
   o = vpx.xua.getElement.call(this, o);

   if (isDefined(h)) {
      vpx.xua.style(o, "height", h + "px");

      if (o.offsetHeight > h) {
	 var diff = o.offsetHeight - h;
	 vpx.xua.style(o, "height", (h - diff) + "px");
      }
   }

   return o.offsetHeight;
};


/**
 * Given an HTML element determine its minimum possible height before
 * scrollbars will appear.
 *
 * @param o     an HTML element (usually a document body)
 */

vpx.xua.getMinHeight = function (o) {
   o = vpx.xua.getElement.call(this, o);

   var min  = o.scrollHeight + (o.offsetHeight - o.clientHeight);

   if (min > o.offsetHeight) {
      min = o.offsetHeight;
   }

   return min;
};


/**
 * Given an HTML element or a string identifying one, get, possibly after
 * setting, the element's dimensions.
 *
 * @param o		the HTML element
 * @param arguments	either a single Dimension object, or width and height
 *			specifying the desired dimensions, if any
 * @return		the actual dimensions of the element as a Dimension object
 */

vpx.xua.getDimension = vpx.xua.setDimension = vpx.xua.dim = function (o) {
   o = vpx.xua.getElement.call(this, o);

   if (isView(o)) {
      return vpx.xua.viewport.dim(o);
   }

   var w, h;

   if (arguments.length == 3) {
      w = arguments[1];
      h = arguments[2];
   } else if (arguments.length == 2) {
      w = arguments[1].w;
      h = arguments[1].h;
   }

   return new Dimension(vpx.xua.w(o, w), vpx.xua.h(o, h));
};


/**
 * Given an HTML element, return a Limit object representing the top-left and
 * bottom-right coordinates of the element.
 *
 * @param o     an HTML element
 */

vpx.xua.getBoundary = function (o) {
   o = vpx.xua.getElement.call(this, o);

   if (isView(o)) {
      return vpx.xua.viewport.getBoundary(o);
   }

   var min = vpx.xua.getPosition(o);
   var max = vpx.xua.getDimension(o).toPosition();

   return new Limit(min, max);
};


/**
 * Given an HTML element or a string identifying one, get, possibly after
 * setting, the element's opacity.
 *
 * @param o	the HTML element
 * @param v	the desired opacity of the element between 0 and 1
 * @return	the current opacity of the element
 */

vpx.xua.getOpacity = vpx.xua.setOpacity = vpx.xua.opacity = function (o, v) {
   o = vpx.xua.getElement.call(this, o);

  if (isDefined(v)) {
    // IE/Win
    vpx.xua.style(o, "filter", "alpha(opacity:"+ (v * 100) +")");

    // Safari<1.2, Konqueror
    vpx.xua.style(o, "KHTMLOpacity", v);

    // Older Mozilla and Firefox
    vpx.xua.style(o, "MozOpacity", v);

    // Safari 1.2, newer Firefox and Mozilla, CSS3
    vpx.xua.style(o, "opacity", v);
  }

  return vpx.xua.style(o, "opacity");
};


/**
 * Given an HTML element or a string identifying one, get, possibly after
 * setting, one or more attributes for the element in question.
 *
 * @param o		the HTML element
 * @param arguments	one or more key-value pairs corresponding to HTML
 *			attributes. For each pair, if value is defined,
 *			it will be set.
 * @return		the current value for a single attribute-value pair,
 *			or a hash of attribute-value pairs for more than one.
 */

vpx.xua.setAttribute = vpx.xua.att = function (o) {
  o = vpx.xua.getElement.call(this, o);

  var result = {};

  for (var i = 1; i < arguments.length; i += 2) {
     var a = arguments[i];
     var v = (i + 1 < arguments.length ? arguments[i + 1] : null);

     if (isDefined(v) && a.toLowerCase() == 'src' &&
	(isDefined(o.nodeName) && o.nodeName.toLowerCase() == 'iframe')) {
	o.setAttribute(a, v);
     }

     // Try to access the attribute directly using an object property. This is the
     // only reliable method when:
     //
     //      Retrieving the "scrolling" attribute for IFRAMEs added dynamically
     //      with 'document.body.appendChild(document.createElement("iframe"));'.
     //
     //      Modifying the name attribute of elements in MSIE (5.5?).
     //
     try {
	if (isDefined(v)) {
	   o[a] = v;
	}

	v = o[a];
     } catch (x) {}

     // Try to access the named attribute manually. This is the only reliable
     // method when:
     //
     //      Setting the frameborder attribute of an IFRAME element in MSIE.
     //
     try {
	if (! isDefined(o.attributes)) {
	   throw null;
	}

	// Retrieve the DOM node for the named attribute.
	var n;
	if (! isDefined(o.attributes.getNamedItem)) {
	   for (var j = 0; o.attributes != null && j < o.attributes.length; j++) {
	      if (o.attributes[j].nodeName == a) {
		 n = o.attributes[j];
		 break;
	      }
	   }
	} else {
	   n = o.attributes.getNamedItem(a);
	}

	if (! isDefined(n)) {
	   throw null;
	}

	// Access the attribute's node value directly.
	if (isDefined(v)) {
	   n.nodeValue = v;
	}

	v = n.nodeValue;
     } catch (x) {
	// Use DOM Element's getAttribute and setAttribute methods. Although
	// these should be the preferred methods, they do not work uniformally
	// and they are not necessary if the first two strategies succeed.
	if (isDefined(o.getAttribute) && isDefined(o.setAttribute)) {
	   if (isDefined(v)) {
	      o.setAttribute(a, v);
	   }

	   v = o.getAttribute(a);
	}
     }

     if (arguments.length <= 3) {
	return v;
     }

     result[a] = v;
  }

  return result;
};

vpx.xua.getAttribute = function (o) {
   var attributes = [o]; // The first argument is not really a property.

   for (var i = attributes.length; i < arguments.length; i++) {
      attributes.push(arguments[i], null);
   }

   return vpx.xua.att.apply(this, attributes);
};


/**
 * Given an HTML element or a string identifying one, get its parent document.
 *
 * @param o     the HTML element
 * @return	the document that contains the element
 */

vpx.xua.getPage = function (o) {
   o = vpx.xua.getElement.call(this, o);

   if (isView(o) && ! areStrictlyEqual(o, o.top)) {
      o = vpx.xua.viewport.getElement(o);
   }

   return (isView(o) ? o.document : o.ownerDocument);
};


/**
 * Given an HTML element or a string identifying one, get the window that
 * contains its parent document.
 *
 * @param o	the HTML element
 * @return	the window that contains the element
 */

vpx.xua.getView = function (o) {
   o = vpx.xua.getElement.call(this, o);

   if (isView(o)) {
      if (areStrictlyEqual(o, o.top)) {
	 return o;
      } else {
	 o = vpx.xua.viewport.getElement(o);
      }
   }

   var p = ["defaultView", "parentWindow"];
   var i;

   for (i = 0; i < p.length; i++) {
      if (isDefined(o[p[i]])) {
	 return o[p[i]];
      }
   }

   if (o.ownerDocument) {
      o = o.ownerDocument;
   }

   for (i = 0; i < p.length; i++) {
      if (isDefined(o[p[i]])) {
	 return o[p[i]];
      }
   }

   return null;
};


/**
 * Encodes a url such that special characters can be safely passed as query
 * string parameters.
 *
 * @param url The url string
 * @return Encoded version of url
 */

vpx.xua.urlEncode = function (url) {
   var view = (isView(this) ? this : self);

	if (isNull(url)) {
		url = view.location.pathname + view.location.search;
	}

	if (view.encodeURIComponent) {
		return view.encodeURIComponent(url);
	} else if (view.escape) {
		return view.escape(url);
	} else {
		return url;
	}
};


/**
 * Decodes an encoded url such that escape sequences for special characters
 * will be coverted back to their original form.
 *
 * @param url The encoded url string
 * @return The decoded url
 */

vpx.xua.urlDecode = function (url) {
   var view = (isView(this) ? this : self);

	if (!url) {
		url = view.location.pathname + view.location.search;
	}

	if (view.decodeURIComponent) {
		return view.decodeURIComponent(url);
	} else if (view.unescape) {
		return view.unescape(url);
	} else {
		return url;
	}
};


vpx.xua.viewport = {};


/**
 * Given a window object or a string identifying one, get its element from its
 * parent window.
 *
 * @param o	the viewport
 * @return	the HTML element that defines the viewport (e.g., an IFRAME
 *              element)
 */

vpx.xua.viewport.getElement = function (o) {

   o = vpx.xua.getElement.call(this, o);

   if (! isView(o)) {
      return o;
   }

   if (areStrictlyEqual(o, o.top)) {
      return o;
   }

   var cpf = o.parent.frames;
   for (var i = 0; i < cpf.length; i++) {
      if (cpf[i] == o) {
         if (isDefined(cpf[i].frameElement)) {
		    return cpf[i].frameElement;
		 } else {
		    return cpf[i];
		 }
      }
   }
   return null;
};


/**
 * Given a window object or a string identifying one, get, possibly after
 * setting, its position.
 *
 * @param o		the viewport
 * @param arguments	either a single Position object, or x, y and z coordinates
 *			specifying the desired position, if any
 * @return		the actual position of the viewport relative to its
 *			parent window; if the viewport specified is top, then
 *			the browser window will be repositioned.
 */

vpx.xua.viewport.getPosition = vpx.xua.viewport.setPosition =
vpx.xua.viewport.pos = function (o) {
   o = vpx.xua.getElement.call(this, o);

   var x, y;

   if (arguments.length == 3) {
      x = arguments[1];
      y = arguments[2];
   } else if (arguments.length == 1) {
      x = arguments[1].x;
      y = arguments[1].y;
   }

   if (! isView(o)) {
      return vpx.xua.pos(o, x, y);
   }

   if (! areStrictlyEqual(o, o.top)) {
      return vpx.xua.pos(vpx.xua.viewport.getElement(o), x, y);
   }

   if (isDefined(x)) {
      if (isDefined(o.moveTo)) {
	 o.moveTo(x, y);
      }

      if (isDefined(o.screenX)) {
	 o.screenX = x;
	 o.screenY = y;
      }
   }

   if (isDefined(o.screenX)) {
      return new Position(o.screenX, o.screenY);
   }

   if (isDefined(o.screenLeft)) {
      return new Position(o.screenLeft, o.screenTop);
   }
};


/**
 * Given an initial window and a target window, determine the position of the
 * initial window relative to the target window along x and y axes.
 *
 * @param o	     the initial window
 * @param target     the target window
 * @param pos	     the current position. This parameter can be used to
 *		     determine the offset of an element contained by the
 *		     initial window, but it is not required.
 * @return	     the position of the initial window relative to the target
 *		     window as a Position object
 */

vpx.xua.viewport.getOffset = function (o, target, pos) {
   if (! isDefined(pos)) {
      pos = new Position(0,0);
   }

   if (areStrictlyEqual(o, o.parent) ||
      (isDefined(target) && areStrictlyEqual(o, target))) {
      return pos;
   }

   var offset;
   var el = vpx.xua.viewport.getElement(o);
   if (isDefined(el)) {
      offset = vpx.xua.pos(el);
   }

   if (! isDefined(offset)) {
      throw "vpx.xua.viewport.getOffset: Could not determine viewport offset.";
   }

   return vpx.xua.viewport.getOffset(o.parent, target, offset.add(pos));
};


/**
 * Given a window object or a string identifying one, get, possibly after
 * setting, its inner width.
 *
 * @param o	the viewport
 * @param w	the desired width
 * @return	the actual inner width of the viewport; if the specified
 *              viewport is top, then the browser window will be resized.
 */

vpx.xua.viewport.getWidth = vpx.xua.viewport.setWidth =
vpx.xua.viewport.w = function (o, w) {
   o = vpx.xua.getElement.call(this, o);

   if (! isView(o)) {
      return vpx.xua.w(o, w);
   }

   if (! areStrictlyEqual(o, o.top)) {
      return vpx.xua.w(vpx.xua.viewport.getElement(o), w);
   }

   if (isDefined(o.innerWidth)) {
      if (isDefined(w)) {
	 o.innerWidth = w;
      }

      return o.innerWidth;
   }

   if (isDefined(o.document) && isDefined(o.document.documentElement) &&
       isDefined(o.document.documentElement.offsetWidth)) {
      if (isDefined(w)) {
	 // The resizeBy can throw an exception if the user clicks too fast
	 // (PR #39015); catch and do nothing since the window is ultimately
	 // resized correctly anyway.
	 try {
	    var ow = o.document.documentElement.offsetWidth;
	    o.resizeBy(w - ow, 0);
	 } catch (x) {}
      }

      return o.document.documentElement.offsetWidth;
   }
};


/**
 * Given a window object or a string identifying one, get, possibly after
 * setting, its inner height.
 *
 * @param o	the viewport
 * @param h	the desired width
 * @return	the actual inner width of the viewport; if the specified
 *              viewport is top, then the browser window will be resized.
 */

vpx.xua.viewport.getHeight = vpx.xua.viewport.setHeight =
vpx.xua.viewport.h = function (o, h) {
   o = vpx.xua.getElement.call(this, o);

   if (! isView(o)) {
      return vpx.xua.h(o, h);
   }

   if (! areStrictlyEqual(o, o.top)) {
      return vpx.xua.h(vpx.xua.viewport.getElement(o), h);
   }

   if (isDefined(o.innerHeight)) {
      if (isDefined(h)) {
	 o.innerHeight = h;
      }

      return o.innerHeight;
   }

   if (isDefined(o.document) && isDefined(o.document.documentElement) &&
       isDefined(o.document.documentElement.offsetHeight)) {
      if (isDefined(h)) {
	 // The resizeBy can throw an exception if the user clicks too fast
	 // (PR #39015); catch and do nothing since the window is ultimately
	 // resized correctly anyway.
	 try {
	    var oh = o.document.documentElement.offsetHeight;
	    o.resizeBy(0, h - oh);
	 } catch (x) {}
      }

      return o.document.documentElement.offsetHeight;
   }
};


/**
 * Given a window object or a string identifying one, get, possibly after
 * setting, its inner dimensions.
 *
 * @param o		the viewport
 * @param arguments	either a single Dimension object, or width and height
 *			specifying the desired inner dimensions, if any
 * @return		the actual dimensions of the viewport; if the specified
 *                      viewport is top, then the browser window will be
 *                      resized.
 */

vpx.xua.viewport.getDimension = vpx.xua.viewport.setDimension =
vpx.xua.viewport.dim = function (o) {
   o = vpx.xua.getElement.call(this, o);

   var w, h;

   if (arguments.length == 3) {
      w = arguments[1];
      h = arguments[2];
   } else if (arguments.length == 2) {
      w = arguments[1].w;
      h = arguments[1].h;
   }

   if (! isView(o)) {
      return vpx.xua.dim(o, w, h);
   }

   if (! areStrictlyEqual(o, o.top)) {
      return vpx.xua.dim(vpx.xua.viewport.getElement(o), w, h);
   }

   return new Dimension(vpx.xua.viewport.w(o, w), vpx.xua.viewport.h(o, h));
};


/**
 * Given an HTML element, return a Limit object representing the top-left and
 * bottom-right coordinates of the element.
 *
 * @param o     an HTML element
 */

vpx.xua.viewport.getBoundary = function (o) {
   o = vpx.xua.getElement.call(this, o);

   if (! isView(o)) {
      return vpx.xua.getBoundary(o);
   }

   var min = (areStrictlyEqual(o, o.top) ? new Position(0,0) : vpx.xua.viewport.getPosition(o));
   var max = vpx.xua.viewport.getDimension(o).toPosition();

   return new Limit(min, max);
};


/**
 * Given a viewport element or a string identifying one, get the window it
 * defines.
 *
 * @param o	the viewport element
 * @return	the element's window object
 */

vpx.xua.viewport.getView = function (o) {
   o = vpx.xua.getElement.call(this, o);

   if (! isDefined(o)) {
      return null;
   }

   if (isView(o)) {
      return o;
   }

   return (isDefined(o.contentWindow) ? o.contentWindow : vpx.xua.getView(o));
};


/**
 * Given a viewport element or a string identifying one, get the document it
 * contains.
 *
 * @param o	the viewport element
 * @return	the viewport's content document object
 */

vpx.xua.viewport.getPage = function (o) {
   o = vpx.xua.getElement.call(this, o);

   if (! isDefined(o)) {
      return null;
   }

   if (! isView(o)) {
      o = vpx.xua.viewport.getView(o);
   }

   return o.document;
};


/**
 * Given a viewport element or a string identifying one, get the position of
 * the first HTML element relative to the top-left corner of the viewport.
 *
 * @param o	the viewport element
 * @return	a Position object representing the position of the first HTML
 *              element in the viewport's content document.
 */

vpx.xua.viewport.getContentOffset = function (o) {
   o = vpx.xua.getElement.call(this, o);

   o = vpx.xua.viewport.getPage(o);

   var dbc = o.body.childNodes;
   for (var i = 0; i < dbc.length; i++) {
      if (dbc[i].nodeType == 1) {
	 return vpx.xua.pos(dbc[i]);
      }
   }

   return null;
};


/**
 * Given a viewport element or a string identifying one, get its top-left-most
 * visible coordinates. If the page is not scrolled this value will be 0,0.
 *
 * @param o	the viewport element
 * @return	the viewport's top-left-most visible coordinates as a Position
 *              object
 */

vpx.xua.viewport.getScroll = vpx.xua.viewport.setScroll =
vpx.xua.viewport.scroll = function (o) {
   o = vpx.xua.getElement.call(this, o);
   return new Position(vpx.xua.viewport.scrollX(o), vpx.xua.viewport.scrollY(o));
};


/**
 * Given a viewport element or a string identifying one, get its left-most
 * visible x coordinate. If the page is not scrolled horizontally, this value
 * will be 0.
 *
 * @param o	the viewport element
 * @return	the viewport's left-most visible x cooridinate
 */

vpx.xua.viewport.getScrollX = vpx.xua.viewport.setScrollX =
vpx.xua.viewport.scrollX = function (o) {
   o = vpx.xua.getElement.call(this, o);

   if (o.pageXOffset) {
      return o.pageXOffset;
   }

   if (o.document.body && o.document.body.scrollLeft) {
      return o.document.body.scrollLeft;
   }

   return 0;
};


/**
 * Given a viewport element or a string identifying one, get its top-most
 * visible y coordinate. If the page is not scrolled vertically, this value
 * will be 0.
 *
 * @param o	the viewport element
 * @return	the viewport's top-most visible y cooridinate
 */

vpx.xua.viewport.getScrollY = vpx.xua.viewport.setScrollY =
vpx.xua.viewport.scrollY = function (o) {
   o = vpx.xua.getElement.call(this, o);

   if (o.pageYOffset) {
      return o.pageYOffset;
   }

   if (o.document.body && o.document.body.scrollTop) {
      return o.document.body.scrollTop;
   }

   return 0;
};


/**
 * Create one or more elements with the tag name(s) given.
 *
 * @param arguments     one or more desired element tag names
 * @return	        the element created, or an array if elements created
 */

vpx.xua.create = function () {
   var view = (isView(this) ? this : self);
   var result = [];

   for (var i = 0; i < arguments.length; i++) {
      var o = view.document.createElement(arguments[i]);

      if (arguments.length == 1) {
	 return o;
      }

      result.push(o);
   }

   return result;
};


/**
 * Given a container and one or more element objects, or strings identifying
 * them, append the elements to the container's children.
 *
 * @param arguments     a container and one or more element objects
 * @return	        the element appended, or an array of elements appended
 */

vpx.xua.append = function () {
   var container = vpx.xua.getElement.call(this, arguments[0]);

   var result = [];

   for (var i = 1; i < arguments.length; i++) {
      var o = container.appendChild(vpx.xua.getElement.call(this, arguments[i]));

      if (arguments.length == 2) {
	 return o;
      }

      result.push(o);
   }

   return result;
};


/**
 * Given one or more element objects, or strings identifying them, delete each
 * node element after removing it from its parent node.
 *
 * @param arguments     one or more element objects
 * @return	        undefined
 */

vpx.xua.destroy = function () {
   for (var i = 0; i < arguments.length; i++) {
      var o = vpx.xua.getElement.call(this, arguments[i]);

      if (isDefined(o) && isDefined(o.parentNode)) {
	 var name = o.nodeName.toLowerCase();

	 // MSIE leaks like a sieve unless an iframe's inner page is unloaded
	 // before its element is removed.
	 if (name == 'iframe' || name == 'frame') {
	    vpx.xua.viewport.getPage(o).location.replace(vpx.xua.BLANK_FRAME_SRC);
	 }

	 try {
	    delete o.parentNode.removeChild(o);
	 } catch (x) {}
      }
   }
};


/**
 * Given one or more element objects, or strings identifying them, remove each
 * element from its parent node.
 *
 * @param arguments     one or more element objects
 * @return	        the element removed, or an array of elements removed
 */

vpx.xua.cut = function () {
   var result = [];

   for (var i = 0; i < arguments.length; i++) {
      var o = vpx.xua.getElement.call(this, arguments[i]);

      if (isDefined(o) && isDefined(o.parentNode)) {
	 o = o.parentNode.removeChild(o);

	 if (arguments.length == 1) {
	    return o;
	 }

	 result.push(o);
      }
   }

   return result;
};


/**
 * Given one or more element objects, or strings identifying them, clone each
 * node.
 *
 * @param arguments     one or more element objects
 * @return	        the element cloned, or an array of elements cloned
 */

vpx.xua.copy = function () {
   var result = [];

   for (var i = 0; i < arguments.length; i++) {
      var o = vpx.xua.getElement.call(this, arguments[i]).cloneNode(true);

      // Remove unique identification. The name attribute is not necessarily
      // unique so leave it in tact.
      vpx.xua.setAttribute(o, "id", "");

      if (arguments.length == 1) {
	 return o;
      }

      result.push(o);
   }

   return result;
};


vpx.xua.event = {
   "MOUSE_BUTTON_NONE":   0,
   "MOUSE_BUTTON_LEFT":   1,
   "MOUSE_BUTTON_RIGHT":  2,
   "MOUSE_BUTTON_MIDDLE": 3
};


/**
 * Given an HTML elelent (or a string identifying one), an event type and a
 * method, execute the specified method when the object is the target of an
 * event of the specified type.
 *
 * @param o     the HTML element or a vpx.xua.event.Handler
 * @param t     the event type
 * @param m     the method to execute
 * @return	a new vpx.xua.event.Handler object
 */

vpx.xua.event.listen = function (o, t, m) {
   if (1 == arguments.length) {
      if (isTypeOf(arguments[0], vpx.xua.event.Handler)) {
	 t = arguments[0].type;
	 m = arguments[0].method;
	 o = arguments[0].element;
      } else {
	 throw "vpx.xua.event.listen: Too few arguments.";
      }
   } else {
      o = vpx.xua.getElement.call(this, o);
   }

   if (o.addEventListener) {
      o.addEventListener(t, m, false);
   } else if (o.attachEvent){
      o.attachEvent("on" + t, m);
   } else {
      eval("o.on" + t + " = " + m + ";");
   }

   return new vpx.xua.event.Handler(o, t, m);
};


/**
 * Given an HTML elelent (or a string identifying one), an event type and a
 * method, stop executing the specified method when the object is the target of
 * an event of the specified type.
 *
 * @param o     the HTML element or a vpx.xua.event.Handler
 * @param t     the event type
 * @param m     the method to stop executing
 * @return	null
 */

vpx.xua.event.ignore = function (o, t, m) {
   if (1 == arguments.length) {
      if (isTypeOf(arguments[0], vpx.xua.event.Handler)) {
	 t = arguments[0].type;
	 m = arguments[0].method;
	 o = arguments[0].element;
      } else {
	 throw "vpx.xua.event.ignore: Too few arguments.";
      }
   } else {
      o = vpx.xua.getElement.call(this, o);
   }

   if (o.removeEventListener) {
      o.removeEventListener(t, m, false);
   } else if (o.detachEvent) {
      o.detachEvent("on" + t, m);
   } else {
      eval("o.on" + t + " = null;");
   }
};


/**
 * Given an event, return the HTML element that either captured the event or
 * was the first to receieve the event.
 *
 * @param e     the event
 */

vpx.xua.event.getSource = function (e) {
   e = e || window.event;

   if (isDefined(e.currentTarget)) {
      return e.currentTarget;
   }

   if (isDefined(e.target)) {
      return e.target;
   }

   if (isDefined(e.srcElement)) {
      return e.srcElement;
   }
};


/**
 * Given an event, return the mouse button pressed to initiate the event.
 *
 * @param e     the event
 */

vpx.xua.event.getButton = function (e) {
   //?? XXX This method is broken (returning LEFT when should be NONE)
   e = e || window.event;

   switch (e.button) {
   case 0:
      return vpx.xua.event.MOUSE_BUTTON_LEFT;
   case 1:
      return (vpx.xua.ie ? vpx.xua.event.MOUSE_BUTTON_LEFT :
              vpx.xua.event.MOUSE_BUTTON_MIDDLE);
   case 2:
      return vpx.xua.event.MOUSE_BUTTON_RIGHT;
   case 3:
      return vpx.xua.event.MOUSE_BUTTON_MIDDLE;
   case 4:
      return vpx.xua.event.MOUSE_BUTTON_MIDDLE;
   case 65535:
      return vpx.xua.event.MOUSE_BUTTON_NONE;
   default:
      return e.button;
   }
};


/**
 * Given an event, return the key (as an integer) pressed to initiate the event.
 *
 * @param e     the event
 */

vpx.xua.event.getKeyCode = function (e) {
   e = e || window.event;

   return e.keyCode;
};


/**
 * Given an event, return the key (as a character) pressed to initiate the event.
 *
 * @param e     the event
 */

vpx.xua.event.getKeyChar = function (e) {
   e = e || window.event;

   return String.fromCharCode(e.keyCode);
};


/**
 * Given an event, return the coordinates of the mouse cursor relative to the
 * viewport where the event was captured.
 *
 * @param e     the event
 */

vpx.xua.event.getPosition = vpx.xua.event.pos = function (e) {
   e = e || window.event;

   if (isDefined(e.pageX)) {
      return new Position(e.pageX, e.pageY);
   }

   var view = vpx.xua.getView(vpx.xua.event.getSource(e));
   return new Position(e.clientX, e.clientY).add(vpx.xua.viewport.scroll(view));
};


/**
 * Given an event, prevent it from propagating any further.
 *
 * @param e     the event
 */

vpx.xua.event.cancel = function (e) {
   e = e || window.event;

   try {
      e.stopPropagation();
      e.preventDefault();
   } catch (x) {}

   try {
      e.cancelBubble = true;
      e.returnValue = false;
   } catch (x) {}
};


/**
 * Constructor for an event handler data structure.
 *
 * @param e     the target of the event
 * @param t     the event type
 * @param m     the event handler method
 */

vpx.xua.event.Handler = function (e, t, m) {
   this.element = e;
   this.type = t;
   this.method = m;
};


vpx.xua.css = {};


/**
 * Given a list of CSS property names (e.g., background-image), convert each to
 * its JavaScript representation (e.g., backgroundImage).
 *
 * @param arguments     one or more CSS property strings
 * @return              a JavaScript CSS property, or an array of them
 */

vpx.xua.css.propertyToJavaScript = function () {
   var result = [];

   for (var i = 0; i < arguments.length; i++) {
      var list = arguments[i].split("-");

      for (var j = 1; j < list.length; j++) {
         list[j] = list[j].substr(0, 1).toUpperCase() + list[j].substr(1);
      }

      if (arguments.length == 1) {
         return list.join('');
      }

      result.push(list.join(''));
   }

   return result;
};


/**
 * Given a list of CSS definitions (e.g., background-color: #fff), convert each
 * to a JavaScript associative array (e.g., {"backgroundColor": "#fff"}).
 *
 * @param arguments     one or more CSS definitions as strings
 * @return              a JavaScript associative array, or an array of them
 */

vpx.xua.css.definitionToHash = function () {
   var result = [];

   for (var i = 0; i < arguments.length; i++) {
      var list = arguments[i].split(';');
      var hash = {};

      for (var j = 0; j < list.length; j++) {
         if (list[j].trim().length == 0) {
            continue;
         }

         var pair = list[j].split(':');

	 hash[vpx.xua.css.propertyToJavaScript(pair[0].trim())] =
	    pair[1].trim();
      }

      if (arguments.length == 1) {
         return hash;
      }

      result.push(hash);
   }

   return result;
};

/**
 * Gets the css value to use, given a css property and a CSS2.1 specification
 * value.  This method exists because some user agents have created their own
 * extensions that correspond directly to specification properties.
 *
 * @param p String
 *    The CSS2.1 specification property
 * @param v String
 *    The CSS2.1 specification value
 * @return String
 *    The value to use for the current user agent
 */
vpx.xua.css.getValue = function(p, v) {
   // TODO convert hard-coded exceptions to generic lookup of exceptions
   if (!vpx.xua.ie && p == "display" && v == "inline-block") {
      return "-moz-inline-box";
   }
   return v;
};
