# **********************************************************
# Copyright 2005-2006 VMware, Inc.  All rights reserved.
# -- VMware Confidential
# **********************************************************

import httplib
import thread
import urlparse
from xml.parsers.expat import ParserCreate
from xml.sax.saxutils import escape
from VmomiSupport import *

SOAP_START = """<?xml version="1.0" encoding="UTF-8"?>
<soapenv:Envelope xmlns:soapenc="http://schemas.xmlsoap.org/soap/encoding/"
 xmlns:soapenv="http://schemas.xmlsoap.org/soap/envelope/"
 xmlns:xsd="http://www.w3.org/2001/XMLSchema"
 xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance">
<soapenv:Body>\n"""
SOAP_END = "\n</soapenv:Body>\n</soapenv:Envelope>"

## Localized MethodFault type
class LocalizedMethodFault(object):
   _wsdlName = "LocalizedMethodFault"
   pass
wsdlTypeMap["LocalizedMethodFault"] = LocalizedMethodFault

## Get the start tag, end tag, and text handlers of a class
def GetHandlers(obj):
   return (obj.StartElementHandler,
           obj.EndElementHandler,
           obj.CharacterDataHandler)

## Set the start tag, end tag, and text handlers of a parser
def SetHandlers(obj, handlers):
   (obj.StartElementHandler,
    obj.EndElementHandler,
    obj.CharacterDataHandler) = handlers


## Serialize an object
#
# This function assumes CheckField(info, val) was already called
# @param val the value to serialize
# @param info the field
# @param version the version
# @return the serialized object as a string
def Serialize(val, info=None, version=None):
   if val is None:
      return ''
   if version is None:
      try:
         if isinstance(val, list):
            itemType = val.Item
            version = itemType._version
         else:
            version = val._version
      except AttributeError:
         version = BASE_VERSION
   if info is None:
      info = Object(name="object", type=object, version=version, flags=0)
      ns = ' xmlns="urn:%s"' % nsMap[version]
   else:
      ns = ''
   if not IsChildVersion(version, info.version):
      return ''

   attr = ''
   if isinstance(val, DataObject):
      dynType = GetCompatibleType(val.__class__, version)
      if dynType != info.type:
         attr = ' xsi:type="%s"' % GetWsdlName(dynType)
      if info.flags & F_LINK:
         result = str(val.key)
      else:
         result = ''.join([Serialize(getattr(val, prop.name), prop, version)
                           for prop in val._GetPropertyList()])
   elif isinstance(val, ManagedObject):
      if not issubclass(info.type, ManagedObject):
         attr = ' xsi:type="ManagedObjectReference"'
      attr += ' type="%s"' % GetWsdlName(val.__class__)
      result = val._moId
   elif isinstance(val, list):
      if info.type is object:
         itemType = val.Item
         if (itemType is ManagedMethod or itemType is PropertyPath \
         or  itemType is type) and not IsChildVersion(version, VERSION1):
            tag = 'string'
            attr = ' xsi:type="ArrayOfString"'
         elif issubclass(itemType, ManagedObject):
            tag = 'ManagedObjectReference'
            attr = ' xsi:type="ArrayOfManagedObjectReference"'
         else:
            tag = GetWsdlName(itemType)
            attr = ' xsi:type="%s"' % GetWsdlName(val.__class__)
         itemInfo = Object(name=tag, type=itemType, version=info.version,
                           flags=info.flags)
         result = ''.join([Serialize(it, itemInfo, version) for it in val])
      else:
         itemInfo = Object(name=info.name, type=info.type.Item,
                           version=info.version, flags=info.flags)
         return ''.join([Serialize(it, itemInfo, version) for it in val])
   elif isinstance(val, type) or isinstance(val, type(Exception)):
      if info.type is object:
         attr = ' xsi:type="%s"' % (IsChildVersion(version, VERSION1) \
                                and GetWsdlName(type) or 'xsd:string')
      result = GetWsdlName(val)
   elif isinstance(val, ManagedMethod):
      if info.type is object:
         attr = ' xsi:type="%s"' % (IsChildVersion(version, VERSION1) \
                                and GetWsdlName(ManagedMethod) or 'xsd:string')
      result = val.info.wsdlName
   elif isinstance(val, PropertyPath) or info.type is PropertyPath:
      if info.type is object:
         attr = ' xsi:type="%s"' % (IsChildVersion(version, VERSION1) \
                                and GetWsdlName(PropertyPath) or 'xsd:string')
      result = val
   elif isinstance(val, Enum):
      if info.type is object:
         attr = ' xsi:type="%s"' % GetWsdlName(val.__class__)
      result = val
   else:
      if info.type is object:
         attr = ' xsi:type="xsd:%s"' % GetWsdlName(Type(val))
      result = escape(str(val))
   return '<%s%s%s>%s</%s>' % (info.name, ns, attr, result, info.name)

## Deserialize an object from a file or string
#
# This function will deserialize one top-level XML node.
# @param self self
# @param data the data to deserialize (a file object or string)
# @param resultType expected result type
# @param stub stub for moRef deserialization
# @return the deserialized object
def Deserialize(data, resultType=object, stub=None):
   parser = ParserCreate()
   ds = SoapDeserializer(stub)
   ds.Deserialize(parser, resultType)
   if isinstance(data, str):
      parser.Parse(data)
   else:
      parser.ParseFile(data)
   return ds.GetResult()

## SOAP -> Python Deserializer
class SoapDeserializer:
   ## Constructor
   #
   # @param self self
   # @param stub Stub adapter to use for deserializing moRefs
   def __init__(self, stub=None):
      self.stub = stub

   ## Deserialize a SOAP object
   #
   # @param self self
   # @param parser an expat parser
   # @param resultType the static type of the result
   # @param isFault true if the response is a fault response
   # @return the deserialized object
   def Deserialize(self, parser, resultType=object, isFault=False):
      self.isFault = isFault
      self.parser = parser
      self.origHandlers = GetHandlers(parser)
      SetHandlers(parser, GetHandlers(self))
      self.resultType = resultType
      self.stack = []
      self.links = {}
      self.linkables = {}
      self.data = ""
      if issubclass(resultType, list):
         self.result = resultType()
      else:
         self.result = None

   ## Get the result of deserialization, resolving links
   def GetResult(self):
      for (obj, name) in self.links:
         val = getattr(obj, name)
         if isinstance(val, list):
            val = val.__class__([self.linkables[k] for k in val])
         else:
            val = self.linkables[val]
         setattr(obj, name, val)
      self.links = {}
      self.linkables = {}
      return self.result

   ## Handle an opening XML tag
   def StartElementHandler(self, tag, attr):
      self.data = ""
      if not self.stack:
         if self.isFault:
            objType = GetWsdlType(tag[:-5])
         else:
            objType = self.resultType
      elif isinstance(self.stack[-1], list):
         objType = self.stack[-1].Item
      elif isinstance(self.stack[-1], DataObject):
         objType = self.stack[-1]._GetPropertyInfo(tag).type
      elif isinstance(self.stack[-1], LocalizedMethodFault):
         objType = tag == "localizedMessage" and str or DataObject
      else:
         raise TypeError("Invalid type")

      try:
         dynType = GetWsdlType(attr[u'xsi:type'].split(':')[-1])
         # Ignore dynamic type for TypeName, MethodName, PropertyPath
         # @bug 150459
         if objType is not type and objType is not ManagedMethod and \
            objType is not PropertyPath and (not issubclass(dynType, list) or \
                                             not issubclass(objType, list)):
            objType = dynType
      except KeyError:
         if issubclass(objType, Exception) and not self.isFault:
            objType = LocalizedMethodFault
         elif issubclass(objType, list):
            objType = objType.Item

      if self.stub:
         objType = GetCompatibleType(objType, self.stub.version)
      if issubclass(objType, ManagedObject):
         self.stack.append(GetWsdlType(attr[u'type']))
      elif issubclass(objType, DataObject) or issubclass(objType, list) or \
           issubclass(objType, LocalizedMethodFault):
         self.stack.append(objType())
      else:
         self.stack.append(objType)

   ## Handle a closing XML tag
   def EndElementHandler(self, tag):
      try:
         obj = self.stack.pop()
      except IndexError:
         SetHandlers(self.parser, self.origHandlers)
         handler = self.parser.EndElementHandler
         del self.parser, self.origHandlers, self.stack, self.resultType
         if handler:
            return handler(tag)
         return
      data = self.data
      if isinstance(obj, type) or isinstance(obj, type(Exception)):
         if issubclass(obj, ManagedObject):
            obj = obj(data, self.stub)
         elif issubclass(obj, Enum):
            obj = getattr(obj, data)
         elif obj is type:
            obj = GetWsdlType(data)
         elif obj is ManagedMethod:
            obj = wsdlMethodMap[data]
         elif obj is bool:
            data = data.lower()
            if data == "false":
               obj = bool(False)
            elif data == "true":
               obj = bool(True)
            else:
               obj = bool(data)
         elif obj is str:
            try:
               obj = str(data)
            except UnicodeError:
               obj = data
         else:
            obj = obj(data)
      elif isinstance(obj, LocalizedMethodFault):
         obj.fault.message = obj.localizedMessage
         obj = obj.fault

      if self.stack:
         top = self.stack[-1]
         if isinstance(top, list):
            top.append(obj)
         elif isinstance(top, DataObject):
            info =  top._GetPropertyInfo(tag)

            if info.flags & F_LINKABLE:
               self.linkables[obj.key] = obj
            if info.flags & F_LINK:
               self.links[(top, tag)] = True
               obj = self.data

            if issubclass(info.type, list) and not isinstance(obj, list):
               getattr(top, info.name).append(obj)
            elif info.flags & F_LINK:
               SetAttr(top, info.name, obj)
            else:
               top._SetProperty(info, top, obj)
         else:
            setattr(top, tag, obj)
      else:
         if issubclass(self.resultType, list) and not isinstance(obj, list):
            self.result.append(obj)
         else:
            self.result = obj
            SetHandlers(self.parser, self.origHandlers)
            del self.parser, self.origHandlers, self.stack, self.resultType

   ## Handle text data
   def CharacterDataHandler(self, data):
      self.data += data

## SOAP Response Deserializer class
class SoapResponseDeserializer:
   ## Constructor
   #
   # @param self self
   # @param stub Stub adapter to use for deserializing moRefs
   def __init__(self, stub):
      self.stub = stub
      self.deser = SoapDeserializer(stub)

   ## Deserialize a SOAP response
   #
   # @param self self
   # @param response the response (a file object or a string)
   # @param resultType expected result type
   # @return the deserialized object
   def Deserialize(self, response, resultType):
      self.resultType = resultType
      self.stack = []
      self.msg = ""
      self.deser.result = None
      self.isFault = False
      self.parser = ParserCreate()
      try: # buffer_text only in python >= 2.3
         self.parser.buffer_text = True
      except AttributeError:
         pass
      SetHandlers(self.parser, GetHandlers(self))
      if isinstance(response, str):
         self.parser.Parse(response)
      else:
         self.parser.ParseFile(response)
      result = self.deser.GetResult()
      if self.isFault:
         if result is None:
            result = GetWsdlType("RuntimeFault")()
         result.message = self.msg
      del self.resultType, self.stack, self.parser, self.msg, self.data
      return result

   ## Handle an opening XML tag
   def StartElementHandler(self, tag, attr):
      self.data = ""
      if tag == "soapenv:Fault":
         self.isFault = True
      elif tag == "detail":
         self.deser.Deserialize(self.parser, object, True)
      elif tag.endswith("Response"):
         self.deser.Deserialize(self.parser, self.resultType, False)

   ## Handle text data
   def CharacterDataHandler(self, data):
      self.data += data

   ## Handle a closing XML tag
   def EndElementHandler(self, tag):
      if tag == "faultstring":
         try:
            self.msg = str(self.data)
         except UnicodeError:
            self.msg = self.data

## SOAP stub adapter object
class SoapStubAdapter:
   ## Constructor
   #
   # The endpoint can be specified individually as either a host/port
   # combination, or with a URL (using a url= keyword).
   #
   # @param self self
   # @param host host
   # @param port port (pass negative port number for no SSL)
   # @param ns API namespace
   # @param path location of SOAP VMOMI service
   # @param url URL (overrides host, port, path if set)
   # @param poolSize size of HTTP connection pool
   def __init__(self, host='localhost', port=443, ns='vim2', path='/sdk',
                url=None, poolSize=5):
      self.cookie = ""
      if url:
         scheme, self.host, self.path = urlparse.urlparse(url)[:3]
         self.scheme = scheme == "http" and httplib.HTTPConnection \
                    or scheme == "https" and httplib.HTTPSConnection
      else:
         port, self.scheme = port < 0 and (-port, httplib.HTTPConnection) \
                                       or (port, httplib.HTTPSConnection)
         self.host = '%s:%d' % (host, port)
      self.version = versionMap[ns]
      self.path = path
      self.poolSize = poolSize
      self.pool = []
      self.lock = thread.allocate_lock()

   ## Serialize a VMOMI request to SOAP
   #
   # @param version API version
   # @param mo the 'this'
   # @param info method info
   # @param args method arguments
   # @return the serialized request
   def SerializeRequest(self, mo, info, args):
      if not IsChildVersion(self.version, info.version):
         raise GetWsdlType("MethodNotFound")(receiver=mo, method=info.name)
      result = [SOAP_START,
                '<%s xmlns="urn:%s">' % (info.wsdlName, nsMap[self.version]),
                Serialize(mo, Object(name="_this", type=ManagedObject,
                                     version=BASE_VERSION), self.version) ]
      for (param, arg) in zip(info.params, args):
         result.append(Serialize(arg, param, self.version))
      result.append('</%s>' % info.wsdlName)
      result.append(SOAP_END)
      return ''.join(result)

   ## Invoke a managed method
   #
   # @param self self
   # @param mo the 'this'
   # @param info method info
   # @param args arguments
   def InvokeMethod(self, mo, info, args):
      req = self.SerializeRequest(mo, info, args)
      conn = self.GetConnection()
      conn.request('POST', self.path, req, {'Cookie' : self.cookie})
      resp = conn.getresponse()
      cookie = resp.getheader('set-cookie')
      status = resp.status
      if cookie:
         self.cookie = cookie
      if status == 200 or status == 500:
         try:
            obj = SoapResponseDeserializer(self).Deserialize(resp, info.result)
         finally:
            resp.read()
            self.ReturnConnection(conn)
         if status == 200:
            return obj
         else:
            raise obj
      else:
         conn.close()
         raise httplib.HTTPException("%d %s" % (resp.status, resp.reason))

   ## Invoke a managed property accessor
   #
   # @param self self
   # @param mo the 'this'
   # @param info property info
   def InvokeAccessor(self, mo, info):
      prop = info.name
      param = Object(name="prop", type=str, version=BASE_VERSION, flags=0)
      info = Object(name=info.name, type=ManagedObject, wsdlName="Fetch",
                    version=info.version, params=(param,), result=info.type)
      return self.InvokeMethod(mo, info, (prop,))

   ## Get a HTTP connection from the pool
   def GetConnection(self):
      self.lock.acquire()
      if self.pool:
         result = self.pool.pop()
         self.lock.release()
      else:
         self.lock.release()
         result = self.scheme(self.host)
      return result

   ## Return a HTTP connection to the pool
   def ReturnConnection(self, conn):
      self.lock.acquire()
      if len(self.pool) < self.poolSize:
         self.pool.append(conn)
         self.lock.release()
      else:
         self.lock.release()
         conn.close()

