# **********************************************************
# Copyright 2005-2006 VMware, Inc.  All rights reserved.
# -- VMware Confidential
# **********************************************************
## VMOMI support code

NoneType = type(None)

WSDL_NAMES = False

(F_LINK,
 F_LINKABLE,
 F_OPTIONAL) = [ 1<<x for x in range(3) ]

BASE_VERSION = 'vmodl.version.version0'
VERSION1      = 'vmodl.version.version1'

# In Python 2.4 and earlier, exceptions are old-style classes, so data objects
# must be old-style too.  For 2.5 and newer, data objects must be new-style
# classes
if issubclass(Exception, object):
   class Base(object): pass
   SetAttr = object.__setattr__
else:
   class Base: pass
   def SetAttr(obj, key, val):
      obj.__dict__[key] = val

## Simple class to store named attributes
class Object:
   ## Constructor
   def __init__(self, **kwargs):
      self.__dict__.update(kwargs)

## Format a python VMOMI object
#
# @param val the object
# @param info the field
# @param indent the level of indentation
# @return the formatted string
def FormatObject(val, info=Object(name="", type=object, flags=0), indent=0):
   start = indent * "   " + (info.name and "%s = " % info.name or "")
   if val == None:
      result = "<unset>"
   elif isinstance(val, DataObject):
      if info.flags & F_LINK:
         result = "<%s:%s>" % (val.__class__.__name__, val.key)
      else:
         result = "(%s) {\n%s\n%s}" % (val.__class__.__name__,
            ',\n'.join([FormatObject(getattr(val, prop.name), prop, indent+1)
                        for prop in val._GetPropertyList()]), indent * "   ")
   elif isinstance(val, ManagedObject):
      result = "'%s:%s'" % (val.__class__.__name__, val._moId)
   elif isinstance(val, list):
      itemType = getattr(val, 'Item', getattr(info.type, 'Item', object))
      if val:
         item = Object(name="", type=itemType, flags=info.flags)
         result = "(%s) [\n%s\n%s]" % (itemType.__name__,
            ',\n'.join([FormatObject(obj, item, indent+1) for obj in val]),
            indent * "   ")
      else:
         result = "(%s) []" % itemType.__name__
   elif isinstance(val, type):
      result = val.__name__
   elif isinstance(val, ManagedMethod):
      result = '%s.%s' % (val.info.type.__name__, val.info.name)
   elif isinstance(val, bool):
      result = val and "true" or "false"
   else:
      result = repr(val)
   return start + result

## Lookup a property for a given object type
#
# @param type the type
# @param name the name of the property
def GetPropertyInfo(type, name):
   while hasattr(type, "_propInfo"):
      try:
         return type._propInfo[name]
      except KeyError:
         type = type.__bases__[0]
   raise AttributeError(name)

## VMOMI Managed Object class
class ManagedObject(object):
   _wsdlName = "ManagedObject"
   _propInfo = {}
   _methodInfo = {}
   _version = BASE_VERSION

   ## Constructor
   #
   # @param[in] self self
   # @param[in] moId The ID of this managed object
   # @param[in] stub The stub adapter, if this is a client stub object
   def __init__(self, moId, stub=None):
      object.__setattr__(self, "_moId", moId)
      object.__setattr__(self, "_stub", stub)

   ## Invoke a managed method
   #
   # @param info method info
   # @param self self
   # @param ... arguments
   def _InvokeMethod(info, self, *posargs, **kwargs):
      if len(posargs) > len(info.params):
         s = "s"*(len(info.params)!=1)
         raise TypeError("%s() takes at most %d argument%s (%d given)" %
            (Capitalize(info.name),  len(info.params), s, len(posargs)))
      args = list(posargs) + [None] * (len(info.params) - len(posargs))
      for (k, v) in kwargs.items():
         try:
            idx = [param.name for param in info.params].index(k)
         except ValueError:
            raise TypeError("%s() got an unexpected keyword argument '%s'" %
                            (Capitalize(info.name),  k))
         if idx < len(posargs):
            raise TypeError("%s() got multiple values for keyword argument '%s'" %
                            (Capitalize(info.name),  k))
         args[idx] = v
      map(CheckField, info.params, args)
      return self._stub.InvokeMethod(self, info, args)
   _InvokeMethod = staticmethod(_InvokeMethod)

   ## Invoke a managed property accessor
   #
   # @param info property info
   # @param self self
   def _InvokeAccessor(info, self):
      return self._stub.InvokeAccessor(self, info)
   _InvokeAccessor = staticmethod(_InvokeAccessor)

   ## Get the ID of a managed object
   def _GetMoId(self):
      return self._moId

   def __setattr__(*args):
      raise Exception("Managed object attributes are read-only")
   __delattr__ = __setattr__

   __str__ = __repr__ = FormatObject
   _GetPropertyInfo = classmethod(GetPropertyInfo)

## VMOMI Data Object class
class DataObject(Base):
   _wsdlName = "DataObject"
   _propInfo = {}
   _propList = []
   _version = BASE_VERSION

   ## Constructor
   #
   # @param info property info
   # @param ... keyword arguments indicate properties
   def __init__(self, **kwargs):
      for info in self._GetPropertyList():
         if issubclass(info.type, list):
            SetAttr(self, info.name, info.type())
         elif info.flags & F_OPTIONAL:
            SetAttr(self, info.name, None)
         elif issubclass(info.type, Enum):
            SetAttr(self, info.name, None)
         elif issubclass(info.type, str):
            SetAttr(self, info.name, "")
         elif issubclass(info.type, long):
            SetAttr(self, info.name, 0L)
         elif issubclass(info.type, float):
            SetAttr(self, info.name, 0.0)
         else:
            SetAttr(self, info.name, None)
      for (k, v) in kwargs.items():
         setattr(self, k, v)

   ## Get a list of all properties of this type and base types
   #
   # @param cls the managed object type
   # @param name moId The ID of this managed object
   # @param stub The stub adapter, if this is a client stub object
   def _GetPropertyList(type):
      result = type._propList
      while type != DataObject:
         type = type.__bases__[0]
         result = type._propList + result
      return result
   _GetPropertyList = classmethod(_GetPropertyList)

   ## Set a property
   def _SetProperty(info, self, val):
      CheckField(info, val)
      SetAttr(self, info.name, val)
   _SetProperty = staticmethod(_SetProperty)

   ## Get a property
   def _GetProperty(info, self):
      return getattr(self, info.name)
   _GetProperty = staticmethod(_GetProperty)

   def __setattr__(self, name, val):
      self._SetProperty(self._GetPropertyInfo(name), self, val)

   _GetPropertyInfo = classmethod(GetPropertyInfo)
   __str__ = __repr__ = FormatObject

## Base class for enum types
class Enum(str): pass

## Base class for array types
class Array(list):
   __str__ = __repr__ = FormatObject

## Class for curried function objects
#
# Instances of this class are curried python function objects.
# If g = Curry(f, a1,...,an), then g(b1,...,bm) = f(a1,...,an, b1,...,bm)
class Curry(object):
   ## Constructor
   #
   # @param self self
   # @param f the function object
   # @param args arguments to fix
   def __init__(self, f, *args):
      self.f = f
      self.args = args

   def __call__(self, *args, **kwargs):
      args = self.args + args
      return self.f(*args, **kwargs)

   def __get__(self, obj, type):
      if obj:
         # curried methods will receive 'self' *after* any fixed arguments
         args = self.args + (obj,)
         return Curry(self.f, *args)
      return self

## Class for managed object methods
class ManagedMethod(Curry):
   ## Constructor
   #
   # @param self self
   # @param info method info
   def __init__(self, info):
      Curry.__init__(self, ManagedObject._InvokeMethod, info)
      self.info = info

## Create the vmodl.MethodFault type
#
# This type must be generated dynamically because it inherits from
# vmodl.DynamicData, which is created dynamically by the emitted code.
#
# @return the new type
def CreateMethodFaultType():
   info = Object(name="message", type="string", version=BASE_VERSION, flags=0)
   dict = {"_wsdlName" : "MethodFault", "_propInfo" : {"message" : info },
           "_propList" : [info], "_version" : BASE_VERSION,
           "GetMessage" : Curry(DataObject._GetProperty, info),
           "SetMessage" : Curry(DataObject._SetProperty, info)}
   name = WSDL_NAMES and "MethodFault" or "vmodl.MethodFault"
   return AddType(type(Exception)(name,
                    (wsdlTypeMap["DynamicData"], Exception), dict), name)

## Create a data object type
#
# @param vmodlName the VMODL name of the type
# @param wsdlName the WSDL name of the type
# @param parent the WSDL name of the parent type
# @param version the version of the type
# @param props properties of the type
# @return the new data object type
def CreateDataType(vmodlName, wsdlName, parent, version, props):
   parent = GetWsdlType(parent)
   propInfo = {}
   propList = [ Object(**dict(zip(("name", "type", "version", "flags"), p)))
                for p in props ]
   dic = {"_wsdlName" : wsdlName, "_propInfo" : propInfo,
          "_propList" : propList, "_version" : version}
   for info in propList:
      propInfo[info.name] = info
      capName = Capitalize(info.name)
      dic["Get" + capName] = Curry(DataObject._GetProperty, info)
      dic["Set" + capName] = Curry(DataObject._SetProperty, info)
   name = WSDL_NAMES and wsdlName or vmodlName
   result = AddType(type(Exception)(name, (parent,), dic), name)
   if wsdlName == "DynamicData":
      CreateMethodFaultType()
      CreateDataType("vmodl.RuntimeFault", "RuntimeFault",
                     "MethodFault", BASE_VERSION, [])
   return result

## Create a managed object type
#
# @param vmodlName the VMODL name of the type
# @param wsdlName the WSDL name of the type
# @param parent the WSDL name of the parent type
# @param version the version of the type
# @param props properties of the type
# @param methods methods of the type
# @return the new managed object type
def CreateManagedType(vmodlName, wsdlName, parent, version, props, methods):
   parent = GetWsdlType(parent)
   propInfo = {}
   methodInfo = {}
   dic = {"_wsdlName" : wsdlName, "_propInfo" : propInfo,
          "_methodInfo" : methodInfo, "_version" : version}
   for prop in props:
      info = Object(**dict(zip(("name","type","version","flags"), prop)))
      propInfo[info.name] = info
      getter = Curry(ManagedObject._InvokeAccessor, info)
      dic["Get" + Capitalize(info.name)] = getter
      dic[info.name] = property(getter)
   for (mName, mWsdl, mVersion, mParams, mResult) in methods:
      mName = WSDL_NAMES and mWsdl or Capitalize(mName)
      if mName.endswith("_Task"):
         mName = mName[:-5]
      params = tuple([Object(**dict(zip(("name","type","version","flags"), p)))
                      for p in mParams])
      info = Object(name=mName, type=wsdlName, wsdlName=mWsdl,
                    version=mVersion, params=params, result=mResult)
      methodInfo[mName] = info
      mm = ManagedMethod(info)
      wsdlMethodMap[info.wsdlName] = mm
      dic[mName] = mm
   name = WSDL_NAMES and wsdlName or vmodlName
   return AddType(type(name, (parent,) , dic), name)

## Create an enum type
#
# @param vmodlName the VMODL name of the type
# @param wsdlName the WSDL name of the type
# @param version the version of the type
# @param values enum values
# @return the new enum type
def CreateEnumType(vmodlName, wsdlName, version, values):
   name = WSDL_NAMES and wsdlName or vmodlName
   result = type(name, (Enum,),
      {"_wsdlName" : wsdlName, "_version" : version})
   result.values = map(result, values)
   for value in result.values:
      setattr(result, value, value)
   return AddType(result, name)

## Create an array type
#
# @param itemType the item type
# @return the new array type
def CreateArrayType(itemType):
   return type("%s[]" % itemType.__name__, (Array,), {'Item' : itemType})

## Add a new type to the type maps, create array constructors
#
# @param type the type object
# @param name the name of the type (optional)
# @return type
def AddType(type, name=None):
   # the name field is needed since python2.2 turns 'a.b.c' into just 'c'
   if name is None:
      name = type.__name__
   type.Array = CreateArrayType(type)
   wsdlTypeMap[type._wsdlName] = type

   # add type to the dynamic wrapper object
   names = name.split('.')
   cur = types
   for name in names[:-1]:
      try:
         cur = getattr(cur, name)
      except AttributeError:
         ns = Object()
         setattr(cur, name, ns)
         setattr(cur, Capitalize(name), ns)
         cur = ns
   setattr(cur, names[-1], type)
   return type

## Check that a value matches a given type, and annotate if neccesary
#
# @param checkType expected type
# @param val object to check
# @throw TypeError if the value does not match the type
def CheckField(info, val):
   valType = Type(val)
   if val == None:
      if not info.flags & F_OPTIONAL:
         raise TypeError('Field "%s" is not optional' % info.name)
      return
   elif info.type is object:
      try:
         GetWsdlName(valType)
         return
      except KeyError:
         raise TypeError('Unknown type: %s' % info.type.__name__)
   elif issubclass(valType, info.type):
      return
   elif issubclass(info.type, list) and not issubclass(valType, Array):
      itemInfo = Object(type=info.type.Item)
      for it in val:
         CheckField(itemInfo, it)
   elif info.type is type and valType is type(Exception) \
     or issubclass(info.type, int) and issubclass(valType, int) \
     or issubclass(info.type, long) and issubclass(valType, long) \
     or issubclass(info.type, float) and issubclass(valType, float) \
     or issubclass(info.type, str) and (issubclass(valType, str) or \
                                        issubclass(valType, unicode)):
      return
   else:
      raise TypeError('Expected %s, but got %s'
                    % (info.type.__name__, valType.__name__))

## Finalize the created types
#
# Due to circular dependencies, all of the method/property info
# objects are created with their types as strings.  Once all the types
# have been generated, replace these strings with the actual generated
# type objects.
def FinalizeTypes():
   for type in wsdlTypeMap.values():
      if issubclass(type, DataObject):
         for info in type._propList:
            info.type = GetWsdlType(info.type)
      elif issubclass(type, ManagedObject):
         for info in type._propInfo.values():
            info.type = GetWsdlType(info.type)
         for info in type._methodInfo.values():
            info.result = GetWsdlType(info.result)
            info.type = GetWsdlType(info.type)
            for param in info.params:
               param.type = GetWsdlType(param.type)

## Get the type of an object, for both new and old-style classes
def Type(obj):
   try:
      return obj.__class__
   except AttributeError:
      return type(obj)

## Lookup a WSDL type name
def GetWsdlType(name):
   try :
      return wsdlTypeMap[name]
   except KeyError:
      if name.startswith("ArrayOf"):
         return wsdlTypeMap[name[7:]].Array
      raise

## Get the WSDL name of a type
def GetWsdlName(type):
   try:
      if issubclass(type, list):
         return "ArrayOf" + Capitalize(GetWsdlName(type.Item))
      else:
         return type._wsdlName
   except AttributeError:
      return wsdlNameMap[type]

## Capitalize a string
def Capitalize(str):
   if str:
      return str[0].capitalize() + str[1:]
   return str

## Add an API version
def AddVersion(version, ns):
  if not (version in parentMap):
      nsMap[version] = ns
      versionMap[ns] = version
      parentMap[version] = {}

## Add a parent version
def AddVersionParent(version, parent):
   parentMap[version][parent] = True

## Check if a version is a child of another
def IsChildVersion(a, b):
   return b in parentMap[a]

## Widen a type to one supported in a given version
def GetCompatibleType(type, version):
   if issubclass(type, DataObject):
      while not IsChildVersion(version, type._version):
         type = type.__bases__[0]
   return type

## Invert an injective mapping
def InverseMap(map):
   return dict([ (v, k) for (k, v) in map.items() ])

types = Object()
nsMap = {}
versionMap = {}
parentMap = {}

if not isinstance(bool, type): # bool not a type in python <= 2.2
   bool = type("bool", (int,),
               {"__new__": lambda cls, val=0: int.__new__(cls, val and 1 or 0)})
byte  = type("byte", (int,), {})
short  = type("short", (int,), {})
double = type("double", (float,), {})
datetime = type("datetime", (str,), {}) # TODO: Use datetime module if available
PropertyPath = type("PropertyPath", (str,), {})

wsdlTypeMap = {
   'void' : NoneType,
   'anyType': object,
   'TypeName' : type,
   'boolean': bool,
   'byte': byte,
   'short' : short,
   'int' : int,
   'long' : long,
   'float' : float,
   'double' : double,
   'string' : str,
   'dateTime' : datetime,
   'MethodName' : ManagedMethod,
   'PropertyPath' : PropertyPath
}
wsdlNameMap = InverseMap(wsdlTypeMap)

for (name, typ) in wsdlTypeMap.items():
   if typ is not NoneType:
      setattr(types, typ.__name__, typ)
      wsdlTypeMap[Capitalize(name)] = typ
      arrayType = CreateArrayType(typ)
      setattr(types, Capitalize(typ.__name__) + "Array", arrayType)
      wsdlTypeMap["ArrayOf" + Capitalize(name)] = arrayType
del name, typ

wsdlMethodMap = {}

AddType(ManagedObject)
AddType(DataObject)
wsdlTypeMap['ManagedObjectReference'] = ManagedObject

checkMap = {
   bool : int,
   byte : int,
   short : int,
   int : int,
   long : long,
   float : float,
   double : float,
   datetime : str,
   PropertyPath : str
}

