#!/usr/bin/python

# **********************************************************
# Copyright 2006 VMware, Inc.  All rights reserved. -- VMware Confidential
# **********************************************************

import getopt
import os
import sys
from pyVmomi import types, Vmodl, Vim, SoapStubAdapter
from pyVmomi.SoapAdapter import Serialize
import pyVim.vimApiTypeMatrix
import pyVim.vimApiGraph

def connect(host, port, user, pwd, adapter="SOAP"):
   if adapter == None:
      adapter = "SOAP"

   if host == None:
      host = "localhost"

   if port == None:
      port = 443

   if user == None:
      user = os.getlogin()

   if adapter != "SOAP":
      raise Exception("Invalid adapter protocol: %s" % str(adapter))

   print "Connecting to :", host, "at", port, "as", user, "over", adapter

   # Create the SOAP stub adapter
   stub = SoapStubAdapter(host, port)

   # Get Service instance
   si = Vim.ServiceInstance("ServiceInstance", stub)
   content = si.RetrieveContent()
   sessionManager = content.GetSessionManager()

   # Login locally
   if pwd == None:
      localTicket = sessionManager.AcquireLocalTicket(user)
      user = localTicket.GetUserName()
      passwordFile = localTicket.GetPasswordFilePath()
      try:
         fd = open(passwordFile, "r")
         pwd = fd.readline()
         fd.close()
      except Exception:
         msg = "Failed to read password file.  A password is required to " + \
               "connect remotely."
         raise Exception(msg)
      
   # Login
   try:
      x = sessionManager.Login(user, pwd, None);
   except Vim.Fault.InvalidLogin:
      print "Invalid login:", user	
   except:
      print "Failed to login as ", user
      sys.exit()

   return si

def GetPropertyCollector(si):
   content = si.RetrieveContent()
   pc = content.GetPropertyCollector()
   return pc

def SortByManagedObject(x, y):
   x = x.GetObj()
   y = y.GetObj()
   result = cmp(x.__class__.__name__, y.__class__.__name__)
   if (result == 0):
      result = cmp(x._GetMoId(), y._GetMoId())
   return result

# [(moType, all), (moType, all), ...]
def MakePropertySpecs(managedObjectSpecs):
   propertySpecs = []

   for managedObjectSpec in managedObjectSpecs:
      moType = managedObjectSpec[0]
      all = managedObjectSpec[1]

      propertySpec = Vmodl.Query.PropertyCollector.PropertySpec()
      propertySpec.SetType(reduce(getattr, moType.split('.'), types))
      propertySpec.SetAll(all)

      propertySpecs.append(propertySpec)

   return propertySpecs

def usage():
   print "Usage: vimdump <options>\n"
   print "Options:"
   print "  -h            help"
   print "  -H <host>     host to connect. default is 'localhost'"
   print "  -O <port>     port to connect. default is 443 for SOAP adapter."
   print "  -U <user>     user to connect as. default is logged in user."
   print "  -P <password> password to login."
   print "  -A <adapter>  adapter type.  Either 'VMDB' or 'SOAP'. default is 'VMDB'"
   print "  -f            only show property collector filter.  Don't execute."

def main():
   try:
      opts, args = getopt.getopt(sys.argv[1:], "hH:O:U:P:A:f",
                                 ["help",
                                  "host=",
                                  "port=",
                                  "user=",
                                  "password=",
                                  "adapter=",
                                  "filteronly"
                                  ])
   except getopt.GetoptError:
      # print help information and exit:
      usage()
      sys.exit(1)

   host = None
   port = None
   user = None
   pwd = None
   adapter = None
   filteronly = False

   for o, a in opts:
      if o in ("-H", "--host"):
         host = a
      if o in ("-O", "--port"):
         port = int(a)
      if o in ("-U", "--user"):
         user = a
      if o in ("-P", "--password"):
         pwd = a
      if o in ("-A", "--adapter"):
         adapter = a
      if o in ("-f", "--filteronly"):
         filteronly = True
      if o in ("-h", "--help"):
         usage()
         sys.exit()
   
   objectSpec = Vmodl.Query.PropertyCollector.ObjectSpec()
   siInfo = Vim.ServiceInstance("ServiceInstance", None)
   objectSpec.SetObj(siInfo)
   objectSpec.SetSkip(False)
   objectSpec.SetSelectSet(pyVim.vimApiGraph.BuildMoGraphSelectionSpec())

   fetchProp = False

   # Build up a property spec that consists of all managed object types
   matrix = pyVim.vimApiTypeMatrix.CreateMoTypeMatrix()
   classNames = matrix.GetClassNames()
   propertySpecs = map(lambda x: [x, fetchProp], classNames)
   propertySpecs = MakePropertySpecs(propertySpecs)

   objectSpecs = [objectSpec]
    
   filterSpec = Vmodl.Query.PropertyCollector.FilterSpec()
   filterSpec.SetPropSet(propertySpecs)
   filterSpec.SetObjectSet(objectSpecs)

   filterSpecs = Vmodl.Query.PropertyCollector.FilterSpec.Array([filterSpec])
   
   if filteronly:
      result = Serialize(filterSpecs)
      header = (
         "<!-- Property collector spec used by vimdump to gather references to -->",
         "<!-- all known managed objects on the system (SOAP serialization).  -->",
         "<!-- Usage: vim-cmd vimsvc/property_dump vimdump-spec.xml -->")
      print '\n'.join(header)
      print result
      return 0

   # Connect
   si = None
   try:
      si = connect(host, port, user, pwd, adapter)
   except Exception, e:
      print e
      sys.exit(2)

   pc = GetPropertyCollector(si)
   objectContents = pc.RetrieveContents(filterSpecs);
   objectContents.sort(lambda x, y: SortByManagedObject(x, y))

#   print objectContents    

   print "\nFound " + str(len(objectContents)) + " managed objects:"
   i = 1
   for objectContent in objectContents:
      obj = objectContent.GetObj()
      print str(i) + ". " + obj.__class__.__name__ + "::" + obj._GetMoId()
      i = i + 1
   print ""

   mos = map(lambda x: x.GetObj(), objectContents)

   i = 1
   for mo in mos:
      objectSpec = Vmodl.Query.PropertyCollector.ObjectSpec()
      objectSpec.SetObj(mo)
      objectSpec.SetSkip(False)
      objectSpec.SetSelectSet([])

      propertySpec = Vmodl.Query.PropertyCollector.PropertySpec()
      propertySpec.SetType(mo.__class__)
      propertySpec.SetAll(True)

      filterSpec = Vmodl.Query.PropertyCollector.FilterSpec()
      filterSpec.SetPropSet([propertySpec])
      filterSpec.SetObjectSet([objectSpec])

      print "=============== " + str(i) + ". " + \
            mo.__class__.__name__ + "::" + mo._GetMoId() + \
            " ===============\n"

      objectContents = None
      try:
         objectContents = pc.RetrieveContents([filterSpec]);
      except Exception:
         print "Failed to retrieve contents."

      if objectContents != None:
         try:
            print objectContents
         except Exception:
            print "Failed to format object."
            return
         
      print ""
      i = i + 1
      

# Start program
if __name__ == "__main__":
   main()
