#!/usr/bin/perl -w

use strict;

#
# OOTB, vmware-hostd is setup to use uid 12 & gid 20 as its 
# datastorePrincipal configuration.  Adjust group & user name
# for these uid & gid and assign uid 12's default group to be
# gid 20.  This user & group are named 'vimuser'
#
# When upgrading from ESX2.x and there are vm files located under
# /root, we set vimuser's default group to root so that foundry
# can load the vm's config files in vimuser context.
# 

sub getGroupOption {
   if (-e "/etc/vmware/vm-list") {
      # upgrading from ESX2.x.  Check if /root/ path is used anywhere.
      `/bin/grep "/root/" /etc/vmware/vm-list`;
      if ($? == 0) { 
         return "-g root";
      }
   }
   return "";
}

sub runCommand($) {
   my $cmd = $_[0];
   my $output = `$cmd 2>&1`;
   if ($? != 0) {
      if (!defined $output) {
         $output = "unknown error";
      }
      system("logger -p crit -t vmware-hostd \"cmd: $cmd failed reason: $output\"");
      exit 1;
   }
}

my $vimuser = "vimuser";
my $groupOption = getGroupOption();


if (length $groupOption == 0) {
   my $group;
   $group = getgrgid(20);
   if (!defined $group) {
      # group with gid 20 does not exist - create a group
      runCommand("/usr/sbin/groupadd -g 20 $vimuser");
      $group = $vimuser;
   } else {
      # if the group name is 'game,' modify it to $vimuser.  Otherwise,
      # leave it alone since it may have been customized
      if ($group eq "games") {
         $group = $vimuser;
         runCommand("/usr/sbin/groupmod -n $group games");  
      }
   }
   $groupOption = "-g $group";
}

#
# Setup user
#
my @userInfo;
@userInfo = getpwuid(12);
if (!@userInfo) {
   # user with uid 12 does not exist - create it
   runCommand("/usr/sbin/useradd -u 12 -s /sbin/nologin -d /dev/null $groupOption $vimuser");
} else {
   if ($userInfo[0] eq "games") {
      # change user name to $vimuser 
      runCommand("/usr/sbin/usermod -c $vimuser -l $vimuser -d /sbin games");
   }
   runCommand("/usr/sbin/usermod $groupOption $vimuser");
}

exit 0;
