#!/bin/sh
#
###################################################
# Copyright 2008-2009 VMware, Inc.  All rights reserved.
# -- VMware Confidential
###################################################

: "${PARTED_UTIL:=/sbin/partedUtil}"
TRUE=0
FALSE=1

retcode=0

ESX_3i_SCRATCH_BEG=1536000
ESX_3i_SCRATCH_END=9922559
ESX_4i_SCRATCH_BEG=1843200

ESX_4i_EXTENDED_END=1843199

SCRATCH_PARTITION=2
BOOTBANK='/bootbank'

ESX_3i_VISOR_LAYOUT="1 8192 1535999 5 0
4 32 8191 4 128
5 8224 106495 6 0
6 106528 204799 6 0
7 204832 430079 252 0
8 430112 1535999 6 0"

LOG_FILE=/var/log/vmware/upgrade.log

Log()
{
   echo "$@" >> "${LOG_FILE}"
}

isGreater()
{
# This function safely compares large integers.
# returns 1 if first argument is larger than the second.
# returns 0 otherwise.

    if [ ${#1} -gt ${#2} ]; then
        return ${TRUE}
    fi
    if [ ${#1} -eq ${#2} ]; then
        if [ ${1} -gt ${2} ]; then
            return ${TRUE}
        fi
    fi

    return ${FALSE}
}

CanCreateScratch()
{
   # check bootbank partition layout
   partitiontable=$("${PARTED_UTIL}" get "${1}" 2>/dev/null | tail -n +2 | sort)
   [ "${partitiontable}" = "${ESX_3i_VISOR_LAYOUT}" ] || {
      Log "Partition has changed, might not be able to create scratch partition"
      return ${FALSE}
   }

   return ${TRUE}
}

#
# Check supported CPU, need 64 bit
#
cpu_test_result="ERROR"
cpu_test_found="False"

Log "Checking compatible CPU..."
edx81=$(vsish  -b -r -e cat /hardware/cpu/cpuList/0 | sed -n '/id81/ {N N N N s/.*edx:\(.*\)/\1/p }')

CPU_MASK_LM=0x20000000
if [ -n "${edx81}" ]; then
   if [ $(( ${edx81} & ${CPU_MASK_LM})) -ne 0 ]; then
      cpu_test_found="True"
      cpu_test_result="SUCCESS"
      Log "Passed CPU test"
   fi
else
   Log "Unable to retrieve CPU info"
   retcode=1
fi

# Check available physical memory

EXPECTED_MEM=2047                # unit in MB. Subtract one for rounding errors 'n such
mem_test_result="ERROR"
mem_test_found=""

# unit is KB
Log "Checking available memory..."
physmem=$(vsish -b -r -e cat /memory/comprehensive | grep 'physmem')
total_mem=${physmem##*physmem:}
if [ -n "${total_mem}" ]; then
    # Round up by 256MB, unit in MB. Because there are holes that are not
    # reported or cannot be accurately measured sometimes. On the other hand,
    # physical memory usually has at least 256MB increment.
    mem_test_found="$((($total_mem + 256*1024 -1)/(256*1024)*256))"
    if [ "${mem_test_found}" -ge "${EXPECTED_MEM}" ]; then
        mem_test_result="SUCCESS"
        Log "Passed memory test."
    fi
else
   Log "Unable to retrieve physical memory info"
   retcode=1
fi

# Check supported hard disk size/layout
Log "Checking supported boot disk..."
bootdisk_test_result="ERROR"
bootdisk_test_found="False"

BootHBA=$(vmkfstools -P "${BOOTBANK}" 2> /dev/null | sed -n '$s/^.\(.\+\):[0-9]$/\1/p')

case "$(vmware -v)" in
  *3i*) BootDisk="/vmfs/devices/disks/${BootHBA}:0" ;;
  *ESXi*) BootDisk="/vmfs/devices/disks/${BootHBA}" ;;
  *) BootDisk="" ;;
esac

if [ -n "${BootHBA}" -a -n "${BootDisk}" ] ; then

   set -- $("${PARTED_UTIL}" get "${BootDisk}" 2> /dev/null | head -1)
   isGreater ${4} ${ESX_4i_EXTENDED_END}
   result=$?
   if [ ${result} -eq ${TRUE} ]; then
      local visorSignature=$(esxcfg-info -b 2> /dev/null | cut -b 7,7 2> /dev/null)
      # '7' is thinESX signature on the bootbank partition
      if [ "${visorSignature}" = "7" ]; then
         if isGreater "${ESX_3i_SCRATCH_END}" "${4}"; then
            Log "Failed boot disk test, ThinESX: no enough space for scratch"
            retcode=1
         else
            set -- $("${PARTED_UTIL}" get "${BootDisk}" 2> /dev/null | grep "^${SCRATCH_PARTITION} ")
            if [ "${3}" -eq "${ESX_3i_SCRATCH_END}" ] \
                 && [ "${2}" -eq "${ESX_3i_SCRATCH_BEG}" -o "${2}" -eq "${ESX_4i_SCRATCH_BEG}" ]; then
               bootdisk_test_result="SUCCESS"
               bootdisk_test_found="True"
               Log "Passed boot disk test, ThinESX"
            elif CanCreateScratch "${BootDisk}" ; then
               bootdisk_test_result="SUCCESS"
               bootdisk_test_found="True"
               Log "No scratch, will try to create scratch during upgrade"
            else
               Log "Failed boot disk test, ThinESX"
               retcode=1
            fi
         fi
      elif [ -n "${visorSignature}" ]; then
         bootdisk_test_result="SUCCESS"
         bootdisk_test_found="True"
         Log "Passed boot disk test, embedded ESX"
      else
         Log "Failed to get ESX 3i bootbank signature"
         retcode=1
      fi
   else
      Log "Failed boot disk test, general"
      [ -z "${4}" ] && retcode=1
   fi
else
   Log "Failed boot disk test, either BootDisk or BootHBA is empty: ${BootDisk}:${BootHBA}"
   retcode=1
fi

cat <<EOF
<?xml version="1.0" ?>
<precheck>
  <tests>
    <test>
      <name>SUPPORTED_CPU</name>
      <expected>
        <value>True</value>
      </expected>
      <found>
        <value>${cpu_test_found}</value>
      </found>
      <result>${cpu_test_result}</result>
    </test>
    <test>
      <name>MEMORY_SIZE</name>
      <expected>
        <value>${EXPECTED_MEM}</value>
      </expected>
      <found>
        <value>${mem_test_found}</value>
      </found>
      <result>${mem_test_result}</result>
    </test>
    <test>
      <name>SUPPORTED_BOOT_DISK</name>
      <expected>
        <value>True</value>
      </expected>
      <found>
        <value>${bootdisk_test_found}</value>
      </found>
      <result>${bootdisk_test_result}</result>
    </test>
  </tests>
</precheck>
EOF

exit ${retcode}

