#!/bin/sh
# Copyright 2008 VMware Inc.,

TRUE=0
FALSE=1
: "${LOG_FILE:=/var/log/vmware/upgrade.log}"

Log()
{
   if [ ${LOG_VERBOSE} -eq ${TRUE} ] ; then
      echo "$*" >&2
   fi

   echo "$*" >> "${LOG_FILE}"
}

Warning()
{
   echo "Warning: $*" >&2
}

Panic()
{
   echo "Error: $*" >&2
   exit 1
}

LOG_VERBOSE=${TRUE}
stagepath="/tmp/confstage"
: "${UPGRADE_LOGFILE:=/var/log/upgrade.log}"

#
# Files to remove during upgrade
#
filestoskip="etc/vmware/vmware.lic
etc/profile
etc/inittab
etc/inetd.conf
etc/motd
etc/issue
etc/vmware/license.cfg
etc/vmware/hostd/config.xml
etc/vmware/hostd/proxy.xml"

ISWRAPPED=${FALSE}

SetupStage ()
{
   local configfile=$1
   local localtgz=$1
   local confpath=$2

   Log "Setup translation staging area..."
   [ -f "${configfile}" ] || Panic "Source configuration file ${1} doesn't exist."

   rm -rf "${stagepath}"
   mkdir -p "${stagepath}"
   if [ "$(basename "${configfile}")" = "state.tgz" ] ; then
      ISWRAPPED=${TRUE}
      tar -C "${stagepath}" -xzf "${configfile}" 2>&1 || {
         Panic "Error untar ${configfile} to ${stagepath}: ${res}"
      }
      localtgz="${stagepath}/local.tgz"
   fi

   [ -f "${localtgz}" ] || Panic "local.tgz ${localtgz} not found."

   mkdir -p "${confpath}"
   tar -C "${confpath}" -xzf "${localtgz}" 2>&1  || {
      Panic "Error untar ${localtgz} to ${confpath}: ${res}"
   }
}

DecorateStage ()
{
   Log "Translating configuration files..."
   (cd "$1"
    echo "${filestoskip}" | while read line ;
    do
       rm -f "${line}"
    done

    # Generate a upgrade postscript postscript.sh under
    # /etc/rc.local.d folder. This script invokes partedUtil
    # to fix partition misalignment problem. It executes
    # only once during first reboot after the upgrade.

    [ -d etc/rc.local.d ] || mkdir -p etc/rc.local.d
    cat > etc/rc.local.d/postscript.sh <<'POSTSCRIPTEOF'
#!/bin/sh

# Maximum retry times for resizing partition.
RESIZE_LOOP_COUNT=100

. /etc/vmware/UpgradeFunctions.sh

IsThinESX || exit 0

hba=$(esxcfg-mpath -m | grep "$(GetBootHBA)" | awk '{ print $2 }')
driver=$(esxcfg-scsidevs -a | grep "^${hba}" | awk '{ print $2 }')

[ "${driver}" = "cciss" ] || exit 0

Upgrade_IsExpandNeeded && Upgrade_AllowVMFSOverwrite && {
   logger postscript "Attempt to resize core dump partition."
   count=0
   while [ $count -lt $RESIZE_LOOP_COUNT ] ; do
      "${PARTED_UTIL}" resize "$(GetBootDisk)" ${CORE_DUMP_PARTITION} ${ESX_4i_CORE_DUMP_BEG} ${ESX_4i_CORE_DUMP_END} 2> /dev/null && {
         Upgrade_PreventVMFSOverwrite
         logger postscript "Core dump partition is resized."
         exit 0
      }
      count=$(( $count + 1 ))
   done
   logger postscript "Failed to resize core dump partition."
   Upgrade_PreventVMFSOverwrite
}
exit 0
POSTSCRIPTEOF
    chmod +x etc/rc.local.d/postscript.sh

    [ -d etc/vmware ] || mkdir -p etc/vmware
    [ -f etc/vmware/config ] || cp /etc/vmware/config etc/vmware/config
    # See pr 314216, need to add key value to config
    $(grep '^libdir' etc/vmware/config 2>&1 > /dev/null) || {
       echo 'libdir = "/usr/lib/vmware"' >> etc/vmware/config
    }

    # We always want to use latest scripts for a major upgrade
    [ -d etc/init.d ] && rm -rf etc/init.d

    # Update chkconfig.db
    if [ -f etc/chkconfig.db ] ; then
       disabled=$(ls /etc/init.d/* | cat - etc/chkconfig.db | sort | uniq -c |sed -n 's:^1 ::p')
    fi
    cp -f "$2" etc/chkconfig.db
    for service in ${disabled} ; do
       if [ -f "${service}" ] ; then
          Log "Disable service : ${service}"
          service="$(echo "${service}" | sed -e 's:/:\\\/:g')"
          sed -i -e "/${service}/d" etc/chkconfig.db
       fi
    done
   )
}

DeployRollbackSupport()
{
   Log "Adding rollback cleanup support..."
   (cd "$1"

    # Cleanup /locker after rollback..
    [ -d etc/rc.local.d ] || mkdir -p etc/rc.local.d
    cat > etc/rc.local.d/clearlocker.sh <<EOF
#!/bin/sh
rm -rf /locker/packages/${PRODUCT_VERSION}
rm -rf /locker/packages/usr
EOF
    chmod +x etc/rc.local.d/clearlocker.sh
   )
}

PackageStage ()
{
   local confpath=${1}
   local outputpath=${2}
   local tgtpath=${3}

   Log "Repackaging staged configuration files..."
   if [ -d "${outputpath}" ] ; then
      rm -rf "${outputpath}"
   fi
   mkdir -p "${outputpath}"
   (
      cd "${confpath}"
      tar -czf "${outputpath}/local.tgz.$$" * 2>&1 || {
         Panic "Failed to create ${outputpath}/local.tgz: ${?}"
      }
   )

   if [ -s "${outputpath}/local.tgz.$$" ]; then
      if [ ${ISWRAPPED} -eq ${TRUE} ] ; then
         mkdir -p "$stagepath/state.$$"
         mv "${outputpath}/local.tgz.$$" "${stagepath}/state.$$/local.tgz"
         tar -C "${stagepath}/state.$$" -czf "${tgtpath}/state.tgz.$$" local.tgz 2>&1 || {
            Panic "Failed to create ${tgtpath}/stage.tgz: ${?}"
         }
         if [ -s "${tgtpath}/state.tgz.$$" ]; then
            mv -f "${tgtpath}/state.tgz.$$" "${tgtpath}/state.tgz"
         else
            Panic "Error creating state.tgz on $tgtpath"
         fi

         # Clean up any temporary files that might have been left behind
         rm -rf "${stagepath}/state.$$" "${tgtpath}/state.tgz.$$"
      else
         mv -f "${outputpath}/local.tgz.$$" "${tgtpath}/local.tgz.$$"
         mv -f "${tgtpath}/local.tgz.$$" "${tgtpath}/local.tgz"
      fi
      sync
   else
      Panic "Error creating local.tgz on $outputpath"
   fi
   rm -rf "${stagepath}"
}

Conf_Translate ()
{
   local confpath="${stagepath}/conf"
   local outputpath="${stagepath}/output"

   local configfile=$1
   local tgtpath=$2
   local chkconfigdb=$3
   [ -n "${tgtpath}" ] || Panic "ERROR tgtpath can't be empty"
   SetupStage "${configfile}" "${confpath}"
   DecorateStage "${confpath}" "${chkconfigdb}"
   PackageStage "${confpath}" "${outputpath}" "${tgtpath}"
}

Conf_RollbackSupport ()
{
   local confpath="${stagepath}/conf"
   local outputpath="${stagepath}/output"

   local configfile=$1
   local tgtpath=$2
   [ -n "${tgtpath}" ] || Panic "ERROR tgtpath can't be empty"
   SetupStage "${configfile}" "${confpath}"
   DeployRollbackSupport "${confpath}"
   PackageStage "${confpath}" "${outputpath}" "${tgtpath}"
}

