#!/bin/sh
# Copyright 2008-2009 VMware Inc.,

# \ref https://wiki.eng.vmware.com/Esx35To40DiskUpgrade
EXTENDED_PARTITION=1

ESX_3i_EXTENDED_BEG=8192
ESX_3i_EXTENDED_END=1535999

ESX_4i_EXTENDED_BEG=${ESX_3i_EXTENDED_BEG}
ESX_4i_EXTENDED_END=1843199             # $(( 900 * 1024 * 1024 / 512 ))

SCRATCH_PARTITION=2

ESX_3i_SCRATCH_BEG=1536000
ESX_3i_SCRATCH_END=9922559

ESX_4i_SCRATCH_BEG=1843200             # ${ESX_4i_EXTENDED_END} + 1
ESX_4i_SCRATCH_END=${ESX_3i_SCRATCH_END}

BOOT_BANK_ONE=5

ESX_3i_BOOT_BANK_ONE_BEG=8224
ESX_3i_BOOT_BANK_ONE_END=106495

ESX_4i_BOOT_BANK_ONE_BEG=${ESX_3i_BOOT_BANK_ONE_BEG}
ESX_4i_BOOT_BANK_ONE_END=520191        # ${ESX_4i_BOOT_BANK_ONE_BEG} + $(( 32 + 250 * 1024 * 1024 / 512 ))

BOOT_BANK_TWO=6

ESX_3i_BOOT_BANK_TWO_BEG=106528
ESX_3i_BOOT_BANK_TWO_END=204799

ESX_4i_BOOT_BANK_TWO_BEG=520224
ESX_4i_BOOT_BANK_TWO_END=1032191        # ${ESX_4i_BOOT_BANK_TWO_BEG} + $(( 32 + 250 * 1024 * 1024 / 512 ))

CORE_DUMP_PARTITION=7

ESX_3i_CORE_DUMP_BEG=204832
ESX_3i_CORE_DUMP_END=430079

ESX_4i_CORE_DUMP_BEG=1032224
ESX_4i_CORE_DUMP_END=1257471

LOCKER_PARTITION=8

ESX_3i_LOCKER_BEG=430112
ESX_3i_LOCKER_END=${ESX_3i_EXTENDED_END}

ESX_4i_LOCKER_BEG=1257504
ESX_4i_LOCKER_END=${ESX_4i_EXTENDED_END}

ESX_3i_VISOR_LAYOUT="1 8192 1535999 5 0
4 32 8191 4 128
5 8224 106495 6 0
6 106528 204799 6 0
7 204832 430079 252 0
8 430112 1535999 6 0"


TRUE=0
FALSE=1

: "${PARTED_UTIL:=/sbin/partedUtil}"
: "${UPGRADE_LOGFILE:=/var/log/upgrade.log}"
: ${LOG_VERBOSE:=${TRUE}}

PARTITION_TYPE_VFAT=6                   # 0x06
PARTITION_TYPE_VMFS=251                 # 0xfb

DATE_FORMAT="+%Y-%m-%d %H:%M:%S"

Log()
{
   if [ ${LOG_VERBOSE} -eq ${TRUE} ] ; then
      echo "$*" >&2
   fi

   echo "$(date "${DATE_FORMAT}") $*" >> "${UPGRADE_LOGFILE}"
}

Warning()
{
   echo "$(date "${DATE_FORMAT}") Warning: $*" >> "${UPGRADE_LOGFILE}"

   echo "Warning: $*" >&2
}

Panic()
{
   echo "$(date "${DATE_FORMAT}") installation aborted Error: $*" >> "${UPGRADE_LOGFILE}"

   echo "Error: $*" >&2
   exit 1
}


GetHBAFromVolume()
{
   echo $(vmkfstools -P "$1" 2> /dev/null | sed -n '$s/^.\(.\+\):[0-9]$/\1/p')
}

GetHBAPFromVolume()
{
   echo $(vmkfstools -P "$1" 2> /dev/null | sed -n '$s/^.\(.\+:[0-9]\)$/\1/p')
}

GetPartitionFromHBAP()
{
   echo $(echo "${1}" | awk -F: '{ print $NF }')
}

GetVolumeFromHBAP()
{
   local Volume=

   for volume in /vmfs/volumes/* ; do
      local VolumeHBAP=

      if [ -h "${volume}" ] ; then
         continue
      fi

      VolumeHBAP=$(GetHBAPFromVolume "${volume}")

      if [ "${VolumeHBAP}" = "${1}" ] ; then
         Volume="${volume}"
         break
      fi
   done

   echo "${Volume}"
}

GetPrimaryBootVolume()
{
   local BootUUID=$(esxcfg-info -b 2> /dev/null)
   local BootVolume=

   if [ -n "${BootUUID}" ] ; then
      BootVolume="/vmfs/volumes/${BootUUID}"
   fi

   echo ${BootVolume}
}

GetSecondaryBootVolume()
{
   echo $(GetVolumeFromHBAP "$(GetSecondaryBootHBAP)")
}

GetPrimaryBootHBAP()
{
   local BootVolume=$(GetPrimaryBootVolume)
   local BootHBAP=

   if [ -n "${BootVolume}" ] ; then
      BootHBAP=$(GetHBAPFromVolume "${BootVolume}")
   fi

   echo ${BootHBAP}
}

GetSecondaryBootHBAP()
{
   local PrimaryBootBankPartition=$(GetPartitionFromHBAP "$(GetPrimaryBootHBAP)")
   local SecondaryBootHBAP=

   if [ "${PrimaryBootBankPartition}" = "5" ] ; then
      SecondaryBootHBAP="$(GetBootHBA):6"
   else
      SecondaryBootHBAP="$(GetBootHBA):5"
   fi

   echo ${SecondaryBootHBAP}
}

GetBootHBA()
{
   local BootVolume=$(GetPrimaryBootVolume)
   local BootHBA=

   if [ -n "${BootVolume}" ] ; then
      BootHBA=$(GetHBAFromVolume "${BootVolume}")
   fi

   echo ${BootHBA}
}


GetBootDisk()
{
   case "$(vmware -v)" in
      *3i*) echo "/vmfs/devices/disks/$(GetBootHBA):0" ;;
      *ESXi*) echo "/vmfs/devices/disks/$(GetBootHBA)" ;;
      *) Panic "Unsupported Image" ;;
   esac
}

IsThinESX()
{
   local BootUUID=$(esxcfg-info -b 2> /dev/null)

   if [ -n "${BootUUID}" ] ; then
      # note; starts from zero.  First nibble, in big endian, should be '7'
      if [ "$(echo "${BootUUID}" | cut -b 7,7 2> /dev/null)" = "7" ] ; then
         return ${TRUE}
      fi
   fi

   return ${FALSE}
}

IsSwapOn()
{
   vmksystemswap --state | awk '/Enabled/{ exit($NF != "Yes"); }'
}

GetSwapFile()
{
   local SwapFile=$(vmksystemswap --state | awk '/FilePath/{ print $NF }')

   if IsSwapOn ; then
      if [ -z "${SwapFile}" ] ; then
         Panic "Swap enabled, but can not determine file path"
      fi
   fi

   echo ${SwapFile}
}

IsSwapOnScratchPartition()
{
   local SwapNode= DefaultNode=

   if IsSwapOn ; then
      if [ -f "/scratch/uwswap" ]; then
         SwapNode=$(stat -c "%i" "$(GetSwapFile)")
         DefaultNode=$(stat -c "%i" "/scratch/uwswap")

         if [ "${SwapNode}" = "${DefaultNode}" ]; then
            return ${TRUE}
         fi
      fi
   fi

   return ${FALSE}
}


UpgradeGetPartitionInfo()
{
   echo $("${PARTED_UTIL}" get "$(GetBootDisk)" 2>/dev/null | grep "^$1 ")
}

UpgradeResizePartition()
{
   for i in 0 1 2 4 8 16 32 64 ; do
      sleep ${i}
      "${PARTED_UTIL}" resize "$(GetBootDisk)" ${1} ${2} ${3} 2> /dev/null && {
         Log "Partition resize required $(( ${i} * 2 - 1 )) second backoff"
         return
      }
   done

   Panic "Failed to resize partiton"
}

Upgrade_IsScratchResized()
{
   set -- $(UpgradeGetPartitionInfo ${SCRATCH_PARTITION})

   if [ ${2} -eq ${ESX_4i_SCRATCH_BEG} -a ${3} -eq ${ESX_4i_SCRATCH_END} ] ; then
      return ${TRUE}
   elif [ ${2} -eq ${ESX_3i_SCRATCH_BEG} -a ${3} -eq ${ESX_3i_SCRATCH_END} ] ; then
      return ${FALSE}
   fi

   Panic "Invalid geometry for scratch partition"
}

UpgradeResizeScratch()
{
   if Upgrade_IsScratchResized ; then
      return
   fi

   if IsSwapOnScratchPartition ; then
      Log "Disabling swap (on scratch) with force option"
      vmksystemswap --off -f "$(GetSwapFile)" || Panic "Failed to disable swap"
   fi

   Log "Resizing Partition ${SCRATCH_PARTITION} (scratch)"
   UpgradeResizePartition ${SCRATCH_PARTITION} ${ESX_4i_SCRATCH_BEG} ${ESX_4i_SCRATCH_END}
}

UpgradeIsExtendedPartitionResized()
{
   set -- $(UpgradeGetPartitionInfo ${EXTENDED_PARTITION})

   if [ ${2} -eq ${ESX_4i_EXTENDED_BEG} -a ${3} -eq ${ESX_4i_EXTENDED_END} ] ; then
      return ${TRUE}
   elif [ ${2} -eq ${ESX_3i_EXTENDED_BEG} -a ${3} -eq ${ESX_3i_EXTENDED_END} ] ; then
      return ${FALSE}
   fi

   Panic "Invalid geometry for extended partition"
}

UpgradeResizeExtendedPartition()
{
   if UpgradeIsExtendedPartitionResized ; then
      return
   fi

   if IsThinESX ; then
      UpgradeResizeScratch
   fi

   Log "Resizing Partition ${EXTENDED_PARTITION} (extended)"
   UpgradeResizePartition ${EXTENDED_PARTITION} ${ESX_4i_EXTENDED_BEG} ${ESX_4i_EXTENDED_END}
}

UpgradeIsLockerPartitionResized()
{
   set -- $(UpgradeGetPartitionInfo ${LOCKER_PARTITION})

   if [ ${2} -eq ${ESX_4i_LOCKER_BEG} -a ${3} -eq ${ESX_4i_LOCKER_END} ] ; then
      return ${TRUE}
   elif [ ${2} -eq ${ESX_3i_LOCKER_BEG} -a ${3} -eq ${ESX_3i_LOCKER_END} ] ; then
      return ${FALSE}
   fi

   Panic "Invalid geometry for locker partition"
}

UpgradeResizeLockerPartition()
{
   if UpgradeIsLockerPartitionResized ; then
      return
   fi

   Log "Resizing Partition ${LOCKER_PARTITION} (locker)"
   UpgradeResizePartition ${LOCKER_PARTITION} ${ESX_4i_LOCKER_BEG} ${ESX_4i_LOCKER_END}
}

UpgradeIsCoreDumpPartitionResized()
{
   set -- $(UpgradeGetPartitionInfo ${CORE_DUMP_PARTITION})

   if [ ${2} -eq ${ESX_4i_CORE_DUMP_BEG} -a ${3} -eq ${ESX_4i_CORE_DUMP_END} ] ; then
      return ${TRUE}
   elif [ ${2} -eq ${ESX_3i_CORE_DUMP_BEG} -a ${3} -eq ${ESX_3i_CORE_DUMP_END} ] ; then
      return ${FALSE}
   fi

   Panic "Invalid geometry for core dump partition"
}

UpgradeResizeCoreDumpPartition()
{
   if UpgradeIsCoreDumpPartitionResized ; then
      return
   fi

   Log "Resizing Partition ${CORE_DUMP_PARTITION} (core dump)"
   UpgradeResizePartition ${CORE_DUMP_PARTITION} ${ESX_4i_CORE_DUMP_BEG} ${ESX_4i_CORE_DUMP_END}
}

UpgradeIsBootBankMigrated()
{
   set -- $(UpgradeGetPartitionInfo ${BOOT_BANK_ONE})

   if [ ${2} -eq ${ESX_4i_BOOT_BANK_TWO_BEG} -a ${3} -eq ${ESX_4i_BOOT_BANK_TWO_END} ] ; then
      return ${TRUE}
   fi

   set -- $(UpgradeGetPartitionInfo ${BOOT_BANK_TWO})

   if [ ${2} -eq ${ESX_4i_BOOT_BANK_TWO_BEG} -a ${3} -eq ${ESX_4i_BOOT_BANK_TWO_END} ] ; then
      return ${TRUE}
   fi

   return ${FALSE}
}

UpgradeMigrateBootBank()
{
   local SecondaryBootBankPartition=

   if UpgradeIsBootBankMigrated ; then
      return
   fi

   SecondaryBootBankPartition=$(GetPartitionFromHBAP "$(GetSecondaryBootHBAP)")

   Log "Resizing Partition ${SecondaryBootBankPartition} (secondary boot bank)"
   UpgradeResizePartition ${SecondaryBootBankPartition} ${ESX_4i_BOOT_BANK_TWO_BEG} ${ESX_4i_BOOT_BANK_TWO_END}
}

UpgradeIsBootBankExpanded()
{
   set -- $(UpgradeGetPartitionInfo ${BOOT_BANK_ONE})

   if [ ${2} -eq ${ESX_4i_BOOT_BANK_ONE_BEG} -a ${3} -eq ${ESX_4i_BOOT_BANK_ONE_END} ] ; then
      return ${TRUE}
   fi

   set -- $(UpgradeGetPartitionInfo ${BOOT_BANK_TWO})

   if [ ${2} -eq ${ESX_4i_BOOT_BANK_ONE_BEG} -a ${3} -eq ${ESX_4i_BOOT_BANK_ONE_END} ] ; then
      return ${TRUE}
   fi

   return ${FALSE}
}

UpgradeExtendBootBank()
{
   local SecondaryBootBankPartition=

   if UpgradeIsBootBankExpanded ; then
      return
   fi

   SecondaryBootBankPartition=$(GetPartitionFromHBAP "$(GetSecondaryBootHBAP)")

   Log "Resizing Partition ${SecondaryBootBankPartition} (secondary boot bank)"
   UpgradeResizePartition ${SecondaryBootBankPartition} ${ESX_4i_BOOT_BANK_ONE_BEG} ${ESX_4i_BOOT_BANK_ONE_END}
}

Upgrade_AllowVMFSOverwrite()
{
   Log "Deactivating dump partition"
   esxcfg-dumppart --deactivate || Panic "Failed to deactivate dump partition"

   Log "Setting PreventVMFSOverwrite to 0"
   vsish -e set /config/Disk/intOpts/PreventVMFSOverwrite 0 > /dev/null 2>&1
}

Upgrade_PreventVMFSOverwrite()
{
   Log "Setting PreventVMFSOverwrite to 1"
   vsish -e set /config/Disk/intOpts/PreventVMFSOverwrite 1 > /dev/null 2>&1

   Log "Enabling dump partition"
   esxcfg-dumppart --activate || Warning "Failed to activate dump partition"
}

Upgrade_MigrateBootBank()
{
   local AlternateBootBankPartition=

   Upgrade_AllowVMFSOverwrite || Panic "Unable to get global disk writing permissions"

   UpgradeResizeExtendedPartition
   UpgradeResizeLockerPartition
   UpgradeResizeCoreDumpPartition
   UpgradeMigrateBootBank

   Upgrade_PreventVMFSOverwrite || Warning "Unable to set global disk writing permissions"
}

Upgrade_ExpandBootBank()
{
   Upgrade_AllowVMFSOverwrite || Panic "Unable to get global disk writing permissions"

   UpgradeExtendBootBank

   Upgrade_PreventVMFSOverwrite || Warning "Unable to set global disk writing permissions"
}

GetESXType()
{
   if IsThinESX; then
      echo "7"
   else
      echo "e"
   fi
}

GenerateHash()
{
   local hash= value=
   local randomnum=$(hexdump -n 8 -e '"%u"'  /dev/urandom)

   hash="$(GetESXType)$(echo $(date '+%s')${randomnum} | md5sum - | cut -b2-32)"

   for i in $(seq 1 2 32) ; do
      value="${value}\\x$(echo ${hash} | cut -b${i}-$(( ${i} + 1 )))"
   done

   echo ${value}
}

Upgrade_IsExpandNeeded()
{
   local PrimaryBootBankPartition=$(GetPartitionFromHBAP "$(GetPrimaryBootHBAP)")
   local SecondaryBootBankPartition=$(GetPartitionFromHBAP "$(GetSecondaryBootHBAP)")
   local PrimaryPartitionEnd=

   set -- $(UpgradeGetPartitionInfo ${PrimaryBootBankPartition})
   PrimaryPartitionEnd=${3}

   set -- $(UpgradeGetPartitionInfo ${SecondaryBootBankPartition})

   # NOTE: Have to use the new end alone because the beginging of new bootbank when upgrading
   # from 5 can be different from upgrading from 6
   if [ ${PrimaryPartitionEnd} -eq ${ESX_4i_BOOT_BANK_TWO_END} ] ; then
      if [ ${2} -eq ${ESX_3i_BOOT_BANK_TWO_BEG} -o ${3} -eq ${ESX_3i_BOOT_BANK_ONE_END} ] ; then
         return ${TRUE}
      fi
   fi

   return ${FALSE}
}

GetSecondaryVolumeNumber()
{
   local PrimaryBootBankPartition=$(GetPartitionFromHBAP "$(GetPrimaryBootHBAP)")
   if [ "${PrimaryBootBankPartition}" = "5" ] ; then
       echo "2"
   else
       echo "1"
   fi
}

SuperDD()
{
   local src="${1}"
   local dest="${2}"
   local cmd=$@
   shift 2

   Upgrade_AllowVMFSOverwrite || Log "INFO Unable to get global disk writing permissions"
   local totalDelay=0
   for i in 1 2 4 8 16 32 64 128 ; do
      dd if="${src}" of="${dest}" "$@" 2> /dev/null && {
         Log "dd $cmd required ${totalDelay} seconds backoff"
         Upgrade_PreventVMFSOverwrite || Log "INFO Unable to set global disk writing permissions"
         sync
         local srcmd5=$(cat "${src}" | md5sum)
         local srcsize=$(stat -c %s "${src}")
         local destmd5=$(dd if="${dest}" bs=1 count="${srcsize}" | md5sum)
         [ "${srcmd5}" = "${destmd5}" ] || {
            Log "Failed to verify dd operation"
            Log "INFO Source md5 :${srcmd5}\n dest md5 :${destmd5}"
            return ${FALSE}
         }
         return ${TRUE}
     }
     sleep ${i}
     totalDelay=$(( ${totalDelay} + ${i} ))
   done
   return ${FALSE}
}

Upgrade_UpdateImageUUID()
{
   # update UUID
   Log "Updating UUID on staged vfat image ${1}"
   [ -f "${1}" ] || Panic "tmp image ${1} not found"

   local bankuuid="VMWARE FAT16    $(GenerateHash)"
   echo -ne "${bankuuid}" | dd conv=notrunc of="${1}" seek=512 bs=1 count=32 >/dev/null 2>&1
   [ $? -eq 0 ] || {
      rm -f "${1}"
      Panic "Failed to update UUID"
   }
}

SetupSecondaryBootBank()
{
    local tgtbootbank="${1}"
    local secondaryVolume=

    Log "Setting up symbolic link to secondary bootbank volume"
    # Rescan filesystem
    vmkfstools -V
    secondaryVolume="$(GetSecondaryBootVolume)"
    [ -d "${secondaryVolume}" ] || Panic "No volume is found for secondary bootbank"
    Log "Set symbolic link ${tgtbootbank} to ${secondaryVolume}"
    rm -f -- "${tgtbootbank}"
    ln -s "${secondaryVolume}" "${tgtbootbank}" || Panic "Failed to create symbolic link: ${tgtbootbank}"
}

Upgrade_InitSecondaryBootbank()
{
   local tmpfatimage="${1}"
   local secondaryHBAP="$(GetSecondaryBootHBAP)"

   # update UUID
   Upgrade_UpdateImageUUID "${tmpfatimage}"

   # Update file system label
   Log "Updating label name on staged image"
   local secondaryVolumeNumber="$(GetSecondaryVolumeNumber)"
   echo -ne "${secondaryVolumeNumber}" | dd conv=notrunc of="${tmpfatimage}" seek=53 bs=1 count=1 >/dev/null 2>&1
   [ $? -eq 0 ] || {
      rm -f "${tmpfatimage}"
      Panic "Failed to update label name"
   }

   # Write to altbootbank
   Log "Formating secondary bootbank :${secondaryHBAP}"
   SuperDD "${tmpfatimage}" "/dev/disks/${secondaryHBAP}" "conv=notrunc"
   local ret=$?
   rm -f "${tmpfatimage}"
   [ $ret -eq 0 ] || {
      Panic "Failed to format altbootbank"
   }

   SetupSecondaryBootBank '/altbootbank'
}

IsGreaterOrEqual()
{
# This function almost safely compares large integers.
# return 1 if first argument is larger than or equal to the second.
# return 0 otherwise.

    if [ ${#1} -gt ${#2} ]; then
        return ${TRUE}
    fi

    if [ ${#1} -eq ${#2} ]; then
        if [ ${1} -ge ${2} ]; then
            return ${TRUE}
        fi
    fi

    return ${FALSE}
}

Upgrade_CreateScratch()
{
   IsThinESX || {
      Log "No need to create scratch for embeddedEsx"
      return ${TRUE}
   }

   set -- $(UpgradeGetPartitionInfo ${SCRATCH_PARTITION})
   [ -z "${1}" ] || {
      Log "Scratch is already created"
      return ${TRUE}
   }

   # check bootbank partition layout
   partitiontable=$("${PARTED_UTIL}" get "$(GetBootDisk)" 2>/dev/null | tail -n +2 | sort)
   [ "${partitiontable}" = "${ESX_3i_VISOR_LAYOUT}" ] || {
      Panic "Partition has changed, won't try to created scratch partition."
   }

   # check available space for scratch partition
   set -- $("${PARTED_UTIL}" get "$(GetBootDisk)" 2>/dev/null | head -n 1)
   IsGreaterOrEqual "${4}" "${ESX_3i_SCRATCH_END}" || {
      Panic "There is no enough space to create scratch partition"
   }

   fdiskscript=$(mktemp /tmp/partitionXXXXXXXX)
   [ -f "${fdiskscript}" ] || {
      Panic "failed to create temp file to create scrach partition"
   }

# new, primary, partition number, start, end, type, partion number, vfat
   echo "n
p
2
751
4845
t
2
6
w" > "${fdiskscript}"

   Upgrade_AllowVMFSOverwrite || Panic "Unable to get global disk writing permissions"
   fdisk $(GetBootDisk) < "${fdiskscript}"
   Upgrade_PreventVMFSOverwrite || Warning "Unable to set global disk writing permissions"

   rm -f "${fdiskscript}"

   # final check the scratch partition layout
   set -- $(UpgradeGetPartitionInfo ${SCRATCH_PARTITION})
   [ "${2}" = "${ESX_3i_SCRATCH_BEG}" -a "${3}" = "${ESX_3i_SCRATCH_END}" ] || {
      Panic "Failed to verify scratch partition"
   }
}

#
# copy and verify
#
copy_and_verify() {
   local src=$1
   local dest=$2

   local method='MD5'
   local dgst_1=
   local dgst_2=

   Log "Copying ${src} to ${dest}..."
   # if the source file is not there, no copy is possible
   if [ ! -f "$src" ]; then
      Panic "File to copy ${src} does not exist!"
   fi

   # try to copy
   cp "$src" "$dest"

   # verify the copy
   cmdline="openssl dgst -${method}"
   dgst_1=$(${cmdline} "${src}" 2>/dev/null | cut -d" " -f 2 )
   dgst_2=$(${cmdline} "${dest}" 2>/dev/null | cut -d" " -f 2 )

   # if any of the files don't get a md5 sum string, something is 
   # wrong
   if [ -z "$dgst_1" ]; then
      Panic "Can't obtain digest for ${src}"
   fi

   if [ -z "$dgst_2" ]; then
      Panic "Can't obtain digest for ${dest}"
   fi

   # if they don't match the copy failed.
   if [ "${dgst_1}" != "${dgst_2}" ]; then
      Panic "Copy of ${src} to ${dest} failed on validation"
   fi
}

