#!/bin/ash
#
###################################################
#
# Installer for VMvisor Firmware
#
#
# Copyright 2007-2008 VMware, Inc.  All rights reserved.
# -- VMware Confidential
#
###################################################


BUILD=502767
PACKAGE_FILE=VMware-image.tar.gz
TARGET_CHKCONFIGDB=chkconfig.db
PRODUCT_VERSION="4.1.0"

PACKAGE="VMware Visor Build $BUILD"

TRUE=0
FALSE=1

# Unless otherwise specified, assume the files to be
# installed exist in the current directory
install_src=$PWD

# Useful for testing
ROOTDIR=""

verbose=${FALSE}
offline=${FALSE}
upgrade=${FALSE}

LOG_FILE="/dev/null"

LEVEL_0="DEBUG"
LEVEL_10="INFO"
LEVEL_20="WARNING"
LEVEL_30="ERROR"
LEVEL_40="CRITICAL"

print_stdout () { echo "$@"; }

print_verbose ()
{
   print_log "${LEVEL_0} $@"
}

print_log () {
    DATE_STR=$(date "+%Y-%m-%d %H:%M:%S")
    [ -n $LOG_FILE -a -e $LOG_FILE ] && echo "${DATE_STR} $@" >> "${LOG_FILE}"
    [ ${verbose} -eq ${TRUE} ] && echo "$@" 1>&2
}

update_panic()
{
   [ ${verbose} -eq ${FALSE} ] && echo "$@" 1>&2
   print_log "${LEVEL_40} installation aborted :$@"
   exit 1
}

progress () {
    local total_files=$1
    local curr_file=$2
    local percentage=0

    while read REPLY;
    do
        percentage=$((100 * $curr_file / $total_files))
        print_stdout "Installing: ${percentage}%"
        if [ $percentage -lt 100 ]; then
            progress=$(printf "0.%02d" $percentage)
        else
            progress="1.0"
        fi
        print_log "${LEVEL_10} status: progress=${progress}; messageID=PACKAGES; Installing ${REPLY}"
        curr_file=$(($curr_file + 1))
    done
    percentage=$((100 * $curr_file / $total_files))
    print_stdout "Installing: ${percentage}%"
}

#
# verify by comparing the md5 sums of two files
#
verify_copy() {
    file1=$1
    file2=$2
    method=$3

    # get the md5 sums for these files
    cmdline="openssl dgst -${method}"
    dgst_1=$(${cmdline} ${file1} 2>/dev/null | cut -d" " -f 2 )
    dgst_2=$(${cmdline} ${file2} 2>/dev/null | cut -d" " -f 2 )

    # if any of the files don't get a md5 sum string, something is 
    # wrong
    if [ -z "$dgst_1" ]; then
        update_panic "Error obtaining digest for ${file1}"
    fi

    if [ -z "$dgst_2" ]; then
        update_panic "Error obtaining digest for ${file2}"
    fi

    # if they don't match the copy failed.
    if [ "${dgst_1}" != "${dgst_2}" ]; then
        update_panic "Copy of ${file1} to ${file2} failed on validation"
    fi
}

#
# copy and verify
#
copy_and_verify() {
    src=$1
    dest=$2

    # if the source file is not there, no copy is possible
    if [ ! -f $src ]; then
        update_panic "File to copy ${src} does not exist!"
    fi

    # try to copy
    cp $src $dest

    # verify the copy
    verify_copy $src $dest MD5

}

#
# verify tar extraction is successful by checking file digest
#
install_verify() {
    tgtdir=$1
    dgst_list="${tgtdir}/$2"
    alg=$3

    if [ ! -f "${dgst_list}" ];
    then
        update_panic "Digest checksum file ${dgst_list} does not exist!"
    fi

    while read line;
    do
        dgst_expected=$(echo ${line} | cut -d" " -f 1)
        file_name="${tgtdir}/$(echo ${line} | cut -d" " -f 2)"

        cmdline="openssl dgst -${alg} ${file_name}"
        dgst_ret=$(${cmdline})
        sig=$(echo "${dgst_ret}" | cut "-b1-${#alg}")

        if [ "${sig}" = "${alg}" ]
        then
            dgst_actual=$(echo "${dgst_ret}" | cut -d" " -f 2)
            if [ "${dgst_actual}" != "${dgst_expected}" ]
            then
                update_panic "${file_name} failed on ${alg} digest check; Expected :${dgst_expected}; got ${dgst_actual}"
            fi
        else
            update_panic "Error during ${alg} digest calculation for cmd: ${cmdline}."
        fi
    done < "${dgst_list}"
}

#
# Run install.sh in esx 3.5 environment for upgrade
#
is_upgrade()
{
   case "$(vmware -v)" in
      *3i*) return 0;;
      *)  return 1;;
   esac
}

IsWrapped()
{
   esxcfg-info -e 2>&1 > /dev/null || return ${TRUE}
   return ${FALSE}
}

format_altbootbank()
{
   local secondaryHBAP=$(GetSecondaryBootHBAP)
   print_log "Formating secondary bootbank :${secondaryHBAP}"
   vmkfstools -C vfat "/dev/disks/${secondaryHBAP}" > /dev/null 2>&1
}

#
# Usage:
#     install.sh [--rootdir=<dir>] [--source=<location>] [--off-line]
#                [--log-file=<log>]
#
for optarg in "$@"; do
    case $optarg in
        --rootdir=*)
            ROOTDIR=$(echo "$optarg" | cut -d"=" -f2-)
            ;;
        --source=*)
            src_arg=$(echo "$optarg" | cut -d"=" -f2-)
            ;;
        --log-file=*)
            LOG_FILE=$(echo "$optarg" | cut -d"=" -f2-)
            ;;
        -v)
            verbose=${TRUE}
            ;;
        --off-line)
            offline=${TRUE}
            ;;
        -*)
            ;;
        *)
            ;;
    esac
done

UPGRADE_LOGFILE=${LOG_FILE}

#
# Active and target boot banks
#
curbootbank="/bootbank"
tgtbootbank="/altbootbank"
: ${configdir:=$tgtbootbank}

print_verbose "Installing $PACKAGE on ${tgtbootbank}"

#
# Check for alternate install source
#
if [ -n "${src_arg}" ]; then
    install_src="${src_arg}"
    print_verbose "Installation Source: $src_arg"
fi


#
# Make sure we have our package file
#
if [ ! -f "$install_src/$PACKAGE_FILE" ]; then
    update_panic "Could not find package file $PACKAGE_FILE in ${install_src}"
fi


#
# Check firmware states
#
curstate=$(sed -n 's/^bootstate=//p' $curbootbank/boot.cfg)
if [ -z "$curstate" ]; then curstate=0; fi

if [ $curstate -ne 0 ]; then
    update_panic "The current firmware has not yet been validated. It is unsafe to continue. Aborting."
fi


#
# Get "updated" field from current boot bank
# and increment for target boot bank
#
curserial=$(sed -n 's/^updated=//p' $curbootbank/boot.cfg)
if [ -z "$curserial" ]; then curserial=0; fi
updated=$(($curserial + 1))

# check if running for upgrade
if [ ${offline} -ne ${TRUE} ]; then
   if is_upgrade; then
      upgrade=${TRUE}
   fi
fi


#
# Check target boot bank
#
if [ ! -d "$tgtbootbank" ]; then
   update_panic "Target boot bank cannot be found"
else
   rm -rf ${tgtbootbank}/*
fi

#
# Perform install
#
print_verbose "Unpacking tar file $PACKAGE_FILE to ${tgtbootbank}"
if [ -z $number_of_files ]; then
    number_of_files=$(tar -tzf $PACKAGE_FILE | wc -l)
fi

if [ -z $number_of_installed ];then
    number_of_installed="0"
fi
tar -xvzf $PACKAGE_FILE -C $tgtbootbank | progress $number_of_files $number_of_installed

# Verify installed bits
#
print_log "${LEVEL_10} Verify installed bits..."
install_verify "${tgtbootbank}" "checksum.md5" "MD5"

#
# Copy the license, vpxa.vgz and oem files from the current bank to the target bank.
# For upgrade, DON'T copy old oem and license file.
#
if [ ${upgrade} -eq ${FALSE} ];then
    local filesToCopy="oem.tgz license.tgz vpxa.vgz aam.vgz"
    for fn in ${filesToCopy} ; do
       if [ -f "${curbootbank}/${fn}" ] ; then
          copy_and_verify $curbootbank/${fn} $tgtbootbank/${fn}
       fi
    done
fi

#
# Save state
#
: "${BACKUP_UTIL:=$ROOTDIR/sbin/backup.sh}"
if [ -x "$BACKUP_UTIL" ] && [ ${offline} -eq ${FALSE}  ]; then
    print_log "${LEVEL_10} Saving current state on $tgtbootbank"
    "${BACKUP_UTIL}" 0 "${configdir}" 1>&2
    print_verbose "STATE CHANGE BETWEEN NOW AND REBOOT WILL NOT BE SAVED ON NEW BANK"
else
    # save state for offline mode, embedded
    if [ -f "${curbootbank}/local.tgz" ]; then
        cp "${curbootbank}/local.tgz" "${configdir}/local.tgz";
    fi

    # installable
    if [ -f "${curbootbank}/state.tgz" ]; then
        cp "${curbootbank}/state.tgz" "${configdir}/state.tgz";
    fi
fi

# Prepare base boot.cfg
cp "${curbootbank}/boot.cfg" "${tgtbootbank}/boot-new.cfg"

if is_upgrade ; then
   print_log "${LEVEL_10} Translating configuration files..."
   [ -f "${install_src}/transconf.sh" ] || update_panic "Can't find configuration translation script"
   . "${install_src}/transconf.sh"

   [ -f "${install_src}/UpgradeFunctions.sh" ] ||  update_panic "Can't find upgrade support script"
   .  "${install_src}/UpgradeFunctions.sh"

   #
   # For upgrade, configdir is set by upgrade install script, which is not /altbootbank.
   # This temporary local/state.tgz is reworked by Conf_Translate function and repackaged to
   # /altbootbank
   #
   configfile=${configdir}/local.tgz
   if IsWrapped ; then
      [ -f "${configdir}/state.tgz" ] || update_panic "No state.tgz in config tmp dir"

      if IsThinESX ; then
         configfile=${configdir}/state.tgz
      else
         # NOTE: Dell shipped visor image, which will use wrapped state file,
         # but it is actually not thinESX. We correct it to use local.tgz instead during upgrade.
         tar -C "${configdir}" -xzf "${configdir}/state.tgz" 2>&1 || {
            update_panic "Error untar ${configdir}/state.tgz to ${configdir}: ${?}"
         }
         rm -f "${configdir}/state.tgz"
      fi
   fi

   chkconfigdbfile="${install_src}/${TARGET_CHKCONFIGDB}"
   [ -f "${chkconfigdbfile}" ] || {
      update_panic "Could not find chkconfig DB ${TARGET_CHKCONFIGDB} in ${install_src}"
   }
   Conf_Translate ${configfile} ${tgtbootbank} "${chkconfigdbfile}"
   [ "${tgtbootbank}" = "${configdir}" ] || rm -f "${configfile}"
fi

#
# Modify bootloader config to boot from the new bank
# vmkboot.gz --> b.z
# vmk.gz --> k.z
# sys.vgz --> s.z
# cim.vgz --> c.z
# mod.tgz --> m.z
#
print_verbose "Updating build number and boot target counter"

# Modify 3.5 boot.cfg to new content
sed -i -e "s/kernel\=.*/kernel\=b.z/g" \
       -e '/^bootstate/cbootstate=1' \
       -e "/^updated/cupdated=${updated}" \
       -e "/^build/cbuild=${PRODUCT_VERSION}-${BUILD}" \
       -e "s/^modules\=.*/modules\=k.z --- s.z --- c.z --- oem.tgz --- license.tgz --- m.z/g" \
       $tgtbootbank/boot-new.cfg

if IsThinESX ; then
   if [ -f "${tgtbootbank}/state.tgz" ] ; then
      sed -i -e "/^modules=/ { /--- state.tgz/!s/$/ --- state.tgz/ }" \
      "${tgtbootbank}/boot-new.cfg"
   else
      update_panic "No state.tgz is saved to ${tgtbootbank}"
   fi
fi

# Copy file to boot bank
# Avoid vfat PR 226024 by pointing boot.cfg file entry at new location
mv $tgtbootbank/boot-new.cfg $tgtbootbank/boot.cfg

# Shut down sfcb to prevent potential failure during reboot
# PR 390592 for detail
/etc/init.d/sfcbd-watchdog stop || {
   print_log "${LEVEL_20} sfcbd-watchdog stop exits with ${?}"
}
return 0
