#!/bin/sh
# Copyright 2008-2009 VMware Inc.,

TRUE=0
FALSE=1

TOOLS_PKG="VMware-tools.vib"
PRODUCT_VERSION="4.1.0"

UPGRADE_FUNCTIONS="UpgradeFunctions.sh"
CONF_TRANSLATION="transconf.sh"
PARTEDUTIL="partedUtil"
BOOTBANK_VFAT_IMAGE="vfat_image_250.dd"
SWAP_IMAGE="ThinESXSWP_upgrade.dd"
LOCKER_VFAT_IMAGE="vfat_locker_286.dd"
PKGDB_FILE="/locker/db/packages.xml"
TMP_PKGDB="/bootbank/packages.xml"
TGTBOOTBANK="/altbootbank"
OBDSOLETED_VER="3.5.0"
HOST_PROFILE="/etc/profile"
LOG_FILE="/var/log/vmware/upgrade.log"
AAM_UNINSTALL="/opt/vmware/uninstallers/VMware-aam-ha-uninstall.sh"
VPXA_UNINSTALL="/opt/vmware/uninstallers/VMware-vpxa-uninstall.sh"

verbose=${FALSE}

GetAbsPath()
{
    local parentdir=$(dirname "${1}")
    local absparentdir=$(cd "${parentdir}"; pwd)
    echo "${absparentdir}/$(basename "${1}")"
}

Panic()
{
   echo "Error: $*" >&2
   echo "$(date "+%Y-%m-%d %H:%M:%S") installation aborted Error: $*" >> "${UPGRADE_LOGFILE}"
   exit 1
}

#
# Usage:
#     install.sh --component=<package> [--log-file=<logfile>] [--source=<location>]
#                 [--verbose]
#
for optarg in "$@"; do
   case $optarg in
      --component=*)
         component=$(echo "$optarg" | cut -d"=" -f2-)
         ;;
      --log-file=*)
         LOG_FILE=$(echo "$optarg" | cut -d"=" -f2-)
         ;;
      --source=*)
         src_dir=$(echo "$optarg" | cut -d"=" -f2-)
         ;;
      --verbose)
         verbose=${TRUE}
         ;;
      *)
         ;;
    esac
done

UPGRADE_LOGFILE="${LOG_FILE}"

if [ -z "${src_dir}" ]; then
   src_dir="$(dirname $(GetAbsPath "${0}"))"
fi

if [ ! -f  "${src_dir}/${UPGRADE_FUNCTIONS}" ]; then
   Panic "Hepler script ${src_dir}/${UPGRADE_FUNCTIONS} not found."
fi
. "${src_dir}/${UPGRADE_FUNCTIONS}"


LogLevel()
{
    case "${1}" in
        "0")  echo "DEBUG";;
        "10") echo "INFO";;
        "20") echo "WARNING";;
        "30") echo "ERROR";;
        "40") echo "CRITICAL";;
        *) echo "INFO";;
    esac
}

GetESXVersion()
{
   echo $(vmware -v 2> /dev/null) | sed -n "s:.\+\([0-9]\.[0-9]\.[0-9]\).\+:\1:p"
}

GetESXProduct()
{
   echo $(vmware -v 2> /dev/null) | sed -n "s:\(.\+\) [0-9]\.[0-9]\.[0-9].\+:\1:p"
}

IsRunningCurrentESXi()
{
   [ "$(GetESXVersion)" = "${PRODUCT_VERSION}" -a "$(GetESXProduct)" = "VMware ESXi" ]
}

ResizeAltbootPartition()
{
   if Upgrade_IsExpandNeeded ; then
      Log "$(LogLevel) Expand ESX 3i bootbanks"
      Upgrade_ExpandBootBank
   else
      Log "$(LogLevel) Migrate altbootbank to new location"
      Upgrade_MigrateBootBank
   fi
}

FormatSwapPartition()
{
   if Upgrade_IsScratchResized ; then
      local scratchHBAP=$(GetHBAPFromVolume "/scratch/")
      local scratchpartition=$(GetPartitionFromHBAP ${scratchHBAP})
      if [ "${SCRATCH_PARTITION}" = "${scratchpartition}" ] ; then
         # don't (can't) format scratch partition if it is resized and working.
         # A rollback from a successful thinESX upgrade will be very likely in this situation.
         Log "$(LogLevel) No need to format swap partition; scratch is already working"
      else
         local tmpswapimage=/tmp/tmpswapimage.dd
         copy_and_verify "${1}/${SWAP_IMAGE}" "${tmpswapimage}"
         Upgrade_UpdateImageUUID "${tmpswapimage}"

         Log "$(LogLevel) Formating swap partition :${2}"
         SuperDD "${tmpswapimage}" "/dev/disks/${2}" "conv=notrunc"
         rm -f "${tmpswapimage}"
      fi
   else
      Log "$(LogLevel) Don't need to format swap partition"
   fi
}

PurgeToolsAndViclient()
{
   Log "$(LogLevel 10) Removing packages in /locker/packages/$OBDSOLETED_VER"
   rm -rf "/locker/packages/${OBDSOLETED_VER}"/*

   # XXX: esxupdate is not happy if tools or viclient is removed from DB,
   # Create a faked package entry as a place holder.
   # look for package element and then read 4 lines and substitute
   sed -i -e "/package / { N; N; N; N; \
    s:.*package.*tools.*package.*:\
    <package ID=\"TOOLS-UNINSTALLED\">\n \
     <name>tools</name>\n \
     <version>3.5.0</version>\n \
     <rel>00000</rel>\n \
   </package>:; \
   s:.*package.*viclient.*package.*:\
   <package ID=\"VICLIENT-UNINSTALLED\">\n \
     <name>viclient</name>\n \
     <version>2.5.0</version>\n \
     <rel>00000</rel>\n \
   </package>: }" "${PKGDB_FILE}"
   cp -- "${PKGDB_FILE}" "${TMP_PKGDB}"
}

RestoreLockerDB()
{
   Log "Restoring locker package DB"
   [ -d /locker/db ] || mkdir -p /locker/db
   cp -- "${TMP_PKGDB}" "${PKGDB_FILE}.tmp"
   mv "${PKGDB_FILE}.tmp" "${PKGDB_FILE}"
   [ -f "${PKGDB_FILE}" ] || Log "$(LogLevel 20) packagesDB is not restored"
   rm -f -- "${TMP_PKGDB}"
}

ReserveMemoryForNoswap()
{
    #Rserve more memory for vim group
    if [ ! -x /sbin/configRP ] ; then
        Log "$(LogLevel 20) Can not locate configRP, upgrade might fail."
        return ${FALSE}
    fi
    /sbin/configRP noswap
}

GetScratchHBAP()
{
    echo "$(GetBootHBA):${SCRATCH_PARTITION}"
}

FormatLockerPartition()
{
   Log "$(LogLevel 10) Formating locker partition"
   local LockerHBAP="$(GetBootHBA):${LOCKER_PARTITION}"

   # Update temp image UUID
   local tmplockerimage=/tmp/tmplockerimage.dd
   copy_and_verify "${1}/${LOCKER_VFAT_IMAGE}" "${tmplockerimage}"
   Upgrade_UpdateImageUUID "${tmplockerimage}"

   SuperDD "${tmplockerimage}" "/dev/disks/${LockerHBAP}" "conv=notrunc"
   rm -f "${tmplockerimage}"
}

UninstallAAMAndVPXA()
{
   Log "DEBUG uninstalling AAM or VPXA"
   if [ -x "${AAM_UNINSTALL}" ]; then
      Log "INFO uninstalling AAM"
      "${AAM_UNINSTALL}"
   fi

   if [ -x "${VPXA_UNINSTALL}" ]; then
      Log "INFO unstalling VPXA"
      "${VPXA_UNINSTALL}"
   fi

   # Temporary diectory in ramdisk to install VPXA/AAM
   rm -f /opt
   rm -f /etc/opt
   mkdir -p /tmp/opttmp/opt && ln -s /tmp/opttmp/opt /opt || {
      Log "DEBUG Unable to set up temporary path for /opt"
   }
   mkdir -p /tmp/opttmp/etc/opt && ln -s /tmp/opttmp/etc/opt /etc/opt || {
      Log "DEBUG Unable to set up temporary path for /etc/opt"
   }
}

SetupLockerPartition()
{
   Log "$(LogLevel 10)Setting up symbolic link to locker partition"
   vmkfstools -V
   local LockerVolume="$(GetVolumeFromHBAP "$(GetBootHBA):${LOCKER_PARTITION}")"
   [ -d "${LockerVolume}" ] || Panic "NO locker volume found..."
   rm -f /locker
   ln -s "${LockerVolume}" "/locker" || Panic "Failed to create symbolic link to /locker"
}

InstallFirmware()
{
   local esxproduct=$(GetESXProduct)

   if [ ! "${esxproduct}" = "VMware ESX Server 3i" ]; then
       Panic "Only support upgrading VMware ESX Server 3i"
   fi

   export PARTED_UTIL="${1}/${PARTEDUTIL}"
   chmod +x "${1}/${PARTEDUTIL}"
   [ -x "${PARTED_UTIL}" ] || Panic "${PARTED_UTIL} is not executable"
   PurgeToolsAndViclient
   ReserveMemoryForNoswap

   if Upgrade_IsExpandNeeded ; then
      Log "$(LogLevel) Expanding bootbank"
      Upgrade_ExpandBootBank || Panic "Failed to expand altbootbank"
   else
      UninstallAAMAndVPXA

      # Create scratch if necessary for thinESX
      Upgrade_CreateScratch

      Log "$(LogLevel) Migrating altbootbank to new location"
      Upgrade_MigrateBootBank || Panic "Failed to migrate altbootbank"
      if IsThinESX ; then
         FormatSwapPartition "${1}" "$(GetScratchHBAP)" || Panic "Failed to create file system on swap partition"
      fi

      FormatLockerPartition "${1}" || Panic "Failed to format locker partition"
      SetupLockerPartition
   fi

   RestoreLockerDB

   # Format secondary bootbank and setup symbolic link
   local tmpfatimage=/tmp/tmpbootbankfat.dd
   copy_and_verify "${1}/${BOOTBANK_VFAT_IMAGE}" "${tmpfatimage}"
   Upgrade_InitSecondaryBootbank "${tmpfatimage}"

   chmod +x "${1}/firmware_install.sh"
   #
   # Redirect local/stage.tgz to /tmp/conftmp for modification during upgrade
   #
   [ -d /tmp/conftmp ] && rm -rf /tmp/conftmp
   mkdir -p /tmp/conftmp
   export configdir="/tmp/conftmp"
   "${1}/firmware_install.sh"  "--log-file=${LOG_FILE}"
}

AddRollbackCleanup()
{
   local altversion=
   local configfile=local.tgz
   if [ -f /altbootbank/boot.cfg ]; then
      altversion=$(sed -n "s:^build=\([0-9]\.[0-9]\.[0-9]\)-.\+:\1:p" /altbootbank/boot.cfg)
   fi

   if [ "${altversion}" = "${OBDSOLETED_VER}" ]; then
      Log "INFO attempt to add rollback cleanup to altbootbank"
      if [ ! -f  "${src_dir}/${CONF_TRANSLATION}" ]; then
         Panic "Hepler script ${src_dir}/${CONF_TRANSLATION} not found."
      fi
      . "${src_dir}/${CONF_TRANSLATION}"

      # Dell visor machine will have different state file in altbootbank
      if IsThinESX || [ -f "/altbootbank/state.tgz" -a ! -f "/altbootbank/local.tgz" ] ; then
         configfile=state.tgz
      fi

      if [ -f "/altbootbank/$configfile" ]; then
         tar -tzf "/altbootbank/$configfile" >/dev/null 2>&1 || {
            Log "WARNING /altbootbank/$configfile might be corrupted"
            return ${FALSE}
         }
         Log "INFO adding rollback cleanup script to /altbootbank/${configfile}"
         [ -d /tmp/conftmp ] || mkdir -p /tmp/conftmp
         Conf_RollbackSupport /altbootbank/$configfile /tmp/conftmp
         if [ -f "/tmp/conftmp/$configfile" ]; then
            Log "INFO updating /altbootbank/${configfile}"
            copy_and_verify /tmp/conftmp/$configfile /altbootbank/$configfile.tmp
            tar -tzvf /altbootbank/$configfile.tmp >/dev/null 2>&1
            if [ $? -eq 0 ]; then
               mv -f /altbootbank/$configfile.tmp /altbootbank/$configfile
            else
               Log "WARNING Failed to add rollback cleanup support"
               rm -f  /altbootbank/$configfile.tmp
            fi
         fi
         rm -rf /tmp/conftmp
      fi
   fi
}

InstallVib()
{
   # esxupdate need pytonpath
   [ -f "${HOST_PROFILE}" ] || Panic "${HOST_PROFILE} is missing..."
   . ${HOST_PROFILE}

   IsRunningCurrentESXi || {
      Panic "Need to run on ESXi ${PRODUCT_VERSION}, currently running: $(vmware -v)"
   }
   Log "$(LogLevel) status: progress=0 messageID=PACKAGES; installing ${2} with esxupdate..."

   esxupdate --nosigcheck -b "${1}/${2}" update >> "${LOG_FILE}" 2>&1 || {
      Panic "Install ${2} failed with: ${?}"
   }
}


Log "Start upgrading with options :$@"

cd "${src_dir}"
case "${component}" in
   'firmware')
      InstallFirmware "${src_dir}" "${LOG_FILE}"
      ;;
   'tools')
      AddRollbackCleanup
      InstallVib "${src_dir}" "${TOOLS_PKG}"
      ;;
   *)
      Panic "component must be one of firmware or tools"
      ;;
esac

ret=$?

if [ ${ret} -ne 0 ]; then
    Panic "${component} installation script failed with exit code :${ret}"
fi

Log "$(LogLevel 10) status: progress=1.0 messageID=PACKAGES; Finish copying ${component} files."

