#!/bin/sh
# Copyright 2008 VMware Inc.,

TRUE=0
FALSE=1

PRODUCT_VERSION="4.1.0"

LOG_FILE=/var/log/vmware/upgrade.log
verbose=${FALSE}
LOCKER_DB='/locker/packages/usr/lib/ipkg/status'

Log()
{
   if [ ${verbose} -eq ${TRUE} ] ; then
      echo "$@" >&2
   fi

   echo "$(date "+%Y-%m-%d %H:%M:%S")" "$@" >> "${LOG_FILE}"
}

Warning()
{
   echo "Warning: $@" >&2
}

Panic()
{
   echo "Error: $@" >&2
   Log "CRITICAL $@"
   exit 1
}

GetESXVersion()
{
echo $(vmware -v 2> /dev/null) | sed -n "s:.\+\([0-9]\+\.[0-9]\+\.[0-9]\+\).\+:\1:p"
}

GetESXProduct()
{
   echo $(vmware -v | awk '{ print $2 }')
}

IsRunningCurrentESXi()
{
   [ "$(GetESXVersion)" = "${PRODUCT_VERSION}" -a "$(GetESXProduct)" = "ESXi" ]
}

GetPackageStatus ()
{
   echo $(sed -n -e "/Package: vmware-esx-${1}/,/^$/ s/Status: [a-zA-Z]* [a-zA-Z]* //p" "${LOCKER_DB}")
}

IsPackageInstalled ()
{
   if [ "$(GetPackageStatus "${1}")" = 'installed' ]; then
    Log "Package ${1} is already installed."
    return ${TRUE}
   fi

   Log "Package ${1} is not installed."
   return ${FALSE}
}

VerifyFirmware()
{
   # ESXi 4.0.0 is running
   IsRunningCurrentESXi && {
      Log "Host is running current ESXi :${PRODUCT_VERSION}"
      return ${TRUE}
   }

   Log "Failed to verify firmware version ${PRODUCT_VERSION}"
   return ${FALSE}
}

component=
#
# Usage:
#       verify.sh --component=<firmware|tools>
#
#           Fails with non-zero exit code.
#
for optarg in "$@"; do
    case $optarg in
        --log-file=*)
            LOG_FILE=$(echo "$optarg" | cut -d"=" -f2-)
            ;;
        --component=*)
            component=$(echo "$optarg" | cut -d"=" -f2-)
            ;;
       --verbose)
            verbose=${TRUE}
            ;;
        -*)
            ;;
        *)
            ;;
    esac
done

Log "Verify the installation of ${component}"
case "${component}" in
    "firmware")
       VerifyFirmware
    ;;
    "tools")
       IsPackageInstalled 'tools'
    ;;
    *)
       echo "Must specify --component=[firmware|tools]"
       exit ${FALSE}
    ;;
esac
