/*
 * dump-scli.c --
 *
 *      Operations to generate MIB module stubs for the scli package
 *
 * Copyright (c) 2001 J. Schoenwaelder, Technical University of Braunschweig.
 *
 * See the file "COPYING" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * @(#) $Id: dump-scli.c,v 1.3 2001/11/30 18:31:48 harrie Exp $
 */

/*
 * TODO:
 *	  - generate range/size checking code
 */

#include <config.h>

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_WIN_H
#include "win.h"
#endif

#include "smi.h"
#include "smidump.h"


#include <sys/types.h>
#include <regex.h>


static char *pattern = NULL;
static regex_t _regex, *regex = NULL;



static char *
getStringTime(time_t t)
{
    static char   s[27];
    struct tm	  *tm;

    tm = gmtime(&t);
    sprintf(s, "%04d-%02d-%02d %02d:%02d",
	    tm->tm_year + 1900, tm->tm_mon + 1, tm->tm_mday,
	    tm->tm_hour, tm->tm_min);
    return s;
}



static void
printCommentString(FILE *f, char *s)
{
    int i, len;

    if (s) {
	fprintf(f, " *   ");
	len = strlen(s);
	for (i = 0; i < len; i++) {
	    fputc(s[i], f);
	    if (s[i] == '\n') {
		fprintf(f, " *   ");
	    }
	}
	fputc('\n', f);
    }
}



static void
printTopComment(FILE *f, SmiModule *smiModule)
{
    SmiRevision *smiRevision;
    char *date;

    fprintf(f,
	    "/*	\t\t\t\t\t\t-- DO NOT EDIT --\n"
	    " * This file has been generated by smidump\n"
	    " * version " SMI_VERSION_STRING " for the scli package.\n"
	    " *\n"
	    " * Derived from %s:\n",
	    smiModule->name);

    printCommentString(f, smiModule->description);

    for (smiRevision = smiGetFirstRevision(smiModule);
	 smiRevision;
	 smiRevision = smiGetNextRevision(smiRevision)) {
	date = getStringTime(smiRevision->date);
	fprintf(f,
		" *\n"
		" * Revision %s:\n", date);
	printCommentString(f, smiRevision->description);
    }

    fprintf(f,
	    " *\n * $I" "d$\n"
	    " */\n"
	    "\n");
}



static char*
translate(char *m)
{
    char *s;
    int i;

    s = xstrdup(m);
    for (i = 0; s[i]; i++) {
	if (s[i] == '-') s[i] = '_';
    }
  
    return s;
}



static char*
translateUpper(char *m)
{
    char *s;
    int i;

    s = xstrdup(m);
    for (i = 0; s[i]; i++) {
	if (s[i] == '-') s[i] = '_';
	if (islower((int) s[i])) {
	    s[i] = toupper(s[i]);
	}
    }
  
    return s;
}



static char*
translateLower(char *m)
{
    char *s;
    int i;

    s = xstrdup(m);
    for (i = 0; s[i]; i++) {
	if (s[i] == '-') s[i] = '_';
	if (isupper((int) s[i])) {
	    s[i] = tolower(s[i]);
	}
    }
  
    return s;
}



static char*
translateFileName(char *m)
{
    char *s;
    int i;

    s = xstrdup(m);
    for (i = 0; s[i]; i++) {
	if (s[i] == '_') s[i] = '-';
	if (isupper((int) s[i])) {
	    s[i] = tolower(s[i]);
	}
    }
  
    return s;
}



static FILE *
createFile(char *name, char *suffix)
{
    char *fullname;
    FILE *f;

    fullname = xmalloc(strlen(name) + (suffix ? strlen(suffix) : 0) + 2);
    strcpy(fullname, name);
    if (suffix) {
        strcat(fullname, suffix);
    }
    if (!access(fullname, R_OK)) {
        fprintf(stderr, "smidump: %s already exists\n", fullname);
        xfree(fullname);
        return NULL;
    }
    f = fopen(fullname, "w");
    if (!f) {
        fprintf(stderr, "smidump: cannot open %s for writing: ", fullname);
        perror(NULL);
        xfree(fullname);
        exit(1);
    }
    xfree(fullname);
    return f;
}



static int
isGroup(SmiNode *smiNode)
{
    SmiNode *childNode;

    if (regex) {
	int status;
	status = regexec(regex, smiNode->name, (size_t) 0, NULL, 0);
	if (status != 0) {
	    return 0;
	}
    }

    if (smiNode->nodekind == SMI_NODEKIND_ROW) {
	return 1;
    }
    
    for (childNode = smiGetFirstChildNode(smiNode);
	 childNode;
	 childNode = smiGetNextChildNode(childNode)) {
	if (childNode->nodekind == SMI_NODEKIND_SCALAR) {
	    return 1;
	}
    }

    return 0;
}



static int
isAccessible(SmiNode *groupNode)
{
    SmiNode *smiNode;
    int num = 0;
    
    for (smiNode = smiGetFirstChildNode(groupNode);
	 smiNode;
	 smiNode = smiGetNextChildNode(smiNode)) {
	if ((smiNode->nodekind == SMI_NODEKIND_SCALAR
	     || smiNode->nodekind == SMI_NODEKIND_COLUMN)
	    && (smiNode->access == SMI_ACCESS_READ_ONLY
		|| smiNode->access == SMI_ACCESS_READ_WRITE)) {
	    num++;
	}
    }

    return num;
}



static int
isIndex(SmiNode *groupNode, SmiNode *smiNode)
{
    SmiElement *smiElement;
    
    /*
     * Perhaps this test needs to be more sophisticated if you have
     * really creative cross-table indexing constructions...
     */

    for (smiElement = smiGetFirstElement(groupNode);
	 smiElement; smiElement = smiGetNextElement(smiElement)) {
	if (smiNode == smiGetElementNode(smiElement)) {
	    return 1;
	}
    }

    return 0;
}



static int
isWritable(SmiNode *treeNode, SmiNodekind nodekind)
{
    SmiNode *smiNode;
    
    for (smiNode = smiGetFirstChildNode(treeNode);
	 smiNode;
	 smiNode = smiGetNextChildNode(smiNode)) {
	if (smiNode->nodekind & (nodekind)
	    && (smiNode->access >= SMI_ACCESS_READ_WRITE)) {
	    break;
	}
    }

    return (smiNode != NULL);
}



static unsigned int
getMinSize(SmiType *smiType)
{
    SmiRange *smiRange;
    SmiType  *parentType;
    unsigned int min = 65535, size;
    
    switch (smiType->basetype) {
    case SMI_BASETYPE_BITS:
	return 0;
    case SMI_BASETYPE_OCTETSTRING:
    case SMI_BASETYPE_OBJECTIDENTIFIER:
	size = 0;
	break;
    default:
	return -1;
    }

    for (smiRange = smiGetFirstRange(smiType);
	 smiRange ; smiRange = smiGetNextRange(smiRange)) {
	if (smiRange->minValue.value.unsigned32 < min) {
	    min = smiRange->minValue.value.unsigned32;
	}
    }
    if (min < 65535 && min > size) {
	size = min;
    }

    parentType = smiGetParentType(smiType);
    if (parentType) {
	unsigned int psize = getMinSize(parentType);
	if (psize > size) {
	    size = psize;
	}
    }

    return size;
}



static unsigned int
getMaxSize(SmiType *smiType)
{
    SmiRange *smiRange;
    SmiType  *parentType;
    SmiNamedNumber *nn;
    unsigned int max = 0, size;
    
    switch (smiType->basetype) {
    case SMI_BASETYPE_BITS:
    case SMI_BASETYPE_OCTETSTRING:
	size = 65535;
	break;
    case SMI_BASETYPE_OBJECTIDENTIFIER:
	size = 128;
	break;
    default:
	return -1;
    }

    if (smiType->basetype == SMI_BASETYPE_BITS) {
	for (nn = smiGetFirstNamedNumber(smiType);
	     nn;
	     nn = smiGetNextNamedNumber(nn)) {
	    if (nn->value.value.unsigned32 > max) {
		max = nn->value.value.unsigned32;
	    }
	}
	size = (max / 8) + 1;
	return size;
    }

    for (smiRange = smiGetFirstRange(smiType);
	 smiRange ; smiRange = smiGetNextRange(smiRange)) {
	if (smiRange->maxValue.value.unsigned32 > max) {
	    max = smiRange->maxValue.value.unsigned32;
	}
    }
    if (max > 0 && max < size) {
	size = max;
    }

    parentType = smiGetParentType(smiType);
    if (parentType) {
	unsigned int psize = getMaxSize(parentType);
	if (psize < size) {
	    size = psize;
	}
    }

    return size;
}



static char*
getSnmpType(SmiType *smiType)
{
    struct {
	char *module;
	char *name;
	char *tag;
    } typemap[] = {
	{ "RFC1155-SMI",	"Counter",	"G_SNMP_COUNTER32" },
	{ "SNMPv2-SMI",		"Counter32",	"G_SNMP_COUNTER32" },
	{ "RFC1155-SMI",	"TimeTicks",	"G_SNMP_TIMETICKS" },
	{ "SNMPv2-SMI",		"TimeTicks",	"G_SNMP_TIMETICKS" },
	{ "RFC1155-SMI",	"Opaque",	"G_SNMP_OPAQUE" },
	{ "SNMPv2-SMI",		"Opaque",	"G_SNMP_OPAQUE" },
	{ "RFC1155-SMI",	"IpAddress",	"G_SNMP_IPADDRESS" },
	{ "SNMPv2-SMI",		"IpAddress",	"G_SNMP_IPADDRESS" },
	{ NULL, NULL, NULL }
    };

    SmiBasetype basetype = smiType->basetype;
    
    do {
	int i;
	for (i = 0; typemap[i].name; i++) {
	    if (smiType->name
		&& (strcmp(smiType->name, typemap[i].name) == 0)) {
		return typemap[i].tag;
	    }
	}
    } while ((smiType = smiGetParentType(smiType)));

    switch (basetype) {
    case SMI_BASETYPE_INTEGER32:
    case SMI_BASETYPE_ENUM:
	return "G_SNMP_INTEGER32";
    case SMI_BASETYPE_UNSIGNED32:
	return "G_SNMP_UNSIGNED32";
    case SMI_BASETYPE_INTEGER64:
	return NULL;
    case SMI_BASETYPE_UNSIGNED64:
	return "G_SNMP_COUNTER64";
    case SMI_BASETYPE_OCTETSTRING:
	return "G_SNMP_OCTET_STRING";
    case SMI_BASETYPE_BITS:
	return "G_SNMP_OCTET_STRING";
    case SMI_BASETYPE_OBJECTIDENTIFIER:
	return "G_SNMP_OBJECT_ID";
    case SMI_BASETYPE_FLOAT32:
    case SMI_BASETYPE_FLOAT64:
    case SMI_BASETYPE_FLOAT128:
	return NULL;
    case SMI_BASETYPE_UNKNOWN:
	return NULL;
    }
    return NULL;
}



static void
printIndexParams(FILE *f, SmiNode *smiNode)
{
    SmiElement *smiElement;
    SmiNode *iNode;
    SmiType *iType;
    char *cName;
    unsigned minSize, maxSize;
    
    for (smiElement = smiGetFirstElement(smiNode);
	 smiElement; smiElement = smiGetNextElement(smiElement)) {
	iNode = smiGetElementNode(smiElement);
	if (iNode) {
	    iType = smiGetNodeType(iNode);
	    if (iType) {
		cName = translate(iNode->name);
		switch (iType->basetype) {
		case SMI_BASETYPE_OBJECTIDENTIFIER:
		    maxSize = getMaxSize(iType);
		    minSize = getMinSize(iType);
		    fprintf(f, ", guint32 *%s", cName);
		    if (minSize != maxSize) {
			fprintf(f, ", guint16 _%sLength", cName);
		    }
		    break;
		case SMI_BASETYPE_OCTETSTRING:
		case SMI_BASETYPE_BITS:
		    maxSize = getMaxSize(iType);
		    minSize = getMinSize(iType);
		    fprintf(f, ", guchar *%s", cName);
		    if (minSize != maxSize) {
			fprintf(f, ", guint16 _%sLength", cName);
		    }
		    break;
		case SMI_BASETYPE_ENUM:
		case SMI_BASETYPE_INTEGER32:
		    fprintf(f, ", gint32 %s", cName);
		    break;
		case SMI_BASETYPE_UNSIGNED32:
		    fprintf(f, ", guint32 %s", cName);
		    break;
		default:
		    fprintf(f, "/* ?? %s */", cName);
		    break;
		}
		xfree(cName);
	    }
	}
    }
}



static void
printHeaderEnumerations(FILE *f, SmiModule *smiModule)
{
    SmiNode  *smiNode, *parentNode;
    SmiType  *smiType;
    SmiNamedNumber *nn;
    int      cnt = 0;
    char     *cName, *cModuleName;
    char     *dName, *dModuleName;

    cModuleName = translateLower(smiModule->name);
    dModuleName = translateUpper(smiModule->name);

    for (smiNode = smiGetFirstNode(smiModule,
				   SMI_NODEKIND_SCALAR | SMI_NODEKIND_COLUMN);
	 smiNode;
	 smiNode = smiGetNextNode(smiNode,
				  SMI_NODEKIND_SCALAR | SMI_NODEKIND_COLUMN)) {
	parentNode = smiGetParentNode(smiNode);
	if (! parentNode || ! isGroup(parentNode)) {
	    continue;
	}
	smiType = smiGetNodeType(smiNode);
	if (smiType && smiType->basetype == SMI_BASETYPE_ENUM) {
	    if (! cnt) {
		fprintf(f,
			"/*\n"
			" * Tables to map enumerations to strings and vice versa.\n"
			" */\n"
			"\n");
	    }
	    cnt++;
	    cName = translate(smiNode->name);
	    dName = translateUpper(smiNode->name);
	    for (nn = smiGetFirstNamedNumber(smiType); nn;
		 nn = smiGetNextNamedNumber(nn)) {
		char *dEnum = translateUpper(nn->name);
		fprintf(f, "#define %s_%s_%s\t%d\n",
			dModuleName, dName, dEnum,
			(int) nn->value.value.integer32);
		xfree(dEnum);
	    }
	    fprintf(f, "\nextern GSnmpEnum const %s_enums_%s[];\n\n",
		    cModuleName, cName);
	    xfree(dName);
	    xfree(cName);
	}
    }
    
    if (cnt) {
	fprintf(f, "\n");
    }

    xfree(dModuleName);
    xfree(cModuleName);
}



static void
printHeaderIdentities(FILE *f, SmiModule *smiModule)
{
    SmiNode      *smiNode, *moduleIdentityNode, *parentNode;
    int          cnt = 0;
    unsigned int i;
    char         *dName, *dModuleName;
    char         *cModuleName;

    moduleIdentityNode = smiGetModuleIdentityNode(smiModule);
    
    dModuleName = translateUpper(smiModule->name);

    for (smiNode = smiGetFirstNode(smiModule, SMI_NODEKIND_NODE);
	 smiNode;
	 smiNode = smiGetNextNode(smiNode, SMI_NODEKIND_NODE)) {
	parentNode = smiGetParentNode(smiNode);
	if (! parentNode || ! isGroup(parentNode)) {
	    continue;
	}
	if (smiNode->status == SMI_STATUS_UNKNOWN) {
	    continue;
	}
	if (smiNode == moduleIdentityNode) {
	    continue;
	}
	if (! cnt) {
	    fprintf(f,
		    "/*\n"
		    " * Tables to map identities to strings and vice versa.\n"
		    " */\n"
		    "\n");
	}
	cnt++;
	dName = translateUpper(smiNode->name);
	fprintf(f, "#define %s_%s\t", dModuleName, dName);
	for (i = 0; i < smiNode->oidlen; i++) {
	    fprintf(f, "%s%u", i ? "," : "", smiNode->oid[i]);
	}
	fprintf(f, "\n");
	xfree(dName);
    }
    
    if (cnt) {
	cModuleName = translateLower(smiModule->name);
	fprintf(f,
		"\n"
		"extern GSnmpIdentity const %s_identities[];\n"
		"\n",
		cModuleName);
	xfree(cModuleName);
    }

    xfree(dModuleName);
}



static void
printHeaderTypedefMember(FILE *f, SmiNode *smiNode,
			 SmiType *smiType, int isIndex)
{
    char *cName, *dNodeName, *dModuleName;
    unsigned minSize, maxSize;
    SmiModule *smiModule;

    smiModule = smiGetNodeModule(smiNode);

    cName = translate(smiNode->name);
    dNodeName = translateUpper(smiNode->name);
    dModuleName = translateUpper(smiModule ? smiModule->name : "");
    switch (smiType->basetype) {
    case SMI_BASETYPE_OBJECTIDENTIFIER:
	maxSize = getMaxSize(smiType);
	minSize = getMinSize(smiType);
	if (isIndex && maxSize > 128 - smiNode->oidlen) {
	    maxSize = 128 - smiNode->oidlen;
	}
	if (maxSize == minSize) {
	    fprintf(f,
		    "#define %s_%sLENGTH %u\n",
		    dModuleName, dNodeName, maxSize);
	}
	if (isIndex) {
	    fprintf(f,
		    "    guint32  %s[%u];\n", cName, maxSize);
	} else {
	    fprintf(f,
		    "    guint32  *%s;\n", cName);
	}
	if (maxSize != minSize) {
	    fprintf(f,
		    "    guint16  _%sLength;\t/* (%u..%u) */\n",
		    cName, minSize, maxSize);
	}
	break;
    case SMI_BASETYPE_OCTETSTRING:
    case SMI_BASETYPE_BITS:
	maxSize = getMaxSize(smiType);
	minSize = getMinSize(smiType);
	if (isIndex && maxSize > 128 - smiNode->oidlen) {
	    maxSize = 128 - smiNode->oidlen;
	}
	if (maxSize == minSize) {
	    fprintf(f,
		    "#define %s_%sLENGTH %u\n",
		    dModuleName, dNodeName, maxSize);
	}
	if (isIndex) {
	    fprintf(f,
		    "    guchar   %s[%u];\n", cName, maxSize);
	} else {
	    fprintf(f,
		    "    guchar   *%s;\n", cName);
	}
	if (maxSize != minSize) {
	    fprintf(f,
		    "    guint16  _%sLength;\t/* (%u..%u) */\n",
		    cName, minSize, maxSize);
	}
	break;
    case SMI_BASETYPE_ENUM:
    case SMI_BASETYPE_INTEGER32:
	fprintf(f,
		"    gint32   %s%s;\n", isIndex ? "" : "*", cName);
	break;
    case SMI_BASETYPE_UNSIGNED32:
	fprintf(f,
		"    guint32  %s%s;\n", isIndex ? "" : "*", cName);
	break;
    case SMI_BASETYPE_INTEGER64:
	fprintf(f,
		"    gint64   *%s; \n", cName);
	break;
    case SMI_BASETYPE_UNSIGNED64:
	fprintf(f,
		"    guint64  *%s; \n", cName);
	break;
    default:
	fprintf(f,
		"    /* ?? */  _%s; \n", cName);
	break;
    }
    xfree(cName);
}



static void
printHeaderTypedefIndex(FILE *f, SmiNode *smiNode)
{
    SmiElement *smiElement;
    SmiNode *iNode;
    SmiType *iType;
    
    for (smiElement = smiGetFirstElement(smiNode);
	 smiElement; smiElement = smiGetNextElement(smiElement)) {
	iNode = smiGetElementNode(smiElement);
	if (iNode) {
	    iType = smiGetNodeType(iNode);
	    if (iType) {
		printHeaderTypedefMember(f, iNode, iType, 1);
	    }
	}
    }
}



static void
printHeaderTypedef(FILE *f, SmiModule *smiModule, SmiNode *groupNode)
{
    SmiNode *smiNode, *indexNode;
    SmiType *smiType;
    char    *cModuleName, *dModuleName, *cGroupName, *dGroupName, *dNodeName;
    int     writable = 0, count = 1;

    cModuleName = translateLower(smiModule->name);
    dModuleName = translateUpper(smiModule->name);
    cGroupName = translate(groupNode->name);
    dGroupName = translateUpper(groupNode->name);

    fprintf(f,
	    "/*\n"
	    " * C type definitions for %s::%s.\n"
	    " */\n\n",
	    smiModule->name, groupNode->name);
    
    for (smiNode = smiGetFirstChildNode(groupNode);
	 smiNode;
	 smiNode = smiGetNextChildNode(smiNode)) {
	if (smiNode->nodekind & (SMI_NODEKIND_COLUMN | SMI_NODEKIND_SCALAR)
	    && (smiNode->access >= SMI_ACCESS_READ_ONLY)) {
#if 0
	    if (isIndex(groupNode, smiNode)) {
		continue;
	    }
#endif
	    dNodeName = translateUpper(smiNode->name);
	    fprintf(f, "#define %s_%s\t0x%x \n", dModuleName, dNodeName, count);
	    xfree(dNodeName);
	    count *= 2;
	}	    
    }

    if (count > 1) {
	fprintf(f, "\n");
    }

    fprintf(f, "typedef struct {\n");

    /*
     * print index objects that are not part of the group
     */

    switch (groupNode->indexkind) {
    case SMI_INDEX_INDEX:
    case SMI_INDEX_REORDER:
	printHeaderTypedefIndex(f, groupNode);
	break;
    case SMI_INDEX_EXPAND:	/* TODO: we have to do more work here! */
	break;
    case SMI_INDEX_AUGMENT:
    case SMI_INDEX_SPARSE:
	indexNode = smiGetRelatedNode(groupNode);
	if (indexNode) {
	    printHeaderTypedefIndex(f, indexNode);
	}
	break;
    case SMI_INDEX_UNKNOWN:
	break;
    }

	    
    for (smiNode = smiGetFirstChildNode(groupNode);
	 smiNode;
	 smiNode = smiGetNextChildNode(smiNode)) {
	if (smiNode->nodekind & (SMI_NODEKIND_COLUMN | SMI_NODEKIND_SCALAR)
	    && (smiNode->access >= SMI_ACCESS_READ_ONLY)) {
	    if (isIndex(groupNode, smiNode)) {
		continue;
	    }
	    if (smiNode->access == SMI_ACCESS_READ_WRITE) {
		writable++;
	    }
	    smiType = smiGetNodeType(smiNode);
	    if (! smiType) {
		continue;
	    }
	    printHeaderTypedefMember(f, smiNode, smiType, 0);
	}	    
    }

    fprintf(f, "} %s_%s_t;\n\n", cModuleName, cGroupName);

    if (groupNode->nodekind == SMI_NODEKIND_ROW) {
	char *cTableName;
	SmiNode *tableNode;

	tableNode = smiGetParentNode(groupNode);
	if (tableNode) {
	    cTableName = translate(tableNode->name);
	    fprintf(f, "extern void\n"
		    "%s_get_%s(GSnmpSession *s, %s_%s_t ***%s, gint mask);\n\n",
		    cModuleName, cTableName,
		    cModuleName, cGroupName, cGroupName);
	    fprintf(f, "extern void\n"
		    "%s_free_%s(%s_%s_t **%s);\n\n",
		    cModuleName, cTableName,
		    cModuleName, cGroupName, cGroupName);
	    xfree(cTableName);
	}
    }
    fprintf(f, "extern %s_%s_t *\n"
	    "%s_new_%s(void);\n\n",
	    cModuleName, cGroupName, cModuleName, cGroupName);
    fprintf(f, "extern void\n"
	    "%s_get_%s(GSnmpSession *s, %s_%s_t **%s",
	    cModuleName, cGroupName,
	    cModuleName, cGroupName,
	    cGroupName);
    if (groupNode->nodekind == SMI_NODEKIND_ROW) {
	switch (groupNode->indexkind) {
	case SMI_INDEX_INDEX:
	case SMI_INDEX_REORDER:
	    printIndexParams(f, groupNode);
	    break;
	case SMI_INDEX_EXPAND:	/* TODO: we have to do more work here! */
	    break;
	case SMI_INDEX_AUGMENT:
	case SMI_INDEX_SPARSE:
	    indexNode = smiGetRelatedNode(groupNode);
	    if (indexNode) {
		printIndexParams(f, indexNode);
	    }
	    break;
	case SMI_INDEX_UNKNOWN:
	    break;
	}
    }
    fprintf(f, ", gint mask);\n\n");
    if (writable) {
	fprintf(f, "extern void\n"
		"%s_set_%s(GSnmpSession *s, %s_%s_t *%s, gint mask);\n\n",
		cModuleName, cGroupName,
		cModuleName, cGroupName, cGroupName);
    }
    fprintf(f, "extern void\n"
	    "%s_free_%s(%s_%s_t *%s);\n\n",
	    cModuleName, cGroupName,
	    cModuleName, cGroupName, cGroupName);
	    
    xfree(dGroupName);
    xfree(cGroupName);
    xfree(dModuleName);
    xfree(cModuleName);
}



static void
printHeaderTypedefs(FILE *f, SmiModule *smiModule)
{
    SmiNode   *smiNode;
    int       cnt = 0;
    
    for (smiNode = smiGetFirstNode(smiModule, SMI_NODEKIND_ANY);
	 smiNode;
	 smiNode = smiGetNextNode(smiNode, SMI_NODEKIND_ANY)) {
	if (isGroup(smiNode) && isAccessible(smiNode)) {
	    cnt++;
	    printHeaderTypedef(f, smiModule, smiNode);
	}
    }
    
    if (cnt) {
	fprintf(f, "\n");
    }
}



static void
dumpHeader(SmiModule *smiModule, char *baseName)
{
    char *pModuleName;
    FILE *f;
    
    pModuleName = translateUpper(smiModule->name);

    f = createFile(baseName, ".h");
    if (! f) {
	return;
    }

    printTopComment(f, smiModule);
    
    fprintf(f,
	    "#ifndef _%s_H_\n"
	    "#define _%s_H_\n"
	    "\n"
	    "#include \"g_snmp.h\"\n"
	    "\n",
	    pModuleName, pModuleName);

    printHeaderEnumerations(f, smiModule);
    printHeaderIdentities(f, smiModule);
    printHeaderTypedefs(f, smiModule);

    fprintf(f,
	    "#endif /* _%s_H_ */\n",
	    pModuleName);

    fclose(f);
    xfree(pModuleName);
}



static void
printStubEnumerations(FILE *f, SmiModule *smiModule)
{
    SmiNode   *smiNode, *parentNode;
    SmiType   *smiType;
    SmiNamedNumber *nn;
    char      *cName, *cModuleName;
    char      *dName, *dModuleName;
    int       cnt = 0;
    
    cModuleName = translateLower(smiModule->name);
    dModuleName = translateUpper(smiModule->name);
    
    for (smiNode = smiGetFirstNode(smiModule,
				   SMI_NODEKIND_SCALAR | SMI_NODEKIND_COLUMN);
	 smiNode;
	 smiNode = smiGetNextNode(smiNode,
				  SMI_NODEKIND_SCALAR | SMI_NODEKIND_COLUMN)) {
	parentNode = smiGetParentNode(smiNode);
	if (! parentNode || ! isGroup(parentNode)) {
	    continue;
	}
	smiType = smiGetNodeType(smiNode);
	if (smiType && smiType->basetype == SMI_BASETYPE_ENUM) {
	    cnt++;
	    cName = translate(smiNode->name);
	    dName = translateUpper(smiNode->name);
	    fprintf(f, "GSnmpEnum const %s_enums_%s[] = {\n",
		    cModuleName, cName);
	    for (nn = smiGetFirstNamedNumber(smiType); nn;
		 nn = smiGetNextNamedNumber(nn)) {
		char *dEnum = translateUpper(nn->name);
		fprintf(f, "    { %s_%s_%s,\t\"%s\" },\n",
			dModuleName, dName, dEnum, nn->name);
		xfree(dEnum);
	    }
	    fprintf(f,
		    "    { 0, NULL }\n"
		    "};\n"
		    "\n");
	    xfree(dName);
	    xfree(cName);
	}
    }
    
    if (cnt) {
	fprintf(f, "\n");
    }

    xfree(dModuleName);
    xfree(cModuleName);
}



static void
printStubIdentities(FILE *f, SmiModule *smiModule)
{
    SmiNode   *smiNode, *moduleIdentityNode, *parentNode;
    char      *cName, *cModuleName;
    char      *dName, *dModuleName;
    int       cnt = 0;
    
    moduleIdentityNode = smiGetModuleIdentityNode(smiModule);
    
    cModuleName = translateLower(smiModule->name);
    dModuleName = translateUpper(smiModule->name);
    
    for (smiNode = smiGetFirstNode(smiModule, SMI_NODEKIND_NODE);
	 smiNode;
	 smiNode = smiGetNextNode(smiNode, SMI_NODEKIND_NODE)) {
	parentNode = smiGetParentNode(smiNode);
	if (! parentNode || ! isGroup(parentNode)) {
	    continue;
	}
	if (smiNode->status == SMI_STATUS_UNKNOWN) {
	    continue;
	}
	if (smiNode == moduleIdentityNode) {
	    continue;
	}
	cnt++;
	cName = translate(smiNode->name);
	dName = translateUpper(smiNode->name);
	fprintf(f,
		"static guint32 const %s[]\n\t= { %s_%s };\n",
		cName, dModuleName, dName);
	xfree(dName);
	xfree(cName);
    }

    if (cnt) {
	fprintf(f,
		"\n"
		"GSnmpIdentity const %s_identities[] = {\n",
		cModuleName);
    
	for (smiNode = smiGetFirstNode(smiModule, SMI_NODEKIND_NODE);
	     smiNode;
	     smiNode = smiGetNextNode(smiNode, SMI_NODEKIND_NODE)) {
	    if (smiNode->status == SMI_STATUS_UNKNOWN) {
		continue;
	    }
	    if (smiNode == moduleIdentityNode) {
		continue;
	    }
	    cName = translate(smiNode->name);
	    fprintf(f, "    { %s,\n"
		    "      sizeof(%s)/sizeof(guint32),\n"
		    "      \"%s\" },\n",
		    cName, cName, smiNode->name);
	    xfree(cName);
	}
	
	fprintf(f,
		"    { 0, 0, NULL }\n"
		"};\n"
		"\n"
		"\n");
    }

    xfree(dModuleName);
    xfree(cModuleName);
}



static void
printAttribute(FILE *f, SmiNode *smiNode, SmiNode *groupNode, int flags)
{
    SmiType *smiType;
    char *snmpType;
    char *dModuleName, *dNodeName;

    smiType = smiGetNodeType(smiNode);
    if (!smiType) {
	return;
    }

    snmpType = getSnmpType(smiType);
    if (!snmpType) {
	return;
    }

    /*
     * Suppress all INDEX objects as if they were not-accessible.
     */

    if (flags) {
	if (isIndex(groupNode, smiNode)) {
	    return;
	}
    }

    dModuleName = translateUpper(smiGetNodeModule(smiNode)->name);
    dNodeName = translateUpper(smiNode->name);

    fprintf(f, "    { %u, %s, %s_%s, \"%s\" },\n",
	    smiNode->oid[smiNode->oidlen-1], snmpType,
	    dModuleName, dNodeName, smiNode->name);

    xfree(dNodeName);
    xfree(dModuleName);
}



static void
printScalarsAttributes(FILE *f, SmiModule *smiModule, SmiNode *groupNode)
{
    SmiNode *smiNode;
    
    for (smiNode = smiGetFirstChildNode(groupNode);
	 smiNode;
	 smiNode = smiGetNextChildNode(smiNode)) {
	printAttribute(f, smiNode, groupNode, 0);
    }
}



static void
printTableAttributes(FILE *f, SmiModule *smiModule, SmiNode *rowNode)
{
    SmiNode *smiNode;
    int     idx, cnt;
    
    for (smiNode = smiGetFirstChildNode(rowNode), idx = 0, cnt = 0;
	 smiNode;
	 smiNode = smiGetNextChildNode(smiNode)) {
	if (isIndex(rowNode, smiNode)) idx++;
	cnt++;
    }

    for (smiNode = smiGetFirstChildNode(rowNode);
	 smiNode;
	 smiNode = smiGetNextChildNode(smiNode)) {
	printAttribute(f, smiNode, rowNode, cnt > idx);
    }
}



static void
printStubUtilities(FILE *f, SmiModule *smiModule)
{
    SmiNode *smiNode;

    smiNode = smiGetFirstNode(smiModule,
			      SMI_NODEKIND_SCALAR | SMI_NODEKIND_COLUMN);
    if (! smiNode) {
	return;
    }

    fprintf(f,
	    "typedef struct {\n"
	    "    guint32 const     subid;\n"
	    "    GSnmpVarBindType  type;\n"
	    "    gint              tag;\n"
	    "    gchar            *label;\n"
	    "} attribute_t;\n"
	    "\n");

    fprintf(f,
	    "static void\n"
	    "add_attributes(GSnmpSession *s, GSList **vbl, guint32 *base, guint8 len,\n"
	    "                guint idx, attribute_t *attributes, gint mask)\n"
	    "{\n"
	    "    int i;\n"
	    "\n"
	    "    for (i = 0; attributes[i].label; i++) {\n"
	    "        if (! mask || (mask & attributes[i].tag)) {\n"
	    "            if (attributes[i].type != G_SNMP_COUNTER64\n"
	    "                || s->version > G_SNMP_V1) {\n"
	    "                base[idx] = attributes[i].subid;\n"
	    "                g_snmp_vbl_add_null(vbl, base, len);\n"
	    "            }\n"
	    "        }\n"
	    "    }\n"
	    "}\n"
	    "\n");

    fprintf(f,
	    "static int\n"
	    "lookup(GSnmpVarBind *vb, guint32 const *base, guint16 const base_len,\n"
	    "	    attribute_t *attributes, guint32 *idx)\n"
	    "{\n"
	    "    int i;\n"
	    "\n"
	    "    if (vb->type == G_SNMP_ENDOFMIBVIEW\n"
	    "        || (vb->type == G_SNMP_NOSUCHOBJECT)\n"
	    "        || (vb->type == G_SNMP_NOSUCHINSTANCE)) {\n"
	    "        return -1;\n"
	    "    }\n"
	    "    \n"
	    "    if (memcmp(vb->id, base, base_len * sizeof(guint32)) != 0) {\n"
	    "	return -2;\n"
	    "    }\n"
	    "\n"
	    "    for (i = 0; attributes[i].label; i++) {\n"
	    "	if (vb->id_len > base_len && vb->id[base_len] == attributes[i].subid) {\n"
	    "	    if (vb->type != attributes[i].type) {\n"
	    "		g_warning(\"type tag 0x%%02x does not match 0x%%02x (%%s)\",\n"
	    "			  vb->type, attributes[i].type, attributes[i].label);\n"
	    "		return -3;\n"
	    "	    }\n"
	    "	    *idx = attributes[i].subid;\n"
	    "	    return 0;\n"
	    "	}\n"
	    "    }\n"
	    "    \n"
	    "    return -4;\n"
	    "}\n"
	    "\n");
}



static void
printStubAttributes(FILE *f, SmiModule *smiModule)
{
    SmiNode *smiNode;
    char    *cName;
    int     i, cnt = 0;
    
    for (smiNode = smiGetFirstNode(smiModule, SMI_NODEKIND_ANY);
	 smiNode;
	 smiNode = smiGetNextNode(smiNode, SMI_NODEKIND_ANY)) {
	if (isGroup(smiNode) && isAccessible(smiNode)) {
	    cnt++;
	    cName = translate(smiNode->name);

	    fprintf(f, "static guint32 const oid_%s[] = {", cName);
	    for (i = 0; i < smiNode->oidlen; i++) {
		fprintf(f, "%s%u", i ? ", " : "", smiNode->oid[i]);
	    }
	    fprintf(f, "};\n\n");
	    
	    fprintf(f,
		    "static attribute_t attr_%s[] = {\n",
		    cName);
	    if (smiNode->nodekind == SMI_NODEKIND_ROW) {
		printTableAttributes(f, smiModule, smiNode);
	    } else {
		printScalarsAttributes(f, smiModule, smiNode);
	    }
	    fprintf(f,
		    "    { 0, 0, 0, NULL }\n"
		    "};\n"
		    "\n");
	    xfree(cName);
	}
    }
    
    if (cnt) {
	fprintf(f, "\n");
    }
}



static void
printUnpackMethod(FILE *f, SmiModule *smiModule, SmiNode *groupNode)
{
    SmiElement *smiElement;
    SmiNode *indexNode = NULL;
    SmiNode *iNode;
    SmiType *iType;
    char    *cModuleName, *cGroupName, *cName;
    unsigned maxSize, minSize;
    int last = 0;

    cModuleName = translateLower(smiModule->name);
    cGroupName = translate(groupNode->name);

    switch (groupNode->indexkind) {
    case SMI_INDEX_INDEX:
    case SMI_INDEX_REORDER:
	indexNode = groupNode;
	break;
    case SMI_INDEX_EXPAND:	/* TODO: we have to do more work here! */
	indexNode = NULL;
	break;
    case SMI_INDEX_AUGMENT:
    case SMI_INDEX_SPARSE:
	indexNode = smiGetRelatedNode(groupNode);
	break;
    case SMI_INDEX_UNKNOWN:
	indexNode = NULL;
	break;
    }

    for (smiElement = smiGetFirstElement(indexNode);
	 smiElement; smiElement = smiGetNextElement(smiElement)) {
	iNode = smiGetElementNode(smiElement);
	if (iNode) {
	    iType = smiGetNodeType(iNode);
	    if (iType && iType->basetype == SMI_BASETYPE_OCTETSTRING) {
		break;
	    }
	}
    }

    fprintf(f,
	    "static int\n"
	    "unpack_%s(GSnmpVarBind *vb, %s_%s_t *%s)\n"
	    "{\n"
	    "    int idx = %u;\n"
	    "%s"
	    "\n",
	    cGroupName, cModuleName, cGroupName, cGroupName,
	    groupNode->oidlen + 1,
	    smiElement ? "    guint16 i, len;\n" : "");

    for (smiElement = smiGetFirstElement(indexNode);
	 smiElement; smiElement = smiGetNextElement(smiElement)) {
	iNode = smiGetElementNode(smiElement);
	last = (smiGetNextElement(smiElement) == NULL);
	if (iNode) {
	    iType = smiGetNodeType(iNode);
	    if (! iType) {
		continue;
	    }
	    cName = translate(iNode->name);
	    switch (iType->basetype) {
	    case SMI_BASETYPE_ENUM:
	    case SMI_BASETYPE_INTEGER32:
		fprintf(f,
			"    if (vb->id_len < idx) return -1;\n"
			"    %s->%s = vb->id[idx++];\n",
			cGroupName, cName);
		break;
	    case SMI_BASETYPE_UNSIGNED32:
		fprintf(f,
			"    if (vb->id_len < idx) return -1;\n"
			"    %s->%s = vb->id[idx++];\n",
			cGroupName, cName);
		break;
	    case SMI_BASETYPE_OCTETSTRING:
		maxSize = getMaxSize(iType);
		minSize = getMinSize(iType);
		if (maxSize > 128 - iNode->oidlen) {
		    maxSize = 128 - iNode->oidlen;
		}
		if (minSize == maxSize) {
		    fprintf(f,
			    "    len = %u;\n"
			    "    if (vb->id_len < idx + len) return -1;\n",
			    minSize);
		} else if (last && indexNode->implied) {
		    fprintf(f,
			    "    if (vb->id_len < idx) return -1;\n"
			    "    len = vb->id_len - idx;\n");
		} else {
		    fprintf(f,
			    "    if (vb->id_len < idx) return -1;\n"
			    "    len = vb->id[idx++];\n");
		}
		if (minSize == maxSize) {
		    fprintf(f, "    if (len != %u) return -1;\n", minSize);
		} else {
		    if (minSize > 0 && maxSize < 65535) {
			fprintf(f,
				"    if (len < %u || len > %u) return -1;\n",
				minSize, maxSize);
		    } else if (minSize > 0 && maxSize == 65535) {
			fprintf(f,
				"    if (len < %u) return -1;\n", minSize);
		    } else if (minSize == 0 && maxSize < 65535) {
			fprintf(f,
				"    if (len > %u) return -1;\n", maxSize);
		    }
		}
		fprintf(f,
			"    if (vb->id_len < idx + len) return -1;\n"
			"    for (i = 0; i < len; i++) {\n"
			"        %s->%s[i] = vb->id[idx++];\n"
			"    }\n",
			cGroupName, cName);
		if (minSize != maxSize) {
		    fprintf(f, 
			    "    %s->_%sLength = len;\n", cGroupName, cName);
		}
		break;
	    case SMI_BASETYPE_OBJECTIDENTIFIER:
		maxSize = getMaxSize(iType);
		minSize = getMinSize(iType);
		if (maxSize > 128 - iNode->oidlen) {
		    maxSize = 128 - iNode->oidlen;
		}
		if (minSize == maxSize) {
		    fprintf(f,
			    "    len = %u;\n"
			    "    if (vb->id_len < idx + len) return -1;\n",
			    minSize);
		} else if (last && indexNode->implied) {
		    fprintf(f,
			    "    if (vb->id_len < idx) return -1;\n"
			    "    len = vb->id_len - idx;\n");
		} else {
		    fprintf(f,
			    "    if (vb->id_len < idx) return -1;\n"
			    "    len = vb->id[idx++];\n"
			    "    if (vb->id_len < idx + len) return -1;\n");
		}
		if (minSize == maxSize) {
		    fprintf(f, "    if (len != %u) return -1;\n", minSize);
		} else {
		    if (minSize > 0 && maxSize < 65535) {
			fprintf(f,
				"    if (len < %u || len > %u) return -1;\n",
				minSize, maxSize);
		    } else if (minSize > 0 && maxSize == 65535) {
			fprintf(f,
				"    if (len < %u) return -1;\n", minSize);
		    } else if (minSize == 0 && maxSize < 65535) {
			fprintf(f,
				"    if (len > %u) return -1;\n", maxSize);
		    }
		}
		fprintf(f,
			"    for (i = 0; i < len; i++) {\n"
			"        %s->%s[i] = vb->id[idx++];\n"
			"    }\n",
			cGroupName, cName);
		if (minSize != maxSize) {
		    fprintf(f,
			    "    %s->_%sLength = len;\n", cGroupName, cName);
		}
		break;
	    default:
		fprintf(f,
			"    /* XXX how to unpack %s->%s ? */\n",
			cGroupName, cName);
		break;
	    }
	    xfree(cName);
	}
    }

    fprintf(f,
	    "    if (vb->id_len > idx) return -1;\n"
	    "    return 0;\n"
	    "}\n\n");

    xfree(cGroupName);
    xfree(cModuleName);
}



static void
printPackMethod(FILE *f, SmiModule *smiModule, SmiNode *groupNode)
{
    SmiElement *smiElement;
    SmiNode *indexNode = NULL;
    SmiNode *iNode;
    SmiType *iType;
    char    *cModuleName, *cGroupName, *cName;
    unsigned maxSize, minSize;
    int last = 0;

    cModuleName = translateLower(smiModule->name);
    cGroupName = translate(groupNode->name);

    switch (groupNode->indexkind) {
    case SMI_INDEX_INDEX:
    case SMI_INDEX_REORDER:
	indexNode = groupNode;
	break;
    case SMI_INDEX_EXPAND:	/* TODO: we have to do more work here! */
	indexNode = NULL;
	break;
    case SMI_INDEX_AUGMENT:
    case SMI_INDEX_SPARSE:
	indexNode = smiGetRelatedNode(groupNode);
	break;
    case SMI_INDEX_UNKNOWN:
	indexNode = NULL;
	break;
    }

    for (smiElement = smiGetFirstElement(indexNode);
	 smiElement; smiElement = smiGetNextElement(smiElement)) {
	iNode = smiGetElementNode(smiElement);
	if (iNode) {
	    iType = smiGetNodeType(iNode);
	    if (iType && iType->basetype == SMI_BASETYPE_OCTETSTRING) {
		break;
	    }
	}
    }

    fprintf(f,
	    "static int\n"
	    "pack_%s(guint32 *base",
	    cGroupName);
    printIndexParams(f, indexNode);
    fprintf(f,
	    ")\n"
	    "{\n"
	    "    int idx = %u;\n"
	    "%s"
	    "\n",
	    groupNode->oidlen + 1,
	    smiElement ? "    guint16 i, len;\n" : "");

    for (smiElement = smiGetFirstElement(indexNode);
	 smiElement; smiElement = smiGetNextElement(smiElement)) {
	iNode = smiGetElementNode(smiElement);
	last = (smiGetNextElement(smiElement) == NULL);
	if (iNode) {
	    iType = smiGetNodeType(iNode);
	    if (! iType) {
		continue;
	    }
	    cName = translate(iNode->name);
	    switch (iType->basetype) {
	    case SMI_BASETYPE_ENUM:
	    case SMI_BASETYPE_INTEGER32:
		fprintf(f,
			"    base[idx++] = %s;\n",
			cName);
		break;
	    case SMI_BASETYPE_UNSIGNED32:
		fprintf(f,
			"    base[idx++] = %s;\n",
			cName);
		break;
	    case SMI_BASETYPE_OCTETSTRING:
		maxSize = getMaxSize(iType);
		minSize = getMinSize(iType);
		if (maxSize > 128 - iNode->oidlen) {
		    maxSize = 128 - iNode->oidlen;
		}
		if (minSize == maxSize) {
		    fprintf(f,
			    "    len = %u;\n",
			    minSize);
		} else if (last && indexNode->implied) {
		    fprintf(f,
			    "    len = _%sLength;\n",
			    cName);
		} else {
		    fprintf(f,
			    "    len = _%sLength;\n"
			    "    base[idx++] = len;\n",
			    cName);
		}
		if (minSize == maxSize) {
		    fprintf(f,
			    "    if (len != %u) return -1;\n",
			    minSize);
		} else {
		    if (minSize > 0 && maxSize < 65535) {
			fprintf(f,
				"    if (len < %u || len > %u) return -1;\n",
				minSize, maxSize);
		    } else if (minSize > 0 && maxSize == 65535) {
			fprintf(f,
				"    if (len < %u) return -1;\n", minSize);
		    } else if (minSize == 0 && maxSize < 65535) {
			fprintf(f,
				"    if (len > %u) return -1;\n", maxSize);
		    }
		}
		fprintf(f,
			"    for (i = 0; i < len; i++) {\n"
			"        base[idx++] = %s[i];\n"
			"        if (idx >= 128) return -1;\n"
			"    }\n",
			cName);
		break;
	    case SMI_BASETYPE_OBJECTIDENTIFIER:
		maxSize = getMaxSize(iType);
		minSize = getMinSize(iType);
		if (maxSize > 128 - iNode->oidlen) {
		    maxSize = 128 - iNode->oidlen;
		}
		if (minSize == maxSize) {
		    fprintf(f,
			    "    len = %u;\n",
			    minSize);
		} else if (last && indexNode->implied) {
		    fprintf(f,
			    "    len = _%sLength;\n",
			    cName);
		} else {
		    fprintf(f,
			    "    len = _%sLength;\n"
			    "    base[idx++] = len;\n",
			    cName);
		}
		if (minSize == maxSize) {
		    fprintf(f,
			    "    if (len != %u) return -1;\n",
			    minSize);
		} else {
		    if (minSize > 0 && maxSize < 65535) {
			fprintf(f,
				"    if (len < %u || len > %u) return -1;\n",
				minSize, maxSize);
		    } else if (minSize > 0 && maxSize == 65535) {
			fprintf(f,
				"    if (len < %u) return -1;\n", minSize);
		    } else if (minSize == 0 && maxSize < 65535) {
			fprintf(f,
				"    if (len > %u) return -1;\n", maxSize);
		    }
		}
		fprintf(f,
			"    for (i = 0; i < len; i++) {\n"
			"        base[idx++] = %s[i];\n"
			"        if (idx >= 128) return -1;\n"
			"    }\n",
			cName);
		break;
	    default:
		fprintf(f,
			"    /* XXX how to pack %s ? */\n", cGroupName);
		break;
	    }
	    xfree(cName);
	}
    }

    fprintf(f,
	    "    return idx;\n"
	    "}\n\n");

    xfree(cGroupName);
    xfree(cModuleName);
}



static void
printVariableAssignement(FILE *f, SmiNode *groupNode)
{
    SmiNode *smiNode;
    SmiType *smiType;
    char    *cGroupName, *cName, *snmpType;
    unsigned maxSize, minSize;

    cGroupName = translate(groupNode->name);

    for (smiNode = smiGetFirstChildNode(groupNode);
	 smiNode;
	 smiNode = smiGetNextChildNode(smiNode)) {
	if (smiNode->nodekind & (SMI_NODEKIND_COLUMN | SMI_NODEKIND_SCALAR)
	    && (smiNode->access == SMI_ACCESS_READ_ONLY
		|| smiNode->access == SMI_ACCESS_READ_WRITE)) {

	    smiType = smiGetNodeType(smiNode);
	    if (!smiType) {
		continue;
	    }

	    if (isIndex(groupNode, smiNode)) {
		continue;
	    }
	    
	    snmpType = getSnmpType(smiType);
	    if (!snmpType) {
		continue;
	    }

	    cName = translate(smiNode->name);

	    fprintf(f,
		    "        case %d:\n", smiNode->oid[smiNode->oidlen-1]);
	    
	    switch (smiType->basetype) {
	    case SMI_BASETYPE_INTEGER32:
	    case SMI_BASETYPE_ENUM:
		fprintf(f,
			"            %s->%s = &(vb->syntax.i32[0]);\n",
			cGroupName, cName);
		break;
	    case SMI_BASETYPE_UNSIGNED32:
		fprintf(f,
			"            %s->%s = &(vb->syntax.ui32[0]);\n",
			cGroupName, cName);
		break;
	    case SMI_BASETYPE_INTEGER64:
		fprintf(f,
			"            %s->%s = &(vb->syntax.i64[0]);\n",
			cGroupName, cName);
		break;
	    case SMI_BASETYPE_UNSIGNED64:
		fprintf(f,
			"            %s->%s = &(vb->syntax.ui64[0]);\n",
			cGroupName, cName);
		break;
	    case SMI_BASETYPE_OCTETSTRING:
	    case SMI_BASETYPE_BITS:
		maxSize = getMaxSize(smiType);
		minSize = getMinSize(smiType);
		if (minSize == maxSize) {
		    fprintf(f,
			    "            if (vb->syntax_len != %u) break;\n", minSize);
		} else {
		    if (minSize > 0 && maxSize < 65535) {
			fprintf(f,
				"            if (vb->syntax_len < %u || vb->syntax_len > %u) break;\n",
				minSize, maxSize);
		    } else if (minSize > 0 && maxSize == 65535) {
			fprintf(f,
				"            if (vb->syntax_len < %u) break;\n", minSize);
		    } else if (minSize == 0 && maxSize < 65535) {
			fprintf(f,
				"            if (vb->syntax_len > %u) break;\n", maxSize);
		    }
		}
		if (minSize != maxSize) {
		    fprintf(f,
			    "            %s->_%sLength = vb->syntax_len;\n",
			    cGroupName, cName);
		}
		fprintf(f,
			"            %s->%s = vb->syntax.uc;\n",
			cGroupName, cName);
		break;
	    case SMI_BASETYPE_OBJECTIDENTIFIER:
		maxSize = getMaxSize(smiType);
		minSize = getMinSize(smiType);
		if (minSize == maxSize) {
		    fprintf(f,
			    "            if (vb->syntax_len != %u) break;\n", minSize);
		} else {
		    if (minSize > 0 && maxSize < 65535) {
			fprintf(f,
				"            if (vb->syntax_len < %u || vb->syntax_len > %u) break;\n",
				minSize, maxSize);
		    } else if (minSize > 0 && maxSize == 65535) {
			fprintf(f,
				"            if (vb->syntax_len < %u) break;\n", minSize);
		    } else if (minSize == 0 && maxSize < 65535) {
			fprintf(f,
				"            if (vb->syntax_len > %u) break;\n", maxSize);
		    }
		}
		if (minSize != maxSize) {
		    fprintf(f,
			    "            %s->_%sLength = vb->syntax_len / sizeof(guint32);\n",
			    cGroupName, cName);
		}
		fprintf(f,
			"            %s->%s = vb->syntax.ui32;\n",
			cGroupName, cName);
		break;
	    default:
		break;
	    }
	    fprintf(f,
		    "            break;\n");
	    
	    xfree(cName);
	}
    }

    xfree(cGroupName);
}



static void
printAssignMethod(FILE *f, SmiModule *smiModule, SmiNode *groupNode)
{
    char *cModuleName, *cGroupName;

    cModuleName = translateLower(smiModule->name);
    cGroupName = translate(groupNode->name);

    if (groupNode->nodekind == SMI_NODEKIND_ROW) {
	printUnpackMethod(f, smiModule, groupNode);
	printPackMethod(f, smiModule, groupNode);
    }
    
    fprintf(f,
	    "static %s_%s_t *\n"
	    "assign_%s(GSList *vbl)\n"
	    "{\n"
	    "    GSList *elem;\n"
	    "    %s_%s_t *%s;\n"
	    "    guint32 idx;\n"
	    "    char *p;\n"
	    "\n",
	    cModuleName, cGroupName, cGroupName,
	    cModuleName, cGroupName, cGroupName);

    fprintf(f,
	    "    %s = %s_new_%s();\n"
	    "    if (! %s) {\n"
	    "        return NULL;\n"
	    "    }\n"
	    "\n",
	    cGroupName, cModuleName, cGroupName, cGroupName);

    fprintf(f,
	    "    p = (char *) %s + sizeof(%s_%s_t);\n"
	    "    * (GSList **) p = vbl;\n"
	    "\n", cGroupName, cModuleName, cGroupName);

    if (groupNode->nodekind == SMI_NODEKIND_ROW) {
	fprintf(f,
		"    if (unpack_%s((GSnmpVarBind *) vbl->data, %s) < 0) {\n"
		"        g_warning(\"illegal %s instance identifier\");\n"
		"        g_free(%s);\n"
		"        return NULL;\n"
		"    }\n\n",
		cGroupName, cGroupName, cGroupName, cGroupName);
    }
    
    fprintf(f,
	    "    for (elem = vbl; elem; elem = g_slist_next(elem)) {\n"
	    "        GSnmpVarBind *vb = (GSnmpVarBind *) elem->data;\n"
	    "\n"
	    "        if (lookup(vb, oid_%s, sizeof(oid_%s)/sizeof(guint32),\n"
	    "                   attr_%s, &idx) < 0) continue;\n"
	    "\n"
	    "        switch (idx) {\n",
	    cGroupName, cGroupName, cGroupName);
    
    printVariableAssignement(f, groupNode);

    fprintf(f,
	    "        };\n"
	    "    }\n"
	    "\n"
	    "    return %s;\n"
	    "}\n"
	    "\n", cGroupName);

    xfree(cGroupName);
    xfree(cModuleName);
}
 



static void
printGetTableMethod(FILE *f, SmiModule *smiModule, SmiNode *rowNode)
{
    SmiNode      *tableNode;
    char         *cModuleName, *cRowName, *cTableName;
    unsigned int i;

    tableNode = smiGetParentNode(rowNode);
    if (! tableNode) {
	return;
    }

    cModuleName = translateLower(smiModule->name);
    cRowName = translate(rowNode->name);
    cTableName = translate(tableNode->name);

    fprintf(f,
	    "void\n"
	    "%s_get_%s(GSnmpSession *s, %s_%s_t ***%s, gint mask)\n"
	    "{\n"
	    "    GSList *in = NULL, *out = NULL;\n",
	    cModuleName, cTableName, cModuleName, cRowName, cRowName);

    fprintf(f,
	    "    GSList *row;\n"
	    "    int i;\n");

    fprintf(f, "    static guint32 base[] = {");
    for (i = 0; i < rowNode->oidlen; i++) {
	fprintf(f, "%u, ", rowNode->oid[i]);
    }
    fprintf(f, "0};\n");

    fprintf(f,
	    "\n"
	    "    *%s = NULL;\n"
	    "\n",
	    cRowName);

    fprintf(f,
	    "    add_attributes(s, &in, base, %u, %u, attr_%s, mask);\n",
	    rowNode->oidlen+1, rowNode->oidlen, cRowName);

    fprintf(f,
	    "\n"
	    "    out = gsnmp_gettable(s, in);\n"
	    "    /* g_snmp_vbl_free(in); */\n"
	    "\n");
    fprintf(f,
	    "    if (out) {\n"
	    "        *%s = (%s_%s_t **) g_malloc0((g_slist_length(out) + 1) * sizeof(%s_%s_t *));\n"
	    "        if (! *%s) {\n"
            "            s->error_status = G_SNMP_ERR_INTERNAL;\n"
	    "            g_snmp_vbl_free(out);\n"
	    "            return;\n"
	    "        }\n"
	    "        for (row = out, i = 0; row; row = g_slist_next(row), i++) {\n"
	    "            (*%s)[i] = assign_%s(row->data);\n"
	    "        }\n"
	    "    }\n"
	    "}\n"
	    "\n",
	    cRowName, cModuleName, cRowName,
	    cModuleName, cRowName, cRowName,
	    cRowName, cRowName);
    
    xfree(cTableName);
    xfree(cRowName);
    xfree(cModuleName);
}



static void
printGetRowMethod(FILE *f, SmiModule *smiModule, SmiNode *rowNode)
{
    SmiNode    *indexNode = NULL, *iNode;
    SmiType    *iType;
    SmiElement *smiElement;
    char       *cModuleName, *cRowName, *cName;
    unsigned   maxSize, minSize;
    
    cModuleName = translateLower(smiModule->name);
    cRowName = translate(rowNode->name);

    switch (rowNode->indexkind) {
    case SMI_INDEX_INDEX:
    case SMI_INDEX_REORDER:
	indexNode = rowNode;
	break;
    case SMI_INDEX_EXPAND:	/* TODO: we have to do more work here! */
	indexNode = NULL;
	break;
    case SMI_INDEX_AUGMENT:
    case SMI_INDEX_SPARSE:
	indexNode = smiGetRelatedNode(rowNode);
	break;
    case SMI_INDEX_UNKNOWN:
	indexNode = NULL;
	break;
    }

    fprintf(f,
	    "void\n"
	    "%s_get_%s(GSnmpSession *s, %s_%s_t **%s",
	    cModuleName, cRowName, cModuleName, cRowName, cRowName);
    if (indexNode) {
	printIndexParams(f, indexNode);
    }
    fprintf(f,
	    ", gint mask)\n"
	    "{\n"
	    "    GSList *in = NULL, *out = NULL;\n");

    fprintf(f,
	    "    guint32 base[128];\n"
	    "    gint8 len;\n"
	    "\n"
	    "    memset(base, 0, sizeof(base));\n"
	    "    memcpy(base, oid_%s, sizeof(oid_%s));\n",
	    cRowName, cRowName);

    fprintf(f,
	    "    len = pack_%s(base",
	    cRowName);
    
    for (smiElement = smiGetFirstElement(indexNode);
	 smiElement; smiElement = smiGetNextElement(smiElement)) {
	iNode = smiGetElementNode(smiElement);
	if (iNode) {
	    iType = smiGetNodeType(iNode);
	    if (! iType) {
		continue;
	    }
	    cName = translate(iNode->name);
	    fprintf(f, ", %s", cName);
	    switch (iType->basetype) {
	    case SMI_BASETYPE_BITS:
	    case SMI_BASETYPE_OCTETSTRING:
	    case SMI_BASETYPE_OBJECTIDENTIFIER:
		maxSize = getMaxSize(iType);
		minSize = getMinSize(iType);
		if (minSize != maxSize) {
		    fprintf(f, ", _%sLength", cName);
		}
		break;
	    default:
		break;
	    }
	    xfree(cName);
	}
    }

    fprintf(f,
	    ");\n"
	    "    if (len < 0) {\n"
	    "        g_warning(\"illegal %s index values\");\n"
	    "        s->error_status = G_SNMP_ERR_INTERNAL;\n"
	    "        return;\n"
	    "    }\n",
	    cRowName);

    fprintf(f,
	    "\n"
	    "    *%s = NULL;\n"
	    "\n",
	    cRowName);

    fprintf(f,
	    "    add_attributes(s, &in, base, len, %u, attr_%s, mask);\n",
	    rowNode->oidlen, cRowName);

    fprintf(f,
	    "\n"
	    "    out = g_snmp_session_sync_get(s, in);\n"
	    "    g_snmp_vbl_free(in);\n"
	    "    if (out) {\n"
	    "        *%s = assign_%s(out);\n"
	    "    }\n"
	    "}\n"
	    "\n", cRowName, cRowName);
    
    xfree(cRowName);
    xfree(cModuleName);
}



static void
printSetRowMethod(FILE *f, SmiModule *smiModule, SmiNode *rowNode)
{
    SmiNode      *indexNode = NULL, *iNode;
    SmiType      *iType;
    SmiElement   *smiElement;
    char         *cModuleName, *cRowName, *cTableName, *cSmiNodeName, *cName;
    SmiNode      *tableNode;
    SmiNode	 *smiNode;
    SmiType	 *smiType;
    unsigned int minSize, maxSize;

    tableNode = smiGetParentNode(rowNode);
    if (! tableNode) {
	return;
    }

    cModuleName = translateLower(smiModule->name);
    cRowName = translate(rowNode->name);
    cTableName = translate(tableNode->name);

    switch (rowNode->indexkind) {
    case SMI_INDEX_INDEX:
    case SMI_INDEX_REORDER:
	indexNode = rowNode;
	break;
    case SMI_INDEX_EXPAND:	/* TODO: we have to do more work here! */
	indexNode = NULL;
	break;
    case SMI_INDEX_AUGMENT:
    case SMI_INDEX_SPARSE:
	indexNode = smiGetRelatedNode(rowNode);
	break;
    case SMI_INDEX_UNKNOWN:
	indexNode = NULL;
	break;
    }

    fprintf(f,
	    "void\n"
	    "%s_set_%s(GSnmpSession *s, %s_%s_t *%s, gint mask)\n"
	    "{\n"
	    "    GSList *in = NULL, *out = NULL;\n",
	    cModuleName, cRowName, cModuleName, cRowName, cRowName);

    fprintf(f,
	    "    guint32 base[128];\n"
	    "    gint8 len;\n"
	    "\n"
	    "    memset(base, 0, sizeof(base));\n"
	    "    memcpy(base, oid_%s, sizeof(oid_%s));\n",
	    cRowName, cRowName);

    fprintf(f,
	    "    len = pack_%s(base", cRowName);

    for (smiElement = smiGetFirstElement(indexNode);
	 smiElement; smiElement = smiGetNextElement(smiElement)) {
	iNode = smiGetElementNode(smiElement);
	if (iNode) {
	    iType = smiGetNodeType(iNode);
	    if (! iType) {
		continue;
	    }
	    cName = translate(iNode->name);
	    fprintf(f, ", %s->%s", cRowName, cName);
	    switch (iType->basetype) {
	    case SMI_BASETYPE_BITS:
	    case SMI_BASETYPE_OCTETSTRING:
	    case SMI_BASETYPE_OBJECTIDENTIFIER:
		maxSize = getMaxSize(iType);
		minSize = getMinSize(iType);
		if (minSize != maxSize) {
		    fprintf(f, ", %s->_%sLength", cRowName, cName);
		}
		break;
	    default:
		break;
	    }
	    xfree(cName);
	}
    }

    fprintf(f,
	    ");\n"
	    "    if (len < 0) {\n"
	    "        g_warning(\"illegal %s index values\");\n"
	    "        s->error_status = G_SNMP_ERR_INTERNAL;\n"
	    "        return;\n"
	    "    }\n"
	    "\n",
	    cRowName);

    for (smiNode = smiGetFirstChildNode(rowNode);
	 smiNode;
	 smiNode = smiGetNextChildNode(smiNode)) {
	if (smiNode->nodekind & (SMI_NODEKIND_COLUMN)
	    && (smiNode->access >= SMI_ACCESS_READ_WRITE)) {

	    smiType = smiGetNodeType(smiNode);
	    if (!smiType) {
		continue;
	    }
	    
	    if (isIndex(rowNode, smiNode)) {
		continue;
	    }
	    
	    cSmiNodeName = translate(smiNode->name);
	    fprintf(f,
		    "    if (%s->%s) {\n"
		    "        base[%u] = %u;\n"
		    "        g_snmp_vbl_add(&in, base, len, %s,\n"
		    "                       %s->%s,\n",
		    cRowName, cSmiNodeName,
		    smiNode->oidlen-1, smiNode->oid[smiNode->oidlen-1],
		    getSnmpType(smiType),
		    cRowName, cSmiNodeName);
	    switch (smiType->basetype) {
	    case SMI_BASETYPE_OCTETSTRING:
	    case SMI_BASETYPE_BITS:
	    case SMI_BASETYPE_OBJECTIDENTIFIER:
		maxSize = getMaxSize(smiType);
		minSize = getMinSize(smiType);
		if (maxSize == minSize) {
		    fprintf(f,
			    "                       %d);\n",
			    maxSize);
		} else {
		    fprintf(f,
			    "                       %s->_%sLength);\n",
			    cRowName, cSmiNodeName);
		}
		break;
	    default:
		fprintf(f,
			"                       0);\n");
		break;
	    }
	    fprintf(f, "    }\n");
		
	    xfree(cSmiNodeName);
	}	    
    }

    fprintf(f,
	    "\n"
	    "    out = g_snmp_session_sync_set(s, in);\n"
	    "    g_snmp_vbl_free(in);\n"
	    "    if (out) {\n"
	    "        g_snmp_vbl_free(out);\n"
	    "    }\n"
	    "}\n"
	    "\n");

    xfree(cTableName);
    xfree(cRowName);
    xfree(cModuleName);
}



static void
printGetScalarsMethod(FILE *f, SmiModule *smiModule, SmiNode *groupNode)
{
    char         *cModuleName, *cGroupName;
    unsigned int i;

    cModuleName = translateLower(smiModule->name);
    cGroupName = translate(groupNode->name);

    fprintf(f,
	    "void\n"
	    "%s_get_%s(GSnmpSession *s, %s_%s_t **%s, gint mask)\n"
	    "{\n"
	    "    GSList *in = NULL, *out = NULL;\n",
	    cModuleName, cGroupName, cModuleName, cGroupName, cGroupName);

    fprintf(f, "    static guint32 base[] = {");
    for (i = 0; i < groupNode->oidlen; i++) {
	fprintf(f, "%u, ", groupNode->oid[i]);
    }
    fprintf(f, "0};\n");

    fprintf(f,
	    "\n"
	    "    *%s = NULL;\n"
	    "\n",
	    cGroupName);

    fprintf(f,
	    "    add_attributes(s, &in, base, %u, %u, attr_%s, mask);\n",
	    groupNode->oidlen + 1, groupNode->oidlen, cGroupName);

    fprintf(f,
	    "\n"
	    "    out = g_snmp_session_sync_getnext(s, in);\n"
	    "    g_snmp_vbl_free(in);\n"
	    "    if (out) {\n"
	    "        *%s = assign_%s(out);\n"
	    "    }\n"
	    "}\n"
	    "\n", cGroupName, cGroupName);
    
    xfree(cGroupName);
    xfree(cModuleName);
}



static void
printSetScalarsMethod(FILE *f, SmiModule *smiModule, SmiNode *groupNode)
{
    char         *cModuleName, *cGroupName, *cSmiNodeName;
    SmiNode	 *smiNode;
    SmiType	 *smiType;
    unsigned int i, minSize, maxSize;

    cModuleName = translateLower(smiModule->name);
    cGroupName = translate(groupNode->name);

    fprintf(f,
	    "void\n"
	    "%s_set_%s(GSnmpSession *s, %s_%s_t *%s, gint mask)\n"
	    "{\n"
	    "    GSList *in = NULL, *out = NULL;\n",
	    cModuleName, cGroupName, cModuleName, cGroupName, cGroupName);

    fprintf(f, "    static guint32 base[] = {");
    for (i = 0; i < groupNode->oidlen; i++) {
	fprintf(f, "%u, ", groupNode->oid[i]);
    }
    fprintf(f, "0, 0};\n\n");

    for (smiNode = smiGetFirstChildNode(groupNode);
	 smiNode;
	 smiNode = smiGetNextChildNode(smiNode)) {
	if (smiNode->nodekind & (SMI_NODEKIND_SCALAR)
	    && (smiNode->access >= SMI_ACCESS_READ_WRITE)) {

	    smiType = smiGetNodeType(smiNode);
	    if (!smiType) {
		continue;
	    }
	    
	    if (isIndex(groupNode, smiNode)) {
		continue;
	    }
	    
	    cSmiNodeName = translate(smiNode->name);
	    fprintf(f,
		    "    if (%s->%s) {\n"
		    "        base[%u] = %u;\n"
		    "        g_snmp_vbl_add(&in, base, sizeof(base)/sizeof(guint32),\n"
		    "                       %s,\n"
		    "                       %s->%s,\n",
		    cGroupName, cSmiNodeName,
		    smiNode->oidlen-1, smiNode->oid[smiNode->oidlen-1],
		    getSnmpType(smiType),
		    cGroupName, cSmiNodeName);
	    switch (smiType->basetype) {
	    case SMI_BASETYPE_OCTETSTRING:
	    case SMI_BASETYPE_BITS:
	    case SMI_BASETYPE_OBJECTIDENTIFIER:
		maxSize = getMaxSize(smiType);
		minSize = getMinSize(smiType);
		if (maxSize == minSize) {
		    fprintf(f,
			    "                       %d);\n",
			    maxSize);
		} else {
		    fprintf(f,
			    "                       %s->_%sLength);\n",
			    cGroupName, cSmiNodeName);
		}
		break;
	    default:
		fprintf(f,
			"                       0);\n");
		break;
	    }
	    fprintf(f, "    }\n");
		
	    xfree(cSmiNodeName);
	}	    
    }

    fprintf(f,
	    "\n"
	    "    out = g_snmp_session_sync_set(s, in);\n"
	    "    g_snmp_vbl_free(in);\n"
	    "    if (out) {\n"
	    "        g_snmp_vbl_free(out);\n"
	    "    }\n"
	    "}\n"
	    "\n");

    xfree(cGroupName);
    xfree(cModuleName);
}



static void
printNewMethod(FILE *f, SmiModule *smiModule, SmiNode *groupNode)
{
    char *cModuleName, *cGroupName;

    cModuleName = translateLower(smiModule->name);
    cGroupName = translate(groupNode->name);

    fprintf(f,
	    "%s_%s_t *\n"
	    "%s_new_%s()\n"
	    "{\n"
	    "    %s_%s_t *%s;\n"
	    "\n",
	    cModuleName, cGroupName,
	    cModuleName, cGroupName,
	    cModuleName, cGroupName, cGroupName);

    fprintf(f,
	    "    %s = (%s_%s_t *) g_malloc0(sizeof(%s_%s_t) + sizeof(gpointer));\n"
	    "    return %s;\n"
	    "}\n"
	    "\n",
	    cGroupName, cModuleName, cGroupName,
	    cModuleName, cGroupName, cGroupName);

    xfree(cGroupName);
    xfree(cModuleName);
}



static void
printFreeTableMethod(FILE *f, SmiModule *smiModule, SmiNode *groupNode)
{
    SmiNode *tableNode;
    char *cModuleName, *cGroupName, *cTableName;

    tableNode = smiGetParentNode(groupNode);
    if (! tableNode) {
	return;
    }

    cModuleName = translateLower(smiModule->name);
    cGroupName = translate(groupNode->name);
    cTableName = translate(tableNode->name);

    fprintf(f,
	    "void\n"
	    "%s_free_%s(%s_%s_t **%s)\n"
	    "{\n"
	    "    int i;\n"
	    "\n",
	    cModuleName, cTableName, cModuleName, cGroupName, cGroupName);

    fprintf(f,	    
	    "    if (%s) {\n"
	    "        for (i = 0; %s[i]; i++) {\n"
	    "            %s_free_%s(%s[i]);\n"
	    "        }\n"
	    "        g_free(%s);\n"
	    "    }\n"
	    "}\n"
	    "\n",
	    cGroupName, cGroupName, cModuleName,
	    cGroupName, cGroupName, cGroupName);

    xfree(cTableName);
    xfree(cGroupName);
    xfree(cModuleName);
}
 



static void
printFreeMethod(FILE *f, SmiModule *smiModule, SmiNode *groupNode)
{
    char *cModuleName, *cGroupName;

    cModuleName = translateLower(smiModule->name);
    cGroupName = translate(groupNode->name);

    fprintf(f,
	    "void\n"
	    "%s_free_%s(%s_%s_t *%s)\n"
	    "{\n"
	    "    GSList *vbl;\n"
	    "    char *p;\n"
	    "\n",
	    cModuleName, cGroupName, cModuleName, cGroupName, cGroupName);

    fprintf(f,	    
	    "    if (%s) {\n"
	    "        p = (char *) %s + sizeof(%s_%s_t);\n"
	    "        vbl = * (GSList **) p;\n"
	    "        g_snmp_vbl_free(vbl);\n"
	    "        g_free(%s);\n"
	    "    }\n"
	    "}\n"
	    "\n",
	    cGroupName, cGroupName, cModuleName, cGroupName, cGroupName);

    xfree(cGroupName);
    xfree(cModuleName);
}
 



static void
printStubMethods(FILE *f, SmiModule *smiModule)
{
    SmiNode   *smiNode;
    int       cnt = 0;
    
    for (smiNode = smiGetFirstNode(smiModule, SMI_NODEKIND_ANY);
	 smiNode;
	 smiNode = smiGetNextNode(smiNode, SMI_NODEKIND_ANY)) {
	if (isGroup(smiNode) && isAccessible(smiNode)) {
	    cnt++;
	    printNewMethod(f, smiModule, smiNode);
	    printAssignMethod(f, smiModule, smiNode);
	    if (smiNode->nodekind == SMI_NODEKIND_ROW) {
		printGetTableMethod(f, smiModule, smiNode);
		printGetRowMethod(f, smiModule, smiNode);
		if (isWritable(smiNode, SMI_NODEKIND_COLUMN)) {
		    printSetRowMethod(f, smiModule, smiNode);
		}
	    } else {
		printGetScalarsMethod(f, smiModule, smiNode);
		if (isWritable(smiNode, SMI_NODEKIND_SCALAR)) {
		    printSetScalarsMethod(f, smiModule, smiNode);
		}
	    }
	    printFreeMethod(f, smiModule, smiNode);
	    if (smiNode->nodekind == SMI_NODEKIND_ROW) {
		printFreeTableMethod(f, smiModule, smiNode);
	    }
	}
    }
    
    if (cnt) {
	fprintf(f, "\n");
    }
}



static void
dumpStubs(SmiModule *smiModule, char *baseName)
{
    FILE *f;

    f = createFile(baseName, ".c");
    if (! f) {
        return;
    }

    printTopComment(f, smiModule);

    fprintf(f,
	    "#include \"%s.h\"\n"
	    "\n",
	    baseName);

    printStubEnumerations(f, smiModule);
    printStubIdentities(f, smiModule);

    printStubUtilities(f, smiModule);
    printStubAttributes(f, smiModule);
    printStubMethods(f, smiModule);
    
    fclose(f);
}



static void
dumpScli(int modc, SmiModule **modv, int flags, char *output)
{
    char	*baseName;
    int		i, code;

    if (pattern) {
	regex = &_regex;
	code = regcomp(regex, pattern, REG_EXTENDED|REG_NOSUB);
	if (code != 0) {
	    char buffer[256];
	    regerror(code, regex, buffer, sizeof(buffer));
	    fprintf(stderr, "smidump: regular expression error: %s\n", buffer);
	    exit(1);
	}
    }

    if (flags & SMIDUMP_FLAG_UNITE) {
	/* not implemented yet */
    } else {
	for (i = 0; i < modc; i++) {
	    baseName = output ? output : translateFileName(modv[i]->name);
	    dumpHeader(modv[i], baseName);
	    dumpStubs(modv[i], baseName);
	    if (! output) xfree(baseName);
	}
    }

    if (regex) {
	regfree(regex);
	regex = NULL;
    }
}



void initScli()
{
    static SmidumpDriverOption opt[] = {
	{ "match", OPT_STRING, &pattern, 0,
	  "produce stubs for groups matching a pattern"},
        { 0, OPT_END, 0, 0 }
    };

    static SmidumpDriver driver = {
	"scli",
	dumpScli,
	0,
	SMIDUMP_DRIVER_CANT_UNITE,
	"ANSI C manager stubs for the scli package",
	opt,
	NULL
    };

    smidumpRegisterDriver(&driver);
}
