// Copyright (c) 2016 VMware, Inc.  All rights reserved. -- VMware Confidential

angular.module('com.vmware.platform.ui').directive('vxPortlets', [
   '$log', 'jsUtils', 'persistenceService', 'navigation', 'defaultUriSchemeUtil', '$rootScope', '$timeout',
   function ($log, jsUtils, persistenceService, navigation, defaultUriSchemeUtil, $rootScope, $timeout) {
      'use strict';
      var KEY_CODE_SPACE = 32;
      return {
         scope: true,
         restrict: 'A',
         replace: true,

         templateUrl: 'resources/ui/components/portlet/portlets.html',

         link: function (scope, element, attributes) {
            var portletOptionsPropName = attributes.vxPortlets;
            var PORTLETS_STATE_CATEGORY = 'portletsStateCategory';
            var PORTLETS_CLASSNAME = 'vx-portlets';
            var PORTLET_CLASSNAME = 'vx-portlet';

            var portletsPersistParams = {
               extensionId: navigation.getRouteFromUrl().extensionId,
               entityType: defaultUriSchemeUtil.getEntityType($rootScope._route.objectId)
            };

            // toggles
            // - collapsed/expanded state
            // - collapsed/expanded classname
            // persists the updated portlets state
            scope.toggleCollapsed = function (portlet, event) {
               if (event.keyCode && event.keyCode === KEY_CODE_SPACE) {
                  event.preventDefault();
               }
               portlet.collapsed = !portlet.collapsed;
               var portletNode = angular.element(event.currentTarget).closest('.' + PORTLET_CLASSNAME);
               portletNode.toggleClass('collapsed');
               persistPortletsState(scope.columns);
               if (portlet.collapsed) {
                  var portletContent = angular.element(
                        '[portlet-id="' + portlet._view.uid + '"] .vx-portlet-content');
                  if (portletContent) {
                     portletContent.remove();
                  }
               }
            };

            // Returns true if link has label
            scope.hasLabel = function (link) {
               return (link.label && link.label.trim().length > 0);
            };

            scope.clickAction = function (event, link) {
               if (angular.isDefined(link.onClick) &&
                  angular.isFunction(link.onClick)) {
                  link.onClick.call(undefined, event, link);
               } else {
                  $log.warn('No onClick action defined for link: ' + link.label);
               }
            };

            scope.getLimits = function (array) {
               return [
                  Math.floor(array.length / 2) + 1,
                  -Math.floor(array.length / 2)
               ];
            };

            scope.sortableOptions = {
               handle: '> .portlet-titlebar',
               cursor: 'move',
               placeholder: PORTLET_CLASSNAME + " placeholder",
               connectWith: ".col",
               start: function (e, ui) {
                  ui.item.css('transform', 'rotate(2deg)');
                  ui.item.css('transform', 'translateX(10px)');
                  ui.item.closest('.' + PORTLETS_CLASSNAME).addClass('ui-sorting');
               },
               stop: function (e, ui) {
                  ui.item.css('transform', 'rotate(0deg)');
                  ui.item.css('transform', 'translateX(0px)');

                  $timeout(function () {
                     ui.item.closest('.' + PORTLETS_CLASSNAME).removeClass('ui-sorting');
                     persistPortletsState(scope.columns);
                  }, 0);
               }
            };

            var getPortletsPersistanceKey = function () {
               return portletsPersistParams.entityType + '-' + portletsPersistParams.extensionId + '-portlets';
            };

            var persistPortletsState = function (cols) {
               var portletsToPersist = [];
               _.each(cols, function (column) {
                  var portletsInColumn = [];
                  portletsToPersist.push(portletsInColumn);
                  _.each(column, function (portlet) {
                     var id = portlet._view.uid;
                     var collapsed = portlet.collapsed;
                     portletsInColumn.push({
                        id: id,
                        collapsed: collapsed
                     });
                  });
               });

               persistenceService.setData(
                   getPortletsPersistanceKey(),
                   angular.toJson(portletsToPersist),
                   {},
                   PORTLETS_STATE_CATEGORY
               );
            };

            scope.$watchCollection(portletOptionsPropName + '.portlets', function () {

               scope.columns = [[], []];

               var portletOptions = scope.$eval(portletOptionsPropName);
               if (portletOptions && portletOptions.portlets &&
                     portletOptions.portlets.length > 0) {

                  var persistenceObj = {
                     "key": 'uid',
                     "op": '=',
                     "value": getPortletsPersistanceKey()
                  };

                  persistenceService.getData(persistenceObj).then(function (data) {
                     var portletUid;

                     var portletsById = {};
                     _.each(portletOptions.portlets, function (portlet, index) {
                        portletUid = portlet._view.uid;
                        if (portletsById[portletUid]) {
                           $log.debug("Duplicate portlet with ID " + portletUid + " found in the scope!");
                        }
                        // Populate default values for collapsed state and column.
                        portlet.collapsed = !portlet._view.activateViewByDefault;
                        portlet.columnIndex = index % 2;
                        portletsById[portletUid] = portlet;
                     });

                     var visiblePortletsById = {};
                     _.each(scope.columns, function (column) {
                        _.each(column, function (visiblePortlet) {
                           portletUid = visiblePortlet._view.uid;
                           if (visiblePortletsById[portletUid]) {
                              $log.warn("Duplicate portlet with ID " + portletUid + " found in the visible portlet columns!");
                           }
                           visiblePortletsById[portletUid] = true;
                        });
                     });

                     if (data[0] && data[0].value) {
                        var persistedLayoutData = angular.fromJson(data[0].value);
                        _.each(persistedLayoutData, function (persistedColumnData, columnIndex) {
                           _.each(persistedColumnData, function (persistedPortletData) {
                              var portlet = portletsById[persistedPortletData.id];
                              // If the persisted data is for a portlet not in
                              // the current scope, it can safely be skipped.
                              if (!portlet) {
                                 return;
                              }
                              // Populate persisted values for collapsed state and column.
                              if (persistedPortletData.collapsed !== undefined
                                    && persistedPortletData.collapsed !== null) {
                                 portlet.collapsed = persistedPortletData.collapsed;
                              }
                              portlet.columnIndex = columnIndex;
                           });
                        });
                     }

                     _.each(portletOptions.portlets, function (portlet) {
                        portletUid = portlet._view.uid;
                        if (!portletsById[portletUid] || visiblePortletsById[portletUid]) {
                           return;
                        }
                        portlet = portletsById[portletUid];
                        delete portletsById[portletUid];
                        scope.columns[portlet.columnIndex].push(portlet);
                        visiblePortletsById[portletUid] = true;

                        updatePortlet(portlet);
                     });
                  });
               }
            });

            /**
             * Updates the portlet properties and state.
             * A hook method that is invoked once for each portlet.
             *
             * @param portlet The portlet whose state will be updated.
             */
            function updatePortlet(portlet) {
               var portletSpanSizeRegExp = /^(\d+):(\d+)$/i;

               portlet.style = {height: "auto"};

               var size = ((portlet._view.contentSpec || {}).metadata || {}).size;
               if (typeof(size) !== 'string' ||
                     !portletSpanSizeRegExp.test(size)) {
                  return;
               }

               var portletSpanSizeGroups = size.match(portletSpanSizeRegExp);
               // Unused for now
               var spanWidth = parseInt(portletSpanSizeGroups[1], 10);
               var spanHeight = parseInt(portletSpanSizeGroups[2], 10);

               if (spanHeight < 1 || 2 < spanHeight) {
                  return;
               }

               // This default value is taken from the upcoming specification
               // for the Clarity card UI elements.
               var BASE_HEIGHT_IN_PIXELS = 156;

               var pixelHeight = BASE_HEIGHT_IN_PIXELS * spanHeight;
               portlet.style.height = pixelHeight + "px";
            }
         }
      };
   }
]);
