/* Copyright (c) 2018 VMware, Inc. All rights reserved. -- VMware Confidential */

module platform {
   import IComponentOptions = angular.IComponentOptions;
   import IComponentController = angular.IComponentController;

   class HtmlBridgePluginModalIFrame implements IComponentOptions {
      public bindings = {
         "url": "<",
         "closeModal": "&"
      };
      public template = `<iframe ng-src="{{$ctrl.url}}" class="sandbox-iframe"></iframe>`;
      public controller = HtmlBridgePluginModalIFrameController;
   }

   class HtmlBridgePluginModalIFrameController implements IComponentController {
      public static $inject = [
         "$element",
         "h5SdkCommonService",
         "telemetryService"
      ];

      public url: string;
      public closeModal: () => void;

      constructor(private $element: any,
            private h5SdkCommonService: H5SdkCommonService,
            private telemetryService: TelemetryService) {
      }

      public $onInit(): void {
         let element: HTMLElement = <HTMLElement> this.$element[0];
         element.classList.add(`plugin-resource(${this.url})`);

         let iframe: HTMLIFrameElement = element.querySelector("iframe");
         iframe.addEventListener("load", (event: Event) => {
            iframe.focus();

            if (!iframe.contentWindow ||
                  this.h5SdkCommonService.isCrossDomainIFrame(iframe)) {
               return;
            }

            iframe.contentWindow.addEventListener("keyup", (evt: KeyboardEvent) => {
               if (evt.keyCode === 27 || evt.key === "Escape") {
                  this.closeModal();
               }
            });

            this.telemetryService
                  .startTrackingWindowEvents(iframe.contentWindow);

            iframe.contentWindow.addEventListener("unload", () => {
               if (iframe.contentWindow &&
                     !this.h5SdkCommonService.isCrossDomainIFrame(iframe)) {
                  this.telemetryService
                        .stopTrackingWindowEvents(iframe.contentWindow);
               }
            });
         });
      }
   }

   angular.module("com.vmware.platform.ui").component(
         "htmlBridgePluginModalIframe",
         new HtmlBridgePluginModalIFrame()
   );
}
