/* Copyright (c) 2017-2019 VMware, Inc. All rights reserved. -- VMware Confidential */

module platform {

   import IDirective = angular.IDirective;
   import IComponentController = angular.IComponentController;
   import IScope = angular.IScope;
   import ILocationService = angular.ILocationService;

   function pluginIframe(): IDirective {
      return {
         restrict: "E",
         template: "",
         scope: {
            viewUrl: "<",
            isModal: "<?",
            viewUrlParams: "<?",
            context: "<?",
            viewId: "<?",
            remotePluginExtensionContext: "<?"
         },
         controller: PluginIframeController,
         controllerAs: "ctrl"
      };
   }

   export interface IdVersionPair {
      id: string;
      version: string;
   }

   export interface RemotePluginExtensionContext {
   }

   export interface RemotePluginSingleInstanceExtensionContext extends
         RemotePluginExtensionContext {
      contextId: string;
      fullyQualifiedPluginInstanceId: string;
      pluginNavigableExtensionIdsPrefix: string;
   }

   export interface RemotePluginSingleInstanceExtensionDescriptor {
      pluginRef: IdVersionPair;
      extensionId: string;
      extensionContext: RemotePluginSingleInstanceExtensionContext;
      url: string;
      instanceLabel: string;
      instanceUrl: string;
      versionLabel: string;
      vCenterNames: string[];
   }

   export interface RemotePluginMultiInstanceExtensionContext extends
         RemotePluginExtensionContext {
      label: string;
      descriptors: RemotePluginSingleInstanceExtensionDescriptor[];
   }

   /**
    * Defines what is in the scope of PluginIframe
    */
   export interface PluginIframeScope extends IScope {
      /**
       * The url of the view.
       */
      viewUrl: string;

      /**
       * The url params of the view.
       */
      viewUrlParams: any;

      /**
       * The data to which the iframe has access.
       */
      context: PluginIframeContext | undefined;

      /**
       * The id of the view (extension id).
       */
      viewId: string | undefined;

      /**
       * Whether the plugin iframe is modal
       */
      isModal: boolean | undefined;

      /**
       * The view's containing plugin info.
       */
      remotePluginExtensionContext: RemotePluginSingleInstanceExtensionContext | undefined;
   }

   /**
    * Describes what is in the "context" of the PluginIframe object
    */
   export interface PluginIframeContext {
      /**
       * If the plugin view is a modal opened by another plugin view using the
       * modal.open API, this function closes the modal and invokes the `onClosed`
       * callback provided by the view that opened the dialog.
       */
      closeModal?: (result?: any) => void;

      /**
       * An optional field to store an array of context objects that can be passed
       * to the modal when it is opened.
       */
      contextObjects?: any[];

      /**
       * Set options to the already opened modal
       */
      setOptions?: (options: DynamicModalConfig) => void;

      /**
       * An optional additional data that can be passed
       * to the modal when it is opened.
       */
      customData?: () => any;
   }

   export interface MouseEventProperties {
      type: String;
      button: number;
      buttons: number;
      screenX: number;
      screenY: number;
      clientX: number;
      clientY: number;
   }

   /**
    * The class is used to create a plugin pinger that pings periodically a given plugin and
    * associates the plugin session with the client session making an HTTP request.
    */
   class PingPlugin {

      private PING_INTERVAL: number = 25 * 60 * 1000; // 25 minutes

      private timerId: number | null = null;

      constructor(private pluginUrlService: any,
            private vxZoneService: any,
            private pingUrl: any) {
      }

      /**
       * Starts a timer and starts pinging the specified plugin url.
       */
      start(): void {
         this.stop();

         this.vxZoneService.runOutsideAngular(() => {
            this.timerId =
                  setInterval(this.onTimerTriggerEvent.bind(this), this.PING_INTERVAL);
         });
      }

      /**
       * Stops the timer and stops pinging the specified plugin url.
       */
      stop(): void {
         this.vxZoneService.runOutsideAngular(() => {
            if (this.timerId !== null) {
               clearInterval(this.timerId);
               this.timerId = null;
            }
         });
      }

      /**
       * Associates a plugin session with the client session
       * once the ping interval passed.
       */
      onTimerTriggerEvent(): void {
         this.pluginUrlService.associatePluginSessionWithClientId(this.pingUrl);
      }
   }

   /**
    * This controller defines how plugin iframe is created and destroyed.
    */
   class PluginIframeController implements IComponentController {
      public static $inject = [
         "$scope",
         "$element",
         "h5SdkApiService",
         "configurationService",
         "pluginUrlService",
         "vxZoneService",
         "iframeNavigationDataCache",
         "$q",
         "$compile",
         "sessionTimeoutService",
         "userSessionService",
         "$log",
         "$location",
         "h5RemoteSdkAdapterService",
         "h5SdkCommonService",
         "telemetryService"
      ];

      private static REMOTE_PLUGIN_PROXIED_URL_REGEX: RegExp =
            new RegExp("^(/plugins/[^/]+)(/[^/]+)(/.*$)", "i");

      private static LOCAL_PLUGIN_IFRAME_TEMPLATE: string =
            "<iframe ng-src=\"{{ctrl.trustedUrl}}\" class=\"sandbox-iframe\"></iframe>";
      private static REMOTE_PLUGIN_IFRAME_TEMPLATE: string =
            "<iframe ng-src=\"{{ctrl.trustedUrl}}\" sandbox=\"allow-scripts allow-same-origin allow-forms\" class=\"sandbox-iframe\"></iframe>";

      private h5SdkInitializer: H5SdkInitializer;
      private htmlClientSdk: H5SdkApi;
      private h5RemoteSdkAdapter: H5RemoteSdkAdapter;

      private shadowRoot: ShadowRoot;
      private iframe: HTMLIFrameElement;

      private isRemotePlugin: boolean;

      private trustedUrl: any;
      private locale: string;
      private pingPlugin: PingPlugin;
      private onIframeUnloadBound = this.onIframeUnload.bind(this);
      private extendUserSessionBound = this.extendUserSession.bind(this);
      private redispatchIframeMouseEventBound = this.redispatchIframeMouseEvent.bind(this);
      private userSessionExtendedTime: number = new Date().getTime();
      private lastSdkReloadPluginExtensionsValue: string = "";
      private EXTEND_USER_SESSION_INTERVAL: number = 5 * 60 * 1000; // 5 minutes
      private static readonly mouse_event_names: String[] = ["mousedown", "mouseup", "mousemove", "click"];

      private get supportsShadowDom() {
         return !!this.$element[0].attachShadow;
      }

      constructor(private $scope: PluginIframeScope,
            private $element: JQuery,
            private h5SdkApiService: H5SdkApiService,
            private configurationService: any,
            private pluginUrlService: any,
            private vxZoneService: any,
            private iframeNavigationDataCache: IframeNavigationDataCache,
            private $q: ng.IQService,
            private $compile: ng.ICompileService,
            private sessionTimeoutService: any,
            private userSessionService: any,
            private $log: any,
            private $location: ILocationService,
            private h5RemoteSdkAdapterService: H5RemoteSdkAdapterService,
            private h5SdkCommonService: H5SdkCommonService,
            private telemetryService: TelemetryService) {

         this.isRemotePlugin = !!this.$scope.remotePluginExtensionContext;
         this.pingPlugin =
               new PingPlugin(pluginUrlService, vxZoneService, this.$scope.viewUrl);

         let sdkFeaturesEnabledPromise = this.configurationService
               .getProperty('experimental.sdk.features.enabled')
               .then((propVal) => this.$q.when(propVal === 'true'));

         let userSessionLocalePromise = this.userSessionService
               .getUserSession()
               .then((userSession: any) => this.$q.when(userSession.locale));

         this.$q.all({
            sdkFeaturesEnabled: sdkFeaturesEnabledPromise,
            userSessionLocale: userSessionLocalePromise
         }).then((result: any) => {
            // session association is not needed for remote plugins since they come
            // from a different domain
            if (!this.isRemotePlugin) {
               return this.pluginUrlService
                     .associatePluginSessionWithClientId(this.$scope.viewUrl)
                     .then(() => result);
            } else {
               return result;
            }

         }).then((result: any) => {
            this.$scope.$on("viewDidLoad", (event, viewId) => {
               if (this.$scope.viewId !== viewId) {
                  return;
               }

               this.replaceNavigationData();
            });

            this.$scope.$on("viewDidUpdate", (event, viewId) => {
               if (this.$scope.viewId !== viewId) {
                  return;
               }

               /*
                * See comment in H5SdkApplicationApi.navigateTo() for more
                * information why this check is performed.
                */
               let route: any = this.$location.search();
               if (!route || !route.sdkReloadPluginExtensions ||
                     route.sdkReloadPluginExtensions === this.lastSdkReloadPluginExtensionsValue) {
                  return;
               }

               this.lastSdkReloadPluginExtensionsValue =
                     route.sdkReloadPluginExtensions;

               this.replaceNavigationData();
               /*
                * When a "viewDidUpdate" event comes we need to reload the
                * plugin view in case the navigation data has changed.
                * See https://jira.eng.vmware.com/browse/VUSP-2156 or
                * https://bugzilla.eng.vmware.com/show_bug.cgi?id=2273357
                */
               this.reloadPluginView();
            });

            this.replaceNavigationData();

            return result;
         }).then((result: any) => {
            let iframeTemplateString: string = this.isRemotePlugin ?
                  PluginIframeController.REMOTE_PLUGIN_IFRAME_TEMPLATE :
                  PluginIframeController.LOCAL_PLUGIN_IFRAME_TEMPLATE;

            this.iframe = <HTMLIFrameElement> this.$compile(iframeTemplateString)(this.$scope).get(0);
            this.locale = result.userSessionLocale;
            this.setIframeUrl();
            this.$scope.$watch('viewUrlParams.vcSelectorSelection', (newValue: any, oldValue: any) => {
               if (newValue === oldValue) {
                  return;
               }

               this.setIframeUrl();
            });

            let classList: DOMTokenList = (<Element> $element[0]).classList;
            if (this.$scope.viewId) {
               classList.add(`plugin-extension(${this.$scope.viewId})`);
            } else {
               let viewUrl: string = this.trustedUrl.valueOf();
               if (this.isRemotePlugin) {
                  if (!PluginIframeController.REMOTE_PLUGIN_PROXIED_URL_REGEX.test(viewUrl)) {
                     $log.warn(
                           `PluginIframeController: viewUrl: ${viewUrl}` +
                           ` for remote plugin extension context id: ${this.$scope.remotePluginExtensionContext.contextId}` +
                           ` is not proxied!`
                     );
                  }

                  viewUrl = viewUrl.replace(
                        PluginIframeController.REMOTE_PLUGIN_PROXIED_URL_REGEX,
                        "$1/<obfuscated>$3"
                  );
               }

               classList.add(`plugin-resource(${viewUrl})`);
            }

            this.h5SdkInitializer =
                  this.h5SdkApiService.getH5SdkApi(this.$scope);
            this.htmlClientSdk = this.h5SdkInitializer.sdk;

            if (this.isRemotePlugin && this.supportsShadowDom) {
               this.iframe.style.display = "block";
               this.iframe.style.boxSizing = "border-box";
               this.iframe.style.width = "100%";
               this.iframe.style.height = "100%";
               this.iframe.style.border = "0";

               this.shadowRoot = this.$element[0].attachShadow({
                  mode: h5.debug ? "open" : "closed"
               });
               this.shadowRoot.appendChild(this.iframe);
            } else {
               this.$element.append(this.iframe);
            }

            return result;
         }).then((result: any) => {
            // Closes the modal dialog on ESC button click inside an iframe
            if (this.$scope.isModal) {
               this.iframe.addEventListener("load", (event: Event): void => {
                  // focus on the iframe, after the iframe is loaded
                  // (also fixing a bug when a modal is opened and the selected
                  // object is behind the modal)
                  this.iframe.focus();
               });
            }

            if (this.isRemotePlugin) {
               this.bootstrapRemotePluginSdk();
            } else {
               this.bootstrapLocalPluginSdk();
            }

            return result;
         });
      }

      public $onInit() {
         // pinging is not needed for remote plugins since they come
         // from a different domain
         if (!this.isRemotePlugin) {
            this.pingPlugin.start();
         }
      }

      public $onDestroy() {
         this.pingPlugin.stop();
         this.pingPlugin = null;

         if (this.isRemotePlugin) {
            if (this.h5RemoteSdkAdapter) {
               this.h5RemoteSdkAdapter.destroy();
               this.h5RemoteSdkAdapter = null;
            }
         } else {
            if (this.iframe) {
               (<any> this.iframe).htmlClientSdk = null;
               if (this.iframe.contentWindow &&
                     !this.h5SdkCommonService.isCrossDomainIFrame(this.iframe)) {
                  this.iframe.contentWindow.removeEventListener("unload", this.onIframeUnloadBound);
                  this.iframe.contentWindow.removeEventListener("mousedown", this.extendUserSessionBound);

                  for (let mouseEventName: String of PluginIframeController.mouse_event_names) {
                     this.iframe.contentWindow.removeEventListener(mouseEventName, this.redispatchIframeMouseEventBound);
                  }

                  this.telemetryService.stopTrackingWindowEvents(this.iframe.contentWindow);
               }
            }
         }

         if (this.h5SdkInitializer) {
            this.h5SdkInitializer.destroy();
            this.h5SdkInitializer = null;
         }
      }

      /**
       * Replace the value returned by getNavigationData() API.
       */
      private replaceNavigationData(): void {
         this.iframeNavigationDataCache.updateNavigationData(this.$scope.viewId);
      }

      /**
       * Handles the unload of the iframe content
       */
      private onIframeUnload(): void {
         if (!this.h5SdkCommonService.isCrossDomainIFrame(this.iframe) &&
               this.iframe.contentWindow) {
            this.telemetryService.stopTrackingWindowEvents(this.iframe.contentWindow);
         }
         if (this.h5SdkInitializer) {
            this.h5SdkInitializer.onIframeUnload();
         }
      }

      /**
       * Extends the user session in the iframe so that it does not expire while
       * in a plugin's view
       */
      private extendUserSession(): void {
         let currentTime = new Date().getTime();
         let timeDifference = Math.abs(currentTime - this.userSessionExtendedTime);
         if (timeDifference > this.EXTEND_USER_SESSION_INTERVAL) {
            this.sessionTimeoutService.resetSessionTimeout();
            this.userSessionExtendedTime = currentTime;
         }
      }

      private redispatchIframeMouseEvent(mouseEventProperties: MouseEventProperties): void {
         let iframeOffset: any = this.iframe.getBoundingClientRect();
         let clientX: number = iframeOffset.left + mouseEventProperties.clientX;
         var clientY: number = iframeOffset.top + mouseEventProperties.clientY;
         let redispatchedEvent: MouseEvent;

         // handle IE's lack of support for new MouseEvent()
         if (typeof MouseEvent !== 'function') {
            redispatchedEvent = document.createEvent('MouseEvents');
            // https://developer.mozilla.org/en-US/docs/Web/API/MouseEvent/initMouseEvent
            redispatchedEvent.initMouseEvent(
               mouseEventProperties.type, // type
               true, // canBubble
               true, // cancelable
               window, // view
               0, // detail
               mouseEventProperties.screenX, // screenX
               mouseEventProperties.screenY, // screenY
               clientX, // clientX
               clientY, // clientY
               false, // ctrlKey
               false, // altKey
               false, // shiftKey
               false, // metaKey
               mouseEventProperties.button, // button
               null // relatedTarget
            );
         } else {
            redispatchedEvent = new MouseEvent(mouseEventProperties.type, {
               bubbles: true,
               cancelable: true,
               button: mouseEventProperties.button,
               buttons: mouseEventProperties.buttons,
               screenX: mouseEventProperties.screenX,
               screenY: mouseEventProperties.screenY,
               clientX: clientX,
               clientY: clientY
            });
         }
         this.$element[0].dispatchEvent(redispatchedEvent);
      }

      private bootstrapLocalPluginSdk() {
         // Bootstrap JavaScript APIs.
         (<any> this.iframe).htmlClientSdk = Object.freeze(this.htmlClientSdk);

         this.iframe.addEventListener("load", (event: Event): void => {
            if (this.h5SdkCommonService.isCrossDomainIFrame(this.iframe)) {
               return;
            }

            if (this.$scope.isModal) {
               this.iframe.contentWindow.addEventListener("keyup", (evt: KeyboardEvent) => {
                  if (evt.keyCode === 27 || evt.key === "Escape") {
                     (<any> this.iframe).htmlClientSdk.modal.close();
                  }
               });
            }

            this.iframe.contentWindow.addEventListener("unload", this.onIframeUnloadBound);
            this.iframe.contentWindow.addEventListener("mousedown", this.extendUserSessionBound);

            for (let mouseEventName: String of PluginIframeController.mouse_event_names) {
               this.iframe.contentWindow.addEventListener(mouseEventName, this.redispatchIframeMouseEventBound);
            }

            this.telemetryService.startTrackingWindowEvents(this.iframe.contentWindow);
         });
      }

      private bootstrapRemotePluginSdk() {
         this.h5RemoteSdkAdapter = this.h5RemoteSdkAdapterService.createH5RemoteSdkAdapter(
               this.iframe, this.htmlClientSdk, this.$scope,
               this.extendUserSessionBound, this.onIframeUnloadBound,
               this.redispatchIframeMouseEventBound
         );
      }

      private reloadPluginView(): void {
         if (!this.iframe || !this.iframe.parentNode) {
            return;
         }

         const iframeParentNode: Node = <Node> this.iframe.parentNode;
         iframeParentNode.removeChild(this.iframe);
         iframeParentNode.appendChild(this.iframe);
      }

      private setIframeUrl(): void {
         if (!this.$scope.viewUrlParams) {
            this.$scope.viewUrlParams = {};
         }
         this.$scope.viewUrlParams.locale = this.locale;

         if (this.isRemotePlugin) {
            this.trustedUrl = this.pluginUrlService.trustPluginUrl(
                  this.$scope.viewUrl
            );
         } else {
            this.trustedUrl = this.pluginUrlService.buildUrl(
                  this.$scope.viewUrl,
                  this.$scope.viewUrlParams
            );
         }
      }
   }

   angular.module("com.vmware.platform.ui").directive("pluginIframe", pluginIframe);
}
