(function () {
   var absolutePathRegExp = /^http(s?):/i;
   var rootRelativePathRegExp = /^\//;

   var initializationState = false;
   var cachedApiData = null;
   var isModal = false;

   var onInitCallbacks = [];

   var lastModalId = 0;
   var modalIdToCloseHandler = {};

   var lastGetSessionInfoRequestId = 0;
   var sessionInfoRequestIdIdToHandler = {};

   var globalRefreshHandler = null;
   var themeChangedCallback = null;

   var telemetryTrackedMouseEventTypes = ["click", "contextmenu"];
   var redispatchedMouseEventNames = ["mousedown", "mouseup", "mousemove", "click"];

   function getElementDirectTextContents(element) {
      var textContents = [];
      for (var i = 0; i < element.childNodes.length; i++) {
         var currentNode = element.childNodes[i];
         if (currentNode.nodeType === Node.TEXT_NODE &&
               typeof currentNode.textContent === "string") {
            textContents.push(currentNode.textContent);
         }
      }
      return textContents;
   }

   function getElementIndex(element) {
      var index = 0;
      var sibling = element.previousElementSibling;
      while (sibling) {
         index++;
         sibling = sibling.previousElementSibling;
      }
      return index;
   }

   function trackTelemetryEvent$MouseEvent(event) {
      var targetNodesDataChain = [];

      var currentNode = event.target;
      while (currentNode) {
         var currentNodeData = {
            "id": null,
            "classList": null,
            "index": null,
            "nodeType": currentNode.nodeType,
            "nodeName": currentNode.nodeName,
            "textContents": null
         };

         if (currentNode.nodeType === Node.ELEMENT_NODE) {
            currentNodeData["id"] = currentNode.id;
            currentNodeData["classList"] = Array.prototype.slice.apply(currentNode.classList);
            currentNodeData["index"] = getElementIndex(currentNode);
            currentNodeData["textContents"] = getElementDirectTextContents(currentNode);
         }

         targetNodesDataChain.push(currentNodeData);

         currentNode = currentNode.parentNode;
      }

      sendTrackTelemetryMouseEvent(event.type, targetNodesDataChain);
   }

   function createApplicationApi() {
      var api = {};

      api.navigateTo = function (navigationOptions) {
         sendAppNavigateToMessage(navigationOptions);
      };

      api.getNavigationData = function () {
         return cachedApiData.app.navigationData;
      };

      api.getClientInfo = function () {
         return cachedApiData.app.clientInfo;
      };

      api.getContextObjects = function () {
         return cachedApiData.app.contextObjects;
      };

      api.getClientLocale = function () {
         return cachedApiData.app.clientLocale;
      };

      api.getTheme = function() {
         return cachedApiData.app.theme;
      };

      api.getSessionInfo = function (callback) {
         var requestId = ++lastGetSessionInfoRequestId;
         sessionInfoRequestIdIdToHandler[requestId] = callback;

         sendAppGetSessionInfo(requestId);
      };

      api.getApiEndpoints = function () {
         return cachedApiData.app.apiEndpoints;
      };

      return api;
   }

   function createModalApi() {
      var api = {};

      api.open = function (modalConfig) {
         var modalId = ++lastModalId;
         var onClosedHandler = null;
         if (modalConfig && typeof modalConfig.onClosed === "function") {
            onClosedHandler = modalConfig.onClosed;
         }
         var modalConfigCopy = JSON.parse(JSON.stringify(modalConfig));
         modalConfig = undefined;

         if (modalConfigCopy) {
            modalConfigCopy.onClosed = !!onClosedHandler;
            if (typeof modalConfigCopy.url === "string") {
               if (modalConfigCopy.url.match(absolutePathRegExp)) {
                  throw new Error(
                        "htmlClientSdk.modal.open() does not support absolute urls," +
                        " use a non-root relative one."
                  );
               }

               if (modalConfigCopy.url.match(rootRelativePathRegExp)) {
                  throw new Error(
                        "htmlClientSdk.modal.open() does not support root relative urls," +
                        " use a non-root relative one."
                  );
               }

               var linkElement = document.createElement("a");
               linkElement.href = modalConfigCopy.url;
               modalConfigCopy.url = linkElement.href;
            }
         }

         if (onClosedHandler) {
            modalIdToCloseHandler[modalId] = onClosedHandler;
         }

         sendModalOpenMessage(modalId, modalConfigCopy);
      };

      api.close = function (result) {
         sendModalCloseMessage(result);
      };

      api.getCustomData = function () {
         return cachedApiData.modal.customData;
      };

      api.setOptions = function (newModalConfig) {
         sendModalSetOptionsMessage(newModalConfig);
      };

      return api;
   }

   function createEventApi() {
      var api = {};

      api.onGlobalRefresh = function (callback) {
         globalRefreshHandler = callback;
         sendEventOnGlobalRefreshMessage();
      };

      api.onThemeChanged = function (callback) {
         themeChangedCallback = callback;
      };

      return api;
   }

   function sendInitRequestMessage() {
      postMessage("htmlClientSdk.internal.init");
   }

   function sendExtendUserSessionMessage() {
      postMessage("htmlClientSdk.internal.extendUserSession");
   }

   function sendRedispatchIframeMouseEvent(mouseEventProperties) {
      postMessage("htmlClientSdk.internal.redispatchIframeMouseEvent", {
         mouseEventProperties: mouseEventProperties
      });
   }

   function sendTrackTelemetryMouseEvent(eventType, targetNodesDataChain) {
      postMessage("htmlClientSdk.internal.trackTelemetryEvent$MouseEvent", {
         eventType: eventType,
         targetNodesDataChain: targetNodesDataChain
      });
   }

   function sendEventOnGlobalRefreshMessage() {
      postMessage("htmlClientSdk.event.onGlobalRefresh");
   }

   function sendModalSetOptionsMessage(newModalConfig) {
      postMessage("htmlClientSdk.modal.setOptions", {
         newModalConfig: newModalConfig
      });
   }

   function sendModalOpenMessage(modalId, modalConfig) {
      postMessage("htmlClientSdk.modal.open", {
         modalId: modalId,
         modalConfig: modalConfig
      });
   }

   function sendModalCloseMessage(result) {
      postMessage("htmlClientSdk.modal.close", {
         closeResult: result
      });
   }

   function sendAppNavigateToMessage(navigationOptions) {
      postMessage("htmlClientSdk.app.navigateTo", {
         navigationOptions: navigationOptions
      });
   }

   function sendAppGetSessionInfo(requestId) {
      postMessage("htmlClientSdk.app.getSessionInfo", {
         requestId: requestId
      });
   }

   function postMessage(operation, payload) {
      window.parent.postMessage({
         operation: operation,
         payload: payload
      }, "*");
   }

   function onMessage(event) {
      if (event.source !== window.parent) {
         return;
      }

      var operation = event.data.operation;
      var payload = event.data.payload;

      if (operation === "htmlClientSdk.internal.init$response" && initializationState === "initializing") {
         initializationState = true;
         cachedApiData = payload.cachedApiData;
         isModal = payload.isModal;
         htmlClientSdk.app = createApplicationApi();
         htmlClientSdk.modal = createModalApi();
         htmlClientSdk.event = createEventApi();

         if (isModal) {
            window.addEventListener("keyup", function (event) {
               if (event.keyCode === 27 || event.key === "Escape") {
                  htmlClientSdk.modal.close();
               }
            });
         }

         window.addEventListener("mousedown", function (event) {
            sendExtendUserSessionMessage();
         });

         var i;

         for (i = 0; i < telemetryTrackedMouseEventTypes.length; i++) {
            window.addEventListener(
                  telemetryTrackedMouseEventTypes[i],
                  trackTelemetryEvent$MouseEvent
            );
         }

         for (i = 0; i < redispatchedMouseEventNames.length; i++) {
            /*jshint loopfunc: true */
            window.addEventListener(redispatchedMouseEventNames[i], function (event) {
               var mouseEventProperties = {
                  type: event.type,
                  button: event.button,
                  buttons: event.buttons,
                  screenX: event.screenX,
                  screenY: event.screenY,
                  clientX: event.clientX,
                  clientY: event.clientY
               };

               sendRedispatchIframeMouseEvent(mouseEventProperties);
            });
         }

         while (onInitCallbacks.length > 0) {
            var callback = onInitCallbacks.shift();

            try {
               callback();
            } catch (e) {
               window.console.error(e);
            }
         }
      }

      if (operation === "htmlClientSdk.internal.onThemeChanged$event") {
         if (initializationState !== true) {
            return;
         }

         if (!payload || !payload.theme) {
            return;
         }

         var theme = payload.theme;
         cachedApiData.app.theme = theme;

         try {
            if (!themeChangedCallback) {
               return;
            }

            themeChangedCallback(theme);
         } catch (e) {
            window.console.error(e);
         }
      }

      if (operation === "htmlClientSdk.app.getSessionInfo$response") {
         if (!payload || !payload.requestId || !sessionInfoRequestIdIdToHandler[payload.requestId]) {
            return;
         }

         var handler = sessionInfoRequestIdIdToHandler[payload.requestId];
         try {
            handler(payload.sessionInfo);
         } catch (e) {
            window.console.error(e);
         } finally {
            delete sessionInfoRequestIdIdToHandler[payload.requestId];
         }
      }

      if (operation === "htmlClientSdk.modal.open$onClosed()") {
         if (!payload || !payload.modalId || !modalIdToCloseHandler[payload.modalId]) {
            return;
         }

         var closeHandler = modalIdToCloseHandler[payload.modalId];
         try {
            closeHandler(payload.modalResult);
         } catch (e) {
            window.console.error(e);
         } finally {
            delete modalIdToCloseHandler[payload.modalId];
         }
      }

      if (operation === "htmlClientSdk.event.onGlobalRefresh$callback()") {
         try {
            globalRefreshHandler();
         } catch (e) {
            window.console.error(e);
         }
      }
   }

   function initialize(callback) {
      if (initializationState === true) {
         if (callback) {
            callback();
         }
         return;
      }

      if (initializationState === "initializing") {
         if (callback) {
            onInitCallbacks.push(callback);
         }
         return;
      }

      if (callback) {
         onInitCallbacks.push(callback);
      }
      initializationState = "initializing";
      window.addEventListener("message", onMessage);
      sendInitRequestMessage();
   }

   var htmlClientSdk = {
      initialize: function (callback) {
         initialize(callback);
      },
      isInitialized: function () {
         return initializationState === true;
      }
   };

   window.htmlClientSdk = htmlClientSdk;
})();
