/* Copyright (c) 2017 VMware, Inc. All rights reserved. */
module platform {

   import ILogService = angular.ILogService;
   import IHttpService = angular.IHttpService;
   import TelemetryService = platform.TelemetryService;
   import TelemetryHelperService = platform.TelemetryHelperService;
   import TelemetryDomNodeWrapperInterface = platform.TelemetryDomNodeWrapperInterface;

   interface QueryParam {
      name: string;
      value: string;
   }

   interface UiApiEndpoint {
      origin: string;
      pathname: string;
      queryParams: Array<QueryParam>;
      fullUrl: string;
   }

   interface ApiEndpoints {
      uiApiEndpoint: UiApiEndpoint;
   }

   interface ApiInitData {
      isModal: boolean | undefined;
      cachedApiData: {
         app: {
            apiEndpoints: ApiEndpoints;
            navigationData: any;
            clientInfo: ClientInfo;
            contextObjects: any[] | undefined;
            clientLocale: string;
         },
         modal: {
            customData: any;
         }
      };
   }

   export class H5RemoteSdkAdapterService {
      public static $inject = [
         '$http',
         '$log',
         'telemetryService',
         'telemetryHelperService'
      ];

      constructor(private $http: IHttpService,
            private $log: ILogService,
            private telemetryService: TelemetryService,
            private telemetryHelperService: TelemetryHelperService) {
      }

      public createH5RemoteSdkAdapter(pluginIFrame: HTMLIFrameElement,
            htmlClientSdk: H5SdkApi,
            $scope: PluginIframeScope,
            extendUserSession: () => void,
            onIframeUnload: () => void,
            redispatchIframeMouseEvent: (mouseEvent: MouseEventProperties) => void): H5RemoteSdkAdapter {

         return new H5RemoteSdkAdapter(
               this.$http,
               this.$log,
               this.telemetryService,
               this.telemetryHelperService,
               window,
               pluginIFrame,
               htmlClientSdk,
               $scope,
               extendUserSession,
               onIframeUnload,
               redispatchIframeMouseEvent
         );
      }
   }

   export class H5RemoteSdkAdapter {
      private onPluginIFrameMessageBound: (event: MessageEvent) => void;
      private removeContentWindowValueWatch: (() => void) | null;

      constructor(private $http: IHttpService,
            private $log: ILogService,
            private telemetryService: TelemetryService,
            private telemetryHelperService: TelemetryHelperService,
            private appWindow: Window,
            private pluginIFrame: HTMLIFrameElement,
            private htmlClientSdk: H5SdkApi,
            private $scope: PluginIframeScope,
            private extendUserSession: () => void,
            private onIframeUnload: () => void,
            private redispatchIframeMouseEvent: (mouseEventProperties: MouseEventProperties) => void) {

         this.onPluginIFrameMessageBound = this.onPluginIFrameMessage.bind(this);

         this.appWindow.addEventListener("message", this.onPluginIFrameMessageBound);
         this.removeContentWindowValueWatch = (<any> $scope).$watch(
               (): any => this.pluginIFrame.contentWindow,
               (newValue: any, oldValue: any): void => {
                  if (!newValue && oldValue) {
                     this.onIframeUnload();
                  }
               }
         );
      }

      public destroy(): void {
         this.appWindow.removeEventListener("message", this.onPluginIFrameMessageBound);
         if (this.removeContentWindowValueWatch) {
            this.removeContentWindowValueWatch();
            this.removeContentWindowValueWatch = null;
         }
      }

      private onPluginIFrameMessage(event: MessageEvent): void {
         if (!this.pluginIFrame ||
               !event.source ||
               event.source !== this.pluginIFrame.contentWindow) {
            return;
         }

         let operation: string = event.data.operation;
         let payload: any = event.data.payload;

         /*
          * INTERNAL
          */
         if (operation === "htmlClientSdk.internal.init") {
            this.postMessage("htmlClientSdk.internal.init$response", <ApiInitData> {
               isModal: this.$scope.isModal,
               cachedApiData: {
                  app: {
                     apiEndpoints: {
                        uiApiEndpoint: {
                           origin: window.location.origin,
                           pathname: "/api/ui",
                           queryParams: [],
                           fullUrl: window.location.origin + "/api/ui"
                        }
                     },
                     navigationData: this.htmlClientSdk.app.getNavigationData(),
                     clientInfo: this.htmlClientSdk.app.getClientInfo(),
                     contextObjects: this.htmlClientSdk.app.getContextObjects(),
                     clientLocale: this.htmlClientSdk.app.getClientLocale(),
                     theme: this.htmlClientSdk.app.getTheme()
                  },
                  modal: {
                     customData: this.htmlClientSdk.modal.getCustomData()
                  }
               }
            });

            this.htmlClientSdk.event.onThemeChanged(this.onThemeChanged.bind(this));
         }

         if (operation === "htmlClientSdk.internal.extendUserSession") {
            this.extendUserSession();
         }

         if (operation === "htmlClientSdk.internal.redispatchIframeMouseEvent") {
            if (!payload || !payload.mouseEventProperties) {
               return;
            }

            this.redispatchIframeMouseEvent(payload.mouseEventProperties);
         }

         if (operation === "htmlClientSdk.internal.trackTelemetryEvent$MouseEvent") {
            if (!payload ||
                  !payload.eventType ||
                  !payload.targetNodesDataChain ||
                  !payload.targetNodesDataChain.length) {
               return;
            }

            let eventType: string = payload.eventType;
            let currentNodeWrapper: TelemetryDomNodeWrapperInterface;
            let parentNodeWrapper: TelemetryDomNodeWrapperInterface =
                  new TelemetryDomNodeWrapper(
                        this.pluginIFrame,
                        this.telemetryHelperService
                  );

            for (let i = payload.targetNodesDataChain.length - 1; i >= 0; i--) {
               let currentNodeData: any = payload.targetNodesDataChain[i];
               currentNodeWrapper = {
                  id: currentNodeData.id,
                  classList: currentNodeData.classList,
                  index: currentNodeData.index,
                  nodeType: currentNodeData.nodeType,
                  nodeName: currentNodeData.nodeName,
                  textContents: currentNodeData.textContents,
                  isEmbeddedNodeWrapper: true,
                  parentNodeWrapper: parentNodeWrapper
               };
               parentNodeWrapper = currentNodeWrapper;
            }

            this.telemetryService.trackMouseEvent(eventType, currentNodeWrapper);
         }

         /*
          * APP
          */
         if (operation === "htmlClientSdk.app.navigateTo") {
            if (!payload || !payload.navigationOptions) {
               return;
            }

            this.htmlClientSdk.app.navigateTo(<NavigationOptions> payload.navigationOptions);
         }

         if (operation === "htmlClientSdk.app.getSessionInfo") {
            if (!payload || !payload.requestId) {
               return;
            }

            let sessionInfoUrl: string = "/ui/plugins/" +
                  this.$scope.remotePluginExtensionContext.fullyQualifiedPluginInstanceId +
                  "/session";
            this.$http.get(sessionInfoUrl).then((httpResponse: any) => {
               this.postMessage("htmlClientSdk.app.getSessionInfo$response", {
                  requestId: payload.requestId,
                  sessionInfo: httpResponse.data
               });
            });
         }

         /*
          * MODAL
          */
         if (operation === "htmlClientSdk.modal.open") {
            if (!payload || !payload.modalId || !payload.modalConfig) {
               return;
            }

            let modalId: string = payload.modalId;
            let modalConfig: ModalConfig = <ModalConfig> payload.modalConfig;

            if ((<any> modalConfig.onClosed) === true) {
               modalConfig.onClosed = (result?: any) => {
                  this.postMessage("htmlClientSdk.modal.open$onClosed()", {
                     modalId: modalId,
                     modalResult: result
                  });
               };
            } else {
               modalConfig.onClosed = undefined;
            }

            this.htmlClientSdk.modal.open(modalConfig);
         }

         if (operation === "htmlClientSdk.modal.close") {
            let closeResult: any = null;
            if (payload) {
               closeResult = payload.closeResult;
            }

            this.htmlClientSdk.modal.close(closeResult);
         }

         if (operation === "htmlClientSdk.modal.setOptions") {
            if (!payload || !payload.newModalConfig) {
               return;
            }

            this.htmlClientSdk.modal
                  .setOptions(<DynamicModalConfig> payload.newModalConfig);
         }

         /*
          * EVENT
          */
         if (operation === "htmlClientSdk.event.onGlobalRefresh") {
            this.htmlClientSdk.event.onGlobalRefresh(() => {
               this.postMessage("htmlClientSdk.event.onGlobalRefresh$callback()");
            });
         }
      }

      private postMessage(operation: string, payload?: any): void {
         if (!this.pluginIFrame.contentWindow) {
            this.$log.warn(`Can't post message with operation id '${operation}'` +
                  ` to remote plugin extension '${this.$scope.viewId}'` +
                  ` because iframe.contentWindow is null!`);
            return;
         }

         this.pluginIFrame.contentWindow.postMessage({
            operation: operation,
            payload: payload
         }, "*");
      }

      private onThemeChanged(theme: PluginTheme): void {
         this.postMessage(
               "htmlClientSdk.internal.onThemeChanged$event",
               { theme: theme }
         );
      }
   }

   angular
         .module("com.vmware.platform.ui")
         .service("h5RemoteSdkAdapterService", H5RemoteSdkAdapterService);
}
