/* Copyright (c) 2018 VMware, Inc. All rights reserved. */
module platform {

   import IHttpService = angular.IHttpService;
   import ActionEvaluation = com.vmware.vise.actionsfw.ActionEvaluation;

   /**
    * This service class is used to initialize the SDK action invokers registered with the
    * actionsService.
    */
   export class H5SdkActionService {

      public static $inject = [
         "$http",
         "h5SdkModalService"];

      constructor(private $http: IHttpService,
            private h5SdkModalService: H5SdkModalService) {
      }

      public bindActionHandlers(): void {
         h5.actions["com.vmware.vsphere.client.HtmlPluginModalAction"] = this.handleHtmlPluginModalAction.bind(this);
         h5.actions["com.vmware.vsphere.client.HtmlPluginHeadlessAction"] = this.handleHtmlPluginHeadlessAction.bind(this);
         h5.actions["com.vmware.vsphere.client.pluginActions"] = this.handlePluginActions.bind(this);
      }

      private handleHtmlPluginModalAction(actionEval: ActionEvaluation, availableTargets): void {
         if (actionEval.additionalData) {
            let contextObjects: any[] = _.map(availableTargets, (targetId) => {
               return {
                  id: targetId
               };
            });

            let modalOptions: ModalConfig = {
               url: actionEval.additionalData.actionUrl,
               title: actionEval.additionalData.dialogTitle,
               closable: (actionEval.additionalData.closable !== "false"),
               contextObjects: contextObjects
            };

            if (actionEval.additionalData.dialogSize) {
               let sizeArray = actionEval.additionalData.dialogSize.split(",");
               let size = {
                  width: parseInt(sizeArray[0], 10),
                  height: parseInt(sizeArray[1], 10)
               };
               modalOptions.size = size;
            }

            this.h5SdkModalService.openModal(
                  modalOptions,
                  actionEval.remotePluginExtensionContext
            );
         }
      };

      private handleHtmlPluginHeadlessAction (actionEval: ActionEvaluation, availableTargets): void {
         if (actionEval.additionalData) {
            let params = { targetIds: availableTargets.toString() };
            this.$http.post(actionEval.additionalData.actionUrl, null, {params: params});
         }
      };

      private handlePluginActions (actionEval: ActionEvaluation, availableTargets): void {
         const modalSizeRegExp: RegExp = /^(\d+),(\d+)$/i;

         if (actionEval.pluginData) {
            if (actionEval.pluginData.headless === "true") {
               let params = { targetIds: availableTargets.toString() };
               this.$http.post(actionEval.pluginData.url, null, {params: params});
            } else {
               let contextObjects: any[] = _.map(availableTargets, (targetId) => {
                  return {
                     id: targetId
                  };
               });

               let closable: boolean | null = null;
               if (typeof actionEval.pluginData.closable === "string") {
                  try {
                     closable = JSON.parse(actionEval.pluginData.closable);
                  } catch (e) {
                  }
               }
               if (typeof closable !== "boolean") {
                  closable = null;
               }

               let modalOptions: ModalConfig = {
                  url: actionEval.pluginData.url,
                  title: actionEval.pluginData.title,
                  closable: closable,
                  contextObjects: contextObjects
               };

               if (typeof(actionEval.pluginData.size) === "string" &&
                     modalSizeRegExp.test(actionEval.pluginData.size)) {
                  let modalSizeGroups: string[] =
                        actionEval.pluginData.size.match(modalSizeRegExp);
                  let pixelWidth = parseInt(modalSizeGroups[1], 10);
                  let pixelHeight = parseInt(modalSizeGroups[2], 10);

                  if (0 < pixelWidth && 0 < pixelHeight) {
                     modalOptions.size = {
                        width: pixelWidth,
                        height: pixelHeight
                     };
                  }
               }

               this.h5SdkModalService.openModal(
                     modalOptions,
                     actionEval.remotePluginExtensionContext
               );
            }
         }
      };
   }

   angular
         .module("com.vmware.platform.ui")
         .service("h5SdkActionService", H5SdkActionService);
}
