// Copyright (c) 2015 VMware, Inc.  All rights reserved. -- VMware Confidential
(function() {
   'use strict';

   angular.module('com.vmware.platform.ui')
         .service('helpService', HelpService);

   HelpService.$inject = ['$window', '$httpParamSerializer', 'configurationService'];

   /**
    * Help service for opening a help page
    */
   function HelpService($window, $httpParamSerializer, configurationService) {

      // --------------- public APIs  ----------------

      return {
         showHelpPage: showHelpPage,
         getHelpPageUrl: getHelpPageUrl,
         showVmcOnAwsPage: showVmcOnAwsPage
      };

      // --------------- API implementation ----------------
      /**
       * Open a specific help page by providing a contextId.
       * If you do not provide a specific context then the global documentation page is
       * opened.
       * @param contextId The specific id for a help page
       * Example:
       * showHelpPage('com.vmware.vsphere.vm_admin.doc.vc_312') -> opens a page with the
       * provided context id
       * showHelpPage() -> opens the default documentation page
       */
      function showVmcOnAwsPage(){
         return configurationService.getProperty('vmcOnAws.url').then(function(vmcOnAwsPageUrl) {
            $window.open(vmcOnAwsPageUrl, '_blank');
         });
      }

      function showHelpPage(contextId) {
         createPageUrl(contextId).then(function(url) {
            $window.open(url, '_blank');
         });
      }

      function getHelpPageUrl(contextId) {
         return createPageUrl(contextId);
      }

      function createPageUrl(contextId) {
         if (contextId) {
            return configurationService.getProperty('help.url.contextual').then(function(helpUrl) {
               var url = getFormattedHelpUrl(helpUrl);
               url = insertLocaleInUrl(url);

               var contextParam = $httpParamSerializer({
                  id: contextId
               });
               url += 'context?' + contextParam;

               return url;
            });
         } else {
            return configurationService.getProperty('help.url').then(function(helpUrl) {
               var url = getFormattedHelpUrl(helpUrl);
               url = insertLocaleInUrl(url);
               return url;
            });
         }
      }

      function getLocale() {
         var fullLocale = h5.locale || "en-US";
         fullLocale = fullLocale.toLowerCase();
         var localeLangCode;
         switch(fullLocale){
            case "zh-cn":
               localeLangCode = fullLocale.split("-")[1];
               break;
            case "zh-tw":
               localeLangCode = fullLocale.split("-")[1];
               break;
            default:
               localeLangCode = fullLocale.split("-")[0];
               break;
         }
         return localeLangCode;
      }

      function getFormattedHelpUrl(url) {
         if(url !== undefined){
            var result = url;
            if (result.lastIndexOf("/") !== (url.length - 1)) {
               result = url + "/";
            }
            return result;
         }
         else {
            return "http://docs.vmware.com/";
         }
      }

      function insertLocaleInUrl(url) {
         var result = url.replace("{0}", getLocale());
         return result;
      }
   }
})();
