// Copyright (c) 2015-2017 VMware, Inc.  All rights reserved. -- VMware Confidential
(function() {
   'use strict';

   angular.module('com.vmware.platform.ui')
      .service('htmlBridgeService', htmlBridgeService);

   htmlBridgeService.$inject = ['$http', '$rootScope', 'userSessionService',
         'clarityModalService', 'notificationService', 'defaultUriSchemeUtil',
         'pluginUrlService', 'logService'];

   /*
    * Service for supporting 6.0 Html plugins.
    * This service is in addition to platformService.js which exposes the WEB_PLATFORM API.
    */
   function htmlBridgeService ($http, $rootScope, userSessionService, clarityModalService,
                               notificationService, defaultUriSchemeUtil, pluginUrlService,
                               logService) {

      // Cached userSession and server infos
      var userSession = null;
      var userSessionServerInfo = null;

      // Cached dialog options when modal dialog is opened
      var htmlDialogOptions = null;

      // Cached modal dialog and action dialog scope
      var htmlModalDialog = null;
      var htmlActionDialog = null;

      // Cached action targets to be able to send modelChange events
      var targetsByActionId = {};

      // Cached current objectId when opening a modal dialog
      var modalDialogObjectId = null;

      var FORWARD_SLASH_ENCODED2 = "%252F";
      var FORWARD_SLASH_REGEX = /\//g;

      var DEFAULT_WIDTH = 500;
      var DEFAULT_HEIGHT = 400;

      var log = logService('htmlBridgeService');

      // --------------- public APIs  ----------------

      var service = {
         init: init,
         callActionsController: callActionsController,
         getUserSession: getUserSession,
         sendNavigationRequest: sendNavigationRequest,
         openModalDialog: openModalDialog,
         closeDialog: closeDialog,
         setDialogTitle: setDialogTitle,
         setDialogSize: setDialogSize,
         sendModelChangeEvent: sendModelChangeEvent
      };

      return service;

      // --------------- API implementation ----------------

      function init() {
         // Since the platform SDK API used to give the userSession with vc server info, lets continue to do that for
         // backward compatibility.
         // TODO smarathe - Ideally we need to introduce another public API method which returns a Promise with the serverInfos.

         // For now - doing all this in init() since the callers expect it to be a synchronous function.
         // Cache userSession
         userSessionService.getUserSession().then(function(session) {
            userSession = getSdkUserSession(session);
         });

         userSessionService.getAllServersInfo().then(function(info) {
            userSessionServerInfo = getSdkUserSessionServerInfo(info);
         });

         // Register the HtmlActionDelegate used by plugins
         window.h5.actions["com.vmware.vsphere.client.htmlbridge.HtmlActionDelegate"] =
            function(actionEval, targets) {
               // targets is an array of 0 or more objectIds
               targetsByActionId[actionEval.action.uid] = targets;

               if (!actionEval.additionalData.dialogTitle) {
                  // Direct server call, no UI involved
                  handleHeadlessAction(actionEval);
               } else {
                  // Modal dialog. All server calls will be made within the plugin's own code.
                  openActionDialog(actionEval, targets);
               }
            };
      }

      function getSdkUserSession(rawUserSessionInfo) {
         // com.vmware.vsphere.client.usersession.
         var sdkUserSession = rawUserSessionInfo;

         if (!sdkUserSession) {
            return sdkUserSession;
         }

         sdkUserSession = _.pick(
               sdkUserSession,
               [
                  "userName",
                  "locale",
                  "clientId",
                  "hashedClientId"
               ]
         );

         return sdkUserSession;
      }

      function getSdkUserSessionServerInfo(rawUserSessionServerInfo) {
         // com.vmware.vsphere.client.usersession.UserSessionServerInfo
         var sdkUserSessionServerInfo = rawUserSessionServerInfo;

         if (!sdkUserSessionServerInfo) {
            return sdkUserSessionServerInfo;
         }

         sdkUserSessionServerInfo = _.pick(
               sdkUserSessionServerInfo,
               [
                  "serversInfo",
                  "linkedServersInfo"
               ]
         );

         if (sdkUserSessionServerInfo["serversInfo"]) {
            sdkUserSessionServerInfo["serversInfo"] =
                  getSdkServersInfo(sdkUserSessionServerInfo["serversInfo"]);
         }

         if (sdkUserSessionServerInfo["linkedServersInfo"]) {
            sdkUserSessionServerInfo["linkedServersInfo"] =
                  getSdkLinkedServersInfo(sdkUserSessionServerInfo["linkedServersInfo"]);
         }

         return sdkUserSessionServerInfo;
      }

      function getSdkLinkedServersInfo(rawLinkedServersInfo) {
         // com.vmware.vise.vim.commons.vcservice.ServerInfoEx[][]
         var sdkLinkedServersInfo = rawLinkedServersInfo;

         if (!sdkLinkedServersInfo) {
            return sdkLinkedServersInfo;
         }

         sdkLinkedServersInfo = _.map(
               sdkLinkedServersInfo,
               getSdkServersInfo
         );

         return sdkLinkedServersInfo;
      }

      function getSdkServersInfo(rawServersInfo) {
         // com.vmware.vise.vim.commons.vcservice.ServerInfoEx[]
         var sdkServersInfo = rawServersInfo;

         if (!sdkServersInfo) {
            return sdkServersInfo;
         }

         sdkServersInfo = _.map(
               sdkServersInfo,
               getSdkServerInfo
         );

         return sdkServersInfo;
      }

      function getSdkServerInfo(rawServerInfo) {
         // com.vmware.vise.vim.commons.vcservice.ServerInfoEx
         var sdkServerInfo = rawServerInfo;

         if (!sdkServerInfo) {
            return sdkServerInfo;
         }

         sdkServerInfo = _.pick(
               sdkServerInfo,
               [
                  // com.vmware.vise.usersession.ServerInfo
                  "name",
                  "serviceUrl",
                  "serviceGuid",
                  "thumbprint",
                  "version",
                  "sessionKey",
                  "sessionCookie",

                  // com.vmware.vise.vim.commons.vcservice.ServerInfoEx
                  "ssoDomain",
                  "nodeId",
                  "serviceInstance",
                  "content",
                  "errorCode"
               ]
         );

         if (sdkServerInfo["ssoDomain"]) {
            sdkServerInfo["ssoDomain"] = getSdkSsoDomain(
                  sdkServerInfo["ssoDomain"]
            );
         }

         // com.vmware.vim.binding.vmodl.ManagedObjectReference
         if (sdkServerInfo["serviceInstance"]) {
            sdkServerInfo["serviceInstance"] = angular.copy(
                  sdkServerInfo["serviceInstance"]
            );
         }

         // com.vmware.vim.binding.vim.ServiceInstanceContent
         if (sdkServerInfo["content"]) {
            sdkServerInfo["content"] = angular.copy(
                  sdkServerInfo["content"]
            );
         }

         return sdkServerInfo;
      }

      function getSdkSsoDomain(rawSsoDomain) {
         // com.vmware.sso.tokenmgmt.SsoDomain
         var sdkSsoDomain = rawSsoDomain;

         if (!sdkSsoDomain) {
            return sdkSsoDomain;
         }

         sdkSsoDomain = _.pick(
               sdkSsoDomain,
               [
                  "id",
                  "name",
                  "type",
                  "isLocal"
               ]
         );

         return sdkSsoDomain;
      }

      // Ideally we need to introduce a public API method which returns a Promise for this call and even split this call
      // into 2 kinsd - get normal session information and getVCServersInfo(). SDK team has these items in
      // the backlog. Someone needs to mark this methods as deprecated.
      function getUserSession() {
         if (userSession) {
            return angular.extend(userSession, userSessionServerInfo);
         }
         return null;
      }

      function handleHeadlessAction(actionEval) {
         var actionUrl = actionEval.additionalData.actionUrl;
         if (actionUrl.indexOf('?') < 0) {
            actionUrl += '?';
         }
         actionUrl += '&actionUid=' + actionEval.action.uid + '&locale=' + userSession.locale;

         callActionsController(actionUrl, null, actionEval.action.uid);
      }

      function callActionsController(url, jsonData, actionUid) {

         var params = {};

         // When this is called from a modal dialog (and not an action dialog) we must set
         // targetsByActionId to the cached modalDialogObjectId since this is the real target
         if (modalDialogObjectId) {
            targetsByActionId[actionUid] = [modalDialogObjectId];
         }
         var targetObjects = targetsByActionId[actionUid];


         if (!angular.isArray(targetObjects)) {
            throw new Error("invalid targets for action " + actionUid);
         }

         if (targetObjects.length > 0) {
            // Pass object ids as a comma separated string. Better as params than on Url
            // No need to encode objectIds
            params.targets = targetObjects.toString();
         }

         if (jsonData) {
            // convert jsonData to an object so that $httpParamSerializer adds it to the request
            params.json = JSON.parse(jsonData);
         }

         $http.post(url, null, { params: params }).then(function(resp) {
            // Handle the response like HtmlActionDelegate.as in NGC.
            // See various ActionResult use cases in the chassisA-service ActionController.java
            if (resp.data) {
               var actionSuccessfull = false;
               var targetObjects = null;
               if (resp.data.actionUid) {
                  targetObjects = targetsByActionId[resp.data.actionUid];
                  if (!targetObjects) {
                     log.error("Missing target objects or invalid action id: " + resp.data.actionUid);
                  }
               }
               if (resp.data.result) {
                  // result is either true/false for edit & delete, or the new object URI
                  var result = resp.data.result;
                  if (typeof result === 'boolean') {
                     actionSuccessfull = result;
                  } else if (typeof result === 'string' && resp.data.uriType) {
                     actionSuccessfull = true;
                     var objectId = result;
                     targetObjects = [ objectId ];
                  }
               }

               if (actionSuccessfull && resp.data.operationType && targetObjects) {
                  handleModelChange(resp.data, targetObjects);
               }
               if (resp.data.errorMessage) {
                  displayActionError(resp.data);
               }
               // Done with this action
               if (targetObjects && resp.data.actionUid) {
                  targetsByActionId[resp.data.actionUid] = null;
               }
            }

         });
      }

      // Open a vUI dialog in "sandbox" mode to support 6.0 Html plugins.
      // - actionEval.additionalData contains parameters parsed from the 6.0 plugin.xml:
      //
      //  <actionUrl>/vsphere-client/chassisa/resources/editChassisAction.html</actionUrl>
      //  <dialogTitle>#{chassis.editAction}</dialogTitle>
      //  <dialogSize>500,300</dialogSize>
      //  <dialogIcon>#{editChassis}</dialogIcon>
      //
      // - The sandbox flag added to dialogOptons below tells vuiDialogService to use an iFrame and to
      // remove the default footer buttons because OK/Cancel is handled by the plugin code
      function openActionDialog(actionEval, targetObjects) {
         var width = DEFAULT_WIDTH;
         var height = DEFAULT_HEIGHT;

         if (actionEval.additionalData.dialogSize) {
            var size = actionEval.additionalData.dialogSize.split(",");
            width = parseInt(size[0], 10);
            if (isNaN(width) || width <= 0) {
               width = DEFAULT_WIDTH;
            }
            height = parseInt(size[1], 10);
            if (isNaN(height) || height <= 0) {
               height = DEFAULT_HEIGHT;
            }
         }

         var actionUrl = pluginUrlService.buildUrl(actionEval.additionalData.actionUrl,
             {objectId: null, locale: userSession.locale}) ;

         var additionalParams = {
            'actionUid': actionEval.action.uid
         };
         if (targetObjects) {
            // Double encoding of "/" required as elsewhere for objectIds
            additionalParams['targets'] = targetObjects.toString().replace(FORWARD_SLASH_REGEX, FORWARD_SLASH_ENCODED2);
         }

         actionUrl = pluginUrlService.appendParams(actionUrl, additionalParams);

         htmlDialogOptions = {
            title: actionEval.additionalData.dialogTitle,
            url: actionUrl,
            pluginSize: {
               width: width,
               height: height
            },
            // size: getSize(size[0]),
            iconClass: (actionEval.additionalData.dialogIcon) ? actionEval.additionalData.dialogIcon : actionEval.action.icon
         };

         pluginUrlService.associatePluginSessionWithClientId(actionEval.additionalData.actionUrl)
            .then(function() {
               htmlActionDialog = _openModal(actionEval, targetObjects, htmlDialogOptions);
            });
      }

      /**
       * Navigate to the given view
       * @param extensionId
       * @param objectId
       */
      function sendNavigationRequest(extensionId, objectId) {
            $rootScope._navigateToViewAndObject(extensionId, objectId);
      }

      /**
       * Triggers an internal modelChange event.  This is mostly useful to refresh custom
       * objects in the RHS navigator (because vSphere objects are refreshed automatically)
       * after a plugin made some changes AND didn't use the standard ActionController which
       * supports such model update (See ActionsController.java in samples)
       *
       * @param objectId   The object affected by the change
       * @param opType     "add", "change", "delete" or "relationship_change"
       */
      function sendModelChangeEvent(objectId, opType) {
         if (!objectId) {
            return;
         }
         var objectType = defaultUriSchemeUtil.getEntityType(objectId);
         var targetObjects = [objectId];
         var data = {operationType: opType, uriType: objectType};
         handleModelChange(data, targetObjects);
      }

      /**
       * Propagate modelChanged event after an action completes successfully
       */
      function handleModelChange(data, targetObjects) {
         var opType = data.operationType.toUpperCase();
         if (opType !== "ADD" && opType !== "CHANGE" && opType !== "DELETE" &&
            opType !== 'RELATIONSHIP_CHANGE') {
            throw new Error("Invalid operation type " + data.operationType +
               (data.actionUid ? (' returned for action ' + data.actionUid) : ''));
         }
         // Supports only 1 object for now
         // TODO see story https://www.pivotaltracker.com/story/show/121523243 for multiple objects
         var objectId = targetObjects[0];

         var objectChangeInfo = { objectId: objectId, operationType: opType };
         if (opType === "ADD") {
            objectChangeInfo.object = { type: data.uriType};
         }

         $rootScope.$broadcast('modelChanged', objectChangeInfo);
      }

      /**
       * Display the error message sent back by the plugin's ActionController
       */
      function displayActionError(data) {
         // The message is either a string already localized or an object
         var errorMsg = null;
         if (angular.isString(data.errorMessage)) {
            errorMsg = data.errorMessage;
         } else {
            var key = data.errorMessage.key;
            var bundleName = data.errorMessage.bundleName;
            var params = data.errorMessage.params;
            if (key && bundleName) {
               errorMsg = $rootScope.i18n(bundleName, key, params);
            } else {
               log.error("Invalid errorMessage object returned by action " + data.actionUid);
               return;
            }
         }

         // TODO for now we use the non-intrusive notification popup but we could
         // use a H5C modal dialog here or the banner (it wasn't possible in NGC)
         notificationService.notifyError(null, errorMsg);
      }

      /**
       * Open modal dialog in this scope.
       *
       * @param title    Modal dialog title
       * @param url      URL to html page.
       * @param width    Modal dialog width, need to be number
       * @param height   Modal dialog height, need to be number
       * @param objectId
       * TODO add icon parameter
       */
      function openModalDialog(title, url, width, height, objectId) {
         if (!url) {
            return;
         }
         // TODO this logic doesn't work anymore because ClarityModalService allows to close the dialog without
         // going through CloseDialog().  Unfortunately there is no quick way to solve this so I will leave it as
         // bug to fix in a future Fling and document in SDK 6.5

         // if (htmlModalDialog){
         //    console.error('Modal dialog is already opened');
         //    return;
         // }

         // Remember the objectId to use as an action target in callActionsController
         modalDialogObjectId = objectId;
         var availableTargets = [objectId];

         var trustedUrl = pluginUrlService.buildUrl(url, {objectId: objectId, locale: userSession.locale});

         if (!angular.isNumber(width) || width <= 0) {
            width = DEFAULT_WIDTH;
         }
         if (!angular.isNumber(height) || height <= 0) {
            height = DEFAULT_HEIGHT;
         }

         var actionEval = {
            action: { }
         };
         htmlDialogOptions = {
            title: title,
            url: trustedUrl,
            pluginSize: {
               width: width,
               height: height
            }
            // size: getSize(width)
         };

         pluginUrlService.associatePluginSessionWithClientId(url).then(function() {
               htmlModalDialog = _openModal(actionEval, availableTargets, htmlDialogOptions);
            });
      }

      function _openModal(actionEval, availableTargets, dialogOptions) {
         var dialogScope = clarityModalService.openModal(
               actionEval, availableTargets, dialogOptions, 'resources/ui/views/plugins/htmlBridgePluginModalTemplate.html');

         // Handler called by clarityModalService.closeModal(). When user closes the modal with ESC or click
         // in X box in header we must call closeDialog ourselves since their plugin code won't do it.
         dialogScope.onCancelModal = function() {
            _closeDialog(true);
         };

         return dialogScope;
      }

      /**
       * Public API to close a modal dialog (either a dialog from a menu action, or from openModalDialog)
       */
      function closeDialog() {
         _closeDialog(false);
      }

      /**
       * Internal version of closeDialog API which is either called from plugin code, or when the user
       * clicks on the X button in the dialog header.
       *
       * @param fromCancelModal  true when this is called internally from clarityModalService.closeModal(),
       *          this is necessary to avoid an infinite loop!
       * @private
       */
      function _closeDialog(fromCancelModal){
         // Plugin can only have 1 dialog at a time.  This is a documented SDK limitation.
         if (htmlModalDialog) {
            if (!fromCancelModal) {
               htmlModalDialog.closeModal();
            }
            htmlModalDialog = null;
            modalDialogObjectId = null;
            return;
         }
         if (htmlActionDialog) {
            if (!fromCancelModal) {
               htmlActionDialog.closeModal();
            }
            htmlActionDialog = null;
         }
      }

      /**
       * Set the dialog's title.
       * @param title
       */
      function setDialogTitle(title){
         if (htmlDialogOptions === null) {
            log.error("Invalid use of setDialogTitle API");
            return;
         }
         htmlDialogOptions.title = title;

         //trigger the angular digest cycle in order to reflect the changes
         $rootScope.$apply();
      }

      /**
       * Set the dialog's width and height
       * @param width
       * @param height
       */
      function setDialogSize(width, height){
         log.error("Not supported in HTML Client SDK 6.5");

         /*
         if (htmlDialogOptions === null ||
            !(angular.isNumber(width) && width > 0) ||
            !(angular.isNumber(height) && height > 0)) {
            console.error("Invalid use of setDialogSize API");
            return;
         }
         htmlDialogOptions.width = width+'px';
         htmlDialogOptions.height = height+'px';
         //trigger the angular digest cycle in order to reflect the changes
         $rootScope.$apply();
         */
      }

      /* NOT USED FOR NOW
       * Get the predefined size for the clarity modal, see http://clarity.vmware.com/documentation/modals
       *
       * @param width
       * @returns {modal size}: 'sm', '<none>', 'lg' or 'xl'
       *
      function getSize(width) {
         if(width <= 288) {
            return 'sm';
         } else if(width <= 576){
            return '<none>';
         } else if(width <= 864) {
            return 'lg';
         }
         return 'xl'; // 'xl' is 1152, only in Clarity 0.5.1
      }
      */
   }
})();
