/* global html2canvas, window, document, console */
// Copyright (c) 2015 VMware, Inc.  All rights reserved. -- VMware Confidential
(function () {
   'use strict';
   // 2/11/19 - cpham, new feedback tool is in use and old files will be removed later
   angular.module('com.vmware.platform.ui').controller('VfeedDialogController',
      ['$scope', '$element', '$rootScope', 'vuiConstants', '$document', '$http', 'feedbackService', 'i18nService', 'themeService',
         function ($scope, $element, $rootScope, vuiConstants, $document, $http, feedbackService, i18nService, themeService) {
            var MAXLENGTH = 1000;
            //TODO: remove styling from js logic
            var blueBorder = {border: '1px solid blue'};
            var darkThemeBorder = {border: '1px solid #49AFD9'};

            var canvas = null;
            var context;
            var offCanvas;
            var offContext;
            var rating = 0;
            var clickX = [];
            var clickY = [];
            var clickDrag = [];
            var clickPen = [];
            var canvasElt = $element.find('.canvas');
            var paint = true;
            var controlMouseDown = false;
            var controlMouseMove = false;

            $scope.smileyStyles = themeService.getCurrentTheme() === themeService.getKeys().light ? [blueBorder, {}, {}] : [darkThemeBorder, {}, {}];
            $scope.selectSmiley = selectSmiley;
            $scope.addScreenshot = addScreenshot;
            $scope.removeScreenshot = removeScreenshot;
            $scope.clearAllLines = clearAllLines;
            $scope.undo = undo;

            // Detect iFrames and show extra warning for screenshot.
            // Special case of the dashboard-ui for which the screenshot will focus on the iFrame itself.
            var iFrames = window.document.getElementsByClassName('sandbox-iframe');
            var useiFrameContent = false;
            if (iFrames && iFrames.length > 0) {
               var src = iFrames[0].getAttribute("ng-src");
               useiFrameContent = (src && src.indexOf("/ui/dashboard-ui") === 0);
               $scope.showiFrameWarning = !useiFrameContent;
            }

            $scope.email = feedbackService.getEmail();
            if (!$scope.email) {
               $scope.emailPlaceholder = i18nService.getString('Common', 'feedback.optionalEmail');
            }
            $scope.cc = null;

            var stackTrace = null;
            if ($rootScope.errMessage) {
               // Feedback dialog used in "error reporting" mode after user clicked on "Send details to VMware" in the popup.
               // We initialize the text field with the current error message, and give a chance to users to add more info.
               $scope.message = i18nService.getString('Common', 'feedback.errorMessage') + ' ' + $rootScope.errMessage;
               $rootScope.errMessage = null;
               stackTrace = $rootScope.stackTrace;
               $rootScope.stackTrace = null;
               rating = -1;
            } else {
               $scope.message = '';
            }
            $scope.charsLeft = MAXLENGTH;

            var MARKER_PEN = 0;
            var ERASER_PEN = 1;
            var penMode = MARKER_PEN;
            $scope.useMarkerPen = function() {
               penMode = MARKER_PEN;
            };
            $scope.useEraserPen = function() {
               penMode = ERASER_PEN;
            };

            $scope.hasCanvas = function () {
               return (canvas !== null);
            };

            $scope.handleKeypress = function($event) {
               if ($event.which === 13) {
                  $event.stopPropagation();
               }
            };

            $scope.signPostParams = {
               message: i18nService.getString('Common', 'feedback.helpMessageJSVersion',
                  i18nService.getString('Common', 'feedback.helpMessageUrl')),
               title: i18nService.getString('Common', 'feedback.help')
            };

            $scope.$watch('message', function(newValue) {
               if (newValue && newValue.length > MAXLENGTH) {
                  $scope.message = newValue.substring(0, MAXLENGTH);
               }
               $scope.charsLeft = MAXLENGTH - $scope.message.length;
            });

            toggleButtons();
            $scope.useMarkerPen();

            // TODO fix with proper VUI or Clarity option to avoid this hack!
            // Increase dialog max width and height because vui-bootstrap sets it to 80%
            var dialogNodes = document.getElementsByClassName('vui-dialog');
            if (dialogNodes.length > 0) {
               var dialogElt = angular.element(dialogNodes[0]);
               dialogElt.css("max-width", "90%");
               dialogElt.css("max-height", "90%");
               dialogElt.css("overflow-y", "auto");
               var titleElt = angular.element(dialogNodes[0].getElementsByClassName('titlebar-text'));
               titleElt.css("max-width", "100%");
            }


            function selectSmiley(index) {
               $scope.smileyStyles[rating] = {};
               rating = index;
               $scope.smileyStyles[rating] = themeService.getCurrentTheme() === themeService.getKeys().light ? blueBorder : darkThemeBorder;
            }

            function addScreenshot() {
               // Convert DOM to canvas, remove last modal-backdrop temporarily (the one with feedback dialog)
               // When useiFrameContent is true the screenshot will only show iFrame document
               var body = useiFrameContent ? iFrames[0].contentDocument.body : document.body;
               var bodyElt = angular.element(body);

               if (!useiFrameContent) {
                  var backdrops = document.getElementsByClassName('modal-backdrop');
                  var modalBackdrop = angular.element(backdrops[backdrops.length-1]);
                  modalBackdrop.detach();
               }

               // Work-around to the screenshot problem with css 'wizard-steps-list', we remove the class temporarily
               var wizardToc = document.getElementsByClassName('wizard-steps-list');
               var wizardTocElt = null;
               if (wizardToc.length > 0) {
                  wizardTocElt =  angular.element(document.getElementsByClassName('wizard-steps-list')[0]);
                  wizardTocElt.removeClass('wizard-steps-list');
               }

               var options = {
                  allowTaint: true
               };

               html2canvas(bodyElt, options).then(function (_canvas) {
                  if (!useiFrameContent) {
                     bodyElt.append(modalBackdrop);
                  }
                  if (wizardTocElt) {
                     wizardTocElt.addClass('wizard-steps-list');
                  }
                  canvas = _canvas;
                  var canvasDiv = $element.find('.canvas-div');
                  canvas.setAttribute('class', 'canvas');
                  canvasDiv.append(canvas);
                  prepareCanvas();
                  redraw();
                  toggleButtons();
               });
            }

            function toggleButtons() {
               // TODO remove this hack once ng-show works in vui-dialog
               var addDiv = $element.find('.add-screenshot');
               var removeDiv = $element.find('.remove-screenshot');
               var canvasDiv = $element.find('.canvas-div');
               var contentDiv = $element;

               if ($scope.hasCanvas()) {
                  contentDiv.css("width",'100%');
                  canvasDiv.show();
                  addDiv.hide();
                  removeDiv.show();
               } else {
                  contentDiv.css("width",'320px');
                  canvasDiv.hide();
                  addDiv.show();
                  removeDiv.hide();
               }
            }

            function removeScreenshot() {
               canvas = null;
               var canvasDiv =  $element.find('.canvas-div');
               canvasDiv.empty();
               clearAllLines();
               toggleButtons();
            }


            // Need this flag to avoid sending email twice when ENTER key is used (vui-dialog bug?)
            var feedbackSent = false;

            $scope.dialogOptions.confirmOptions.onClick = function () {
               if (feedbackSent) {
                  return true;
               }
               var invalidCode = validateInput();
               if (invalidCode === 0) {
                  var feedbackData = {
                     type: 'feedback',
                     email: $scope.email.substring(0,100),
                     cc_emails: $scope.cc_emails,
                     rating: rating+1,
                     message: $scope.message,
                     stack: stackTrace,
                     canvas: canvas
                  };
                  feedbackService.sendFeedback(feedbackData);

                  feedbackSent = true;
                  $scope.dialogOptions.data.message = $scope.message;
                  $scope.dialogOptions.data.email = $scope.email;
                  return true;
               } else {
                  $scope.dialogOptions.validationBanner.messages = [{
                     text: (invalidCode === 1 ? i18nService.getString('Common', 'feedback.typeMessage') :
                        (invalidCode === 2 ? i18nService.getString('Common', 'feedback.enterValidEmail') :
                           'CC only vmware.com addresses')),
                     type: vuiConstants.validationBanner.type.ERROR
                  }];
                  return false;
               }
            };

            // Returns 0 for valid input, 1 for missing message, 2 for invalid email format,
            // 3 for invalid CC emails (not used at the moment)
            function validateInput() {
               $scope.cc_emails = [];

               if (!$scope.message) {
                  return 1;
               }
               if ($scope.feedbackForm.$error.email) {
                  return 2;
               }
               if (!$scope.cc) {
                  return 0;
               }
               var emails = null;
               if ($scope.cc.indexOf(',') > 0) {
                  emails = $scope.cc.split(',');
               } else {
                  emails = $scope.cc.split(' ');
               }

               // Build array of CC addresses, accept only @vmware.com and @pivotal.io emails!
               for (var i = 0; i < emails.length; i++) {
                  var s = emails[i].trim();
                  var k = s.length - 11;
                  var v = s.indexOf("@vmware.com");
                  var p = s.indexOf("@pivotal.io");
                  if ((v <= 0 && p <= 0) || (v > 0 && v !== k) || (p > 0 && p !== k)) {
                     return 3;
                  }
                  $scope.cc_emails.push(s);
               }
               return 0;
            }

            canvasElt.mousedown(function (e) {
               if(controlMouseDown === false){
                  // prepare for next segment
                  clickX.push([]);
                  clickY.push([]);
                  clickDrag.push([]);
                  clickPen.push([]);

                  var panelContent = canvas.parentElement.parentElement.parentElement.parentElement.parentElement;
                  var vuiPopup = panelContent.parentElement;
                  var mouseX = e.pageX - canvas.offsetParent.offsetLeft - vuiPopup.offsetLeft + panelContent.scrollLeft - 4;
                  var mouseY = e.pageY - canvas.offsetParent.offsetTop - vuiPopup.offsetTop + panelContent.scrollTop - 2;
                  paint = true;
                  controlMouseDown = true;

                  addClick(mouseX, mouseY);
                  redraw();
               }
            });

            canvasElt.mousemove(function (e) {
               if (paint === true && controlMouseMove === false) {
                  var panelContent = canvas.parentElement.parentElement.parentElement.parentElement.parentElement;
                  var vuiPopup = panelContent.parentElement;
                  var mouseX = e.pageX - canvas.offsetParent.offsetLeft - vuiPopup.offsetLeft + panelContent.scrollLeft - 4;
                  var mouseY = e.pageY - canvas.offsetParent.offsetTop - vuiPopup.offsetTop + panelContent.scrollTop - 2;
                  controlMouseMove = true;

                  addClick(mouseX, mouseY, true);
                  redraw();
               }else if(controlMouseMove === true){
                  controlMouseMove = false;
               }
            });

            canvasElt.mouseup(function () {
               controlMouseMove = false;
               controlMouseDown = false;
               paint = false;
               redraw();
            });

            canvasElt.mouseleave(function () {
               paint = false;
            });

            function prepareCanvas() {
               // Keep offscreen copy of screenshot
               offCanvas = document.createElement('canvas');
               offCanvas.width = canvas.width;
               offCanvas.height = canvas.height;
               offContext = offCanvas.getContext('2d');
               try {
                  offContext.drawImage(canvas, 0, 0);
               } catch (e) {
                  console.log(e);
               }
               context = canvas.getContext('2d');
            }

            function addClick(x, y, dragging) {
               var index = clickX.length-1;
               clickX[index].push(x);
               clickY[index].push(y);
               clickDrag[index].push(dragging);
               clickPen[index].push(penMode);
            }

            function redraw() {
               context.clearRect(0, 0, context.canvas.width, context.canvas.height);
               if (offCanvas) {
                  context.drawImage(offCanvas, 0, 0);
               }
               context.lineJoin = 'round';

               for (var s = 0; s < clickX.length; s++) {
                  for (var i = 0; i < clickX[s].length; i++) {
                     context.strokeStyle = (clickPen[s][i] === MARKER_PEN ? '#df4b26' : '#cccccc');
                     context.lineWidth = (clickPen[s][i] === MARKER_PEN ? 5 : 20);

                     context.beginPath();
                     if (clickDrag[s][i] && i) {
                        context.moveTo(clickX[s][i - 1], clickY[s][i - 1]);
                     } else {
                        context.moveTo(clickX[s][i] - 1, clickY[s][i]);
                     }
                     context.lineTo(clickX[s][i], clickY[s][i]);
                     context.closePath();
                     context.stroke();
                  }
               }
            }

            function undo() {
               clickX.pop();
               clickY.pop();
               clickDrag.pop();
               clickPen.pop();
               redraw();
            }

            function clearAllLines() {
               clickX = [];
               clickY = [];
               clickDrag = [];
               clickPen = [];
               redraw();
            }
         }]);
})();
