﻿/*
 * Copyright 2015, 2016 VMware, Inc.  All rights reserved.
 */

namespace vmware.samples
{
    using System;
    using System.Collections.Generic;
    using System.Configuration;
    using System.Linq;
    using System.Net;
    using System.Net.Security;
    using System.Security.Cryptography.X509Certificates;
    using System.Text;
    using vmware.samples.common;

    public class Application
    {
        private static readonly string SAMPLES_NAMESPACE_PREFIX =
            "vmware.samples";
        private static readonly string[] SAMPLE_TYPE_NAMESPACES =
            new string[] { "tagging", "vcenter.vm.create.defaultvm", "vcenter.vm.create.basicvm", "vcenter.vm.create.exhaustivevm" };

        public static void Main(string[] args)
        {
            SetupServerCertificateValidation();

            if (args.Length == 0)
            {
                DisplayHelp();
                return;
            }

            // get sample type from name
            var type = GetSampleType(args[0]);
            if (type == null)
            {
                Console.WriteLine("Could not find sample with name '{0}'",
                    args[0]);
                DisplayHelp();
                return;
            }

            // create sample object using its type
            var sample = Activator.CreateInstance(type) as SamplesBase;

            // run sample
            if (CommandLine.Parser.Default.ParseArguments(args, sample))
            {
                sample.Connect();
                sample.Run();
                sample.Disconnect();
                Console.WriteLine("Finished running sample.");
            }
        }

        private static void SetupServerCertificateValidation()
        {
            // validates or ignores server certificate errors based on the
            // value of config key "ValidateServerCertificate"
            var validateServerCert = ConfigurationManager.AppSettings[
                "ValidateServerCertificate"];

            if (!string.IsNullOrWhiteSpace(validateServerCert) &&
                validateServerCert.Equals(bool.TrueString,
                StringComparison.CurrentCultureIgnoreCase))
            {
                // validates server certificate
                ServicePointManager.ServerCertificateValidationCallback +=
                    Validate;
            }
            else
            {
                // ignores server certificate errors
                ServicePointManager.ServerCertificateValidationCallback =
                    (sender, certificate, chain, sslPolicyErrors) => true;
            }
        }

        private static bool Validate(object sender,
            X509Certificate certificate, X509Chain chain,
            SslPolicyErrors sslPolicyErrors)
        {
            var result = true;

            if (sslPolicyErrors == SslPolicyErrors.None)
            {
                return result;
            }

            if ((sslPolicyErrors &
                SslPolicyErrors.RemoteCertificateNameMismatch) != 0)
            {
                Console.WriteLine("SSL policy error {0}." +
                    " Make sure that your application is using the correct" +
                    " server host name.",
                    SslPolicyErrors.RemoteCertificateNameMismatch);
                result = result && false;
            }

            if ((sslPolicyErrors &
                SslPolicyErrors.RemoteCertificateChainErrors) != 0)
            {
                var chainStatusList = new List<string>();
                if (chain != null && chain.ChainStatus != null)
                {
                    foreach (var status in chain.ChainStatus)
                    {
                        if ((certificate.Subject == certificate.Issuer))
                        {
                            // Self signed certificates with an untrusted root
                            // are valid.
                            continue;
                        }
                        chainStatusList.Add(status.Status.ToString());
                    }
                }
                if (chainStatusList.Count > 0)
                {
                    Console.WriteLine(
                        "SSL policy error {0}. Fix the following errors {1}",
                        SslPolicyErrors.RemoteCertificateChainErrors,
                        string.Join(", ", chainStatusList));
                    result = result && false;
                }
            }

            if ((sslPolicyErrors &
                SslPolicyErrors.RemoteCertificateNotAvailable) != 0)
            {
                Console.WriteLine("SSL policy error {0}." +
                    " The server certificate is not available for validation.",
                    SslPolicyErrors.RemoteCertificateNotAvailable);
                result = result && false;
            }

            return result;
        }

        private static void DisplayHelp()
        {
            // displays comma seperated sample names
            var groupedByNamespace =
                AppDomain.CurrentDomain.GetAssemblies().SelectMany(
                assembly => assembly.GetTypes()).Where(type =>
                    typeof(SamplesBase).IsAssignableFrom(type) && type.IsClass
                    && !type.IsAbstract).GroupBy(concreteType =>
                        concreteType.Namespace);

            var usage = new StringBuilder();
            usage.AppendLine();
            usage.AppendLine(
                "To get sample specific usage type: vSphereAutomationSamples.exe <sample name>");
            usage.AppendLine();
            usage.AppendLine("--- Available Samples (grouped by type) ---");
            foreach (var group in groupedByNamespace)
            {
                usage.AppendLine(group.Key);
                foreach (var type in group)
                    usage.AppendLine("  " + type.Name);  // logger.SimpleInfo("  {0}", type.Name);
            }
            Console.WriteLine(usage);
        }

        private static Type GetSampleType(string simpleName)
        {
            // search sample's type
            foreach (var sampleType in SAMPLE_TYPE_NAMESPACES)
            {
                var qualifiedName = SAMPLES_NAMESPACE_PREFIX + "." +
                    sampleType + "." + simpleName;
                var type = Type.GetType(qualifiedName, false, true);
                if (type != null)
                    return type;
            }
            return null;
        }
    }
}
