﻿/*
 * Copyright 2015, 2016 VMware, Inc.  All rights reserved.
 */

namespace vmware.samples.common
{
    using System;
    using System.Security.Cryptography;
    using vmware.vapi.bindings;
    using vmware.vapi.dsig;
    using vmware.vapi.protocol;
    using vmware.vapi.security;

    /// <summary>
    /// vAPI service endpoint utility class.
    /// </summary>
    /// <typeparam name="T"></typeparam>
    public abstract class ServiceEndpoint<T> where T : IService
    {
        private string url;
        private StubFactory stubFactory;
        private StubConfiguration stubConfiguration;
        private char[] sessionId;

        public ServiceEndpoint(string url)
        {
            this.url = url;
            var protocolFactory = new ProtocolConnectionFactory();
            var protocolConnection = protocolFactory.GetConnection(
                Protocol.Http, url, new CspParameters());
            stubFactory = new StubFactory(protocolConnection.GetApiProvider());
            stubConfiguration = new StubConfiguration();
        }

        public void Login(SsoConnection ssoConnection)
        {
            var tmp = new StubConfiguration();
            tmp.SetSecurityContext(new SamlTokenSecurityContext(
                ssoConnection.SamlToken, ssoConnection.PrivateKey));
            T session = stubFactory.CreateStub<T>(tmp);
            sessionId = Login(session);
            stubConfiguration.SetSecurityContext(
                new SessionSecurityContext(sessionId));
        }

        public void Logout()
        {
            T session = stubFactory.CreateStub<T>(stubConfiguration);
            Logout(session);
        }

        protected abstract char[] Login(T session);
        protected abstract void Logout(T session);

        public S GetService<S>() where S : IService
        {
            Console.WriteLine(
                "Instantiating vAPI Service - " + typeof(S).Name);
            return stubFactory.CreateStub<S>(stubConfiguration);
        }

        public override string ToString()
        {
            return string.Format("vAPI Connection ({0})", url);
        }
    }
}
