﻿/*
 * Copyright 2015, 2016 VMware, Inc.  All rights reserved.
 */

namespace vmware.samples.common
{
    using System;
    using System.IdentityModel.Tokens;
    using System.Net;
    using System.ServiceModel;
    using System.ServiceModel.Channels;
    using System.ServiceModel.Description;
    using vmware.vim25;

    /// <summary>
    /// VIM utility that uses a cookie to create a new connection.
    /// </summary>
    public class VimConnection
    {
        private string vimUrl;

        public VimPortType VimPortType { get; private set; }
        public ManagedObjectReference SvcRef { get; private set; }
        public ServiceContent ServiceContent { get; private set; }

        public string VimUuid { get; private set; }
        public Cookie VimCookie { get; private set; }
        public UserSession VimUserSession { get; private set; }

        public VimConnection(string vimUrl)
        {
            this.vimUrl = vimUrl;

            SvcRef = new ManagedObjectReference();
            SvcRef.type = "ServiceInstance";
            SvcRef.Value = "ServiceInstance";
        }

        public void Login(SsoConnection ssoConnection)
        {
            var securityToken = new GenericXmlSecurityToken(
                ssoConnection.SamlTokenXml, null,
                DateTime.Parse(ssoConnection.RequestSecurityTokenResponse.Lifetime.Created.Value),
                DateTime.Parse(ssoConnection.RequestSecurityTokenResponse.Lifetime.Expires.Value),
                null, null, null);

            var factory = GetChannelFactory<VimPortType>(SecurityKeyType.BearerKey);
            this.VimPortType = factory.CreateChannelWithIssuedToken(securityToken);

            this.ServiceContent = this.VimPortType.RetrieveServiceContent(this.SvcRef);
            if (this.ServiceContent.sessionManager != null)
            {
                var userSession = this.VimPortType.LoginByToken(this.ServiceContent.sessionManager, null);
                this.VimUserSession = userSession;
            }
        }

        public bool IsConnected()
        {
            return this.VimPortType != null && this.ServiceContent != null;
        }

        public void Logout()
        {
            if (this.VimPortType != null)
            {
                if (this.ServiceContent != null)
                {
                    this.VimPortType.Logout(
                        this.ServiceContent.sessionManager);
                    Console.WriteLine("Logged out successfully.");
                }
                this.VimPortType = null;
                this.ServiceContent = null;
            }
        }

        public override string ToString()
        {
            return string.Format("VIM Connection ({0})", vimUrl);
        }

        public ChannelFactory<T> GetChannelFactory<T>(
            SecurityKeyType securityKeyType)
        {
            var binding = GetCustomBinding(securityKeyType);
            var endpoint = new EndpointAddress(new Uri(this.vimUrl));

            var factory = new ChannelFactory<T>(binding, endpoint);
            factory.Endpoint.Behaviors.Add(
                new ClientViaBehavior(new Uri(this.vimUrl)));
            factory.Credentials.SupportInteractive = false;

            return factory;
        }

        private Binding GetCustomBinding(SecurityKeyType securityKeyType)
        {
            var ws2007FederationHttpBinding =
                GetWS2007FederationHttpBinding(securityKeyType);
            var customBinding = new CustomBinding(
                ws2007FederationHttpBinding.CreateBindingElements());

            foreach (var e in customBinding.Elements)
            {
                if (e is MessageEncodingBindingElement)
                {
                    ((TextMessageEncodingBindingElement)e).MessageVersion =
                        MessageVersion.Soap11;
                }
                if (e is SecurityBindingElement)
                {
                    ((TransportSecurityBindingElement)e).AllowInsecureTransport = true;
                    ((TransportSecurityBindingElement)e).EnableUnsecuredResponse = true;
                    ((TransportSecurityBindingElement)e).IncludeTimestamp = true;
                }
                if (e is HttpsTransportBindingElement)
                {
                    ((HttpsTransportBindingElement)e).AllowCookies = true;
                }
            }
            return customBinding;
        }

        private Binding GetWS2007FederationHttpBinding(SecurityKeyType securityKeyType)
        {
            var binding = new WS2007FederationHttpBinding(
                WSFederationHttpSecurityMode.TransportWithMessageCredential);
            binding.Security.Message.NegotiateServiceCredential = false;
            binding.Security.Message.EstablishSecurityContext = false;
            binding.Security.Message.IssuedKeyType = securityKeyType;
            return binding;
        }
    }
}
