"""
Generic client connector
"""

__author__ = 'VMware, Inc.'
__copyright__ = 'Copyright (c) 2015 VMware, Inc.  All rights reserved.'


import logging

from vmware.vapi.protocol.client.connector import Connector
from vmware.vapi.core import ExecutionContext, SecurityContext
from vmware.vapi.lib.context import create_default_application_context

logger = logging.getLogger(__name__)


class GenericConnector(Connector):
    """ A generic protocol connector """

    def __init__(self, rpc_provider, api_provider):
        """
        Generic protocol connector init

        :type  rpc_provider: :class:`vmware.vapi.protocol.client.rpc.provider.RpcProvider`
        :param rpc_provider: rpc provider object
        :type  api_provider: :class:`vmware.vapi.core.ApiProvider`
        :param api_provider: api provider object
        """
        Connector.__init__(self)
        self.rpc_provider = rpc_provider
        self.api_provider = api_provider
        self.session = None
        self._application_ctx = None
        self._security_ctx = None

    def connect(self):
        """ rpc provider connect """
        self.rpc_provider.connect()

    def disconnect(self):
        """ rpc provider disconnect """
        self.rpc_provider.disconnect()

    def get_api_provider(self):
        """
        get api provider

        :rtype:  :class:`vmware.vapi.core.ApiProvider`
        :return: api provider
        """
        return self.api_provider

    def set_application_context(self, ctx):
        """
        Set the application context

        All the subsequent calls made using this
        connector will use this as the application
        context in the ExecutionContext

        :type  ctx: :class:`vmware.vapi.core.ApplicationContext`
        :param ctx: New application context
        """
        self._application_ctx = ctx

    def set_security_context(self, ctx):
        """
        Set the security context

        All the subsequent calls made using this
        connector will use this as the security
        context in the ExecutionContext

        :type  ctx: :class:`vmware.vapi.core.SecurityContext`
        :param ctx: New security context
        """
        self._security_ctx = ctx

    def new_context(self):
        """
        create new execution context object

        :rtype:  :class:`vmware.vapi.core.ExecutionContext`
        :return: execution context
        """
        app_ctx = self._application_ctx
        # Create a default application context only if
        # the user has not provided anything
        if app_ctx is None:
            app_ctx = create_default_application_context()
        sec_ctx = self._security_ctx
        if sec_ctx is None:
            sec_ctx = SecurityContext()
        return ExecutionContext(app_ctx, sec_ctx)
