"""
API Provider filter
"""

__author__ = 'VMware, Inc.'
__copyright__ = 'Copyright (c) 2015 VMware, Inc.  All rights reserved.'

import logging

from vmware.vapi.core import ApiProvider
from vmware.vapi.data.serializers.introspection import convert_data_def_to_data_value
from vmware.vapi.provider.lib import augment_method_result_with_errors

logger = logging.getLogger(__name__)


class ApiProviderFilter(ApiProvider):
    """
    ApiProviderFilter is a base class for all ApiProvider filters.
    This handles all the common methods and also takes care of augmenting
    errors reported by an ApiProvider filter.

    :type next_provider: :class:`vmware.vapi.core.ApiProvider`
    :ivar next_provider: Next API Provider in the chain
    """
    def __init__(self, next_provider=None, errors_to_augment=None):
        """
        Initialize ApiProviderFilter

        :type  next_provider: :class:`vmware.vapi.core.ApiProvider` or ``None``
        :param next_provider: API Provider to invoke the requests
        :type  errors_to_augment: :class:`list` of
            :class:`vmware.vapi.data.definition.ErrorDefinition`
        :param errors_to_augment: List of error definitions to be added to method
            definitions
        """
        ApiProvider.__init__(self)
        self.next_provider = next_provider
        self._error_defs_to_augment = errors_to_augment
        self._error_values_to_augment = [
            convert_data_def_to_data_value(error_def)
            for error_def in self._error_defs_to_augment
        ]

    def invoke(self, service_id, operation_id, input_value, ctx):
        """
        Invoke an API request. Derived classes of ApiProviderFilter
        should call this method to invoke the request. This can be done
        by: ApiProviderFilter.invoke(self, ctx, method_id, input_value).

        This method calls the next API Provider. If the request is made to
        "get" operation of vAPI Operation Introspection service, errors are
        augmented to the method result.

        :type  service_id: :class:`str`
        :param service_id: Service identifier
        :type  operation_id: :class:`str`
        :param operation_id: Operation identifier
        :type  input_value: :class:`vmware.vapi.data.value.StructValue`
        :param input_value: Method input parameters
        :type  ctx: :class:`vmware.vapi.core.ExecutionContext`
        :param ctx: Execution context for this method

        :rtype: :class:`vmware.vapi.core.MethodResult`
        :return: Result of the method invocation
        """
        method_result = self.next_provider.invoke(
            service_id, operation_id, input_value, ctx)
        return augment_method_result_with_errors(
            service_id, operation_id, method_result,
            self._error_values_to_augment)
