﻿/*

 ********************************************************

 * Copyright (c) VMware, Inc.2010, 2016. All Rights Reserved.

 ********************************************************

 *

 * DISCLAIMER. THIS PROGRAM IS PROVIDED TO YOU "AS IS" WITHOUT

 * WARRANTIES OR CONDITIONS OF ANY KIND, WHETHER ORAL OR WRITTEN,

 * EXPRESS OR IMPLIED. THE AUTHOR SPECIFICALLY DISCLAIMS ANY IMPLIED

 * WARRANTIES OR CONDITIONS OF MERCHANTABILITY, SATISFACTORY QUALITY,

 * NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE.

 */



namespace LoginByToken

{

    using AcquireBearerTokenByUserCredentialSample;

    using AcquireHoKTokenByUserCredentialSample;

    using System;

    using System.IdentityModel.Tokens;

    using System.Net;

    using System.Security.Cryptography.X509Certificates;

    using System.ServiceModel;

    using System.ServiceModel.Channels;

    using System.ServiceModel.Description;

    using System.Xml;

    using Vim25Api;

    using vmware.sso;

    using VMware.Binding.WsTrust;



    public class LoginByTokenSample

    {

        #region Private members



        private VimPortType _service;

        private ManagedObjectReference _svcRef;

        private ServiceContent _sic;

        private string _serverUrl;



        #endregion Private members



        # region Private function definition



        private static VimPortType GetVimService(

            string url, string username = null, string password = null,

            X509Certificate2 signingCertificate = null, XmlElement rawToken = null)

        {

            var binding = SamlTokenHelper.GetWcfBinding();

            var address = new EndpointAddress(url);



            var factory = new ChannelFactory<VimPortType>(binding, address);



            // Attach the behaviour that handles the WS-Trust 1.4 protocol for VMware Vim Service

            factory.Endpoint.Behaviors.Add(new WsTrustBehavior(rawToken));



            SamlTokenHelper.SetCredentials(username, password, signingCertificate, factory.Credentials);



            var service = factory.CreateChannel();



            return service;

        }



        #endregion



        # region Public function definition



        /// <summary>

        /// Creates new LoginByTokenSample object

        /// </summary>

        /// <param name="serverUrl">Server url to login to</param>

        public LoginByTokenSample(string serverUrl)

        {

            _serverUrl = serverUrl;

            _svcRef = new ManagedObjectReference();

            _svcRef.type = "ServiceInstance";

            _svcRef.Value = "ServiceInstance";

        }



        /// <summary>

        /// Calls the right login method based on the SAML token type (bearer or holder-of-Key)

        /// </summary>

        /// <param name="token">Token</param>

        public void LoginByToken(RequestSecurityTokenResponseType token)

        {



            _service = GetVimService(

                _serverUrl, null, null, SamlTokenHelper.GetCertificate(), token.RequestedSecurityToken);



            _sic = _service.RetrieveServiceContent(this._svcRef);

            if (_sic.sessionManager != null)

            {

                _service.LoginByToken(_sic.sessionManager, null);

            }

        }



        /// <summary>

        /// Simple method to query and print the server time

        /// </summary>

        public void PrintTime()

        {

            Console.WriteLine("Getting server time!");

            Console.WriteLine("Server time is {0}", _service.CurrentTime(_svcRef));

        }



        /// <summary>

        /// Disconnects the Connection

        /// </summary>

        public void Logout()

        {

            if (_service != null)

            {

                _service.Logout(_sic.sessionManager);

                Console.WriteLine("Logged out successfully");

            }

        }



        /// <summary>

        /// Main method

        /// </summary>

        /// <param name="args">Expects four arguments

        /// (vCenter single sign on server url, username, password, and vCenter server url)</param>

        public static void Main(string[] args)

        {

            if (args != null && args.Length >= 4)

            {

                var ssoArgs = new string[] { args[0], args[1], args[2] };

                var serverUrl = args[3];



                var program = new LoginByTokenSample(serverUrl);

                Console.WriteLine("------------------------------------");

                Console.WriteLine("Acquiring Bearer token");

                var bearerToken = AcquireBearerTokenByUserCredential.

                    GetBearerToken(ssoArgs[0], ssoArgs[1], ssoArgs[2]);

                SamlTokenHelper.PrintToken(bearerToken.RequestedSecurityToken);

                Console.WriteLine("Performing loginByToken using the Bearer token above");

                program.LoginByToken(bearerToken);

                program.PrintTime();

                program.Logout();

                Console.WriteLine("------------------------------------");

                Console.WriteLine("Acquiring HolderOfKey token");

                var hokToken = AcquireHoKTokenByUserCredential.GetToken(ssoArgs);

                SamlTokenHelper.PrintToken(hokToken.RequestedSecurityToken);

                Console.WriteLine("Performing loginByToken using the HolderOfKey token above");

                program.LoginByToken(hokToken);

                program.PrintTime();

                program.Logout();

                Console.ReadLine();

            }

            else

            {

                //print usage

                Console.WriteLine("Usage: LoginByToken [sso url] [sso username] [sso password] [server url]");

                Console.ReadLine();

            }

        }



        # endregion

    }

}