#!/usr/bin/env python3
'''
Copyright (c) 2021 VMware, Inc.  All rights reserved. -- VMware Confidential

Simple human-friendly wrapper on DataSets operations.

Note: this sample doesn't support batch mode, 'append' for large values, or
'context' for list-keys.

'''

import sys

from util import *

LIST_COMMAND = 'datasets-list'
QUERY_COMMAND = 'datasets-query'
SET_COMMAND = 'datasets-set-entry'
GET_COMMAND = 'datasets-get-entry'
DELETE_COMMAND = 'datasets-delete-entry'
LIST_KEYS_COMMAND = 'datasets-list-keys'

def usage():
   sys.exit('Usage: datasets [query|list|listkeys|set|get|delete] <args>')

def do_get(dsname, key):
   get_args = {'dataset': dsname}
   get_args['keys'] = [ key ]
   reply = run_rpc_json(GET_COMMAND, get_args)

   if (check_result(reply, True)):
      entries = json_str_to_obj(reply, 'entries')
      if (len(entries) != 1):
         print("Key %s not found in dataset %s" % (key, dsname))
      else:
         e = entries[0]
         print("%s" % e[key])
   else:
      err_msg = json_str_to_obj(reply, 'error-message')
      print("Get failed: %s" % err_msg)

def do_set(dsname, key, value):
   # XXX this doesn't handle huge values; those need 'append'
   set_args = {'dataset': dsname}
   entries = [ {'key': key, 'value': value} ]
   set_args['entries'] = entries
   reply = run_rpc_json(SET_COMMAND, set_args)

   if (check_result(reply, False)):
      err_msg = json_str_to_obj(reply, 'error-message')
      print("Set failed: %s" % err_msg)

def do_delete(dsname, key):
   delete_args = {'dataset': dsname}
   delete_args['keys'] = [ key ]
   reply = run_rpc_json(DELETE_COMMAND, delete_args)

   if (check_result(reply, False)):
      err_msg = json_str_to_obj(reply, 'error-message')
      print("Delete failed: %s" % err_msg)

def do_listkeys(dsname):
   # XXX this can't handle a more keys than can fit in a reply
   listkeys_args = {'dataset': dsname}
   reply = run_rpc_json(LIST_KEYS_COMMAND, listkeys_args)

   if (check_result(reply, True)):
      keys = json_str_to_obj(reply, 'keys')
      for k in keys:
         print("%s" % k)
   else:
      err_msg = json_str_to_obj(reply, 'error-message')
      print("Listkeys failed: %s" % err_msg)

def do_query(dsname):
   query_args = {'dataset': dsname}
   reply = run_rpc_json(QUERY_COMMAND, query_args)

   if (check_result(reply, True)):
      info = json_str_to_obj(reply, 'info')
      print("Dataset '%s':" % dsname)
      print("\tDescription: %s" % info['description'])
      print("\tUsed: %s" % info['used'])
      print("\tHost access: %s" % info['hostAccess'])
      print("\tGuest access: %s" % info['guestAccess'])
      print("\tomitFromSnapshotAndClone: %s\n" %
            info['omitFromSnapshotAndClone'])
   else:
      err_msg = json_str_to_obj(reply, 'error-message')
      print("Query failed: %s" % err_msg)

def do_list():
   cmd = LIST_COMMAND
   reply = run_rpc(cmd)
   if (check_result(reply, True)):
      datasets = json_str_to_obj(reply, 'datasets')
      for ds in datasets:
         print("Dataset: %s: %s" % (ds['name'], ds['description']))
   else:
      print("List failed: %s" % reply['error-mesage'])

def main():
   num_args = len(sys.argv)

   if (num_args < 2):
      usage()

   cmd = sys.argv[1]

   if (cmd == 'query'):
      if (num_args != 3):
          usage()
      do_query(sys.argv[2])
   elif (cmd == 'list'):
      if (num_args != 2):
         usage()
      do_list()
   elif (cmd == 'listkeys'):
      if (num_args != 3):
          usage()
      do_listkeys(sys.argv[2])
   elif (cmd == 'get'):
      if (num_args != 4):
          usage()
      do_get(sys.argv[2], sys.argv[3])
   elif (cmd == 'set'):
      if (num_args != 5):
          usage()
      do_set(sys.argv[2], sys.argv[3], sys.argv[4])
   elif (cmd == 'delete'):
      if (num_args != 4):
          usage()
      do_delete(sys.argv[2], sys.argv[3])
   else:
      print("Unknown command: %s" % cmd)
      usage()

if __name__ == '__main__':
    main()

