/* Copyright (c) 2019-2023 VMware, Inc. All rights reserved. */
import {
   HttpErrorResponse,
   HttpEvent,
   HttpHandler, HttpHeaders,
   HttpInterceptor,
   HttpRequest
} from '@angular/common/http';
import { catchError, map, mergeMap } from 'rxjs/operators';
import {
   Observable,
   ObservableInput,
   throwError
} from 'rxjs';
import { ResourceService } from '~services/resource.service';
import { Injectable } from '@angular/core';
import { GlobalService } from '~services/global.service';

export interface SessionInfo {
   sessionToken: string;
}

export interface PluginBackendInfo {
   allPluginBackendServers: Array<PluginBackendServerInfo>;
   backendServersPerVc: { [vcGuid: string]: Array<PluginBackendServerInfo> };
}

export interface PluginBackendServerInfo {
   proxiedBaseUrl: string;
   type: string;
}

/**
 * Intercepts error responses and locales the message if possible.
 */
@Injectable()
export class RemotePluginHttpInterceptpr implements HttpInterceptor {

   constructor(private globalService: GlobalService,
         private resourceService: ResourceService) {
   }

   intercept(request: HttpRequest<any>, next: HttpHandler): Observable<HttpEvent<any>> {
      if (request.url && request.url.startsWith('./')) {
         return next.handle(request);
      }
      return this.getHttpHeaders().pipe(mergeMap((httpHeaders: HttpHeaders) => {
         return this.getPluginBackends().pipe(mergeMap((backendInfo: PluginBackendInfo) => {
            if (!backendInfo.allPluginBackendServers.length) {
               return throwError(new Error(this.resourceService.getString('errors.general')));
            }
            const chassisSampleServerProxiedBaseUrl: string =
                  backendInfo.allPluginBackendServers[0].proxiedBaseUrl;
            const url =
                  `${window.location.origin}${chassisSampleServerProxiedBaseUrl}/sample-ui/rest/${request.url}`;
            const interceptedRequest: HttpRequest<any> = request.clone({
               headers: httpHeaders,
               url: url
            });
            return next.handle(interceptedRequest).pipe(catchError(
                  (response: any) => this.catchResponseError(response)));
         }));
      }));
   }

   private catchResponseError(response: HttpErrorResponse): ObservableInput<any> {
      if (!!response.error && !!response.error.localeKey) {
         response.error.message =
               this.resourceService.getString(response.error.localeKey);
         return throwError(response.error);
      }
      return throwError(response);
   }

   private getHttpHeaders(): Observable<HttpHeaders> {
      return this.getSessionInfo().pipe(map((sessionInfo: SessionInfo) => {
         return new HttpHeaders({
            'Content-Type': 'application/json;charset=utf-8',
            'Accept': 'application/json;charset=utf-8',
            'Cache-Control': 'no-cache',
            'Pragma': 'no-cache',
            'Expires': 'Sat, 01 Jan 2000 00:00:00 GMT',
            'vmware-api-session-id': sessionInfo.sessionToken
         });
      }));
   }

   private getSessionInfo(): Observable<SessionInfo> {
      return new Observable(observer => {
         this.globalService.htmlClientSdk.app.getSessionInfo((sessionInfo: SessionInfo) => {
            observer.next(sessionInfo);
            observer.complete();
         });
      });
   }

   protected getPluginBackends(): Observable<PluginBackendInfo> {
      return new Observable(observer => {
         this.globalService.htmlClientSdk.app.getPluginBackendInfo((backendInfo: PluginBackendInfo) => {
            if (backendInfo) {
               observer.next(backendInfo);
               observer.complete();
            } else {
               observer.error('Error retrieving plugin backends information.');
            }
         });
      });
   }
}
