/* Copyright (c) 2020-2023 VMware, Inc. All rights reserved. */

import {Injectable} from '@angular/core';
import {Observable, of, Subject} from 'rxjs';
import {Message} from '~models/message.model';
import {HttpClient} from '@angular/common/http';

@Injectable()
export class MessagingService {

   public readonly messages$: Observable<Message>;
   public readonly close$: Observable<never>;

   private _webSocket: WebSocket;

   private _initStarted: boolean;
   private readonly _initSubject: Subject<never>;

   private readonly _onInitWebSocketOpenBound: (event: Event) => void;
   private readonly _onInitWebSocketCloseBound: (event: CloseEvent) => void;

   private readonly _onWebSocketMessageBound: (event: MessageEvent) => void;
   private readonly _onWebSocketCloseBound: (event: CloseEvent) => void;

   private readonly _messageSubject: Subject<Message>;
   private readonly _closeSubject: Subject<never>;

   constructor(private http: HttpClient) {
      this._initStarted = false;
      this._initSubject = new Subject<never>();
      this._messageSubject = new Subject<Message>();
      this._closeSubject = new Subject<never>();

      this._onInitWebSocketOpenBound = this.onInitWebSocketOpen.bind(this);
      this._onInitWebSocketCloseBound = this.onInitWebSocketClose.bind(this);

      this._onWebSocketMessageBound = this.onWebSocketMessage.bind(this);
      this._onWebSocketCloseBound = this.onWebSocketClose.bind(this);

      this.messages$ = this._messageSubject.asObservable();
      this.close$ = this._closeSubject.asObservable();
   }

   public init$(): Observable<never> {
      if (this._initStarted) {
         throw new Error('[MessagingService] init$() has already been called.');
      }
      this._initStarted = true;

      // The WebSocket JS API does not support additional headers to be added during
      // initialization. On the other hand sending the PSID in the url as a parameter
      // is not secure, since the URL is logged by proxies and servers.
      // To workaround this security issue a special ticket is generated by the backend which
      // is related to the PSID, however an attacker can not guess the PSID based on the ticket.
      // This ticket is used in the URL parameter during the WebSocket initialization.
      this.http.get(`websocket/session/generate-ticket`)
          .subscribe((sessionTicket: string) => {
             if (!sessionTicket) {
               this._initSubject.error(undefined);
               return;
            }

            const anchorElement = document.createElement('a');
            anchorElement.href = 'messaging' +
                  `?session-ticket=${sessionTicket}`;

            const messagingUrl =
                  `wss://${anchorElement.host}${anchorElement.pathname}${anchorElement.search}`;
            this._webSocket = new WebSocket(messagingUrl);
            this.addInitWebSocketListeners();
          }, (error: Error) => {
             this._initSubject.error(error);
          });

      return this._initSubject.asObservable();
   }

   private onInitWebSocketOpen() {
      console.log('[MessagingService] WebSocket opened.');

      this.removeInitWebSocketListeners();
      this.addWebSocketListeners();

      this._initSubject.complete();
   }

   private onInitWebSocketClose(event: CloseEvent) {
      console.log('[MessagingService] WebSocket closed with:', event);

      this.removeInitWebSocketListeners();

      this._initSubject.error(undefined);
   }

   private addInitWebSocketListeners() {
      this._webSocket.addEventListener('open', this._onInitWebSocketOpenBound);
      this._webSocket.addEventListener('close', this._onInitWebSocketCloseBound);
   }

   private removeInitWebSocketListeners() {
      this._webSocket.removeEventListener('open', this._onInitWebSocketOpenBound);
      this._webSocket.removeEventListener('close', this._onInitWebSocketCloseBound);
   }

   private onWebSocketMessage(event: MessageEvent) {
      console.log('[MessagingService] Received a WebSocket message:', event);

      const message: Message = JSON.parse(event.data);
      this._messageSubject.next(message);
   }

   private onWebSocketClose(event: CloseEvent) {
      console.log('[MessagingService] WebSocket closed with:', event);

      this.removeWebSocketListeners();

      this._closeSubject.complete();
   }

   private addWebSocketListeners() {
      this._webSocket.addEventListener('message', this._onWebSocketMessageBound);
      this._webSocket.addEventListener('close', this._onWebSocketCloseBound);
   }

   private removeWebSocketListeners() {
      this._webSocket.removeEventListener('message', this._onWebSocketMessageBound);
      this._webSocket.removeEventListener('close', this._onWebSocketCloseBound);
   }
}
