/**
 * Copyright 2014 VMware, Inc. All rights reserved. VMware Confidential
 * vui-angular
 * @version v0.1.0 - 2022-02-23
 */
// Initialize all modules and the vui.angular namespace.

// Check for/initialize global vUi namespace
var vui = vui || {};

// Create this project's namespace
vui.angular = {};

// Individual module creation -------------------------------------------------

/**
 * @ngdoc overview
 * @name vui.angular.actions
 * @module vui.angular.actions
 *
 * @description
 *    # vui.angular.actions
 *    This module contains `vuiActionsMenuService` that can be used to create
 *    create an actions menu.
 *
 *    It is also a submodule of
 *    {@link vui.angular.datagrid vui.angular.datagrid}, use properties of this
 *    module to define actions.
 *
 *    For actions menu usage guidelines, see
 *    {@link http://developercenter.vmware.com/web/standards/-/actions-menu Actions Menu}.
 *
 *    To learn more about the service, see
 *    {@link vui.angular.actions:vuiActionsMenuService vuiActionsMenuService}.
 */
vui.angular.actions = angular.module('vui.angular.actions', [
   'kendo.directives',
   'vui.angular.directives',
   'vui.angular.constants',
   'vui.angular.services'
]);

/**
 * @xngdoc overview
 * @name vui.angular.alert
 * @module vui.angular.alert
 *
 * @description
 *    # vui.angular.alert
 *    This is just sample code for now.
 */
vui.angular.alert = angular.module('vui.angular.alert', []);

/**
 * @ngdoc overview
 * @name vui.angular.constants
 * @module vui.angular.constants
 *
 * @description
 *    # vui.angular.constants
 *    This helper module contains `vuiConstants` service that contains constants
 *    to configure properties of directives and services.
 *
 *    To learn more about the service, see
 *    {@link vui.angular.constants.vuiConstants vuiConstants}.
 *
 *    **Note:** Not a standalone module.
 */
vui.angular.constants = angular.module('vui.angular.constants', []);

/**
 * @ngdoc overview
 * @name vui.angular.comboBox
 * @module vui.angular.comboBox
 *
 * @description
 *    # vui.angular.comboBox
 *    This module contains `vui-combo-box` directive to create a comboBox.
 *
 *    For usage guidelines, see
 *    {@link http://developercenter.vmware.com/web/standards/-/drop-down-menus-and-combo-boxes comboBox}.
 *
 *    To learn more about the directive, see
 *    {@link vui.angular.comboBox.directive:vuiComboBox vui-combo-box}.
 */
vui.angular.comboBox = angular.module('vui.angular.comboBox', ['kendo.directives']);

/**
 * @ngdoc overview
 * @name vui.angular.datagrid
 * @module vui.angular.datagrid
 *
 * @description
 *    # vui.angular.datagrid
 *    This module contains the `vui-datagrid` directive that can be used to
 *    render complex data and provide controls to interact with the data.
 *
 *    Features include:
 *
 *    - Single and multiple column sorting
 *    - Row selection using checkbox
 *    - Grouping by column(s)
 *    - Actionbar to perform actions on selected rows
 *    - On-the-fly filtering
 *    - Accessiblity for screenreaders/keyboard access
 *    - Smart handling of column widths
 *    - Column resizing
 *    - Column reordering
 *
 *    For usage guidelines, see
 *    {@link http://developercenter.vmware.com/web/standards/-/datagrids Datagrids}.
 *
 *    To learn more about the directive, see
 *    {@link vui.angular.datagrid.directive:vuiDatagrid vui-datagrid}.
 *
 *    **Note:** vui.angular.constants is already included in this module.
 */
vui.angular.datagrid = angular.module('vui.angular.datagrid', [
   'kendo.directives',
   'vui.angular.services',
   'vui.angular.actions',
   'vui.angular.constants'
]);

/**
 * @ngdoc overview
 * @name vui.angular.dialog
 * @module vui.angular.dialog
 *
 * @description
 *    # vui.angular.dialog
 *    This module contains the `vui-dialog` directive that can be used to
 *    communicate with your users, prompt them for a response, or both.
 *
 *    Dialogs can also be created and displayed via the `vuiDialogService`.
 *
 *    For usage guidelines, see
 *    {@link http://developercenter.vmware.com/web/standards/-/dialogs Dialogs}.
 *
 *    To learn more about the directive, see
 *    {@link vui.angular.dialog.directive:vuiDialog vui-dialog}.
 *
 *    To learn more about the service, see
 *    {@link vui.angular.dialog.vuiDialogService vuiDialogService}
 *
 *    **Note:** vui.angular.constants is already included in this module.
 */
vui.angular.dialog = angular.module('vui.angular.dialog', [
   'vui.angular.modal',
   'vui.angular.services',
   'vui.angular.constants',
   'vui.angular.validationBanner'
]);

/**
 * @xngdoc overview
 * @name vui.angular.directives
 * @module vui.angular.directives
 *
 * @description
 *    # vui.angular.directives
 *    Contains general-purpose utility directives.
 */
vui.angular.directives = angular.module('vui.angular.directives', []);

/**
 * @ngdoc overview
 * @name vui.angular.dropdown
 * @module vui.angular.dropdown
 *
 * @description
 *    # vui.angular.dropdown
 *    This module contains `vui-dropdown` directive to create a dropdown list.
 *
 *    For usage guidelines, see
 *    {@link http://developercenter.vmware.com/web/standards/-/drop-down-menus-and-combo-boxes drop down menus}.
 *
 *    To learn more about the directive, see
 *    {@link vui.angular.dropdown.directive:vuiDropdown vui-dropdown}.
 */
vui.angular.dropdown = angular.module('vui.angular.dropdown', [
   'kendo.directives',
   'vui.angular.services'
]);

/**
 * @ngdoc overview
 * @name vui.angular.modal
 * @module vui.angular.modal
 *
 * @description
 *    # vui.angular.modal
 *    This module contains 'vuiModalService' generic service for creating pop-ups
 *    with show, hide, destroy methods. Used by wizard, dialogs, etc.
 *
 *    To learn more about the service, see
 *    {@link vui.angular.modal.vuiModalService vuiModalService}.
 *
 * @description
 *    # vui.angular.modal.vuiModalService
 *    Contains the
 */
vui.angular.modal = angular.module('vui.angular.modal', [
   'vui.angular.services',
   'vui.angular.constants'
]);

/**
 * @ngdoc overview
 * @name vui.angular.notification
 * @module vui.angular.notification
 *
 * @description
 *    # vui.angular.notification
 *    This module contains `vuiNotificationService` service that can be used to
 *    create notification messages about events and system status.
 *
 *    For usage guidelines, see
 *    {@link http://developercenter.vmware.com/web/standards/-/notifications-and-banners Notification}.
 *
 *    To learn more about the service, see
 *    {@link vui.angular.notification.vuiNotificationService vuiNotificationService}.
 */
vui.angular.notification = angular.module('vui.angular.notification', [
   'vui.angular.services',
   'vui.angular.constants'
]);

/**
 * @ngdoc overview
 * @name vui.angular.portlets
 * @module vui.angular.portlets
 *
 * @description
 *    # vui.angular.portlets
 *    This module contains `vui-portlets` directive that can be used as
 *    a container to present various kinds of information, often about an
 *    object. Components like tabs and stack views can be placed inside a
 *    portlet.
 *
 *    For usage guidelines, see
 *    {@link http://developercenter.vmware.com/web/standards/-/portlets Portlets}.
 *
 *    To learn more about the directive, see
 *    {@link vui.angular.portlets.directive:vuiPortlets vui-portlets}.
 *
 *    **Note:** vui.angular.constants is already included in this module.
 */
vui.angular.portlets = angular.module('vui.angular.portlets', [
   'vui.angular.services',
   'vui.angular.constants'
]);


/**
 * @ngdoc overview
 * @name vui.angular.progressBar
 * @module vui.angular.progressBar
 *
 * @description
 *    # vui.angular.progressBar
 *    This module contains `vui-progress-bar` directive that can be used
 *    to show live progress of a task or a process.
 *
 *    For usage guidelines, see
 *    {@link http://developercenter.vmware.com/web/standards/-/progress-indicators Progress Indicators}.
 *
 *    To learn more about the directive, see
 *    {@link vui.angular.progressBar.directive:vuiProgressBar vui-progress-bar}.
 *
 */
vui.angular.progressBar = angular.module('vui.angular.progressBar', [
   'vui.angular.services'
]);

/**
 * @xngdoc overview
 * @name vui.angular.services
 * @module vui.angular.services
 *
 * @description
 *    # vui.angular.services
 *    This module contains general-purpose services.
 */
vui.angular.services = angular.module('vui.angular.services', []);

/**
 * @ngdoc overview
 * @name vui.angular.slider
 * @module vui.angular.slider
 *
 * @description
 *    # vui.angular.slider
 *    This module contains `vui-slider` directive. With a slider, users drag a thumb along a bar.
 *    The thumb points to the current value, bounded by minimum and maximum values.
 *
 *    For usage guidelines, see
 *    {@link https://vdc-stg-app-vip.vmware.com/web/standards/-/sliders-and-steppers Sliders}.
 *
 *    To learn more about the directive, see
 *    {@link vui.angular.slider.directive:vuiSlider vui-slider}.
 *
 */
vui.angular.slider = angular.module('vui.angular.slider', ['kendo.directives']);

/**
 * @ngdoc overview
 * @name vui.angular.splitter
 * @module vui.angular.splitter
 *
 * @description
 *    # vui.angular.splitter
 *    This module contains `vui-splitter` directive that can be used to create
 *    a split layout where the inner panels can be collapsed or resized.
 *
 *    To learn more about the directive, see
 *    {@link vui.angular.splitter.directive:vuiSplitter vui-splitter}.
 *
 *    **Note:** vui.angular.constants is already included in this module.
 */
vui.angular.splitter = angular.module('vui.angular.splitter', [
   'kendo.directives',
   'vui.angular.constants'
]);

/**
 * @ngdoc overview
 * @name vui.angular.stackView
 * @module vui.angular.stackView
 *
 * @description
 *    # vui.angular.stackView
 *    This module contains the `vui-stack-view` and `vui-stack-block` directives.
 *
 *    A stack view is a container for displaying label/content pairs called stack blocks
 *    in a vertical stack.
 *
 *    Each stack block may also contain child stack blocks revealed by clicking the
 *    stack block label, a visual pattern called progressive disclosure.
 *
 *    For usage guidelines, see
 *    {@link http://developercenter.vmware.com/web/standards/-/stack-view Stack VIew}.
 *
 *    To learn more about the directive, see
 *    {@link vui.angular.stackView.directive:vuiStackView vui-stack-view}.
 *
 */
vui.angular.stackView = angular.module('vui.angular.stackView', [
   'vui.angular.services'
]);

/**
 * @ngdoc overview
 * @name vui.angular.tabs
 * @module vui.angular.tabs
 *
 * @description
 *    # vui.angular.tabs
 *    This module contains `vui-tabs` directive that can be used to
 *    creating three different types of tabs (top-level tabs, tab bar, table
 *    of contents).
 *
 *    For usage guidelines, see
 *    {@link http://developercenter.vmware.com/web/standards/-/common-tab-structure Tabs}.
 *
 *    To learn more about the directive, see
 *    {@link vui.angular.tabs.directive:vuiTabs vui-tabs}.
 *
 *    **Note:** vui.angular.constants is already included in this module.
 */
vui.angular.tabs = angular.module('vui.angular.tabs', [
   'vui.angular.constants',
   'vui.angular.services'
]);

/**
 * @ngdoc overview
 * @name vui.angular.treeView
 * @module vui.angular.treeView
 *
 * @description
 *    # vui.angular.treeView
 *    This module contains `vui-tree-view` directive, used for creating the Tree View navigation control.
 *
 *    For usage guidelines, see
 *    {@link http://developercenter.vmware.com/web/standards/-/trees Trees}.
 *
 *    To learn more about the directive, see
 *    {@link vui.angular.treeView.directive:vuiTreeView vui-tree-view}.
 */
vui.angular.treeView = angular.module('vui.angular.treeView', [
   'vui.angular.services'
]);

/**
 * @ngdoc overview
 * @name vui.angular.validationBanner
 * @module vui.angular.validationBanner
 *
 * @description
 *    # vui.angular.validationBanner
 *    Submodule of {@link vui.angular.dialog vui.angular.dialog} and
 *    {@link vui.angular.wizard vui.angular.wizard} which contains properties
 *    to define validation banner to display error/warning messages.
 *
 *    **Note:** Currently not supported as an standalone module.
 */
vui.angular.validationBanner = angular.module('vui.angular.validationBanner', [
   'vui.angular.constants'
]);

/**
 * @ngdoc overview
 * @name vui.angular.wizard
 * @module vui.angular.wizard
 *
 * @description
 *    # vui.angular.wizard
 *    This module contains `vuiWizardService` that can be used to
 *    create wizards. A wizard presents a multistep workflow that users perform
 *    in a recommended sequence.
 *
 *    For usage guidelines, see
 *    {@link http://developercenter.vmware.com/web/standards/-/wizards Wizards}.
 *
 *    To learn more about the service, see
 *    {@link vui.angular.wizard.vuiWizardService vuiWizardService}.
 *
 *    **Note:** vui.angular.constants is already included in this module.
 *
 *    ### Wizard FAQs
 *    1. Does the wizard allow dynamic pages ?
 *
 *       Pages can be added on the fly, page states can be
 *       updated from one state to another and will be instantly processed by the wizard.
 *
 *    2. When is the onClose function called ?
 *
 *       The onClose function is called when cancel button is clicked or when onFinish
 *       returns true. The wizard does not wait for the onClose to return a value.
 *
 *    3. How to persisit the values in the pages during navigation ?
 *
 *       It is the responsibility of the consumer to persist the state/required local variables in the global object to.
 *       Check out the example in the {@link vui.angular.wizard.vuiWizardService vuiWizardService} service docs.
 *
 *    4. Does the wizard have support for form validation ?
 *
 *       Wizard works independent of the form validation and it is left to the descretion of the consumer/page
 *       author to add any kind of form validation that is required. However, wizard comes with the
 *       validationBanner to display error messages.
 *
 */
vui.angular.wizard = angular.module('vui.angular.wizard', [
   'vui.angular.constants',
   'vui.angular.validationBanner',
   'vui.angular.modal'
]);

// The library module ---------------------------------------------------------
/**
 * @ngdoc overview
 * @name vui.angular
 * @module vui.angular
 *
 * @description
 *    # vui.angular (core module)
 *    `vui.angular` module is the root module, include this in
 *    your application to use all the available components.
 *
 *    If you only want to use one or more components, include the corresponding
 *    submodule(s).
 *
 *    List of vui.angular submodules:
 *
 *    - **{@link vui.angular.actions vui.angular.actions}**
 *       - {@link vui.angular.actions:vuiActionsMenuService vuiActionsMenuService}
 *          (`service`)
 *    - **{@link vui.angular.comboBox vui.angular.comboBox}**
 *       - {@link vui.angular.comboBox.directive:vuiComboBox vui-combo-box}
 *          (`directive`)
 *    - **{@link vui.angular.datagrid vui.angular.datagrid}**
 *       - {@link vui.angular.datagrid.directive:vuiDatagrid vui-datagrid}
 *          (`directive`)
 *    - **{@link vui.angular.dialog vui.angular.dialog}**
 *       - {@link vui.angular.dialog.directive:vuiDialog vui-dialog}
 *          (`directive`)
 *       - {@link vui.angular.dialog.vuiDialogService vuiDialogService} (`service`)
 *    - **{@link vui.angular.dropdown vui.angular.dropdown}**
 *       - {@link vui.angular.dropdown.directive:vuiDropdown vui-dropdown}
 *          (`directive`)
 *    - **{@link vui.angular.modal vui.angular.modal}**
*        - {@link vui.angular.modal.vuiModalService vuiModalService}
 *          (`service`)
 *    - **{@link vui.angular.notification vui.angular.notification}**
 *       - {@link vui.angular.notification.vuiNotificationService vuiNotificationService}
 *          (`service`)
 *    - **{@link vui.angular.portlets vui.angular.portlets}**
 *       - {@link vui.angular.portlets.directive:vuiPortlets vui-portlets}
 *          (`directive`)
 *    - **{@link vui.angular.progressBar vui.angular.progressBar}**
 *       - {@link vui.angular.progressBar.directive:vuiProgressBar vui-progress-bar}
 *          (`directive`)
 *    - **{@link vui.angular.slider vui.angular.slider}**
 *       - {@link vui.angular.slider.directive:vuiSlider vui-slider} (`directive`)
 *    - **{@link vui.angular.splitter vui.angular.splitter}**
 *       - {@link vui.angular.splitter.directive:vuiSplitter vui-splitter} (`directive`)
 *    - **{@link vui.angular.tabs vui.angular.tabs}**
 *       - {@link vui.angular.tabs.directive:vuiTabs vui-tabs} (`directive`)
 *    - **{@link vui.angular.treeView vui.angular.treeView}**
 *       - {@link vui.angular.treeView.directive:vuiTreeView vui-tree-view} (`directive`)
 *    - **{@link vui.angular.wizard vui.angular.wizard}**
 *       - {@link vui.angular.wizard.vuiWizardService vuiWizardService} (`service`)
 */
vui.angular.vuiAngular = angular.module('vui.angular', [
   'vui.angular.actions',
   // remove or rename alert module as required
   'vui.angular.alert',
   'vui.angular.comboBox',
   'vui.angular.constants',
   'vui.angular.datagrid',
   'vui.angular.dialog',
   'vui.angular.directives',
   'vui.angular.dropdown',
   'vui.angular.modal',
   'vui.angular.notification',
   'vui.angular.portlets',
   'vui.angular.progressBar',
   'vui.angular.services',
   'vui.angular.slider',
   'vui.angular.splitter',
   'vui.angular.stackView',
   'vui.angular.tabs',
   'vui.angular.treeView',
   'vui.angular.wizard'
]);
;'use strict';
vui.angular.actions

/**
 * @ngdoc object
 * @name vui.angular.actions:ActionbarOptions
 * @module vui.angular.actions
 *
 * @description
 *    Configuration object for actions in the actionbar.
 *
 *    Example:
 *    ```js
 *    var actionBarOptions = {
 *                               actions: [ ... ]
 *                           };
 *    ```
 *
 * @property {Array.<Object>} actions
 *    Type: `Array.<Object>`
 *
 *    An array containing {@link vui.angular.actions:Action Action} objects.
 *
 *    Use `vuiConstants.actions.SEPARATOR` to insert a separator between
 *    actions.
 *
 *    Example:
 *    ```js
 *    actions: [ action1, vuiConstants.actions.SEPARATOR, action2 ];
 *    ```
 *    `action1` and `action2` are {@link vui.angular.actions:Action Action}
 *    objects.
 */

/**
 * @ngdoc object
 * @name vui.angular.actions:Action
 * @module vui.angular.actions
 *
 * @description
 *    Contains properties to define an action in the actionbar.
 *
 *    Example:
 *    ```js
 *    var action1 = {
 *                      id: action1,
 *                      label: 'Add',
 *                      iconClass: 'vui-icon-action-add',
 *                      onClick: clickFunction
 *                   };
 *    ```
 *
 * @property {string} [id]
 *    Type: `string` **|** Optional
 *
 *    A unique identifier for the action.
 * @property {string} label
 *    Type: `string`
 *
 *    The display text for the action.
 *    <script>$('.properties .label').removeClass('label');</script>
 * @property {string} [tooltipText]
 *    Type: `string` **|** Optional
 *
 *    The text that will be displayed in the action tooltip on hover.
 *    If content is not specified, {@link vui.angular.actions:Action label}
 *    text will be used as tooltipText.
 * @property {string} [iconClass]
 *    Type: `string` **|** Optional
 *
 *    The CSS class name representing the icon displayed in the
 *    action.
 *
 *    **Note:** Please refer to the vui-bootstrap documentation for a list of
 *    available icon classes. A custom class specifying a 16x16 icon can also
 *    be used.
 * @property {Function} onClick
 *    Type: `Function`
 *
 *    A function that will be called when the action is
 *    clicked. The click event and the
 *    {@link vui.angular.actions:Action Action} object will be passed as
 *    parameters to the function.
 *
 *    Example:
 *    ```js
 *    onClick: function (event, action) {
 *                ...
 *             };
 *    ```
 * @property {boolean} [enabled]
 *    Type: `boolean` **|** Optional **|** Default: `true`
 *
 *    Disables the action when set to false.
 * @property {boolean} [visible]
 *    Type: `boolean` **|** Optional **|** Default: `true`
 *
 *    Shows the action when set to true.
 */

/**
 * @xngdoc directive
 * @name vui.angular.actions.directive:vuiActionbar
 * @module vui.angular.actions
 * @restrict A
 * @scope
 *
 * @description
 *    A simple actionbar directive
 *
 * @param {ActionbarOptions} vuiActionbar
 *    {@link vui.angular.actions:ActionbarOptions ActionbarOptions}
 *    Configuration object for the actionbar.
 *
 * @example
      <example module="app">

         <file name="app.js">
            'use strict';
            var app = angular.module('app', ['vui.angular']);

            app.controller('ActionBarCtrl', ['$scope', 'vuiConstants',
                  'dataService', function ($scope, vuiConstants, dataService) {

               var actionSet = dataService.fetchBasicActionSet($scope);

               // options object for the vui-action-bar
               $scope.actionBarOptions = {
                  actions: actionSet
               };
            }])
            .service('dataService', function (vuiConstants) {

               this.fetchBasicActionSet = function (scope) {
                  //generic onClick function
                  var clickFunction = function (event, action) {
                     scope.actionClicked = action.label;
                  };

                  var actionSet = [
                     {
                        id: 1,
                        label: 'Add',
                        tooltipText: 'Add tooltip',
                        enabled: true,
                        iconClass: 'vui-icon-action-add',
                        onClick: clickFunction
                     }, {
                        id: 2,
                        label: 'Edit',
                        tooltipText: 'Edit tooltip',
                        enabled: true,
                        iconClass: 'vui-icon-action-edit',
                        onClick: clickFunction
                     }, {
                        id: 3,
                        label: 'Copy',
                        tooltipText: 'Copy tooltip',
                        enabled: true,
                        iconClass: 'vui-icon-action-copy',
                        onClick: clickFunction
                     }, vuiConstants.actions.SEPARATOR, {
                        id: 4,
                        label: 'Delete',
                        tooltipText: 'Delete tooltip',
                        enabled: false,
                        iconClass: 'vui-icon-action-delete',
                        onClick: clickFunction
                     }
                  ];

                  return actionSet;
               };
            });
         </file>

         <file name="index.html">
            <link rel="stylesheet" href="css/vui-bootstrap.min.css"
                  type="text/css">
            <div ng-controller="ActionBarCtrl">
               <div>
                  <!-- vui-action-bar directive -->
                  <div vui-action-bar="actionBarOptions"></div>
               </div>
               <br/>
               <div>
                  <b> Last action clicked: </b> {{actionClicked}}
               </div>
            </div>
         </file>

      </example>
 */

.directive('vuiActionBar', ['$log', 'jsUtils', 'vuiConstants', '$timeout',
      function ($log, jsUtils, vuiConstants, $timeout) {
   var directiveDefinitionObject = {
      scope: {
         options: '=vuiActionBar'
      },
      restrict: 'A',
      replace: true,
      template: '<div class="vui-action-bar">' +
                  '<ul>' +
                     '<!-- loop over each action -->' +
                     '<li id="{{action.id}}" ' +
                           'ng-repeat="action in options.actions track by $index"' +
                           'ng-if="isVisible(action)">' +
                        '<span ng-if="isSeparator(action)"' +
                              'class="vui-actionbar-separator"></span>' +
                        '<!-- applying a disabled class-->' +
                        '<a ng-if="!isSeparator(action)"' +
                              'ng-attr-tabindex="{{isDisabled(action) ? \'-1\' : undefined}}"' +
                              'ng-class="{disabled: isDisabled(action)}"' +
                              'ng-disabled="isDisabled(action)"' +
                              'class="action-link-container"' +
                              'title="{{action.tooltipText && ' +
                              'action.tooltipText.trim().length ? ' +
                              'action.tooltipText : action.label}}" ' +
                              'aria-disabled="{{isDisabled(action)}}" ' +
                              'aria-label="{{action.tooltipText && ' +
                              'action.tooltipText.trim().length ? ' +
                              'action.tooltipText : action.label}}" ' +
                              'ng-click="clickAction($event,action)" vui-click-once ' +
                              'ng-keydown = "handleKeydown($event)" ' +
                              'role="button" ' +
                              'href="">' +
                           '<!-- the icon -->' +
                           '<span ng-if="action.iconClass"' +
                              ' class="vui-icon-placeholder {{action.iconClass}}">' +
                           '</span>' +
                           '<!-- the label -->' +
                           '<span ng-if="action.label" class="vui-action-label">' +
                              '{{action.label}}</span>' +
                           '<span ng-if="action.isMenu" class="caret"></span>' +
                        '</a>' +
                     '</li>' +
                  '</ul>' +
               '</div>',
      link: function (scope) {
         if (!scope.options.actions) {
            throw new Error('No actions passed.');
         }
         scope.options.actions.forEach(function (action) {
            var isSeparator = action === vuiConstants.actions.SEPARATOR;
            var hasIcon = action.iconClass && action.iconClass.trim().length > 0;
            var hasLabel = action.label && action.label.trim().length > 0;
            if (!isSeparator && !hasIcon && !hasLabel) {
               throw new Error('Icon or label must be specified for each action.');
            }
         });

         //Returns true if action is the string SEPARATOR
         scope.isSeparator = function (action) {
            return action === vuiConstants.actions.SEPARATOR ? true : false;
         };

         //Return true if action is visible
         scope.isVisible = function (action) {
            return jsUtils.getProperty(action, 'visible', true);
         };

         //Return true if action is disabled
         scope.isDisabled = function (action) {
            return !jsUtils.getProperty(action, 'enabled', true);
         };

         /**
         * @name vui.angular.actionbar.VuiActionbar:clickAction
         * @methodOf vui.angular.actionbar.VuiActionbar
         * @param action - The action object
         * @param event - The jquery click event
         * @description
         * A click handler that calls the click function given for the action
         * passing in the event and the action object
         */
         scope.clickAction = function (event, action) {
            // handle event propogation when action is disabled
            if (!action.enabled) {
               event.stopPropagation();
               event.preventDefault();
            } else
            if (angular.isDefined(action.onClick) &&
                  angular.isFunction(action.onClick)) {
               action.onClick.call(undefined, event, action);
            } else {
               $log.warn('No onClick action defined for action id: ' +
                  action.id + ' label: ' + action.label);
            }
         };


         /**
          * @name vui.angular.actionbar.VuiActionbar:handleKeydown
          * @methodOf vui.angular.actionbar.VuiActionbar
          * @param event - The jquery keydown event
          * @description
          * A keypress handler that simulate click action
          * when the enter or space key is pressed
          */
         scope.handleKeydown = function (event) {
            if (event.keyCode === vuiConstants.internal.keys.SPACE ||
                  event.keyCode === vuiConstants.internal.keys.ENTER) {
               event.stopImmediatePropagation();
               event.preventDefault();
               $timeout(function () {
                  $(event.target).click();
               }, 0);
            }
         };
      }
   };

   return directiveDefinitionObject;
}]);
;'use strict';
vui.angular.actions

/**
 * @ngdoc object
 * @name vui.angular.actions:ActionsMenuOptions
 * @module vui.angular.actions
 *
 * @description
 *    Configuration object for
 *    {@link vui.angular.actions:vuiActionsMenuService vuiActionsMenuService}
 *    service.
 *
 *    Describes the order and layout of the items in a menu. Use this recursive
 *    data structure to build a tree of actions, separators and sub-menus.
 *
 *    **Note:** If rendered as a top level menu, the value for
 *    label and iconClass represents a label and an icon for the menu header.
 *
 * @property {string>} [label]
 *    Type: `string` **|** Optional
 *
 *    The display text for the menu.
 *
 * @property {string} [iconClass]
 *    Type: `string` **|** Optional
 *
 *    The CSS class name representing the icon displayed in the menu.
 *
 *    **Note:** Please refer to the vui-bootstrap documentation for a list of
 *    available icon classes. A custom class specifying a 16x16 icon can also
 *    be used.
 *
 * @property {Array<ActionMenuItem|Action|string>} items
 *    Type: `Array<ActionMenuItem|Action|string>`
 *
 *    An array containing {@link vui.angular.actions:Action Action} objects
 *    or {@link vui.angular.actions:ActionsMenuOptions ActionsMenuOptions}
 *    object. Add
 *    {@link vui.angular.actions:ActionsMenuOptions ActionsMenuOptions}
 *    object to create a submenu.
 *
 *    **Note:** {@link vui.angular.actions:Action Action} `visible` property
 *    is not supported in action inside actions menu.
 *
 *    Example:
 *    ```js
 *    items: [ subMenuOptions1,
 *             action1,
 *             vuiConstants.actions.SEPARATOR,
 *             action2 ]
 *    ```
 *    `subMenuOptions1` is an
 *    {@link vui.angular.actions:ActionsMenuOptions ActionsMenuOptions}
 *    object.
 *
 *    `action1` and `action2` are {@link vui.angular.actions:Action Action}
 *    objects.
 */

/**
 * @ngdoc object
 * @name vui.angular.actions:PositionOptions
 * @module vui.angular.actions
 *
 * @description
 *    Configuration object for
 *    {@link vui.angular.actions:vuiActionsMenuService vuiActionsMenuService}
 *    service.
 *
 *    Describes the position on the screen, where the menu should appear.
 *
 * @property {number>} x
 *    Type: `number`
 *
 *    X coordinate in pixels.
 *
 * @property {number} y
 *    Type: `number`
 *
 *    Y coordinate in pixels.
 */

/**
 * @ngdoc method
 * @name showMenu
 * @methodOf vui.angular.actions:vuiActionsMenuService
 *
 * @param {Object} config
 *    Object containing the scope, configObjectName and target.
 *
 *    - **scope** – `{$scope}` – Angular scope object of the component
 *       initiating actions menu.
 *       It can be the scope of the controller or rootScope or a custom
 *       scope.
 *
 *    - **configObjectName** – `{string}` – The name of the scope variable
 *       declared on the above mentioned scope. The variable is of type
 *       {@link vui.angular.actions:ActionsMenuOptions ActionsMenuOptions}.
 *
 *    - **target** - `{string}` - Specifies the dom element on which menu
 *       will appear anchored to the bottom left.
 *
 *    - **coordinates** - `{PositionOptions}` - Optional. Specifies the position,
 *       where menu will appear. If not specified, the `target` element will
 *       be used as an anchor. The variable is of type
 *       {@link vui.angular.actions:PositionOptions PositionOptions}.
 *
 *    - **menuContainerId** - `{string}` - Specifies the identifier of a DOM element
 *       inside the document (mostly a div), to which the context menu is to be appended
 *       to.
 *       If not provided, the document body will be used.
 *
 *    **Note:** If the menu container has '"vui-menu-scroll-wrapper' class, the context menu will support
 *       vertical scrolling when it is bigger than the window size in which it is displayed.
 *       In order the scrolling to behave correctly, the context menu and its items have
 *       to have static position.
 *       Example for css rules:
 *       #vui-actions-menu.vui-menu-scroller {
 *          position: static
 *          overflow-y: auto;
 *          overflow-x: hidden;
 *          > li {
 *             position: static;
 *             > .k-link:first-child {
 *                position: relative;
 *             }
 *             &.k-state-border-down > .k-animation-container {
 *                width:auto;
 *                > ul.k-menu-group:first-child {
 *                   position: relative;
 *                }
 *             }
 *          }
 *       ]
 *
 *
 */

/**
 * @ngdoc service
 * @name vui.angular.actions:vuiActionsMenuService
 * @module vui.angular.actions
 * @restrict A
 * @scope
 *
 * @description
 *    Service to create multiple instances of the actions menu.
 *    Actions menu displays full set of actions that can run on an
 *    object.
 *
 *    Example:
 *    ```js
 *    var config = {
 *                     scope: $scope,
 *                     configObjectName: 'actionsMenuOptions',
 *                     target: '#menu-btn'
 *                 };
 *    ```
 *
 * @example
      <example module="app">

         <file name="app.js">
            'use strict';
            var appModule = angular.module('app', ['vui.angular.actions']);

            appModule.controller('MenuServiceCtrl', ['$scope',
                  'vuiActionsMenuService', 'vuiConstants',
                  function ($scope, vuiActionsMenuService, vuiConstants) {

               var clickFunction = function (event, menuItem) {
                  console.log(menuItem.label);
               };

               $scope.actionsMenuOptions = {
                  label: 'Additional actions',
                  items: [{
                     label: 'Add new record',
                     iconClass: 'vui-icon-action-add',
                     onClick: clickFunction
                  }, {
                     label: 'Edit existing record',
                     iconClass: 'vui-icon-action-edit',
                     onClick: clickFunction
                  }, {
                     label: 'Copy record(s)',
                     items: [{
                        label: 'Copy single record',
                        iconClass: 'vui-icon-action-copy',
                        onClick: clickFunction
                     }, {
                        label: 'Copy multiple records',
                        iconClass: 'vui-icon-action-copy',
                        onClick: clickFunction
                     }]
                  }, vuiConstants.actions.SEPARATOR, {
                     label: 'Delete record',
                     iconClass: 'vui-icon-action-delete',
                     enabled: false,
                     onClick: clickFunction
                  }]
               };

               var options = {
                  scope: $scope,
                  configObjectName: 'actionsMenuOptions',
                  target: '#menu-btn'
               };

               $scope.createMenu = function () {
                  vuiActionsMenuService.showMenu(options);
               };
            }]);
         </file>

         <file name="index.html">
            <div ng-controller="MenuServiceCtrl">
               <div class="container" style="height: 200px">
                  <a id="menu-btn" href="" ng-click="createMenu()">
                     Click here to see menu
                  </a>
               </div>
            </div>
         </file>
      </example>
 */

.service('vuiActionsMenuService', ['$compile', '$window', '$log', 'jsUtils',
      'vuiConstants', 'vuiLocale',
      function ($compile, $window, $log, jsUtils, vuiConstants, vuiLocale) {

   var bodyElement = angular.element($window.document.body);
   //Set to true when the preceding menu item is a 'SEPARATOR'
   var isSeparated = false;

   var actionsMenu = null;
   var menuItemsById = {};
   var isActionsMenuClosing = false;
   var menuScope = null;
   var menuParentScopeListeners = [];
   var menuFocusTarget = null;

   // ----------------------------------------------------------------------
   // Helper functions
   var calculateTopFromMenuContainer = function (subMenuElement, menuItemOffsetTop) {
      var offsetTop;
      var windowHeight = $(window).height();
      if (menuItemOffsetTop + subMenuElement.outerHeight() > windowHeight) {
         offsetTop = Math.max(0, windowHeight - subMenuElement.outerHeight());
      } else {
         offsetTop = menuItemOffsetTop;
      }

      return offsetTop;
   };

   var positionSubMenuInContextMenuWithScroller = function (menuItem) {
      var subMenu = menuItem.find('.k-menu-group:first');
      if (subMenu[0]) {
         var offsetTop;
         var subMenuContainer = menuItem.find('> .k-animation-container');
         if (subMenuContainer[0]) {
            offsetTop = calculateTopFromMenuContainer(subMenuContainer, menuItem.offset().top);
         } else {
            offsetTop = calculateTopFromMenuContainer(subMenu, menuItem.offset().top);
         }

         subMenu.offset({ top: offsetTop });

         //Set container top offset of sub menu
         setTimeout(function () {
            subMenuContainer = menuItem.find('> .k-animation-container');
            if (subMenuContainer[0]) {
               subMenuContainer.offset({ top: offsetTop });
               subMenu.css({
                  top: 0
               });
            }
         }, 0);
      }
   };

   var destroyMenu = function () {
      isActionsMenuClosing = true;
      menuItemsById = {};

      if (actionsMenu !== null) {
         // amarinov: Manually close the menu so we can safely remove the element after the close has finished
         actionsMenu.close();
         actionsMenu.destroy();
         actionsMenu.element.remove();
         actionsMenu = null;
      }

      // Destroy child scope when menu is removed from dom
      if (menuScope !== null) {
         menuScope.$destroy();
         menuScope = null;
      }

      if (menuParentScopeListeners.length > 0) {
         for (var i = 0; i < menuParentScopeListeners.length; i++) {
            // deregister the listener (see scope.$on and scope.$watch)
            menuParentScopeListeners[i]();
         }
         menuParentScopeListeners = [];
      }

      isActionsMenuClosing = false;
   };

   var createUserMenuItem = function (menuItem) {
      if (!jsUtils.isEmpty(menuItem)) {
         // Map menuItem properties (Kendo) to action properties
         return {
            label: menuItem.text,
            iconClass: menuItem.spriteCssClass,
            onClick: menuItem.onClick
         };
      }
      return;
   };

   var clickAction = function (event, menuItem) {
      var userMenuItem = {};

      if (menuItem.onClick) {
         userMenuItem = createUserMenuItem(menuItem);
         userMenuItem.onClick.call(undefined, event, userMenuItem);
      } else {
         console.error('No onClick action defined for menu item ' +
               'labelled: ' + menuItem.text);
      }

      //Destroy menu
      destroyMenu();
   };

   var onSelect = function (e) {
      // Prevent default action of select event
      // when clicked item is menu header or submenu title
      if ($(e.item).hasClass('menu-header') ||
         $(e.item).find('ul.k-menu-group').length > 0) {
         e.preventDefault();
      } else {
         var selectedItem = $(e.item);
         var selectedItemId = selectedItem.find(
            '> span.k-link > span.vui-menuitem-label-text').attr('menu-item-id');
         var menuItem = menuItemsById[selectedItemId];
         clickAction(e, menuItem);
      }
   };

   var getCssClass = function (menuItem) {
      var cssClass = '';
      if (isSeparated) {
         cssClass = cssClass + ' is-separated';
         isSeparated = false;
      }
      if (!jsUtils.getProperty(menuItem, 'enabled', true)) {
         cssClass = cssClass + ' k-state-disabled';
      }

      if (menuItem.keyboardShortcut) {
         cssClass = cssClass + ' vui-menuitem-with-shortcut';
      }
      return cssClass;
   };

   var getSpriteCssClass = function (menuItem) {
      return menuItem.iconClass ? menuItem.iconClass : 'vui-icon';
   };

   var createKendoMenuItems = function (menuItems) {
      var kendoMenuItem;
      var kendoMenuItems = [];
      if (menuItems.length > 0) {
         for (var i = 0; i < menuItems.length; i++) {
            var menuItem = menuItems[i];

            // Set isSeparated to true and when a 'SEPARATOR' item is found
            if (menuItem === vuiConstants.actions.SEPARATOR) {
               isSeparated = true;
               continue;
            }
            // When isSeparated is true, additional 'is-separated' class is
            // appended to the cssClass(s) of current menu item
            menuItemsById[menuItem.id] = menuItem;

            kendoMenuItem = {
               text: getFormattedMenuItemText(menuItem),
               encoded: false,
               cssClass: getCssClass(menuItem),
               spriteCssClass: getSpriteCssClass(menuItem),
               onClick: menuItem.onClick
            };
            // Has sub-menu
            if (menuItem.items) {
               kendoMenuItem.items =
                     createKendoMenuItems(menuItem.items);
            }
            kendoMenuItems.push(kendoMenuItem);
         }
      }
      return kendoMenuItems;
   };

   var getFormattedMenuItemText = function (menuItem) {
      // Kendo-endorsed workaround for identifying which menu item has been selected:
      // http://www.telerik.com/forums/menu-item-select-event#5kR2ahcPG0-D4Rboza8Wug
      var idAttr = 'menu-item-id="' + menuItem.id + '" ';

      var menuText = '<span class="vui-menuitem-label-text" ' + idAttr + '>' +
         menuItem.label + '</span>';

      var keyboardShortcut = menuItem.keyboardShortcut;
      if (keyboardShortcut) {
         menuText += '<span class="vui-menuitem-shortcut-text">' + keyboardShortcut + '</span>';
      }

      return menuText;
   };

   var createKendoMenuDataSource = function (menuOptions) {
      var kendoData = [];

      // Actions menu header
      if (menuOptions.label) {
         kendoData.push({
            text: menuOptions.label,
            spriteCssClass: getSpriteCssClass(menuOptions),
            cssClass: 'menu-header'
         });
      }

      // Loop through menu items
      if (menuOptions.items && menuOptions.items.length > 0) {
         kendoData = kendoData
               .concat(createKendoMenuItems(menuOptions.items));
      } else if (!menuOptions.label) {
         // No items in actionsMenu display empty message
         // or no items to show
         kendoData.push({
            text: vuiLocale.actionsMenu.empty,
            cssClass: 'menu-header'
         });
      }
      return kendoData;
   };

   var createKendoMenuOptions = function (menuOptions, menuTarget) {
      return {
         alignToAnchor: true,
         popupCollision: 'fit flip',
         animation: false,
         select: onSelect,
         showOn: 'click',
         target: menuTarget,
         copyAnchorStyles: false,
         activate: function () {
            if (menuOptions.focusable === true) {
               $('#vui-actions-menu').focus();
            }
         },
         open: function (e) {
            // Get context menu if it has scroller (class vc-ui-actions-menu-scroller)
            var mainMenu = $(e.item).parent('.k-menu.k-context-menu.' +
                  vuiConstants.actions.VUI_MENU_SCROLLER_CLASS);
            // Check if the opened menu is sub menu of the context menu with scroller
            if (mainMenu[0] && mainMenu[0] !== $(e.item)[0]) {
               positionSubMenuInContextMenuWithScroller($(e.item));
            }
         },
         close: function (e) {
            // On close destroy menu
            // Prevent close if the menu is the context menu with scroller
            // and the user has clicked on the scroll bar
            if (e.event && e.event.target === $(e.item)[0] &&
                  $(e.item).hasClass(vuiConstants.actions.VUI_MENU_SCROLLER_CLASS)) {
               e.preventDefault();
               return;
            }
            // On close destroy menu
            var menu = $(e.item).closest('.k-menu.k-context-menu');
            // Check if the closed menu is context menu with scroller
            if (menu[0] === $(e.item)[0] && menu.hasClass(vuiConstants.actions.VUI_MENU_SCROLLER_CLASS)) {
               // Reset menu container
               $(menu[0]).parent().css({ top: '', maxHeight: '' });
            }

            // Check if the closed menu is context menu and not a menu group
            // amarinov: If the isActionsMenuClosing flag is true it means that we closing the menu manually i.e. the event is programmatic,
            // so we do not prevent the default behavior and we are also avoiding infinite loop
            if (menu[0] === $(e.item)[0] && !isActionsMenuClosing) {
               // if the menu is focusable and is being closed not as a result
               // of an action click => return the focus to the menu target object
               if (menuOptions.focusable === true) {
                  $(menuFocusTarget).focus();
               }
               // amarinov: Prevent normal close event i.e. close by clicking outside, etc. instead we will close the menu manually in the
               // destroyMenu() method
               e.preventDefault();
               destroyMenu();
            }
         },
         dataSource: createKendoMenuDataSource(menuOptions)
      };
   };

   //TODO aakankshas: Need to persist sub-menu open states
   //Ref http://docs.telerik.com/kendo-ui/api/javascript/ui/contextmenu#methods-enable
   var injectMenu = function (localscope, menuOptions, target, coordinates, menuContainerId) {
      // Source template
      var template = '<ul id="vui-actions-menu" data-role="menu"> </ul>';

      var menuElement = $compile(template)(localscope);

      var menuContainer, menuTarget;
      if (menuContainerId) {
         menuContainer = wrapMenu(menuElement, menuContainerId);
         // note mibryamov: even if target is given, it is not taken into account
         // because otherwise the menu is being added as a child of an auto
         // created div element that is added as a child of the body, so in order
         // to have it appended to the menu contain, the target should be
         // menuContainer element
         menuTarget = menuContainer;

         // if the real target is available, use to give focus back when needed
         menuFocusTarget = target ? target : menuContainer;
      } else {
         menuContainer = bodyElement;
         // Default value for target is 'body'.
         menuTarget = target ? target : 'body';

         // in this case focus target is the menu target
         menuFocusTarget = menuTarget;
      }

      // Initialize and save the kendo context menu.
      menuElement.kendoContextMenu(
            createKendoMenuOptions(menuOptions, menuTarget));

      // Set global kendoContextMenu
      actionsMenu = menuElement.data('kendoContextMenu');
      if (!jsUtils.isEmpty(coordinates)) {
         // If the menu is about to extend through the right edge of the screen,
         // shift it to the left. Vertical fitting is handled by the kendoMenu.
         coordinates.x = Math.min(coordinates.x,
            (bodyElement.outerWidth() - menuElement.outerWidth() - 1));
         actionsMenu.open(coordinates.x, coordinates.y);
      } else {
         actionsMenu.open();
      }

      configureVirtualMenuScroller(menuContainer, menuElement);

   };

   var wrapMenu = function (menuElement, menuContainerId) {
      var menuContainer = angular.element('#' + menuContainerId);
      // The classes k-popup k-animation-container are necessary for this to work.
      // Known by reverse engineering kendo.custom.js
      menuContainer.addClass('k-popup k-animation-container');
      menuContainer.append(menuElement);
      menuContainer = kendo.wrap(menuElement);
      return menuContainer;
   };

   var configureVirtualMenuScroller = function (menuContainer, menuElement) {
      var windowHeight = $(window).height();
      var menuContainerOuterHeight = menuContainer.outerHeight(true);
      if (windowHeight < menuContainerOuterHeight &&
         menuContainer.hasClass(vuiConstants.actions.VUI_MENU_SCROLL_WRAPPER_CLASS)) {
         var menuContainerVerticalEdgeThickness = menuContainerOuterHeight - menuContainer.height();
         menuContainer.css({maxHeight: windowHeight - menuContainerVerticalEdgeThickness, top: 0});
         var menuElementVerticalEdgeThickness = menuElement.outerHeight(true) - menuElement.height();
         menuElement.css({maxHeight: windowHeight - menuElementVerticalEdgeThickness});
         menuElement.addClass(vuiConstants.actions.VUI_MENU_SCROLLER_CLASS);
      }
   };

   // -------------------------------------------------------------------------
   // Initialization
   var createActionsMenu = function (config) {
      var scope = config.scope;
      var menuOptions = scope[config.configObjectName];
      var target = config.target;
      var coordinates = config.coordinates;
      var menuContainerId = config.menuContainerId;

      // new child scope to perform destroy
      menuScope = scope.$new();

      // Remove menu element and destroy scope when parent scope is destroyed
      menuParentScopeListeners.push(scope.$on('$destroy', function () {
         destroyMenu();
      }));

      // set a deep watch on options.items
      menuParentScopeListeners.push(scope.$watch(config.configObjectName + '.items',
            function (newItems, oldItems) {
         if (typeof newItems !== 'undefined' && newItems !== oldItems) {
            updateMenuItems(oldItems, newItems);
         }
      }, true));

      injectMenu(menuScope, menuOptions, target, coordinates, menuContainerId);
   };

   var isUpdatePossible = function (oldItems, newItems) {
      if (!oldItems || oldItems.length === 0) {
         return true;
      }
      if (!newItems) {
         return false;
      }

      if (newItems.length < oldItems.length) {
         return false;
      }

      for (var i = 0; i < oldItems.length; i++) {
         var oldItem = oldItems[i];
         var newItem = newItems[i];
         var changedProperties = getChangedProperties(oldItem, newItem);
         // if anything else is changed besides `enabled` or `items` properties - the menu should be rerendered
         if ((changedProperties.indexOf('enabled') !== -1 && changedProperties.length > 1) ||
            (changedProperties.indexOf('enabled') === -1 && changedProperties.length !== 0)) {
            return false;
         }
      }
      return true;
   };

   var getChangedProperties = function (oldItem, newItem) {
      var changedProps = [];
      Object.keys(oldItem).forEach(function (prop) {
         if (prop !== 'items' && oldItem[prop] !== newItem[prop]) {
            changedProps.push(prop);
         }
      });
      return changedProps;
   };

   var updateMenuItems = function (oldItems, newItems, parentId) {
      var elementsToRemove = [];
      if (!isUpdatePossible(oldItems, newItems)) {
         // If there are changes to the menu that cannot be applied by just
         // changing the enabled state of items or appending new items,
         // remove all items and add the new items in their place.
         oldItems.forEach(function (menuItem) {
            if (menuItem !== 'separator') {
               var elementToRemoveSelector = 'span[menu-item-id="' + menuItem.id + '"]';
               elementsToRemove.push($(elementToRemoveSelector).parent().parent());
            }
         });
         oldItems = [];
      }

      for (var i = 0; i < oldItems.length; i++) {
         var item = newItems[i];
         //skipping the separators
         if (item !== 'separator') {
            var id = oldItems[i].id;
            var selector = 'span[menu-item-id="' + id + '"]';
            actionsMenu.enable($(selector).parent().parent(), item.enabled);
            if (item.items) {
               updateMenuItems(oldItems[i].items, item.items, id);
            }
         }
      }

      var appendedItems = newItems.slice(oldItems.length);
      if (appendedItems.length > 0) {
         var kendoItemsToAppend = createKendoMenuItems(appendedItems);
         if (parentId === undefined || parentId === null) {
            actionsMenu.append(kendoItemsToAppend);
         } else {
            var parentSelector = 'span[menu-item-id="' + parentId + '"]';
            actionsMenu.append(kendoItemsToAppend, $(parentSelector).parent().parent());
         }
      }

      // Remove old items after appending new ones to prevent submenus from closing.
      elementsToRemove.forEach(function (element) {
         actionsMenu.remove(element);
      });
   };

   // -------------------------------------------------------------------------
   // Public methods
   return {
      showMenu: function (config) {
         if (jsUtils.isUndefinedOrNull(config)) {
            $log.warn('Actions menu parameter (config) undefined.');
            return;
         }

         if (!jsUtils.hasProperty(config, 'configObjectName')) {
            $log.warn('Actions menu parameter (config) is missing required ' +
                  'parameter `configObjectName`.');
            return;
         }

         destroyMenu();
         createActionsMenu(config);
      }
   };

}]);
;'use strict';
vui.angular.alert

.controller('AlertController', ['$scope', '$attrs', function ($scope, $attrs) {
   $scope.closeable = 'close' in $attrs;
}])

/**
 * @name vui.angular.alert.directive:alert
 * @module vui.angular.alert
 * @restrict EA
 * @scope
 *
 * @description
 *    Alert is an AngularJS-version of bootstrap's alert. This directive can be used to
 *    generate alerts from the dynamic model data (using the ng-repeat directive);
 *    The presence of the "close" attribute determines if a close button is displayed.
 *
 * @example
      <example module="app">

         <file name="app.js">
            var app = angular.module('app', ['vui.angular.alert']);

            app.controller('AlertDemoCtrl', ['$scope', function($scope) {
               $scope.alerts = [
                  { type: 'danger', msg: 'This is a DANGER message' },
                  { type: 'success', msg: 'Success message' }
               ];

               $scope.addAlert = function() {
                  $scope.alerts.push({msg: 'New alert!'});
               };

               $scope.closeAlert = function(index) {
                  $scope.alerts.splice(index, 1);
               };
            }]);
         </file>

         <file name="index.html">
            <div ng-controller="AlertDemoCtrl">
               <alert ng-repeat="alert in alerts" type="{{alert.type}}" close="closeAlert($index)">{{alert.msg}}</alert>
               <button class='btn btn-default' ng-click="addAlert()">Add Alert</button>
            </div>
         </file>

      </example>
 */

.directive('alert', function () {
   return {
      restrict: 'EA',
      controller: 'AlertController',
      // TODO jaked move template to an external template file
      template:
            '<div class="alert" ' +
               'ng-class="{\'alert-{{type || \'warning\'}}\': true, \'alert-dismissable\': closeable}" role="alert">' +
               '<button ng-show="closeable" type="button" class="close" ng-click="close()">' +
                  '<span aria-hidden="true">&times;</span>' +
                  '<span class="sr-only">Close</span>' +
               '</button>' +
               '<div ng-transclude></div>' +
            '</div>',
      transclude: true,
      replace: true,
      scope: {
         type: '@',
         close: '&'
      }
   };
});


;'use strict';

vui.angular.comboBox

/**
 * @ngdoc object
 * @name vui.angular.comboBox:ComboBoxOptions
 * @module vui.angular.comboBox
 *
 * @description
 *    Configuration object for
 *    {@link vui.angular.comboBox.directive:vuiComboBox vuiComboBox}
 *    directive.
 *    Contains properties to specify a comboBox and populate it with
 *    data.
 *
 *    Example:
 *    ```js
 *    var ComboBoxOptions = {
 *                             data: [ ... ],
 *                             placeHolder: 'Choose a thing.'
 *                          };
 *    ```
 *
 * @property {[*]} data
 *    Type: `Array.<string | number | ComboBoxItem>`
 *
 *    Specifies the items to populate the comboBox. Each item may either be a number,
 *    string or object of type  {@link vui.angular.comboBox:ComboBoxItem ComboBoxItem}.
 *    Each item specifies a selectable item in the combo box.
 *
 *    When an item is a number or string, the number or string given becomes both that item's value and label.
 *    Use a ComboBoxItem object to allow an item's label and value to be different
 *
 *    In order to update the items in the combo box, assign a new value to this property.
 *
 *    Example:
 *    ```js
 *    data: [ {value: 1, label:'VM1'}, {value: 2, label: 'VM2'} ];
 *    (or)
 *    data: ["vm1", "vm2"];
 *    (or)
 *    data: [1, 2, 3]
 *    ```
 *
 * @property {*} [selectedValue]
 *    Type: `*` **|** Optional
 *
 *    Specifies and controls the current value of the combo box.
 *
 *    Setting this to a value that exists in the {@link vui.angular.comboBox:ComboBoxOptions data}
 *    items causes that item's label to be displayed as the default.
 *    The test for existence is done using strict equality (===).
 *
 *    Note: If this property is missing it will be written by the directive.
 *
 * @property {boolean} [enabled]
 *    Type: `boolean` **|** Optional **|** Default: true
 *
 *    Controls whether the widget is enabled or disabled.
 *
 *    Note: If this property is missing it will be written by the directive.
 *
 * @property {string} [placeHolder]
 *    Type: `string` **|** Optional
 *
 *    The initial string to display in the combo box. Leave this as undefined or
 *    null if using 'selectedValue' to control the initial text in the box.
 *
 *    When this property is not undefined or null, the given text is shown in the box
 *    when first rendered and  the 'selectedValue' is set to undefined.
 *
 * @property {string} [width]
 *    Type `string` **|** Optional
 *
 *    The width, in any valid CSS unit, of the combo box.
 *    If not wide enough for the content, the drop down will scroll horizontally.
 *    Don't forget to specify the unit.
 */

/**
 * @ngdoc object
 * @name vui.angular.comboBox:ComboBoxItem
 * @module vui.angular.comboBox
 *
 * @description
 *    Configuration object for a selectable item in a
 *    {@link vui.angular.comboBox.directive:vuiComboBox vuiComboBox}
 *    directive.
 *
 *    Example:
 *    ```js
 *    var ComboBoxItem1 = {
 *                         label: 'Furniture',
 *                         value: 1
 *                        };
 *    ```
 *
 * @property {string} label
 *    Type: `string`
 *
 *    Localized text to display for the item.
 *
 *    <script>$('.properties .label').removeClass('label');</script>
 *
 * @property {*} value
 *    Type: `*`
 *
 *    The value of the item.
 */

/**
 * @ngdoc directive
 * @name vui.angular.comboBox.directive:vuiComboBox
 * @module vui.angular.comboBox
 * @restrict A
 * @element div
 * @scope
 *
 * @description
 *    Directive to create a combo box control.
 *    A combo box is a graphical control element, similar to a list box,
 *    it is a single-line editable textbox, allowing the user to either type
 *    a value directly or select a value from the list. When a combo box is inactive,
 *    it displays a single value. When activated, it displays a list of values,
 *    from which the user may select one. When the user selects a new value, the control reverts
 *    to its inactive state, displaying the selected value.
 *
 * @param {ComboBoxOptions} vuiComboBox
 *    Configuration object for the comboBox. It is of type
 *    {@link vui.angular.comboBox:ComboBoxOptions ComboBoxOptions}.
 *
 * @example
      <example module="app">

         <file name="app.js">
            'use strict';
            var app = angular.module('app', ['vui.angular']);
            app.controller('ComboBoxController', function ($scope, $interval) {
               $scope.comboBoxOptions = {
                  selectedValue: 3,
                  enabled: true,
                  data: [
                     {label: 'Albania', value: 1},
                     {label: 'Andorra', value: 2},
                     {label: 'Armenia', value: 3},
                     {label: 'Austria', value: 4},
                     {label: 'Azerbaijan', value: 5}
                  ],
                  width: '300px'
               };
            });
         </file>

         <file name="index.html">
            <link rel="stylesheet" href="css/vui-bootstrap.min.css"
                  type="text/css">
            <div ng-controller="ComboBoxController">
               <div vui-combo-box="comboBoxOptions"></div><br>
               <div>value : {{comboBoxOptions.selectedValue}}</div>
            </div>
         </file>

      </example>
 */
.directive('vuiComboBox', ['jsUtils', function (jsUtils) {
   return {
      restrict: 'A',
      scope: {
         vuiComboBox: '='
      },
      replace: false,
      template:
         '<select kendo-combo-box ' +
               'ng-model="vuiComboBox.selectedValue"' +
               'ng-disabled="!vuiComboBox.enabled"' +
               'k-placeholder="vuiComboBox.placeHolder"' +
               'k-data-text-field="\'label\'"' +
               'k-data-value-field="\'value\'"' +
               'k-data-source="vuiComboBox.data" ng-style="{width: vuiComboBox.width}">' +
         '</select>',
      controller: ['$scope', function ($scope) {
         // Set default value for enabled if not specified.
         $scope.vuiComboBox.enabled = jsUtils.isBoolean($scope.vuiComboBox.enabled) ?
               $scope.vuiComboBox.enabled : true;
         $scope.vuiComboBox.placeHolder = $scope.vuiComboBox.placeHolder || '';
      }],
      link: function (scope, element) {
         // TODO - Kendo issue ... does not respect ng-style in kendo-combo-box directive
         var widget = element.find('.k-widget');
         widget.css({width: scope.vuiComboBox.width});
      }
   };
}]);
;'use strict';

// Global vUI namespace.
var vui = vui || {};

/**
 * Utils contains static commonly used methods for working with objects.
 * @namespace vui.Utils
 */

// TODO: Move to independent, framework agnostic, reusable js library.
vui.Utils = (function () {
   /**
    * Returns the possibly nested property from the given object, or defaultValue if
    * not defined.
    *
    * @param {Object} object
    *    The object to get the property from.
    * @param {string} property
    *    The property to extract from object. May describe a nested property using '.'
    *    notation.
    * @param {*} [defaultValue]
    *    The default value to return if property is not found on object.
    *
    * @returns {*}
    *    The value of property on object if found, else defaultValue.
    *
    * @throws {string}
    *    Error message if the property parameter is not truthy.
    */
   function getProperty(object, property, defaultValue) {
      if (!property) {
         throw 'No property to get.';
      }

      if (!object) {
         return defaultValue;
      }

      var nestedProps = property.split('.');
      var result = object;
      jQuery.each(nestedProps, function (index, value) {
         if ((typeof result === 'object') && (value in result)) {
            if (result[value] !== undefined) {
               result = result[value];
            } else {
               result = defaultValue;
            }
         } else {
            result = defaultValue;
            // Halts the iteration.
            return false;
         }
      });

      return result;
   }

   /**
    * Writes the given value at the given property name on the given object.
    *
    * @param {object} object
    *    The object to write the property
    * @param {string} property
    *    Name of the property to write on object. Can be nested using dot notation.
    *    Must already exist on object unless `createIfMissing` is true, else nothing is
    *    written.
    * @param {*} value
    *    The value to write at property.
    * @param {boolean} [createIfMissing=false]
    *    When false, if property is not already declared on object this method does
    *    nothing. If true and the property is missing, the property, including its full
    *    path, will be written onto the object.
    */
   function setProperty(object, property, value, createIfMissing) {
      if (!property) {
         throw 'No property to set.';
      }

      if (typeof object !== 'object') {
         throw 'No object to set ' + property + ' on.';
      }

      var nestedProps = property.split('.');
      var pathFound = true;
      var i;
      for (i = 0; i < nestedProps.length - 1; i++) {
         if (nestedProps[i] in object) {
            if (typeof object[nestedProps[i]] !== 'object') {
               if (createIfMissing) {
                  object[nestedProps[i]] = {};
               } else {
                  pathFound = false;
                  break;
               }
            }
         } else if (createIfMissing) {
            object[nestedProps[i]] = {};
         } else {
            pathFound = false;
            break;
         }
         object = object[nestedProps[i]];
      }

      if (!pathFound) {
         return;
      }

      var propToSet = nestedProps[nestedProps.length - 1];
      if (createIfMissing || (propToSet in object)) {
         object[propToSet] = value;
      }
   }

   /**
    * Returns true if the object has the given property, else returns false.
    *
    * @param {object} object
    *    The object to look for property on.
    * @param {string} property
    *    The property to look for on object.
    *
    * @returns {boolean}
    *    True if the property is found on object, else false.
    *
    * @throws {string}
    *    Error message if property is not truthy.
    */
   function hasProperty(object, property) {
      if (!property) {
         throw 'No property to look for.';
      }

      if (!object) {
         return false;
      }

      var notFound = {};
      // Note: using notFound here is safer than testing the returned value for undefined
      // because it prevents false negatives when an object has a property but its value
      // is undefined. For example
      // 'var foo = {bar: undefined}; hasProperty(foo, 'bar')' is true.
      var propertyValue = getProperty(object, property, notFound);
      return propertyValue !== notFound;
   }

   /**
    * Validates that the given object has the given required properties by throwing an
    * exception if any required property is missing from the object.
    *
    * @param {object} object
    *    The object to validate.
    * @param {Array<string|Array>} requiredProps
    *    An array of property names or 'property specs'. A property name is simply the
    *    name of the required property (string). A property spec is an array of length
    *    1, 2 or 3. The required first element is the name of the required property
    *    (string). The second optional element is the required type of the property
    *    (string. Give null to ignore). The third optional element is the required value
    *    of the property.
    *
    * @throws {string}
    *    Error message if: No object parameter is given. An empty property spec parameter
    *    is given. Any of the required properties are missing from the object or are of
    *    the wrong type or hold the wrong value.
    */
   function validateObject(object, requiredProps) {
      if (arguments < 1) {
         throw 'No object to validate.';
      }
      jQuery.each(requiredProps, function (index, value) {
         var requiredProp;
         if (typeof value === 'string') {
            requiredProp = value;
         } else {
            requiredProp = value[0];
         }
         if (!requiredProp) {
            throw 'Invalid property spec.';
         }

         if (!(hasProperty(object, requiredProp))) {
            throw 'Required property ' + requiredProp + ' missing from object.';
         }

         if (typeof value === 'string') {
            // Simple string arg - we are done.
            return;
         }

         var propertyValue = getProperty(object, requiredProp);
         if (value.length > 1) {
            var requiredType = value[1];
            if ((requiredType !== null) && (typeof propertyValue !== requiredType)) {
               throw 'Required property ' + requiredProp + ' not of required type ' + requiredType;
            }
         }

         if (value.length > 2) {
            var requiredValue = value[2];
            if (propertyValue !== requiredValue) {
               throw 'Required property ' + requiredProp + ' does not have required value ' + requiredValue;
            }
         }
      });
   }

   /**
    * Returns true if the value of object is undefined or null, else returns false.
    *
    * @param {object} object
    *    The object to test for a value of undefined or null.
    *
    * @returns {boolean}
    *    True if the object is undefined or null, else false.
    */
   function isUndefinedOrNull(object) {
      if (typeof object === 'undefined' || object === null) {
         return true;
      }
      return false;
   }

   /**
    * Returns true if the object is empty, else false.
    *
    * * null, undefined, booleans and numbers are empty.
    * * '' is empty, all other strings are not empty.
    * * An object is empty if it has no properties.
    * * An object with a 'length' property is empty iff length === 0.
    *  * All other objects are not empty.
    * @param object
    * @returns {boolean}
    */
   function isEmpty(object) {
      if (isUndefinedOrNull(object)) {
         return true;
      }

      var type = typeof object;

      switch (type) {
         case 'number':
         case 'boolean':
         case 'undefined':
            return true;
         case 'string':
            return object.length === 0;
      }

      if (object.hasOwnProperty('length')) {
         return object.length === 0;
      }

      for (var prop in object) {
         if (Object.prototype.hasOwnProperty.call(object, prop)) {
            return false;
         }
      }

      return true;
   }

   /**
    * Returns true if the value is undefined or is empty, else returns false.
    *
    * @param {string|array} value
    *    The value to test undefined or empty.
    *
    * @returns {boolean}
    *    True if the value is undefined or empty, else false.
    */
   function isUndefinedOrEmpty(value) {
      if (value && value.length > 0) {
         return false;
      }
      return true;
   }

   /**
    * Returns true if the type of value is boolean.
    *
    * @param {*} value
    *    The value to test for.
    *
    * @returns {boolean}
    *    True if the value is of type boolean.
    */
   function isBoolean(value) {
      if (typeof value === 'boolean') {
         return true;
      } else {
         return false;
      }
   }

   // Exported API
   return {
      getProperty: getProperty,
      setProperty: setProperty,
      hasProperty: hasProperty,
      validateObject: validateObject,
      isBoolean: isBoolean,
      isEmpty: isEmpty,
      isUndefinedOrEmpty: isUndefinedOrEmpty,
      isUndefinedOrNull: isUndefinedOrNull
   };

})();
;vui.angular.constants

/**
 * @ngdoc service
 * @name vui.angular.constants.vuiConstants
 * @module vui.angular.constants
 *
 * @description
 *    The `vuiConstants` service is already
 *    included in the following modules - vui.angular.datagrid,
 *    vui.angular.dialog, vui.angular.portlets, vui.angular.tabs and
 *    vui.angular.wizard.
 *
 *    List of available `vuiConstants`:
 *
 *    - actions
 *       - **`SEPARATOR`** : Use to insert a separator between actions.
 *       - **`VUI_MENU_SCROLLER_CLASS`** : Use to specify that the context menu has scroller.
 *       - **`VUI_MENU_SCROLL_WRAPPER_CLASS`** : Use define that child context menu, has to
 *            have vertical scroller when menu doesn't fit in the window.
 *    - grid
 *       - dataType
 *          - **`JSON`** : Use to indicate that the data is requested from a
 *             server in the same domain.
 *          - **`JSONP`** : Use to indicate that the data is requested from a
 *             server in a different domain.
 *       - dataMethod
 *          - **`GET`** : Use to indicate that the data is requested using the
 *             'GET' HTTP method
 *          - **`POST`** : Use to indicate that the data is requested using the
 *             'POST' HTTP method (does not work with `dataType` = `JSONP`)
 *       - displayMode
 *          - **`VIRTUAL_SCROLLING`** : Use to indicate that a scrollbar is
 *             used to continuously fetch new pages of data.
 *       - editorType
 *          - **`DROPDOWN`** : Use to specify a dropdown component as custom UI
 *             editor for the column.
 *       - selectionMode
 *          - **`NONE`** : Use to indicate that user cannot select any row in
 *             the Datagrid.
 *          - **`SINGLE`** : Use to indicate that user can select only one row
 *             at a time.
 *          - **`MUTLI`** : Use to indicate that user can select any number of
 *             row in the Datagrid.
 *       - serverSide
 *          -  **`TOTAL`** : Use to indicate the field from the server response which
 *             contains total number of data items.
 *       - sortMode
 *          - **`NONE`** : Use to indicate that user cannot sort data in any
 *             column of the Datagrid.
 *          - **`SINGLE`** : Use to indicate that user can click on the header
 *             cell of each column, sorting the Grid by that column's data.
 *          - **`MUTLI`** : Use to indicate that user can sort multiple columns
 *             by using Shift + Click on the header cell of each column.
 *    - notifications
 *       - type
 *          - **`INFO`** : Use for info notification icon.
 *          - **`SUCCESS`** : Use for success notification icon.
 *          - **`WARNING`** : Use for warning notification icon.
 *          - **`ERROR`** : Use for info notification icon.
 *    - portlets
 *       - layout
 *          - **`TWO_COLUMNS`** : Use for two column layout.
 *          - **`THREE_COLUMNS`** : Use for three column layout.
 *          - **`FOUR_COLUMNS`** : Use for four column layout.
 *    - splitter
 *       - orientation
 *          - **`HORIZONTAL`** : Use to indicate the inner panels will be
 *             oriented horizontally.
 *          - **`VERTICAL`** : Use to indicate the inner panels will be
 *             oriented vertically.
 *    - tabs
 *       - type
 *          - **`PRIMARY`** : Use for a top-level tab style.
 *          - **`SECONDARY`** : Use for a tab bar style.
 *          - **`TERTIARY`** : Use for table of contents (TOC) style.
 *    - validationBanner
 *       - type
 *          - **`ERROR`** : Use to indicate an error or a problem that has
 *             occurred.
 *          - **`WARNING`** : Use to indicate a condition that might cause a
 *             problem in the future.
 *    - wizard
 *       - pageState
 *          - **`COMPLETED`** : Use to indicate that the page is complete.
 *             The page is considered complete when its output data is ready to
 *             be used (by user input or default values).
 *          - **`INCOMPLETE`** : Use to indicate that
 *             the page is incomplete and ready for user input.
 *          - **`DISABLED`** : Use to indicate that the page can not be
 *             activated by the user.
 *          - **`SKIPPED`** : Use to indicate that the page is excluded from
 *             the pages considered by the wizard in all operations.
 *
 *    ##Example
 *
 *    To define primary tab type for `vui-tabs` directive, use
 *    ````js
 *    var tabOptions = {
 *                         tabType: vuiConstants.tabs.type.PRIMARY,
 *                         tabs: [ ... ]
 *                     };
 *    ```
 */

.constant('vuiConstants', {
   actions: {
      VUI_MENU_SCROLLER_CLASS: 'vui-menu-scroller',
      VUI_MENU_SCROLL_WRAPPER_CLASS: 'vui-menu-scroll-wrapper',
      SEPARATOR: 'separator',
      icon: {
         ADD: 'vui-icon-action-add',
         EDIT: 'vui-icon-action-edit',
         COPY: 'vui-icon-action-copy',
         DELETE: 'vui-icon-action-delete',
         MORE: 'vui-icon-action-more'
      }
   },
   grid: {
      dataType: {
         JSON: 'json',
         JSONP: 'jsonp'
      },
      dataMethod: {
         GET: 'get',
         POST: 'post'
      },
      displayMode: {
         PAGES: 'pages',
         VIRTUAL_SCROLLING: 'virtualScrolling'
      },
      editorType: {
         DROPDOWN: 'dropdown'
      },
      selectionMode: {
         NONE: 'none',
         SINGLE: 'single',
         MULTI: 'multi'
      },
      serverSide: {
         TOTAL: 'total'
      },
      sortMode: {
         NONE: 'none',
         SINGLE: 'single',
         MULTI: 'multi'
      }
   },
   notifications: {
      type: {
         INFO: 'info',
         SUCCESS: 'success',
         WARNING: 'warning',
         ERROR: 'error'
      }
   },
   portlets: {
      layout: {
         TWO_COLUMNS: 'two-columns',
         THREE_COLUMNS: 'three-columns',
         FOUR_COLUMNS: 'four-columns'
      }
   },
   splitter: {
      orientation: {
         HORIZONTAL: 'horizontal',
         VERTICAL: 'vertical'
      }
   },
   tabs: {
      type: {
         PRIMARY: 'primary',
         SECONDARY: 'secondary',
         TERTIARY: 'tertiary'
      },
      style: {
         TABS: 'tabs',
         PILLS: 'pills'
      }
   },
   validationBanner: {
      type: {
         ERROR: 'error',
         WARNING: 'warning'
      }
   },
   wizard: {
      pageState: {
         SKIPPED: 'skipped',
         INCOMPLETE: 'incomplete',
         DISABLED: 'disabled',
         COMPLETED: 'completed'
      }
   },
   /* Constants to be used internally
   */
   internal: {
      // Z-index for popups
      zIndex : {
         SINGLE_PAGE_DIALOG: 1051,
         WIZARD: 1050,
         DEFAULT: 1050
      },
      modalPositionClass : {
         CENTER: 'modal-center'
      },
      keys: {
         ENTER: 13,
         ESC: 27,
         SPACE: 32,
         LEFT: 37,
         UP: 38,
         RIGHT: 39,
         DOWN: 40
      }
   }
});
;'use strict';
vui.angular.datagrid

/**
 * @ngdoc object
 * @name vui.angular.datagrid:DatagridOptions
 * @module vui.angular.datagrid
 *
 * @description
 *    Configuration object for
 *    {@link vui.angular.datagrid.directive:vuiDatagrid vuiDatagrid}
 *    directive.
 *    Contains properties to specify a Datagrid and populate it with data.
 *
 *    Example:
 *    ```js
 *    var datagridOptions = {
 *                            columnDefs: [ ... ],
 *                            data: [ ... ],
 *                            actionBarOptions: { actions: [ ... ] },
 *                            selectedItems: []
 *                          };
 *    ```
 *
 * @property {Array.<ColumnSpec>} columnDefs
 *    Type: `Array.<ColumnSpec>`
 *
 *    Array of {@link vui.angular.datagrid:ColumnSpec ColumnSpec} objects
 *    defining the properties of each column in the Datagrid. The columns in
 *    this array must appear in the same order in which they will be displayed
 *    in the Datagrid.
 *
 *    Example:
 *    ```js
 *    columnDefs = [ { field: 'browser', displayName: 'Browser'},
 *                   { field: 'platform', displayName: 'Platform'} ]
 *    ```
 *
 * @property {Array.<Object>|Object<ServerSideOptions>} data
 *    Type: `Array.<Object>`| `Object<ServerSideOptions>`
 *
 *    Array of objects providing data for the Datagrid or a
 *    {@link vui.angular.datagrid:ServerSideOptions ServerSideOptions}
 *    object containing properties to access data and enable server-side
 *    processing.
 *
 *    Each object in the array is mapped to a row being displayed. Data for
 *    each column is mapped by that column's
 *    {@link vui.angular.datagrid:ColumnSpec field} property.
 *
 *    Example of data as array of objects:
 *    ```js
 *    data = [ { browser: 'Firefox 30.0', platform: 'Windows 7' },
 *             { browser: 'Google Chrome 35.0', platform: 'Mac OS X' } ]
 *    ```
 *
 * @property {Object<ActionbarOptions>} [actionBarOptions]
 *    Type: `Object<ActionbarOptions>` **|** Optional
 *
 *    {@link vui.angular.actions:ActionbarOptions ActionbarOptions}
 *    contains properties to define the actions in the Actionbar.
 *
 * @property {string} [sortMode]
 *    Type: `string` **|** Optional **|** Default: `vuiConstants.grid.sortMode.SINGLE`
 *
 *    Specifies the sorting of columns.
 *
 *    Legal values:
 *
 *    - **`vuiConstants.grid.sortMode.NONE`** : User cannot sort data in any
 *       column of the Datagrid.
 *    - **`vuiConstants.grid.sortMode.SINGLE`** : User can click on the header
 *       cell of each column, sorting the Datagrid by that column's data.
 *    - **`vuiConstants.grid.sortMode.MULTI`** : User can sort multiple columns
 *       by using Shift + Click on the header cell of each column.
 *
 * @property {string} [selectionMode]
 *    Type: `string` **|** Optional **|** Default: `vuiConstants.grid.selectionMode.NONE`
 *
 *    Allows the end user to select rows in the Datagrid.
 *
 *    Legal values:
 *
 *    - **`vuiConstants.grid.selectionMode.NONE`** : User cannot select any row
 *       in the Datagrid.
 *    - **`vuiConstants.grid.selectionMode.SINGLE`** : User can select only one
 *       row at a time.
 *    - **`vuiConstants.grid.selectionMode.MULTI`** : User can select any number
 *       of rows in the Datagrid simply by clicking on the rows or using the
 *       corresponding checkbox.
 *
 * @property {Array.<Object>} [selectedItems]
 *    Type: `Array.<Object>` **|** Optional
 *
 *    Read-only array that specifies currently selected data objects in the
 *    Datagrid.
 *
 * @property {boolean} [searchable]
 *    Type: `boolean` **|** Optional **|** Default: `true`
 *
 *    Specifies whether user can filter the data using the filter menu displayed
 *    for each column.
 *
 * @property {boolean} [editable]
 *    Type: `boolean` **|** Optional **|** Default: `false`
 *
 *    Specifies whether the data in the Datagrid is editable by the user.
 *    If set to true {@link vui.angular.datagrid:DatagridOptions idDataField}
 *    is required.
 *
 *    **Note:** Currently editing is only supported for
 *    {@link vui.angular.datagrid:DatagridOptions data} provided as an array of
 *    objects. Any edit operation to data will override user provided
 *    {@link vui.angular.datagrid:DatagridOptions data} with updated data.
 *
 * @property {string} [idDataField]
 *    Type: `string` **|** Optional
 *
 *    Specifies the identification field that has unique value for each
 *    {@link vui.angular.datagrid:DatagridOptions data} item.
 *
 *    **Note:** Must be provided when
 *    {@link vui.angular.datagrid:DatagridOptions editable} is set to true.
 *
 * @property {Object.<PageSpec>} [pageConfig]
 *    Type: `Object.<PageSpec>` **|** Optional
 *
 *    {@link vui.angular.datagrid:PageSpec PageSpec} object specifies how and
 *    when pages of data are loaded into the datagrid. If not specified,
 *    data is not paged but is loaded all at once.
 *
 * @xproperty {boolean} [allowKeyboardNavigation]
 *    Type: `boolean` **|** Optional **|** Default: `true`
 *
 *    Specifies whether the user can navigate the datagrid using keyboard
 *    navigation.
 *
 * @xproperty {boolean} [reorderable]
 *    Type: `boolean` **|** Optional **|** Default: `false`
 *
 *    Specifies whether the user could reorder the columns by dragging their
 *    header cells.
 *
 * @xproperty {boolean} [resizable]
 *    Type: `boolean` **|** Optional **|** Default: `false`
 *
 *    Specifies whether the user could resize the columns by dragging the edges
 *    of their header cells.
 *
 * @property {Array.<OrderSpec>} [groupOrder]
 *    Type: `Array.<OrderSpec>` **|** Optional
 *
 *    Array of {@link vui.angular.datagrid:OrderSpec OrderSpec} objects
 *    specifying the field(s) to group by and direction of sorting for display
 *    aggregated Datagrid rows.
 *
 *    **Note:** To avoid duplicate information remove the column with the
 *    {@link vui.angular.datagrid:OrderSpec field} from
 *    {@link vui.angular.datagrid:DatagridOptions columnDefs}
 *
 * @property {Array.<OrderSpec>} [sortOrder]
 *    Type: `Array.<OrderSpec>` **|** Optional
 *
 *    Array of {@link vui.angular.datagrid:OrderSpec OrderSpec} objects
 *    specifying the sort ordering of the data when loaded into the grid.
 *
 *    **Note:** This property has no effect whilst a sort initiated by the UI
 *    controls is active.
 *
 * @property {Array.<FilterSpec>} [filter]
 *    Type: `Array.<FilterSpec>` **|** Optional
 *
 *    Array of {@link vui.angular.datagrid:FilterSpec FilterSpec} objects
 *    specifying the filters applied to the data when loaded into the datagrid.
 *
 *    **Note:** This property has no effect whilst a filter initiated by the UI
 *    controls is active.
 *
 * @property {string} [height]
 *    Type: `string` **|** Optional
 *
 *    The height of the Datagrid in pixels. If specified it takes preference
 *    over {@link vui.angular.datagrid:DatagridOptions pageSize} property.
 *
 * @property {function} [onChange]
 *    Type: `function` **|** Optional
 *
 *    Callback function to invoke when change is detected on the data grid.
 *    Selected items from the data grid will be available on the callback.
 */

/**
 * @ngdoc object
 * @name vui.angular.datagrid:PageSpec
 * @module vui.angular.datagrid
 *
 * @description
 *    Contains properties to define a page in the Datagrid.
 *
 *    Example:
 *    ```js
 *    pageConfig:[{
 *                   size: 50,
 *                   displayMode: vuiConstants.grid.displayMode.VIRTUAL_SCROLLING
 *                }];
 *    ```
 *
 * @property {number} [size]
 *    Type: `number` **|** Optional **|** Default: 100
 *
 *    Specifies the page size of the data. The datagrid displays one page of
 *    data at a time and when working with a remote data source, requests one
 *    page of data at a time.
 *
 * @property {string} [displayMode]
 *    Type: `string` **|** Optional **|**
 *    Default: `vuiConstants.grid.displayMode.VIRTUAL_SCROLLING`
 *
 *    Specifies the method by which new pages are fetched for display.
 *    At present, the only supported value is
 *    `vuiConstants.grid.displayMode.VIRTUAL_SCROLLING`, which indicates that
 *    pages are continually fetched as the datagrid scollbar moves.
 *
 * @property {boolean} [hidePager]
 *    Type: `boolean` **|** Optional **|** Default: `false`
 *
 *    Specifies whether to show the grid pager or not. This property has effect only if
 *    the PageSpec.displayMode is vuiConstants.grid.displayMode.VIRTUAL_SCROLLING.
 */

/**
 * @ngdoc object
 * @name vui.angular.datagrid:OrderSpec
 * @module vui.angular.datagrid
 *
 * @description
 *    Contains properties to define a group or sort order in the Datagrid during
 *    initialization.
 *
 *    Example:
 *    ```js
 *    groupOrder:[{
 *                   field: 'browser',
 *                   dir: 'asc'
 *               }];
 *    ```
 *
 * @property {string} field
 *    Type: `string`
 *
 *    The name of the field in the
 *    {@link vui.angular.datagrid:ColumnSpec field} that this Datagrid will be
 *    grouped by.
 *
 * @property {string} [dir]
 *    Type: `string` **|** Optional **|** Default: 'asc'
 *
 *    The direction of sort order. The supported values are **`asc`**
 *    (ascending order) and **`desc`** (descending order).
 *
 */

/**
 * @ngdoc object
 * @name vui.angular.datagrid:FilterSpec
 * @module vui.angular.datagrid
 *
 * @description
 *    Contains properties to specify a filter over data.
 *
 *    Example:
 *    ```js
 *    filter:[{
 *                field: 'name',
 *                operator: 'contains',
 *                value: 'Jane'
 *            }];
 *    ```
 *
 * @property {string} field
 *    Type: `string`
 *
 *    The name of the field in the
 *    {@link vui.angular.datagrid:ColumnSpec field} to which the filter operator
 *    is applied.
 *
 * @property {string} operator
 *    Type: `string`
 *
 *    The filter operator (comparison).
 *
 *    The supported operator for string field is `contains`.
 *
 *    The supported operators for numeric field are `eq` (equal to), `neq`
 *    (not equal to), `lt` (less than), `lte` (less than or equal to), `gt`
 *    (greater than) and `gte` (greater than or equal to).
 *
 * @property {string|number|boolean} value
 *    Type: `string|number|boolean`
 *
 *    The value to which the {@link vui.angular.datagrid:FilterSpec field} is
 *    compared.
 *
 *    **Note:** The value must be of the same type as the field.
 */

/**
 * @ngdoc object
 * @name vui.angular.datagrid:ColumnSpec
 * @module vui.angular.datagrid
 *
 * @description
 *    Contains properties to define a column in the Datagrid.
 *
 *    Example:
 *    ```js
 *    var column1 = {
 *                      field: 'browser',
 *                      displayName: 'Browser',
 *                      sortable: true,
 *                      width: 200px
 *                   };
 *    ```
 *
 * @property {string} field
 *    Type: `string`
 *
 *    The name of the property in the
 *    {@link vui.angular.datagrid:DatagridOptions data} model that this
 *    column represents.
 *
 * @property {string} displayName
 *    Type: `string`
 *
 *    The title text for the column header.
 *
 * @property {string} [type]
 *    Type: `string` **|** Optional
 *
 *    The type of data in the column. It is used for filtering
 *    and sorting.
 *
 *    Legal values:
 *
 *    - **`'number'`** : Simple number sorting and filtering.
 *    - **`'string'`** : Default data type if the type in the column is not
 *       specified.
 *    - **`'boolean'`** : Simple boolean sorting and filtering.
 *
 * @property {boolean|string|Function} [sortable]
 *    Type: `boolean|string|Function` **|** Optional **|** Default: `true`
 *
 *    Defines end user's ability to sort data in this column.
 *
 *    Legal values:
 *
 *    - **`true`** : User can sort data in this column.
 *    - **`false`** : User will not be able to sort data in this column.
 *    - **`string`** : The name of a comparator function to sort data in this
 *       column. The function must either be defined on the DatagridOptions
 *       object or on $scope.
 *
 *       Example:
 *       ```js
 *       var customSortFunc = function (a, b) {
 *                               return ((a < b) ? -1 : ((a > b) ?  1 : 0));
 *                            };
 *       ..
 *       ..
 *       sortable: customSortFunc,
 *       ```
 *    - **`Function`** : Comparator function to use when sorting data in this
 *       column. The function must take two inputs `'a'` and `'b'`,
 *       that represent the data to sort and return `>1` if `a < b`, `<1` if
 *       `b < a` and `0` if `a == b`.
 *
 *       Example:
 *       ```js
 *       sortable: function (a, b) {
 *                    return ((a < b) ? -1 : ((a > b) ?  1 : 0));
 *                 };
 *       ```
 *    **Note:** For this column to be sortable,
 *    Datagrid's {@link vui.angular.datagrid:DatagridOptions sortMode} property
 *    must be set to legal values that permit sorting.
 *
 * @property {boolean} [searchable]
 *    Type: `boolean` **|** Optional **|** Default: `true`
 *
 *    If set to true a filter menu will be displayed for this column when
 *    filtering is enabled. If set to false the filter menu will not be
 *    displayed.
 *
 *    **Note:** For this column to be searchable, Datagrid's searchable
 *    property must be set to true.
 *
 * @property {boolean|Object<EditorSpec>} [editable]
 *    Type: `boolean|Object<EditorSpec>` **|** Optional **|** Default: `*`
 *
 *    Specifies whether the data in this column can be edited by the user
 *    If false the data is not editable. If true the data is editable using
 *    the default editor for the field's data type. If an
 *    {@link vui.angular.datagrid:EditorSpec EditorSpec} object, the data is
 *    edited and validated using the given configuration.
 *
 *    **Note:**  For this column to be editable,
 *    Datagrid's {@link vui.angular.datagrid:DatagridOptions editable} property
 *    must be set to true.
 *
 * @property {boolean} [visible]
 *    Type: `boolean` **|** Optional **|** Default: `true`
 *
 *    Specifies whether this column is visible.
 *
 * @property {boolean} [locked]
 *    Type: `boolean` **|** Optional **|** Default: `false`
 *
 *    Specifies whether this column will be displayed as locked in the Datagrid.
 *
 * @property {string|Function} [template]
 *    Type: `string|Function` **|** Optional
 *
 *    The template which renders the column content. By default the
 *    HTML-encoded value of the
 *    {@link vui.angular.datagrid:ColumnSpec field} is displayed in the
 *    column.
 *
 *       Example:
 *       ```js
 *       columnDefs: [{
 *                         field: 'name',
 *                         template: '<strong>#: name # </strong>'
 *                   }],
 *       ```
 *
 * @xproperty {string} [format]
 *    Type: `string` **|** Optional
 *
 *    The format that is applied to the value before it is displayed. Takes the
 *    form "{ 0: format}".
 *
 * @property {string} [width]
 *    Type: `string` **|** Optional
 *
 *    The width of the column in pixels.
 */

/**
 * @ngdoc object
 * @name vui.angular.datagrid:ServerSideOptions
 * @module vui.angular.datagrid
 *
 * @description
 *    Contains properties to request processed data from server to be rendered
 *    in the Datagrid.
 *
 *    Example:
 *    ```js
 *    var serverSideOptions = {
 *                               url: '../..',
 *                               dataType: vuiConstants.grid.dataType.JSON
 *                               method: vuiConstants.grid.dataMethod.GET,
 *                               data: { param1: 'value1', param2: 'value2' }
 *                            };
 *    ```
 *
 *    Paging, sorting, grouping and filtering will be performed on server side.
 *    Values for {@link vui.angular.datagrid:DatagridOptions pageSize},
 *    {@link vui.angular.datagrid:DatagridOptions sortOrder},
 *    {@link vui.angular.datagrid:DatagridOptions groupOrder}, and
 *    {@link vui.angular.datagrid:DatagridOptions filters} are sent to server
 *    following jQuery's
 *    {@link http://api.jquery.com/jQuery.param/ conventions}.
 *
 *    For example filter:
 *    ```js
 *    filter:[{
 *                field: 'name',
 *                operator: 'contains',
 *                value: 'San'
 *            }];
 *    ```
 *    is sent as
 *    ```js
 *    filter[0][field]: name
 *    filter[0][operator]: contains
 *    filter[0][value]: San
 *    ```
 *
 *    For example sortOrder:
 *    ```js
 *    sortOrder:[{
 *                field: 'name',
 *                dir: 'asc'
 *               }];
 *    ```
 *    is sent as
 *    ```js
 *    sort[0][field]: name
 *    sort[0][dir]: asc
 *    ```
 *
 *    groupOrder is sent as group[0][field], group[0][dir].
 *
 * @property {string} [url]
 *    Type: `string` **|** Optional
 *
 *    A string containing URL to which the request is sent to the server.
 *
 * @property {string} [dataType]
 *    Type: `string` **|** Optional **|** Default: `vuiConstants.grid.dataType.JSON`
 *
 *    The type of result expected from the server. The supported types are
 *    `vuiConstants.grid.dataType.JSON` and `vuiConstants.grid.dataType.JSONP`.
 *
 *    **Note:** `vuiConstants.grid.dataType.JSONP` is required for cross-domain
 *     requests, use `vuiConstants.grid.dataType.JSON` for same-domain requests.
 *
 * @property {string} [method]
 *    Type: `string` **|** Optional **|** Default: `vuiConstants.grid.dataMethod.GET`
 *
 *    The HTTP method to use when sending the data request to the server.
 *    The supported types are `vuiConstants.grid.dataMethod.GET` and
 *    `vuiConstants.grid.dataMethod.POST`.
 *
 *    **Note:** `vuiConstants.grid.dataMethod.POST` does not work with `dataType` is
 *    `vuiConstants.grid.dataType.JSONP`.
 *
 * @property {Object} [params]
 *    Type: `Object` **|** Optional **|** Default: `undefined`
 *
 *    The params to send along with the request. Params should be a key-value object.
 *    This object will be converted to a query string and depending on whether `GET`
 *    or `POST` method is used it will be send to the server through the request body
 *    or as a part of the request URL.
 *
 * @property {string} [data]
 *    Type: `string` **|** Optional
 *
 *    The field from the server response which contains the data items.
 *
 *    Example:
 *    ```js
 *    data: 'results' // Response is { results: [ ... ] }
 *    ```
 *
 * @property {string} [total]
 *    Type: `string` **|** Optional **|** Default: `vuiConstants.grid.serverSide.TOTAL`
 *
 *    A string which represents the field from the server response which
 *    contains the total number of data items.
 */

/**
 * @ngdoc object
 * @name vui.angular.datagrid:EditorSpec
 * @module vui.angular.datagrid
 *
 * @description
 *    Specifies the editor and validator for an editable column.
 *
 *    Example:
 *    ```js
 *    var colEditor = {
 *                         validator: {
 *                            required: true
 *                         },
 *                         editor: {
 *                            type: vuiConstants.grid.editorType.dropdown,
 *                            config: {
 *                               data: [ ... ]
 *                            }
 *                         }
 *                    };
 *    ```
 *
 * @property {Object<ValidatorSpec>} [validator]
 *    Type: `Object<ValidatorSpec>` **|** Optional
 *
 *    A {@link vui.angular.datagrid:ValidatorSpec ValidatorSpec} object
 *    specifying the basic validations to be applied to the value provided by
 *    user.
 *
 * @property {Object} [editor]
 *    Type: `Object` **|** Optional
 *
 *    Provides a way to specify a custom editing UI for the column.
 *
 *    The object contains the following two properties:
 *
 *    - **`type`** : Type of the custom editor component.
 *       Currently the supported values for `type` are:
 *       - **`vuiConstants.grid.editorType.dropdown`**
 *          Must provide a `config` object containing an array of
 *          {@link vui.angular.dropdown:DropdownItem DropdownItem} objects in
 *          a `data` property.
 *    - **`config` (Optional)** :
 *       Object containing configuration for the editor. The required
 *       properties of this object depend on the type of the editor component
 *       being congfigured - refer `type` for details.
 */

/**
 * @ngdoc object
 * @name vui.angular.datagrid:ValidatorSpec
 * @module vui.angular.datagrid
 *
 * @description
 *    Specifies how data items should be validated.
 *
 *    Example:
 *    ```js
 *    var validationSpec = {
 *                            required: true
 *                         };
 *    ```
 *
 * @property {boolean} [required]
 *    Type: `boolean` **|** Optional
 *
 *    When true, the data item can not be null or undefined.
 *
 * @property {number} [min]
 *    Type: `number` **|** Optional
 *
 *    Specifies the minimum value for numeric data.
 *
 * @property {number} [max]
 *    Type: `number` **|** Optional
 *
 *    Specifies the maximum value for numeric data.
 */

/**
 * @ngdoc directive
 * @name vui.angular.datagrid.directive:vuiDatagrid
 * @module vui.angular.datagrid
 * @restrict A
 * @element div
 * @scope
 *
 * @description
 *    Directive to create the Datagrid.
 *
 * @param {datagridOptions} vuiDatagrid
 *    Configuration object for the Datagrid. It is of type
 *    {@link vui.angular.datagrid:DatagridOptions DatagridOptions}.
 *
 * @example
      <example module="app">
         <file name="app.js">
            'use strict';
            var app = angular.module('app', ['vui.angular.datagrid']);

            app.controller('DatagridCtrl', ['$scope', 'vuiConstants', 'dataService',
                  function ($scope, vuiConstants, dataService) {

                  var sampleData = dataService.fetchData();

                  var actionSet = dataService.fetchBasicActionSet($scope);

                  $scope.datagridOptions = {
                     columnDefs: [
                        { displayName: 'Virtual Machines', field: 'vm', width: '150px' },
                        { displayName: 'State', field: 'state' },
                        { displayName: 'Status', field: 'status', searchable: false },
                        { displayName: 'Provisioned Space', field: 'provisionedSpace' },
                        { displayName: 'Used Space', field: 'usedSpace', template: '#: usedSpace # GB' },
                        { displayName: 'Host Name', field: 'host', sortable: false }
                     ],
                     data: sampleData,
                     sortMode: vuiConstants.grid.sortMode.SINGLE,
                     actionBarOptions: {
                        actions: actionSet
                     },
                     selectionMode: vuiConstants.grid.selectionMode.NONE,
                     selectedItems: []
                  };
            }]).controller('DatagridWithGroupsCtrl', ['$scope', 'vuiConstants', 'dataService',
                  function ($scope, vuiConstants, dataService) {

                  var sampleData = dataService.fetchData();

                  var actionSet = dataService.fetchBasicActionSet($scope);

                  $scope.datagridOptions = {
                     columnDefs: [
                        { displayName: 'Virtual Machines', field: 'vm', width: '150px' },
                        { displayName: 'State', field: 'state', groupHeaderTemplate: "State: #= value# " },
                        { displayName: 'Status', field: 'status', searchable: false },
                        { displayName: 'Provisioned Space', field: 'provisionedSpace' },
                        { displayName: 'Used Space', field: 'usedSpace', template: '#: usedSpace # GB' },
                        { displayName: 'Host Name', field: 'host', sortable: false }
                     ],
                     data: sampleData,
                     groupOrder: [{field: 'state'}],
                     sortMode: vuiConstants.grid.sortMode.SINGLE,
                     selectionMode: vuiConstants.grid.selectionMode.NONE,
                     selectedItems: []
                  };
            }]);

            app.service('dataService', function (vuiConstants) {
               var data = [
                  {
                     vm: 'Console Test',
                     state: 'Powered On',
                     status: 'Normal',
                     provisionedSpace: '45.12 GB',
                     usedSpace: 23.14,
                     host: 'Host 1'
                  }, {
                     vm: 'San Francisco VM 1',
                     state: 'Powered On',
                     status: 'At Fault',
                     provisionedSpace: '23.14 GB',
                     usedSpace: 23.14,
                     host: 'Host 1'
                  }, {
                     vm: 'San Francisco VM 2',
                     state: 'Powered Off',
                     status: 'Normal',
                     provisionedSpace: '51.32 GB',
                     usedSpace: 40.15,
                     host: 'Host 2'
                  }, {
                     vm: 'MU',
                     state: 'Powered On',
                     status: 'Normal',
                     provisionedSpace: '32.45 GB',
                     usedSpace: 15.12,
                     host: 'Host 1'
                  }, {
                     vm: 'VMC',
                     state: 'Powered On',
                     status: 'At Fault',
                     provisionedSpace: '21.45 GB',
                     usedSpace: 15.12,
                     host: 'Host 2',
                     cluster: 'Cluster 2'
                  }, {
                     vm: 'Stemp',
                     state: 'Powered Off',
                     status: 'Normal',
                     provisionedSpace: '17.42 GB',
                     usedSpace: 15.12,
                     host: 'Host 2'
                  }
               ];

               this.fetchData = function () {
                  return data;
               };

               this.fetchBasicActionSet = function (scope) {
                  // Generic click function
                  var clickFunction = function (event, action) {
                     scope.actionClicked = action.label;
                  };

                  var actionSet = [
                     {
                        id: 1,
                        label: 'Add new record',
                        tooltipText: 'Add new record tooltip',
                        enabled: true,
                        iconClass: 'vui-icon-action-add',
                        onClick: clickFunction
                     }, {
                        id: 2,
                        label: 'Edit',
                        tooltipText: 'Edit tooltip',
                        enabled: true,
                        iconClass: 'vui-icon-action-edit',
                        onClick: clickFunction
                     }, {
                        id: 3,
                        label: 'Copy',
                        tooltipText: 'Copy tooltip',
                        enabled: true,
                        iconClass: 'vui-icon-action-copy',
                        onClick: clickFunction
                     }, vuiConstants.actions.SEPARATOR, {
                        id: 4,
                        label: 'Delete',
                        tooltipText: 'Delete tooltip',
                        enabled: false,
                        iconClass: 'vui-icon-action-delete',
                        onClick: clickFunction
                     }
                  ];
                  return actionSet;
               };
            });
         </file>

         <file name="index.html">
            <h2>You can make plain grids:</h2>
            <div ng-controller="DatagridCtrl">
               <!-- vui-datagrid directive -->
               <div vui-datagrid="datagridOptions"></div>
               <br/>
               <div>
                  <b> Last action clicked: </b> {{actionClicked}}
               </div>
            </div>

            <h2>You can also group columns by a field:</h2>
             <div ng-controller="DatagridWithGroupsCtrl">
                <!-- vui-datagrid directive -->
                <div vui-datagrid="datagridOptions"></div>
                <br/>
             </div>
         </file>
      </example>
 */

.directive('vuiDatagrid', [ function () {
   return {
      restrict: 'A',
      template: '<div kendo-grid k-options="gridOptions"></div>',
      controller: 'VuiDatagridController',
      scope: {
         datagridOptions: '=vuiDatagrid'
      },
      link: function (scope, element, attrs, vuiDatagridCtrl) {
         vuiDatagridCtrl.init();

         element.attr('editable', scope.gridOptions.editable);
      }
   };
}
]);
;/*global $:false */
'use strict';

vui.angular.datagrid

/**
 * @xngdoc
 * @name vui.angular.datagrid.VuiDatagridController
 * @module vui.angular.datagrid
 *
 * @description
 *    Controller for the
 *    {@link vui.angular.datagrid.directive:vuiDatagrid vui-datagrid}
 *    directive.
 */

.controller('VuiDatagridController', ['$scope', '$compile', '$element',
      'jsUtils', 'vuiUtils', 'vuiLocale', 'vuiConstants', 'vuiZoneService',
      function ($scope, $compile, $element, jsUtils, vuiUtils, vuiLocale,
      vuiConstants, vuiZoneService) {

   var self = this;
   var initComplete = false;
   var kendoGridElem;
   var isGridDataUpdated = false;
   var gridSelectionMode;
   var showCheckboxesOnMultiSelection = true;
   var isItemDisabledCallback = null;

   /* Following global variables are set when mapping user provided
    * datagridOptions to kendo options
    */
   var gridOptions = {};
   // Kendo DataSource - an abstraction for using user data (local/ remote)
   var kendoGridDataSource;
   // Array specifying currently selected data objects from DataSource
   var selectedDataItems = [];
   // A collection of following DataSource properties - filter, group,
   // pageSize, sort, schema.model.id and schema.model.field.
   // It is used whenever a new Kendo DataSource is created.
   var dataSourceOptions = {};
   // Reference to datagrid `editable` property
   // It is used to set Kendo `editable` property on DataSource `schema` for
   // each field
   var isGridEditable = false;
   // `noInput` property indicating that the grid contains no input fields
   var containsNoInput = false;
   // -------------------------------------------------------------------------
   // Creating Kendo Grid column - helper functions ---------------------------
   // -------------------------------------------------------------------------

   /* Creates and returns a Kendo column custom editor function based on the
    * columnDef.editable.editor value.
    */
   var createKendoGridColumnEditor = function (columnDef) {

      var editable = jsUtils.getProperty(columnDef, 'editable', isGridEditable);

      if (typeof editable !== 'object' || !editable.hasOwnProperty('editor')) {
         return;
      } else {
         var editorFunc = function (container, options) {
            // TODO aakankshas: Add check for common edit type and field type
            if (editable.editor.type ===
                  vuiConstants.grid.editorType.DROPDOWN) {
               var inputStr = '<input required data-text-field="value"' +
                  'data-value-field="value" data-bind="value:' +
                  options.field + '"/>';

               // create an input element
               var input = $(inputStr);

               // append it to the container
               input.appendTo(container);

               input.kendoDropDownList({
                  autoBind: false,
                  dataSource: {
                     data: editable.editor.config.data
                  }
               });
            } else {
               console.error('Custom editor of type ' +
                     editable.editor.type + ' is not supported.');
            }
         };
         return editorFunc;
      }
   };

   // Creating Kendo Grid column filter options -------------------------------

   /* Filter menu is customized to just include input text box and filter and
    * clear buttons. Filter operator is defaulted to 'contains'.
    * Allows user to input only one criterion (extra: false)
    */
   var customTextFilter = {
      extra: false,
      messages: {
         info:    vuiLocale.datagrid.colFilterInfo,
         filter:  vuiLocale.datagrid.filter,
         clear:   vuiLocale.datagrid.clear
      },
      operators: {
         string: {
            contains: vuiLocale.datagrid.contains
         }
      },
      ui: function (element) {
         var parent = element.parent();
         while (parent.children().length > 1) {
            $(parent.children()[0]).remove();
         }
         parent.prepend('<input data-bind="value:filters[0].value"' +
               ' class="k-textbox" type="text">');
      }
   };

   /* Filter menu is customized to just include operator dropdown, input
    * text box and filter and clear buttons.
    * (By default filter contains - kendo numeric text box which has
    * spinners.)
    * Allows user to input only one criterion (extra: false)
    */
   var customNumericFilter = {
      extra: false,
      messages: {
         info:    vuiLocale.datagrid.colFilterInfo,
         filter:  vuiLocale.datagrid.filter,
         clear:   vuiLocale.datagrid.clear
      },
      operators: {
         number: {
            eq:   vuiLocale.datagrid.eq,
            neq:  vuiLocale.datagrid.neq,
            gte:  vuiLocale.datagrid.gte,
            gt:   vuiLocale.datagrid.gt,
            lte:  vuiLocale.datagrid.lte,
            lt:   vuiLocale.datagrid.lt
         }
      },
      ui: function (element) {
         var parent = element.parent();
         // Adding k-textbox class to filter textbox
         parent.find('input[type="text"]').addClass('k-textbox');
      }
   };

   /* Filter menu is customized to just include values radio buttons and filter
    * and clear buttons
    */
   var customBooleanFilter = {
      extra: false,
      messages: {
         info:    vuiLocale.datagrid.colFilterBooleanInfo,
         filter:  vuiLocale.datagrid.filter,
         clear:   vuiLocale.datagrid.clear,
         isTrue:  vuiLocale.datagrid.isTrue,
         isFalse: vuiLocale.datagrid.isFalse
      },
      operators: {
         enums: {
            eq:   vuiLocale.datagrid.eq,
            neq:  vuiLocale.datagrid.neq
         }
      }
   };

   /* Filter menu is customized to just include values radio buttons and filter
    * and clear buttons
    */
   var customDateFilter = {
      extra: true,
      messages: {
         info:    vuiLocale.datagrid.colFilterBooleanInfo,
         filter:  vuiLocale.datagrid.filter,
         clear:   vuiLocale.datagrid.clear
      }
   };

   /* Creates and returns a Kendo column filterable value based on the
    * columnDef.searchable and columnDef.type value.
    */
   var createKendoGridColumnFilter = function (columnDef) {
      var isFilterable = jsUtils.getProperty(columnDef, 'searchable', 'true');
      if (isFilterable) {
         var colType = jsUtils.getProperty(columnDef, 'type', 'string');
         switch (colType) {
            case 'boolean' :
               return customBooleanFilter;
            case 'date' :
               return customDateFilter;
            case 'number' :
               return customNumericFilter;
            case 'string' :
               return customTextFilter;
            default:
               console.error('Filter for column type ' + colType +
                     ' is not supported.');
               return false;
         }
      }
      return isFilterable;
   };

   /* Creates and returns a Kendo column sortable value based on the
    * columnDef.sortable value.
    */
   var createKendoGridColumnSortable = function (columnDef) {
      var sortable = jsUtils.getProperty(columnDef, 'sortable');
      if (typeof sortable === 'boolean') {
         return sortable;
      } else {
         return {
            compare: sortable
         };
      }
   };

   /* Creates a template stringified version of the variable
    * if no template exists
    * NOTE: This is needed to prevent functions from getting
    * executed as a security measure
    */
   var createTemplateIfNotExist = function (columnDef) {
      var template = jsUtils.getProperty(columnDef, 'template');
      if (jsUtils.isUndefinedOrNull(template)) {
         return '<span ng-non-bindable>#: ' + columnDef.field + ' ? ' + columnDef.field + ' : "" #</span>';
      }

      return template;
   };

   // -------------------------------------------------------------------------
   // Creating Kendo Grid - helper functions ----------------------------------
   // -------------------------------------------------------------------------

   /* Creates and returns a Kendo grid column object initialized using
    * scope.gridOptions.columnDefs
    */
   var createKendoGridColumn = function (columnDef) {
      return {
         editor:     createKendoGridColumnEditor(columnDef),
         field:      columnDef.field,
         filterable: createKendoGridColumnFilter(columnDef),
         format:     jsUtils.getProperty(columnDef, 'format'),
         hidden:     !jsUtils.getProperty(columnDef, 'visible', true),
         locked:     jsUtils.getProperty(columnDef, 'locked', false),
         sortable:   createKendoGridColumnSortable(columnDef),
         template:   createTemplateIfNotExist(columnDef),
         title:      columnDef.displayName,
         width:      jsUtils.getProperty(columnDef, 'width'),
         headerTemplate: columnDef.headerTemplate,
         uid:        columnDef.uid
      };
   };

   // Creates and returns a Kendo grid columns definition
   var createKendoGridColumns = function (columnDefs) {
      var cols = [];
      if (!jsUtils.isUndefinedOrNull(columnDefs)) {
         for (var i = 0; i < columnDefs.length; i++) {
            var columnDef = columnDefs[i];

            var kendoCol = createKendoGridColumn(columnDef);
            cols.push(kendoCol);
         }
      }
      return cols;
   };

   /* Creates and returns a Kendo selectable value based on the
    * gridOptions.selectionMode value.
    */
   var createKendoGridSelectable = function (gridOptions) {
      var selectionMode = jsUtils.getProperty(gridOptions, 'selectionMode',
            vuiConstants.grid.selectionMode.NONE);
      if (selectionMode !== vuiConstants.grid.selectionMode.NONE) {
         return selectionMode ===
               vuiConstants.grid.selectionMode.SINGLE ? 'row' : 'multiple, row';
      } else {
         return;
      }
   };

   /* Creates and returns a Kendo sortable value based on the
    * gridOptions.sortMode value.
    */
   var createKendoGridSortable = function (gridOptions) {
      var sortMode = jsUtils.getProperty(gridOptions, 'sortMode',
            vuiConstants.grid.sortMode.SINGLE);
      if (sortMode !== vuiConstants.grid.sortMode.NONE) {
         return sortMode ===
               vuiConstants.grid.sortMode.SINGLE ? true : { mode: 'multiple' };
      } else {
         return false;
      }
   };

   // -------------------------------------------------------------------------
   // Creating Kendo DataSource Options - helper functions --------------------
   // -------------------------------------------------------------------------

   /* Creates and returns a Kendo dataSource order object initialized using
    * scope.gridOptions.filter
    */
   var createKendoGridFilter = function (filterOptions) {
      var filters = [];
      if (!jsUtils.isUndefinedOrNull(filterOptions)) {
         for (var i = 0; i < filterOptions.length; i++) {
            // A group definition must at least specify one field to filter grid by.
            jsUtils.validateObject(filterOptions[i], [['field', 'string']]);

            var filterInfo = {
               field:      filterOptions[i].field,
               operator:   filterOptions[i].operator,
               value:      filterOptions[i].value
            };
            filters.push(filterInfo);
         }
      }
      return filters;
   };

   /* Creates and returns a Kendo dataSource order object initialized using
    * scope.gridOptions.groupOrder/ scope.gridOptions.sortOrder
    */
   var createKendoGridOrder = function (orderDefs) {
      var order = [];
      if (!jsUtils.isUndefinedOrNull(orderDefs)) {
         for (var i = 0; i < orderDefs.length; i++) {
            // A group definition must at least specify one field to group grid by.
            jsUtils.validateObject(orderDefs[i], [['field', 'string']]);

            var orderInfo = {
               field:   orderDefs[i].field,
               dir:     jsUtils.getProperty(orderDefs[i], 'dir', 'asc')
            };
            order.push(orderInfo);
         }
      }
      return order;
   };

   /* Creates and returns a Kendo column editable value based on the
    * columnDef.editable and datagridOptions.editable value.
    */
   var getColumnEditorConfig = function (columnDef) {
      var editorConfig = {};
      // If not defined set it to editable value on datagrid
      var editable = jsUtils.getProperty(columnDef, 'editable', isGridEditable);
      if (typeof editable === 'object') {
         if (editable.hasOwnProperty('validator')) {
            var validatorObj = editable.validator;
            editorConfig.validation = {};
            for (var key in validatorObj) {
               editorConfig.validation[key] = validatorObj[key];
            }
         }
      } else {
         editorConfig.editable = editable;
      }
      return editorConfig;
   };

   /* Creates and returns a Kendo dataSource schema.model.fields
    *
    * `schema.model.fields` is an array of objects, where each object represents
    * column field and sets each field's configuration using properties from
    * scope.gridOptions.columnDefs.
    * The following configurations are set for each field: type of field,
    * validation rules, editable
    * Example:
    *    fields: {
    *                name: {
    *                   type: 'string',
    *                   validation: {
    *                      required: true
    *                   }
    *                }
    *             }
    * where `name` is the name of the data property that this column represents.
    *
    * Note: Unlike columnSpec.editable.validator that is set on
    * `schema.model.fields`, the custom editor UI property defined in
    * columnSpec.editable.editor is set on `column` in createKendoGridColumnEditor()
    * while creating Kendo grid columns object.
    */
   var createKendoSchema = function (gridOptions) {
      var columnDefs = gridOptions.columnDefs;
      var fieldsDef = {};
      if (!jsUtils.isUndefinedOrNull(columnDefs)) {
         for (var i = 0; i < columnDefs.length; i++) {
            fieldsDef[columnDefs[i].field] = {
               type :
                  jsUtils.getProperty(columnDefs[i], 'type', 'string')
            };

            // Set column editor configuration consisting of editable value,
            // validators to be applied
            var editorConfig = getColumnEditorConfig(columnDefs[i]);
            for (var key in editorConfig) {
               fieldsDef[columnDefs[i].field][key] = editorConfig[key];
            }
         }
      }

      var schema = {};

      if (jsUtils.getProperty(gridOptions.data, 'data') !== undefined) {
         schema.data = jsUtils.getProperty(gridOptions.data, 'data');
      }

      schema.total =  jsUtils.getProperty(gridOptions.data, 'total',
            vuiConstants.grid.serverSide.TOTAL);
      schema.model = {
         fields: fieldsDef
      };

      return schema;
   };

   // -------------------------------------------------------------------------
   // Creating Kendo DataSource -----------------------------------------------
   // -------------------------------------------------------------------------

   /* Creates and returns a Kendo data source initialized using
    * scope.gridOptions.data
    */
   var createKendoGridDataSource = function (data, isEditable) {
      // Set to empty array when data is not defined
      if (typeof data === 'undefined') {
         data = [];
         gridOptions.data = [];
      }

      var isObject = (typeof data === 'object') ? true : false;
      var kendoDataSrc;

      // Case 1 LocalData (Array of objects)
      if (isObject && Array.isArray(data)) {
         kendoDataSrc =
               createKendoGridDataSourceFromLocalData(data, isEditable);
      } else
      // Case 2 RemoteData (ServerSideOptions)
      // Note: By implementation remote data is not editable and processed
      // on server (filter, sort, group, pageSize).
      if (isObject) {
         var dataUrl = jsUtils.getProperty(data, 'url');
         var dataType = jsUtils.getProperty(
               data, 'dataType', vuiConstants.grid.dataType.JSON
            );
         var dataMethod = jsUtils.getProperty(
               data, 'method', vuiConstants.grid.dataMethod.GET
            );
         var dataParams = jsUtils.getProperty(data, 'params', undefined);
         var contentType = jsUtils.getProperty(data, 'contentType', undefined);
         var beforeSend = jsUtils.getProperty(data, 'beforeSend', undefined);
         var processData = jsUtils.getProperty(data, 'processData', true);
         var change = jsUtils.getProperty(data, 'change', undefined);
         var transport = jsUtils.getProperty(data, 'transport', undefined);

         if (!jsUtils.isUndefinedOrEmpty(dataUrl) ||
               !jsUtils.isUndefinedOrNull(transport)) {
            kendoDataSrc = createKendoGridDataSourceFromRemoteData(
                  transport, dataUrl, dataType, dataMethod, dataParams,
                  contentType, beforeSend, processData, change
               );
         } else {
            console.error('URL of the remote service is not defined');
            kendoDataSrc = '';
         }
      }
      return kendoDataSrc;
   };

   var createKendoGridDataSourceFromLocalData = function (data, isGridEditable) {
      var localDataSourceOptions = {
         filter:        dataSourceOptions.filter,
         group:         dataSourceOptions.group,
         pageSize:      dataSourceOptions.pageSize,
         sort:          dataSourceOptions.sort,
         transport: {
            read: function (e) {
               e.success(data);
            }
         }
      };

      // Set schema data item (model) configuration
      localDataSourceOptions.schema = {};
      localDataSourceOptions.schema.model = dataSourceOptions.schema.model;

      /* In case of editable datagrid add the following:
       * 1. `transport.update` configuration which is used when the DataSource
       *    saves updated data items. User data is updated here.
       * 2. `schema.model.id` - required for proper editing
       */
      if (isGridEditable) {
         // Currently there is a bug in Kendo that for angulatJS implementation
         // of grid autoSync does not work as ecpected
         // localDataSourceOptions.autoSync = true;

         localDataSourceOptions.schema.model.id =
               dataSourceOptions.schemaModelId;

         localDataSourceOptions.transport.update =
               function (e) {
                  // Update users data
                  // TODO aakankshas: Replace only the updated user object in
                  // $scope.datagridOptions.data
                  $scope.datagridOptions.data =
                        kendoGridDataSource.data().toJSON();
                  isGridDataUpdated = true;
                  e.success();
               };
      }

      return new kendo.data.DataSource(localDataSourceOptions);
   };

   /* Create and returns a kendo data source set from remote data
    * By design server-side processing is enabled when accessing remote data
    * Note: Currently there is no way of disabling server-side procssing via API
    */
   var createKendoGridDataSourceFromRemoteData =
         function (transport, url, dataType, method, params, contentType, beforeSend, processData, change) {
      var remoteDataSourceOptions = {
            filter:            dataSourceOptions.filter,
            group:             dataSourceOptions.group,
            pageSize:          dataSourceOptions.pageSize,
            serverFiltering:   true,
            serverGrouping:    true,
            serverPaging:      true,
            serverSorting:     true,
            sort:              dataSourceOptions.sort,
            transport:         transport ? transport : {
               read: {
                  url:         url,
                  dataType:    dataType,
                  method:      method,
                  data:        params,
                  contentType: contentType,
                  processData: processData,
                  beforeSend:  beforeSend
               }
            },
            change: change
         };

      // Set schema data item (data, model, total) configuration
      remoteDataSourceOptions.schema = dataSourceOptions.schema;

      return new kendo.data.DataSource(remoteDataSourceOptions);
   };

   // -------------------------------------------------------------------------
   // Custom event handling ---------------------------------------------------
   // -------------------------------------------------------------------------

   var isAlreadySelected = function (selectedItems, uid) {
      var result = $.grep(selectedItems, function (e) {
            return e.uid === uid;
         });
      return (result.length === 1);
   };

   /* Uncheck rows that are not currently selected but were selected previously
    * Addresses case when a new row is selected without Ctrl or Shift
    */
   var uncheckItemsNotCurrentlySelected = function (currSelectedItems) {
      var unselectedId = [];
      var selectDom;

      // Find uid of dataItem(s) to be unchecked by comparing each item in
      // selectedDataItems with every currently selected item
      $.map(selectedDataItems, function (item) {
         var stillSelected = false;

         $.map(currSelectedItems, function (selItem) {
            if ($(selItem).attr('data-uid') === item.uid) {
               stillSelected = true;
            }
         });

         if (!stillSelected) {
            unselectedId.push(item.uid);
         }
      });

      // Uncheck dataItem(s)
      // DataItem is removed from selectedDataItems array in the event handler
      // for checkbox
      for (var i = 0; i < unselectedId.length; i++) {
         selectDom = $element.find('.k-grid-content table tbody tr[data-uid="' +
                  unselectedId[i] + '"]').find('td:first-child input[type="checkbox"]');
         selectDom.prop('checked', false);
         angular.element(selectDom).triggerHandler('click');
      }
   };

   var onChangeCallback = function (items) {
      var callback = $scope.gridOptions.onChange;
      if (callback && typeof callback === 'function') {
         callback(items);
      }
   };

   /* Called when user selects or de-selects a row in the datagrid.
    */
   var onChange = function () {
      var currSelectedItems = this.select();
      var checkBoxDoM;

      /* Handle multi row selection on click
       * This only checks/unchecks the associated checkbox and respective
       * click handler is triggered which handles the logic for selectedItems
       */
      if (gridSelectionMode === vuiConstants.grid.selectionMode.MULTI) {

         if (showCheckboxesOnMultiSelection) {

            if (isItemDisabledCallback && currSelectedItems) {
               var isSelectionModified = false;
               for (var selectedItemIndex = 0; selectedItemIndex < currSelectedItems.length; selectedItemIndex++) {
                  if (isItemDisabledCallback(this.dataItem(currSelectedItems[selectedItemIndex]))) {
                     $(currSelectedItems[selectedItemIndex]).removeClass('k-state-selected');
                     isSelectionModified = true;
                  }
               }
               if (isSelectionModified) {
                  currSelectedItems = this.select();
               }
            }

            uncheckItemsNotCurrentlySelected(currSelectedItems);

            // Check corresponding checkboxes of selected rows
            $.map(currSelectedItems, function (item) {
               // Not to recheck checkbox and triggerHandler when the row is
               // already selected
               if (!isAlreadySelected(selectedDataItems, $(item).attr('data-uid'))) {
                  checkBoxDoM = $(item).find('td:first-child input[type="checkbox"]');
                  checkBoxDoM.prop('checked', true);
                  angular.element(checkBoxDoM).triggerHandler('click');
               }
            });
         } else {
            selectedDataItems = [];

            for (var i = 0; i < currSelectedItems.length; i++) {
               selectedDataItems.push(this.dataItem(currSelectedItems[i]));
            }

            //Update selectedItems
            vuiUtils.updateScopeProperty($scope,
                  'datagridOptions.selectedItems', selectedDataItems);
         }

      }

      /* Handle single row selection on click
       * This also handles the logic for selectedItems
       */
      else if (gridSelectionMode === vuiConstants.grid.selectionMode.SINGLE) {

         //Empty selectedItems array
         selectedDataItems = [];
         // Note: If radio button column is added add logic to check/ uncheck
         // corresponding radiobutton
         if (currSelectedItems.length > 0 && !currSelectedItems.hasClass('k-grouping-row')) {
            selectedDataItems.push(this.dataItem(currSelectedItems[0]));
         }

         //Update selectedItems
         vuiUtils.updateScopeProperty($scope,
               'datagridOptions.selectedItems', selectedDataItems);
      }

      onChangeCallback(selectedDataItems);
   };

   /* Called from the Kendo datagrid, when "change" event is fired.
    * Wrapper around "onChange" function, which guarantee that the "onChange" will
    * be called in Angular zone.
    *
    * The Kendo event handlers are global (attached to the document element).
    * If the "change" handler is initialize by
    * kendo widget which is outside Angular zone (Kendo splitter), this method will be
    * called outside Angular zone.
    *
    */
   var widgetOnChangeHandler = function () {
      vuiZoneService.runInsideAngularZone(onChange.bind(this));
   };

   /* This is used to maintain selection of previously selected rows during
    * pagination/ virtual scrolling.
    */
   var onDataBound = function () {
      var selectedItems = selectedDataItems;
      if (selectedItems) {
         selectedItems.forEach(function (item) {
            var row = $('.k-grid')
                  .find('tr[data-uid="' + item.uid + '"]');
            // Check if row exists
            if (row.length > 0) {
               // Mark it as selected and check corresponding checkbox
               row.addClass('k-state-selected')
                     .find('.row-checkbox')
                     .prop('checked', true);
            }
         });
      }
   };

   // -------------------------------------------------------------------------
   // Creating Kendo Grid Options ---------------------------------------------
   // -------------------------------------------------------------------------

   /* Create the Kendo grid pageable options
    */
   var createKendoGridPageableOptions = function (pageConfig) {
      var displayMode = jsUtils.getProperty(
            pageConfig,
            'displayMode',
            vuiConstants.grid.displayMode.VIRTUAL_SCROLLING
         );

      if (pageConfig &&
            pageConfig.hidePager &&
            displayMode === vuiConstants.grid.displayMode.VIRTUAL_SCROLLING) {
         return false;
      }

      var pageableOptions = {
         numeric: (displayMode === vuiConstants.grid.displayMode.PAGES),
         previousNext: false,
         info: true,
         messages: {
            empty: vuiLocale.datagrid.emptyTable
         }
      };

      if (displayMode === vuiConstants.grid.displayMode.PAGES) {
         pageableOptions.messages.display =
               '{0} - {1} of {2} ' + vuiLocale.datagrid.items;
      } else {
         pageableOptions.messages.display = '{2} ' + vuiLocale.datagrid.items;
      }

      return pageableOptions;
   };

   /* Creates and returns the kendo grid options object with initial properties
    * set from the scope.datagridOptions object and default kendo grid
    * configurations.
    * Grid toolbar is also set if actionBarOptions are defined or when grid is
    * editable (Save button is added).
    */
   var createKendoGridOptions = function (gridOptions) {
      // Set global isGridEditable
      isGridEditable = jsUtils.getProperty(gridOptions, 'editable', false);
      containsNoInput = jsUtils.getProperty(gridOptions, 'noInput', false);

      // Set global dataSourceOptions
      // TODO aakankshas: Add two-way binding for sortOrder, groupOrder and filter
      dataSourceOptions = {
         filter: createKendoGridFilter(gridOptions.filter),
         group: createKendoGridOrder(gridOptions.groupOrder),
         schema: createKendoSchema(gridOptions),
         schemaModelId: jsUtils.getProperty(gridOptions, 'idDataField'),
         sort: createKendoGridOrder(gridOptions.sortOrder),
         pageSize: jsUtils.getProperty(gridOptions.pageConfig, 'size', 100)
      };

      var displayMode = jsUtils.getProperty(
            gridOptions.pageConfig,
            'displayMode',
            vuiConstants.grid.displayMode.VIRTUAL_SCROLLING
         );

      if (displayMode !== vuiConstants.grid.displayMode.VIRTUAL_SCROLLING &&
            displayMode !== vuiConstants.grid.displayMode.PAGES) {
         displayMode = vuiConstants.grid.displayMode.VIRTUAL_SCROLLING;
      }

      if (!jsUtils.isUndefinedOrNull(gridOptions.pageConfig)) {
         gridOptions.pageConfig.displayMode = displayMode;
      }

      // A collection of Kendo grid configuration properties and events
      var options = {
         columns:       createKendoGridColumns(gridOptions.columnDefs),
         editable:      isGridEditable,
         filterable:    jsUtils.getProperty(gridOptions, 'searchable', true),
         height:
               jsUtils.getProperty(gridOptions, 'height'),
         navigatable:
               jsUtils.getProperty(gridOptions, 'allowKeyboardNavigation', true),
         reorderable:
               jsUtils.getProperty(gridOptions, 'reorderable', false),
         resizable:
               jsUtils.getProperty(gridOptions, 'resizable', false),
         selectable:    createKendoGridSelectable(gridOptions),
         sortable:      createKendoGridSortable(gridOptions),

         // Following properties are set using pageConfig
         pageable: createKendoGridPageableOptions(gridOptions.pageConfig),
         scrollable: {
            virtual: (displayMode === vuiConstants.grid.displayMode.VIRTUAL_SCROLLING)
         }
      };

      // Allow other options, supported by kendo, to pass through this wrapper
      options = angular.extend(gridOptions, options);

      // Add toolbar actions if actionBarOptions are defined
      if (gridOptions.actionBarOptions) {
         options.toolbar = [{
               template:   '<div vui-action-bar="datagridOptions.actionBarOptions"' +
                              'class="pull-left">' +
                           '</div>'
            }];
      }

      return options;
   };

   // -------------------------------------------------------------------------
   // Selection Column --------------------------------------------------------
   // -------------------------------------------------------------------------

   /* Returns raw array representation of data items
    */
   var getDataItemsJson = function (dataItems) {
      var dataItemsJson = [];
      for (var i = 0 ; i < dataItems.length; i++) {
         var dataItemJSON = dataItems[i].toJSON();
         // Adding the id of the object if it is missing.
         // The reason for this change is that the toJSON method
         // removes the id of the object if it isn't set in the
         // schema.model.id.
         dataItemJSON.id = dataItemJSON.id || dataItems[i].id;
         dataItemsJson.push(dataItemJSON);
      }

      return dataItemsJson;
   };

   /* Returns selectedItems after filtering out item whose uid is
    * equal to passed uid (item to be removed)
    */
   var removeUnselectedItem = function (selectedItems, uid) {
      var result = $.grep(selectedItems, function (e) {
            return e.uid !== uid;
         });
      return result;
   };

   /* Returns gridOptions with requested additional column type
    * The supported values for colType are checkbox and radiobutton.
    */
   var addColumnForSelection = function ($scope, gridOptions, colType) {
      var hasLockedCol = false;
      var colTemplate = '';
      var colHeaderTemplate = '';

      if (colType === 'checkbox') {
         // In case of multi select add checkbox column
         // Only the <input> elements do not trigger kendo onChange event when clicked
         // We wrap the checkbox into a div and add an input (transparent type="text") as a cell "background"
         // so clicking outside of checkbox(within the cell) behaves the same as clicking the checkbox
         // (onChange won't be triggered and selection won't be lost)
         var enabledCheckboxTemplate = '<div class="vui-checkbox-container"> \
             <input type="checkbox" class="row-checkbox" \
             ng-click="checkboxHandler($event)" /></input>\
             <input tabIndex="-1" ng-click="checkboxHandler($event)"></input>\
              </div>';
         var disabledCheckboxTemplate = '<input type="checkbox" class="row-checkbox" disabled="disabled" />';

         if (!isItemDisabledCallback) {
            colTemplate = enabledCheckboxTemplate;
         } else {
            colTemplate = function (dataItem) {
               if (isItemDisabledCallback(dataItem)) {
                  return disabledCheckboxTemplate;
               } else {
                  return enabledCheckboxTemplate;
               }
            };
         }

         colHeaderTemplate = '<input type="checkbox"' +
            ' ng-click="toggleSelectAll($event)" />';
      } else
      if (colType === 'radiobutton') {
         // In case of single select add radiobutton column
         colTemplate = '<input type="radio" class="row-radiobutton" ' +
               'name="select-row" ng-click="radioButtonHandler($event)" />';
      } else {
         // Simple return gridOptions when no colType is specified
         return gridOptions;
      }

      gridOptions.columns.unshift({
         filterable:       false,
         sortable:         false,
         groupable:        false,
         width:            '27px',
         headerTemplate:   colHeaderTemplate,
         template:         colTemplate
      });

      // Check for locked columns
      gridOptions.columns.forEach(function (column) {
         if (column.locked) {
            hasLockedCol = true;
            return;
         }
      });

      // When user has a locked column, lock checkbox/radio column for
      // for proper placement
      if (hasLockedCol) {
         gridOptions.columns[0].locked = true;
      }

      $scope.checkboxHandler = function ($event) {
         $event.stopPropagation();
         //retrieve the checkbox whenever the actual checkbox or the container was pressed
         var checkbox = $($event.target).is('[type=checkbox]') ? $event.target : $($event.target).siblings()[0];

         if (!$($event.target).is('[type=checkbox]')) {
            //cell background is clicked - changing checkbox value manually
            checkbox.checked = !checkbox.checked;
         }
         var grid = $($event.target).closest('.k-grid');
         var gridView = gridOptions.dataSource.data();
         /* Row with checkbox
          * Note: In case when there is a locked column this row will
          * be in k-grid-content-locked div, so corresponding row in
          * k-grid-content will need to be marked as selected
          */
         var checkedRow = $($event.target).closest('tr');
         var dataItem = grid.data('kendoGrid').dataItem(checkedRow);

         if (checkbox.checked) {
            checkedRow.addClass('k-state-selected');
            // Select corresponding row in k-grid-content
            if (hasLockedCol) {
               grid.find('.k-grid-content table tbody tr[data-uid="' +
                     dataItem.uid + '"]')
                     .addClass('k-state-selected');
            }
            selectedDataItems.push(dataItem);
         } else {
            checkedRow.removeClass('k-state-selected');
            if (hasLockedCol) {
               // Unselect corresponding row in k-grid-content
               grid.find('.k-grid-content table tbody tr[data-uid="' +
                     dataItem.uid + '"]')
                     .removeClass('k-state-selected');
            }
            selectedDataItems = removeUnselectedItem(selectedDataItems, dataItem.uid);
         }

         // Handles case when all row checkboxes are clicked then
         // header checkbox should be checked
         if (selectedDataItems.length === gridView.length) {
            grid.find('thead > tr > th input[type="checkbox"]')
                  .prop('checked', true);
         } else {
            grid.find('thead > tr > th input[type="checkbox"]')
                  .prop('checked', false);
         }

         vuiUtils.updateScopeProperty($scope,
               'datagridOptions.selectedItems',
               getDataItemsJson(selectedDataItems));

         onChangeCallback(selectedDataItems);
      };

      // Function called when a column header checkbox is checked or unchecked.
      $scope.toggleSelectAll = function ($event) {
         var grid = $($event.target).closest('.k-grid');
         /* The data array returned from Kendo is an ObservableArray which
          * contains array object along with tracking capabilities.
          * TODO: use toJSON() returns JS array which represents the contents of the
          * ObservableArray. Using this cases inability to search using UID.
          */
         var gridView = gridOptions.dataSource.data();
         if ($event.target.checked) { // check select status
            // select all
            var rows = grid.find('tbody > tr[role=\'row\']:lt(' + gridView.length + ')');
            var selectedItems;
            if (isItemDisabledCallback && rows) {
               selectedItems = [];

               for (var i = 0; i < rows.length; i++) {
                  var rowItem = rows[i];
                  var dataItem = grid.data('kendoGrid').dataItem(rowItem);
                  if (!isItemDisabledCallback(dataItem)) {
                     $(rowItem).addClass('k-state-selected')
                        .find('.row-checkbox')
                        .prop('checked', true);
                     selectedItems.push(dataItem);
                  }
               }

            } else {
               rows.addClass('k-state-selected')
                  .find('.row-checkbox')
                  .prop('checked', true);
               selectedItems = gridView;
            }

            selectedDataItems = selectedItems;
         } else {
            // deselect all
            grid.find('tbody > tr[role=\'row\']:lt(' + gridView.length + ')')
                  .removeClass('k-state-selected')
                  .find('.row-checkbox')
                  .prop('checked', false);
            selectedDataItems = [];
         }

         vuiUtils.updateScopeProperty($scope,
               'datagridOptions.selectedItems',
               getDataItemsJson(selectedDataItems));

         onChangeCallback(selectedDataItems);
      };

      $scope.radioButtonHandler = function ($event) {
         var grid = $($event.target).closest('.k-grid');
         selectedDataItems = [];

         var selectedRow = $($event.target).closest('tr');
         var dataItem = grid.data('kendoGrid').dataItem(selectedRow);

         /* Note: In case when there is a locked column this row will
          * be in k-grid-content-locked div, so corresponding row in
          * k-grid-content will need to be marked as selected
          */
         // Remove selected class from all other rows
         if (hasLockedCol) {
            grid.find('.k-grid-content-locked table tbody tr')
               .removeClass('k-state-selected');
         }
         grid.find('.k-grid-content table tbody tr')
            .removeClass('k-state-selected');

         // Add selected class
         selectedRow.addClass('k-state-selected');
         if (hasLockedCol) {
            grid.find('.k-grid-content table tbody tr[data-uid="' +
                  dataItem.uid + '"]')
                  .addClass('k-state-selected');
         }
         selectedDataItems.push(dataItem);

         vuiUtils.updateScopeProperty($scope,
               'datagridOptions.selectedItems',
               getDataItemsJson(selectedDataItems));

         onChangeCallback(selectedDataItems);
      };

      return gridOptions;
   };

   // -------------------------------------------------------------------------
   // Grid Initialization -----------------------------------------------------
   // -------------------------------------------------------------------------

   var updateDataSource = function (data) {
      var kendoDataSrc =
            createKendoGridDataSource(data, isGridEditable);

      // Set global kendoGridDataSource
      kendoGridDataSource = kendoDataSrc;

      // Clear selectedItems array when data changes
      selectedDataItems = [];
      vuiUtils.updateScopeProperty($scope,
               'datagridOptions.selectedItems', selectedDataItems);

      // Set the dataSource of the datagrid and rebinds it
      kendoGridElem.data('kendoGrid').setDataSource(kendoDataSrc);
      gridOptions.dataSource = kendoDataSrc;

      // Uncheck select all checkbox when data changes
      kendoGridElem.find('.k-grid-header thead > tr > th input[type="checkbox"]')
         .prop('checked', false);

      onChangeCallback(selectedDataItems);
   };

   var updateData = function (data) {
      if (!Array.isArray(data) ||
         (kendoGridDataSource._data && kendoGridDataSource._data.length === 0) ||
         !containsNoInput) {
         updateDataSource(data);
      } else {
         kendoGridDataSource.data(data);
      }
   };

   function createVuiFooter(kendoGridElem) {
      var footerScope = $scope.$new();
      footerScope.footerOptions = gridOptions.vuiFooter || {};
      footerScope.datagridElement = kendoGridElem;

      var footer = kendoGridElem.find('.k-pager-wrap.k-grid-pager');
      var footerBar = $compile(
         '<vui-datagrid-footer-bar class="vui-action-bar" ' +
         'datagrid-element="datagridElement" ' +
         'options="footerOptions"></vui-datagrid-footer-bar>')(footerScope);
      footer.append(footerBar);

      $scope.$on('$destroy', function () {
         footerScope.$destroy();
         footerBar.remove();
      });
   }

   self.init = function () {
      //Options to be given to the kendo grid from user provided datagridOptions
      if (jsUtils.isEmpty($scope.datagridOptions)) {
         console.error('Datagrid options not provided. ');
      } else {
         gridOptions = createKendoGridOptions($scope.datagridOptions);

         /* Add selection column based on selectionMode value
          * Checkbox column for selectionMode.MULTI
          * Radio button column for selectionMode.SINGLE
          */
         var selectionMode = jsUtils.getProperty($scope.datagridOptions,
               'selectionMode', vuiConstants.grid.selectionMode.NONE);

         // Update global gridSelectionMode
         gridSelectionMode = selectionMode;
         showCheckboxesOnMultiSelection = jsUtils.getProperty(
               $scope.datagridOptions, 'showCheckboxesOnMultiSelection', true);

         isItemDisabledCallback = $scope.datagridOptions.isItemDisabledCallback;

         // Add a checkbox column in case of MULTI selectionMode
         if (selectionMode === vuiConstants.grid.selectionMode.MULTI && showCheckboxesOnMultiSelection) {
            gridOptions =
               addColumnForSelection($scope, gridOptions, 'checkbox');
         }

         $scope.gridOptions = gridOptions;

         var kendoGrid;
         // Kendo UI directives emits this event for each widget which is
         // created
         $scope.$on('kendoWidgetCreated', function (event, widget) {
            kendoGridElem = $element.find('div').eq(0);
            // check that the event is for grid
            if (widget === kendoGridElem.data('kendoGrid')) {
               initComplete = true;
               kendoGrid = widget;
               // Attach internal grid event handlers
               widget.bind('dataBound', onDataBound);
               widget.bind('change', widgetOnChangeHandler);

               updateDataSource($scope.datagridOptions.data);
               createVuiFooter(kendoGridElem);
            }
         });

         /* When the data object changes, update datagrid's data.
          * Note: This is not a deep watch.
          * Datagrid's data is not updated when data is edited
          */
         $scope.$watch('datagridOptions.data', function (newData, oldData) {
            // Not to update datagrid's data when data is edited by user from UI
            if (isGridDataUpdated) {
               isGridDataUpdated = false;
               return;
            }

            if (typeof newData !== 'undefined' && newData !== oldData) {
               if (initComplete) {
                  updateData(newData);
               } else {
                  return;
               }
            }
         });

         $scope.$on('$destroy', function () {
            if (kendoGrid) {
               kendoGrid.unbind('dataBound');
               kendoGrid.unbind('change');
               // there are some cases when the kendo grid element is removed
               // from DOM before destroying the scope (because of a call in
               // kendo.extension.js file in H5 that destroys kendo widgets when
               // their wrapping element is removed from DOM, in general such
               // a monkey patches are not good, but since we don't have support
               // for kendo and we are trying to fix the issues by ourself we run
               // into such situations). If you try to destroy kendo grid more
               // than once you are getting an error, that is why i am adding
               // code to prevent re-destroying of the gird if it is already
               // destroyed
               if (kendoGrid.wrapper) {
                  kendoGrid.destroy(); // destroy the grid object
               }
               kendoGrid = null;
            }
         });
      }
   };

}]);
;'use strict';
(function () {
   vui.angular.datagrid.directive('vuiDatagridFooterAction', [function () {
      return {
         template:
            '<li title="{{tooltip}}">' +
               '<a href="" ng-disabled="isDisabled()" ng-click="action()" class="k-container-flex">' +
                  '<span class="vui-icon-placeholder" ng-class="icon"></span>' +
                  '<span class="vui-action-label">{{::label}}</span>' +
               '</a>' +
            '</li>',
         controller: ['$scope', function ($scope) {
            $scope.isDisabled = function () {
               return $scope.disabled;
            };
         }],
         scope: {
            label: '@',
            tooltip: '=',
            icon: '@',
            action: '&',
            disabled: '='
         }
      };
   }]).directive('vuiDatagridFooterBar', ['vuiLocale', 'vuiFooterActionsService', function (vuiLocale, vuiFooterActionsService) {
      return {
         template:
         '<ul ng-if="actions.length" class="k-container-flex">' +
            '<vui-datagrid-footer-action ng-repeat="action in actions" ' +
               'label="{{::action.label}}" tooltip="action.tooltip" ' +
               'icon="{{::action.icon}}" action="action.action()" ' +
               'disabled="action.disabled">' +
            '</vui-datagrid-footer-action>' +
            '<li>' +
               '<span class="vui-actionbar-separator"></span>' +
            '</li>' +
         '</ul>',
         controller: ['$scope', function ($scope) {
            var actions = [];
            if ($scope.options.actions && $scope.options.actions.length > 0) {
               actions = actions.concat($scope.options.actions);
            }
            if ($scope.options.showCopy) {
               actions.push({
                  label: vuiLocale.datagrid.copy,
                  icon: 'vui-icon-action-copy',
                  action: function () {
                     vuiFooterActionsService.copyAllItems($scope.datagridElement);
                  }
               });
            }

            $scope.actions = actions;
         }],
         scope: {
            options: '=',
            datagridElement: '='
         }
      };
   }
   ]);

})();
;'use strict';


(function () {
   /**
    * @name vui.angular.services.vuiFooterActionsService
    * @module vui.angular.services
    *
    * @description
    *    Contains actions for the vui datagrid footer.
    */
   vui.angular.services.service('vuiFooterActionsService', ['clipboardService', function (clipboardService) {
      var removeHtmlRegex = /<[^>]+>/gm; //used to strip html markup from rendered cells

      function getRowValue(currentRow, column) {
         if (column.template) {
            return column.template(currentRow).replace(removeHtmlRegex, '');
         }
         return currentRow[column.field];
      }

      function getGroupValue(currentRow, column) {
         return column.field === currentRow.field ? currentRow.value : '';
      }

      function buildRowValues(visibleRows, columns, rows, groups) {
         var rowValue = [];
         for (var i = 0; i < visibleRows.length; i++) {
            var currentRow = visibleRows[i];
            var isGroupingRow = !!groups[currentRow.field];

            for (var j = 0; j < columns.length; j++) {
               var column = columns[j];
               if (isGroupingRow) {
                  rowValue.push('"' + getGroupValue(currentRow, column) + '"');
               } else if (column.isGrouping) {
                  rowValue.push('');
               } else if (!column.hidden) {
                  rowValue.push('"' + getRowValue(currentRow, column) + '"');
               }
            }
            rows.push(rowValue.join(','));

            if (isGroupingRow && currentRow.items && currentRow.items.length) {
               rowValue.push(buildRowValues(currentRow.items, columns, rows, groups));
            }
            rowValue.length = 0;
         }
      }

      function copyAllItems(datagridElement) {
         var kendoGrid = datagridElement.data('kendoGrid');
         var columns = kendoGrid.columns;

         var groups = {};
         var headerRowValue = [];

         var columnsData = [];
         //extracting grouping rows
         var grouping = kendoGrid.options.groupOrder;
         if (grouping && grouping.length) {
            angular.forEach(grouping, function (group) {
               groups[group.field] = group;
               var column = $.grep(columns, function (column) {
                  return column.field === group.field;
               })[0];
               headerRowValue.push('"' + column.title + '"');
               columnsData.push({
                  field: column.field,
                  isGrouping: true
               });

            });
         }

         angular.forEach(columns, function (column) {
            if (column && !column.hidden) {
               var columnData = {
                  field: column.field
               };
               if (column.template && column.template !== 'string') {
                  columnData.template = kendo.template(column.template);
               } else {
                  columnData.template = column.template;
               }
               columnsData.push(columnData);
               headerRowValue.push('"' + column.title + '"');
            }
         });

         var rows = [];
         rows.push(headerRowValue.join(','));
         buildRowValues(kendoGrid.dataSource.view(), columnsData, rows, groups);

         clipboardService.copyToClipboard(rows.join('\n'));
      }

      return {
         copyAllItems: copyAllItems
      };

   }]);

})();
;'use strict';

vui.angular.dialog

/**
 * @ngdoc service
 * @name vui.angular.dialog.vuiDialogService
 * @module vui.angular.dialog
 *
 * @description
 *    Service to create multiple instances of the dialog. The service allows
 *    you to create, show, hide and destroy the dialog.
 *
 * # General usage
 *    The `vuiDialogService` is a function which takes a single argument, a
 *    configuration object that is used to create a dialog.
 *
 * @param {Object} config
 *    Object containing the scope and the configObjectName.
 *
 *    - **scope** – `{$scope}` – Angular scope object of the dialog container.
 *       It can be the scope of the controller or rootScope or a custom
 *       scope.
 *
 *       The scope of each page in the dialog prototypically inherits from the
 *    scope that is mentioned here.
 *    - **configObjectName** – `{string}` – The name of the scope variable
 *       declared on the above mentioned scope. The variable is of type
 *       {@link vui.angular.dialog:DialogOptions DialogOptions}.
 *
 *    Example:
 *    ```js
 *    var config = {
 *                     scope: $scope,
 *                     configObjectName: 'dialogOptions'
 *                 };
 *    ```
 *
 * @returns {Object}
 *    Returns a response object which has these methods:
 *
 *    - **show()** – `{Function}` To display the dialog.
 *    - **hide()** – `{Function}` To hide the dialog.
 *    - **destroy()** – `{Function}` To destroy the dialog.
 *
 * @example
      <example module="app">

         <file name="app.js">
            'use strict';
            var appModule = angular.module('app', ['vui.angular.dialog']);

            appModule.controller('DialogServiceCtrl', ['$scope', 'vuiDialogService',
                  function ($scope, vuiDialogService) {

               $scope.dialogOptions = {
                  show: false,
                  title: 'Add Tag',
                  iconClass: 'vui-icon-tag-add',
                  contentUrl: 'dialogContent.html',
                  confirmOptions: {
                     label: 'Add'
                  },
                  rejectOptions: {
                     onClick: function () {
                        console.log('no clicked');
                        return true;
                     }
                  },
                  width: '400px',
                  height: '200px'
               };

               var initDialogData  = function () {
                  $scope.dialogOptions.data = {
                     server: null,
                     tagName: null
                  };
               };

               var dialog = null;
               $scope.statusText = 'Create a dialog';
               var options = {
                  scope: $scope,
                  configObjectName : 'dialogOptions'
               };

               $scope.createDialog = function () {
                  if (dialog === null) {
                     initDialogData();
                     dialog = vuiDialogService(options);
                     $scope.statusText = 'Dialog created';
                  } else {
                     alert('Dialog already exists');
                  }
               };

               $scope.openDialog = function () {
                  if (dialog === null) {
                     alert('Create a dialog first.');
                     return;
                  }
                  dialog.show();
                  $scope.statusText = 'Dialog opened';
               };

               $scope.destroyDialog = function () {
                  if (!dialog) {
                     return;
                  }

                  dialog.destroy();
                  dialog = null;
                  $scope.statusText = 'Dialog destroyed';
               };
            }]);


            appModule.controller('DialogPageCtrl', ['$scope', 'vuiConstants',
                  function($scope, vuiConstants) {
               // Function to be called when OK button is pressed.
               $scope.dialogOptions.confirmOptions.onClick = function () {
                  if ($scope.server && $scope.tagName) {
                     $scope.dialogOptions.data.server = $scope.server;
                     $scope.dialogOptions.data.tagName = $scope.tagName;
                     return true;
                  } else {
                     $scope.dialogOptions.validationBanner.messages  = [{
                        text: 'Fill up the fields to proceed',
                        type: vuiConstants.validationBanner.type.ERROR
                     }];
                     return false;
                  }
               };
            }]);
         </file>

         <file name="index.html">
            <div ng-controller="DialogServiceCtrl">
               <div class="container">
                  <button ng-click="createDialog()">Create Dialog</button>
                  <button ng-click="openDialog()">Open Dialog</button>
                  <button ng-click="destroyDialog()">Destroy Dialog</button>
                  <br><br><br><b style="color:green;">{{statusText}}</b>
               </div>
            </div>
         </file>

         <file name="dialogContent.html">
            <div ng-controller="DialogPageCtrl" class="content-padding">
               <form>
                  <div class="form-group form-group-horizontal">
                     <div class="label-col col-xs-3">
                        <label for="server">Server</label>
                     </div>
                     <div class="input-col col-xs-8">
                        <input type="text" id="server" autofocus ng-model="server"
                              placeholder="pa-server">
                     </div>
                  </div>
                  <div class="form-group form-group-horizontal">
                     <div class="label-col col-xs-3">
                        <label for="tagName">Tag Name</label>
                     </div>
                     <div class="input-col col-xs-8">
                        <input type="text" id="tagName" ng-model="tagName"
                              placeholder="building_26">
                     </div>
                  </div>
               </form>
            </div>
            <style>
               .content-padding {
                  padding-top: 20px;
                  padding-left: 25px;
               }
            </style>
         </file>

      </example>
 */

.provider('vuiDialogService', function () {

   var explicitDefaultConfig = null;
   var explicitDefaultOptions = null;

   this.setDefaultConfig = function (defaultConfig) {
      explicitDefaultConfig = $.extend(true, {}, defaultConfig);
   };

   this.setDefaultDialogOptions = function (defaultOptions) {
      explicitDefaultOptions = $.extend(true, {}, defaultOptions);
   };

   this.$get = ['$window', '$rootScope', '$document', '$compile', '$q', '$timeout', 'vuiModalService', 'jsUtils', 'vuiConstants', 'vuiLocale', function ($window, $rootScope, $document, $compile, $q, $timeout,
         vuiModalService, jsUtils, vuiConstants, vuiLocale) {

      var DEFAULT_CONFIG = {
         showBackdrop: true,
         zIndex: vuiConstants.internal.zIndex.SINGLE_PAGE_DIALOG
      };

      var DEFAULT_OPTIONS = {
         confirmOptions: {
            label : vuiLocale.ok,
            disabled : false,
            confirmOnEnter: true
         },
         rejectOptions: {
            label : vuiLocale.cancel,
            visible  : true,
            rejectOnEsc: true
         }
      };

      function DialogFactory(config) {
         var dialogTemplate;
         var scope = config.scope.$new();
         scope.vuiDialog = config.scope[config.configObjectName];

         var defaultConfig = $.extend(true, { }, DEFAULT_CONFIG, explicitDefaultConfig);
         var defaultOptions = $.extend(true, { }, DEFAULT_OPTIONS, explicitDefaultOptions);

         $.extend(config, $.extend({}, defaultConfig, config));
         $.extend(scope.vuiDialog, $.extend({}, defaultOptions, scope.vuiDialog));
         $.extend(scope.vuiDialog.confirmOptions, $.extend({}, defaultOptions.confirmOptions, scope.vuiDialog.confirmOptions));
         $.extend(scope.vuiDialog.rejectOptions, $.extend({}, defaultOptions.rejectOptions, scope.vuiDialog.rejectOptions));

         dialogTemplate =
            '<div class="vui-popup vui-dialog"' +
                  'ng-style="vuiDialog.getStyle()" role="dialog" tabindex="0" vui-focus-on="vuiModalOpened">' +
               '<div class="titlebar">' +
                  '<span class="titlebar-left-icons">' +
                     '<span ng-if="vuiDialog.iconClass" class="vui-icon" ' +
                           'ng-class="vuiDialog.iconClass">' +
                     '</span>' +
                  '</span>' +
                  '<span class="titlebar-text" title="{{vuiDialog.title}}">' +
                     '{{vuiDialog.title}}</span>' +
              '</div>' +
               (scope.vuiDialog.sandbox ?
               '<iframe id="vuiDialogIFrame" ng-src="{{vuiDialog.contentUrl}}" class="sandbox-iframe-dialog"></iframe>' :
               '<div class="panel-content" ng-if="vuiDialog.show">' +
                  '<div vui-validation-banner="vuiDialog.validationBanner"></div>' +
                  '<div class="dialog-content-view" ng-include="vuiDialog.contentUrl"></div>' +
               '</div>' +
               '<div class="dialog-footer">' +
                  '<button type="button" class="btn" ng-if="vuiDialog.rejectOptions.visible"' +
                        'ng-click="vuiDialog.onCancel()">' +
                        '{{vuiDialog.rejectOptions.label}}</button>' +
                  '<button type="button" class="btn btn-primary" ng-click="vuiDialog.onOk()"' +
                     'ng-disabled="vuiDialog.confirmOptions.disabled">' +
                     '{{vuiDialog.confirmOptions.label}}</button>' +
               '</div>') +
               '<div class="ada-txt" tabindex="0" ng-focus="$broadcast(\'vuiModalOpened\');" style="position:absolute;"></div>' +
            '</div>';

         var clearValidationBanner = function () {
            scope.vuiDialog.validationBanner.messages = [];
         };

         var isCombobox = function (element) {
            return element && element.getAttribute && element.getAttribute('role') === 'combobox';
         };

         var isDropdown = function (element) {
            return element && element.getAttribute && element.getAttribute('role') === 'listbox';
         };

         var escKeyHandler = function (element) {
            // Ignore if disabled
            if (scope.vuiDialog.rejectOptions.rejectOnEsc !== true) {
               return;
            }
            // Ignore on vui.angular.combobox and vui.angular.dropdown
            if (isCombobox(element) || isDropdown(element)) {
               return;
            }

            // Ensure cancel is available
            if (scope.vuiDialog.rejectOptions.visible === true) {
               scope.vuiDialog.onCancel();
            }
         };

         var keypressHandler = function ($event) {
            // Avoid propagation of any keydown event since some other dialog might be
            // listening for it and trigger undesired functionality as closing the other
            // dialog when the current one on is on focus.
            $event.stopImmediatePropagation();

            var code = $event.which || $event.keyCode;
            var focus = $event.currentTarget.activeElement;

            switch (code) {
               case vuiConstants.internal.keys.ESC:
                  escKeyHandler(focus);
                  break;
            }
         };

         // Turn the keypress handler on/off
         var setKeypressHandler = function (state) {
            $document[state]('keydown', keypressHandler);
         };

         var oldFocus;
         /* attaching to angular event getStyle to put focus on dialog properly,
          * does not work on ngInit or onload because it needs to fire every
          * time the dialog is shown or hidden (i.e. when the style is changed)
          * it is also necessary to wait until the spinnner/progress dialog has fully closed first*/
         scope.$watch('vuiDialog.show', function (value) {
            if (value) {
               oldFocus = document.activeElement;

               var waitTillProgressIsGone = function () {
                  $timeout(function () {
                     if (!scope.progress) {
                        scope.progress = $('.modal-spinner');
                     }
                     if (scope.progress.hasClass('ng-hide') || !scope.progress.length) {
                        scope.$broadcast('vuiModalOpened');
                     } else {
                        waitTillProgressIsGone();
                     }
                  }, 100);
               };
               waitTillProgressIsGone();
            }
         });

         var resetFocusOnModalClose = function () {
            if (oldFocus && oldFocus.focus) {
               oldFocus.focus();
            }
         };

         scope.vuiDialog.onOk = function () {
            if (scope.vuiDialog.confirmOptions && scope.vuiDialog.confirmOptions.onClick &&
                  typeof(scope.vuiDialog.confirmOptions.onClick) === 'function') {
               $q.when(scope.vuiDialog.confirmOptions.onClick()).then(function (result) {
                  if (result) {
                     clearValidationBanner();
                     scope.vuiDialog.show = false;
                  }
               });
            } else {
               clearValidationBanner();
               scope.vuiDialog.show = false;
            }
            resetFocusOnModalClose();
         };

         scope.vuiDialog.onCancel = function () {
            if (scope.vuiDialog.rejectOptions && scope.vuiDialog.rejectOptions.onClick &&
                  typeof(scope.vuiDialog.rejectOptions.onClick) === 'function') {
               $q.when(scope.vuiDialog.rejectOptions.onClick()).then(function (result) {
                  if (result) {
                     clearValidationBanner();
                     scope.vuiDialog.show = false;
                  }
               });
            } else {
               clearValidationBanner();
               scope.vuiDialog.show = false;
            }
            resetFocusOnModalClose();
         };

         // style for dialog container
         scope.vuiDialog.getStyle = function () {
            var styles = {};
            if (jsUtils.hasProperty(scope.vuiDialog, 'width')) {
               styles.width = scope.vuiDialog.width;
            }
            if (jsUtils.hasProperty(scope.vuiDialog, 'height')) {
               styles.height = scope.vuiDialog.height;
            }
            return styles;
         };

         // set a watch on show to add/remove keystroke listener
         scope.$watch('vuiDialog.show',
               function (newValue, oldValue) {
                  if (newValue ===  false  && oldValue === true) {
                     setKeypressHandler('off');
                  } else if (newValue ===  true) {
                     setKeypressHandler('on');
                  }
               });

         // Clean up
         scope.$on('$destroy', function () {
            setKeypressHandler('off');
         });

         config._vuiModalTemplate = dialogTemplate;
         config._outerScopeActual = config.scope;
         config.scope = scope;
         config.headerElementSelector = '> .titlebar';
         config.configObjectName = 'vuiDialog';

         var vuiModalApi = vuiModalService(config);

         vuiModalApi.scope.$on('$destroy', function () {
            $rootScope.$evalAsync(function () {
               scope.$destroy();
            });
         });

         return vuiModalApi;
      }

      return DialogFactory;
   }];
});
;'use strict';

vui.angular.dialog

/**
 * @ngdoc object
 * @name vui.angular.dialog:DialogOptions
 * @module vui.angular.dialog
 *
 * @description
 *    **Extends** {@link vui.angular.modal:ModalOptions ModalOptions}.
 *
 *    Configuration object for {@link vui.angular.dialog.directive:vuiDialog vuiDialog} directive.
 *
 *    Example:
 *    ```js
 *    var dialogOptions = {
 *                            title: 'Add new tag',
 *                            iconClass: 'vui-icon-tag-add',
 *                            show: true,
 *                            contentUrl: 'addTagDialog.html',
 *                            width : '400px',
 *                            height: '200px',
 *                            confirmOptions: { ... },
 *                            rejectOptions: { ... }
 *                        };
 *    ```
 *
 * @property {string} title
 *    Type: `string`
 *
 *    Specifies the title of the dialog.
 * @property {string} [iconClass]
 *    Type: `string` **|** Optional
 *
 *    The CSS class name representing the icon displayed in the dialog title bar.
 *
 *    **Note:** Please refer to the vui-bootstrap documentation for a list of
 *    available icon classes. A custom class specifying a 16x16 icon can also
 *    be used.
 * @property {string} contentUrl
 *    Type: `string`
 *
 *    The template URL of web content that will be loaded into and
 *    displayed within the dialog.
 *
 *    **Note:** The dialog's content scope prototypically inherits from the
 *    scope on which {@link vui.angular.dialog.directive:vuiDialog vui-dialog}
 *    directive is declared.
 * @property {string} [width]
 *    Type: `string` **|** Optional **|** Default: `'auto'`
 *
 *    The custom width for the dialog. Min-width is 220px and
 *    max-width is 960px.
 * @property {string} [height]
 *    Type: `string` **|** Optional **|** Default: `'auto'`
 *
 *    The custom height for the dialog. Min-height is 200px and
 *    max-height is 80%.
 *
 *    **Note:** In Internet Explorer the default height of dialog is set to
 *    `'400px'`. This is a workaround to handle an existing browser bug where
 *    the flex container does not honor the containers `min-height`.
 * @property {Object} [confirmOptions]
 *    Type: `Object` **|** Optional
 *
 *    {@link vui.angular.dialog:ConfirmOptions ConfirmOptions} contains
 *    properties to define the confirm (OK) button.
 * @property {Object} [rejectOptions]
 *    Type: `Object` **|** Optional
 *
 *    {@link vui.angular.dialog:RejectOptions RejectOptions} contains
 *    properties to define the reject (Cancel) button.
 * @property {Object} [validationBanner]
 *    Type: `Object` **|** Optional
 *
 *    {@link vui.angular.validationBanner:ValidationBannerOptions ValidationBannerOptions}
 *    contains properties to define the validation banner. In the dialog,
 *    validation banner is used to show an error or warning message.
 */

/**
 * @ngdoc object
 * @name vui.angular.dialog:ConfirmOptions
 * @module vui.angular.dialog
 *
 * @description
 *    Configuration object for confirm (OK) button in the dialog.
 *
 *    Example:
 *    ```js
 *    confirmOptions: {
 *                        label: 'Confirm',
 *                        disabled: false,
 *                        onClick: clickFunction
 *                    }
 *    ```
 *
 * @property {string} [label]
 *    Type: `string` **|** Optional **|** Default: `'OK'`
 *
 *    The custom text for label.
 * @property {boolean} [disabled]
 *    Type: `boolean` **|** Optional **|** Default: `false`
 *
 *    Disables the confirm button when set to true.
 * @property {Function} [onClick]
 *    Type: `Function` **|** Optional
 *
 *    A function that will be called when confirm (OK) button is clicked. It
 *    should return true/false or a promise (resolving to true/false).
 *    If the return value is true the dialog is closed, else its not.
 *
 *    Example:
 *    ```js
 *    onClick: function () {
 *                 ...
 *             };
 *    ```
 * @property {boolean} [confirmOnEnter]
 *    Type: `boolean` **|** Optional **|** Default: `true`
 *
 *    Sets the Enter key as a keyboard shortcut for clicking on the confirm (OK) button.
 *    The disabled property must be set to false (default) for the shortcut to be in effect.
 */

/**
 * @ngdoc object
 * @name vui.angular.dialog:RejectOptions
 * @module vui.angular.dialog
 *
 * @description
 *    Configuration object for reject (Cancel) button in the dialog.
 *
 *    Example:
 *    ```js
 *    rejectOptions: {
 *                       label: 'Cancel',
 *                       visible: true,
 *                       onClick: clickFunction,
 *                       rejectOnEsc: true
 *                   }
 *    ```
 *
 * @property {string} [label]
 *    Type: `string` **|** Optional **|** Default: `'Cancel'`
 *
 *    The custom text for label.
 * @property {boolean} [visible]
 *    Type: `boolean` **|** Optional **|** Default: `true`
 *
 *    Shows the reject button when set to true.
 * @property {Function} [onClick]
 *    Type: `Function` **|** Optional
 *
 *    A function that will be called when reject (Cancel) button is
 *    clicked. It should return true/false or a promise
 *    (resolving to true/false).
 *    If the return value is true the dialog is closed, else its not.
 *
 *    Example:
 *    ```js
 *    onClick: function () {
 *                 ...
 *             };
 *    ```
 * @property {boolean} [rejectOnEsc]
 *    Type: `boolean` **|** Optional **|** Default `true`
 *
 *    Sets the ESC key as a keyboard shortcut for clicking on the cancel button.
 *    The visible property must also be true (default) for the shortcut to be in effect.
 */

/**
 * @ngdoc directive
 * @name vui.angular.dialog.directive:vuiDialog
 * @module vui.angular.dialog
 * @restrict A
 * @scope
 *
 * @description
 *    Directive to create a single page dialog.
 *
 *    The dialog's content scope prototypically inherits from the scope on
 *    which {@link vui.angular.dialog.directive:vuiDialog vui-dialog} directive
 *    is declared.
 *
 * @param {DialogOptions} vuiDialog
 *    Configuration object for a single page dialog.
 *    It is of type {@link vui.angular.dialog:DialogOptions DialogOptions}.
 *
 * @example
      <example module="app">

         <file name="app.js">
            'use strict';
            var app = angular.module('app', ['vui.angular.dialog']);

            app.controller('DialogCtrl', ['$scope', function($scope) {
               $scope.dialogOptions = {
                  show: false,
                  title : 'Add Tag',
                  iconClass : 'vui-icon-tag-add',
                  contentUrl : 'dialogContent.html',
                  confirmOptions : {
                     label: 'Add'
                  },
                  rejectOptions: {
                     onClick: function () {
                        console.log('No Clicked');
                        return true;
                     }
                  },
                  width : '400px',
                  height: '200px'
               };

               $scope.dialogOptions.data = {
                  server : null,
                  tagName : null
               };

               $scope.openDialog = function () {
                  $scope.dialogOptions.show = true;
               };
            }]);

            app.controller('DialogPageCtrl', ['$scope', 'vuiConstants',
                  function($scope, vuiConstants) {
               // Function to be called when OK button is pressed.
               $scope.dialogOptions.confirmOptions.onClick = function () {
                  if ($scope.server && $scope.tagName) {
                     $scope.dialogOptions.data.server = $scope.server;
                     $scope.dialogOptions.data.tagName = $scope.tagName;
                     return true;
                  } else {
                     $scope.dialogOptions.validationBanner.messages  = [{
                        text: 'Fill up the fields to proceed',
                        type: vuiConstants.validationBanner.type.ERROR
                     }];
                     return false;
                  }
               };
            }]);
         </file>

         <file name="index.html">
            <div ng-controller="DialogCtrl">
               <button ng-click="openDialog()">Open Dialog</button>
               <div vui-dialog="dialogOptions"></div>
            </div>
         </file>

         <file name="dialogContent.html">
            <div ng-controller="DialogPageCtrl" class="content-padding">
               <form>
                  <div class="form-group form-group-horizontal">
                     <div class="label-col col-xs-3">
                        <label for="server">Server</label>
                     </div>
                     <div class="input-col col-xs-8">
                        <input type="text" id="server" autofocus ng-model="server"
                              placeholder="pa-server">
                     </div>
                  </div>
                  <div class="form-group form-group-horizontal">
                     <div class="label-col col-xs-3">
                        <label for="tagName">Tag Name</label>
                     </div>
                     <div class="input-col col-xs-8">
                        <input type="text" id="tagName" ng-model="tagName"
                              placeholder="building_26">
                     </div>
                  </div>
               </form>
            </div>
            <style>
               .content-padding {
                  padding-top: 20px;
                  padding-left: 25px;
               }
            </style>
         </file>

    </example>
 */

.directive('vuiDialog', [ 'vuiDialogService', function (vuiDialogService) {
   return {
      restrict: 'A',
      scope: false,
      link: function (scope, element, attr) {
         // Directive options
         var options = {
            scope: scope,
            configObjectName : attr.vuiDialog,
            destroyOnHide: false,
            destroyOuterScope: false
         };

         // Initialize modal
         var dialog = vuiDialogService(options);

         // set a watch on show to show and hide
         scope.$watch(attr.vuiDialog + '.show',
               function (newValue, oldValue) {
            if (newValue ===  false  && oldValue === true) {
               dialog.hide();
            } else if (newValue ===  true  && oldValue === false) {
               dialog.show();
            }
         });

         // Garbage collection
         scope.$on('$destroy', function () {
            dialog.destroy();
            options = null;
            dialog = null;
         });
      }
   };
}])

/**event driven angular enables us to put focus on things
 * after timeout such as on a popup or other object that are
 * not immediately visible
 * example usage: html attribute focus-on="vuiModalOpened"
 * then in js to trigger: $scope.$broadcast('vuiModalOpened');*/

.directive('vuiFocusOn', function () {
   return function (scope, elem, attr) {
      scope.$on(attr.vuiFocusOn, function () {
         var autofocusElements = elem.find('[autofocus]');
         //if the dialog contains elements with autofocus attribute set
         //then focus on the first such element.
         if (autofocusElements.size()) {
            autofocusElements.first().focus();
         } else {
            elem.focus();
         }
      });
   };
});

;'use strict';
vui.angular.directives.directive('vuiClickOnce', function () {
   var KEY_ENTER_CODE = 13;
   return {
      restrict: 'A',
      priority: 100,
      state: { released: true },
      link: function ($scope, element) {
         var self = this;
         element.bind('keydown', function (e) {
            if (e.keyCode === KEY_ENTER_CODE) {
               if (!self.state.released) {
                  e.preventDefault();
               } else {
                  self.state.released = false;
               }
            }
         });

         element.bind('keyup', function (e) {
            if (e.keyCode === KEY_ENTER_CODE) {
               self.state.released = true;
            }
         });
      }
   };
});
;'use strict';

vui.angular.directives

/**
 * @xngdoc directive
 * @name vui.angular.directives.directive:vuiJsonText
 * @module vui.angular.directives
 * @restrict A
 *
 * @description
 *    Use this directive in conjunction with `ngModel` on a text input control in order
 *    to render the text input as JSON and bind the model to the javascript object
 *    obtained by deserializing the JSON text.
 *
 * @example
      <example module="app">

         <file name="app.js">
           'use strict';
           var app = angular.module('app', ['vui.angular.directives']);
           app.controller('TextCtrl', ['$scope', function($scope) {
              $scope.input = {};
              $scope.input.object = {foo: "hello"};
              $scope.getModelType = function() {
                 return typeof $scope.input.object;
              }
           }]);
         </file>

         <file name="index.html">
          <div ng-controller="TextCtrl">
             <textarea rows='10' cols="100" vui-json-text ng-model='input.object'/>
             <h4 class="text-info">Type of input.object: {{getModelType()}}</h4>
          </div>
         </file>

      </example>
 */

.directive('vuiJsonText', function () {
   return {
      restrict: 'A',
      require: 'ngModel',
      link: function (scope, element, attrs, ngModelCtrl) {

         var lastValid;
         var skipModelUpdate = 0;
         var skipJsonUpdate = 0;

         // Deserializes the given JSON string.
         function fromUser(text) {
            if (skipModelUpdate > 0) {
               skipModelUpdate--;
               return lastValid;
            }
            if (!text || text.trim() === '') {
               return {};
            } else {
               try {
                  lastValid = angular.fromJson(text);
                  // The model change triggered when this function returns will be
                  // captured by the watch below, which will in turn trigger an update to the
                  // json text. This is unnecessary however, since this update originated
                  // from that json text changing in the first place. I.e., there is a
                  // cycle here and this control variable lets us break out of it.
                  skipJsonUpdate++;
                  ngModelCtrl.$setValidity('invalidJson', true);
               } catch (e) {
                  ngModelCtrl.$setValidity('invalidJson', false);
               }
               return lastValid;
            }
         }

         // Serializes the given object to JSON string.
         function toUser(object) {
            return angular.toJson(object, true);
         }

         // parse all text entered and deserialize into the model.
         ngModelCtrl.$parsers.push(fromUser);
         // format all text entered as pretty JSON.
         ngModelCtrl.$formatters.push(toUser);

         // Clear any invalid changes on blur (focus out)
         element.bind('blur', function () {
            element.val(toUser(scope.$eval(attrs.ngModel)));
         });

         // Reflect any change to the model in the view.
         scope.$watch(attrs.ngModel, function (newValue, oldValue) {
            if (skipJsonUpdate > 0) {
               skipJsonUpdate--;
               return;
            }
            lastValid = lastValid || newValue;

            if (newValue !== oldValue) {
               // The json change triggered when the next lines execute will result in
               // the 'fromUser' parser executing, which will in turn trigger an update
               // to the model. This is unnecessary however, since this update originated
               // from the model changing in the first place. I.e., there is a
               // cycle here and this control variable lets us break out of it.
               skipModelUpdate++;
               ngModelCtrl.$setViewValue(toUser(newValue));
               ngModelCtrl.$render();
            }
         }, true);
      }
   };
});
;'use strict';

vui.angular.dropdown

/**
 * @ngdoc object
 * @name vui.angular.dropdown:DropdownOptions
 * @module vui.angular.dropdown
 *
 * @description
 *    Configuration object for
 *    {@link vui.angular.dropdown.directive:vuiDropdown vuiDropdown}
 *    directive.
 *    Contains properties to specify a dropdown and populate it with
 *    data.
 *
 *    Example:
 *    ```js
 *    var dropdownOptions = {
 *                             data: [ ... ],
 *                             placeHolder: 'Choose a thing.'
 *                          };
 *    ```
 *
 * @property {[*]} data
 *    Type: `Array.<string | number | DropdownItem>`
 *
 *    Specifies the items to populate the dropdown. Each item may either be a number,
 *    string or object of type  {@link vui.angular.dropdown:DropdownItem DropdownItem}.
 *    Each item specifies a selectable item in the dropdownlist.
 *
 *    When an item is a number or string, the number or string given becomes both that item's value and label.
 *    Use a DropdownItem object to allow an item's label and value to be different
 *
 *    In order to update the items in the dropdown, assign a new value to this property.
 *
 *    Example:
 *    ```js
 *    data: [ {value: 1, label:'VM1'}, {value: 2, label: 'VM2'} ];
 *    (or)
 *    data: ["vm1", "vm2"];
 *    (or)
 *    data: [1, 2, 3]
 *    ```
 *
 * @property {*} [selectedValue]
 *    Type: `*` **|** Optional
 *
 *    Specifies and controls the current value of the dropdown.
 *
 *    Setting this to a value that exists in the {@link vui.angular.dropdown:DropdownOptions data}
 *    items causes that item's label to be displayed as the default.
 *    The test for existence is done using strict equality (===).
 *
 *    Note: If this property is missing it will be written by the directive.
 *
 * @property {boolean} [enabled]
 *    Type: `boolean` **|** Optional **|** Default: true
 *
 *    Controls whether the widget is enabled or disabled.
 *
 *    Note: If this property is missing it will be written by the directive.
 *
 * @property {string} [placeHolder]
 *    Type: `string` **|** Optional
 *
 *    The initial string to display in the dropdown. Leave this as undefined or
 *    null if using 'selectedValue' to control the initial text in the box.
 *
 *    When this property is not undefined or null, the given text is shown in the box
 *    when first rendered and  the 'selectedValue' is set to undefined.
 *
 * @property {string} [width]
 *    Type `string` **|** Optional
 *
 *    The width, in any valid CSS unit, of the dropdown.
 *    If not wide enough for the content, the drop down will scroll horizontally.
 *    Don't forget to specify the unit.
 */

/**
 * @ngdoc object
 * @name vui.angular.dropdown:DropdownItem
 * @module vui.angular.dropdown
 *
 * @description
 *    Configuration object for a selectable item in a
 *    {@link vui.angular.dropdown.directive:vuiDropdown vuiDropdown}
 *    directive.
 *
 *    Example:
 *    ```js
 *    var dropdownItem1 = {
 *                         label: 'Furniture',
 *                         value: 1
 *                        };
 *    ```
 *
 * @property {string} label
 *    Type: `string`
 *
 *    Localized text to display for the item.
 *
 *    <script>$('.properties .label').removeClass('label');</script>
 *
 * @property {*} value
 *    Type: `*`
 *
 *    The value of the item.
 */

/**
 * @ngdoc directive
 * @name vui.angular.dropdown.directive:vuiDropdown
 * @module vui.angular.dropdown
 * @restrict A
 * @element div
 * @scope
 *
 * @description
 *    Directive to create a dropdown-list control.
 *    A dropdown is a graphical control element, similar to a list box,
 *    that allows the user to choose one value from a list.  When a dropdown list is inactive,
 *    it displays a single value. When activated, it displays (drops down) a list of values,
 *    from which the user may select one. When the user selects a new value, the control reverts
 *    to its inactive state, displaying the selected value.
 *
 * @param {dropdownOptions} vuiDropdown
 *    Configuration object for the dropdown. It is of type
 *    {@link vui.angular.dropdown:DropdownOptions DropdownOptions}.
 *
 * @example
      <example module="app">

         <file name="app.js">
            'use strict';
            var app = angular.module('app', ['vui.angular']);
            app.controller('DropdownController', function ($scope, $interval) {
               $scope.dropdownOptions = {
                  selectedValue: 3,
                  enabled: true,
                  data: [
                     {label: 'Albania', value: 1},
                     {label: 'Andorra', value: 2},
                     {label: 'Armenia', value: 3},
                     {label: 'Austria', value: 4},
                     {label: 'Azerbaijan', value: 5}
                  ],
                  width: '300px'
               };
            });
         </file>

         <file name="index.html">
            <link rel="stylesheet" href="css/vui-bootstrap.min.css"
                  type="text/css">
            <div ng-controller="DropdownController">
               <div vui-dropdown="dropdownOptions"></div><br>
               <div>value : {{dropdownOptions.selectedValue}}</div>
            </div>
         </file>

      </example>
 */
.directive('vuiDropdown', ['jsUtils', function (jsUtils) {
   return {
      restrict: 'A',
      scope: {
         vuiDropdown: '='
      },
      replace: false,
      template:
         '<select kendo-drop-down-list ' +
               'ng-model="vuiDropdown.selectedValue"' +
               'ng-disabled="!vuiDropdown.enabled"' +
               'k-option-label="vuiDropdown.placeHolder"' +
               'k-data-text-field="\'label\'"' +
               'k-data-value-field="\'value\'"' +
               'k-data-source="vuiDropdown.data" ng-style="{width: vuiDropdown.width}">' +
         '</select>',
      controller: ['$scope', function ($scope) {
         // Set default value for enabled if not specified.
         $scope.vuiDropdown.enabled = jsUtils.isBoolean($scope.vuiDropdown.enabled) ?
               $scope.vuiDropdown.enabled : true;
         $scope.vuiDropdown.placeHolder = $scope.vuiDropdown.placeHolder || '';
      }],
      link: function (scope, element) {
         // TODO - Kendo bug ... ng-style on template's kendo-drop-down-list is not being respected
         var widget = element.find('.k-widget');
         widget.css({width: scope.vuiDropdown.width});
      }
   };
}]);
;vui.angular.modal.directive('vuiFocusTrap', function () {
   'use strict';
   /**
    * Adding this to the body element enables
    * focus styling
    * @type {string}
    */
   var USER_TABBING_CLASS_NAME = 'user-tabbing';
   return {
      restrict: 'A',
      link: function (scope, $el) {

         function onKeyDown(e) {
            if (e.keyCode === 9) {
               handle(e);
            }
         }

         function handle(e) {
            var focusable = $el.find(':focusable');
            var $first = focusable.first();
            var $last = focusable.last();
            if ($last.get(0) === e.target && e.shiftKey !== true) {
               $first.focus();
               e.stopImmediatePropagation();
               e.preventDefault();
            } else if ($first.get(0) === e.target && e.shiftKey === true) {
               $last.focus();
               e.stopImmediatePropagation();
               e.preventDefault();
            }

            $(document.body).addClass(USER_TABBING_CLASS_NAME);
         }

         $el.on('keydown', onKeyDown);
         scope.$on('$destroy', function () {
            $el.off('keydown', onKeyDown);
         });
      }
   };
});
;'use strict';

vui.angular.notification

/**
 * @ngdoc service
 * @name vui.angular.notification.vuiNotificationService
 * @module vui.angular.notification
 *
 * @description
 *    Used to create and destroy notifications.
 *
 *    A notification is a small window that appears in the lower right corner of the
 *    application and fades out after some duration. It is used to display noncritical
 *    system status information such as events and status.
 *
 *    {@link vui.angular.constants.vuiConstants vuiConstants.notifications.type}
 *    is used to specify type of notification
 *
 *
 * @example
      <example module="app">

         <file name="app.js">
            var app = angular.module('app', ['vui.angular']);
            app.controller('NotificationController', ['$scope', 'vuiNotificationService',
               'vuiConstants',
               function ($scope, vuiNotificationService, vuiConstants) {
                  'use strict';

                  $scope.notificationOptions = {
                     'type': vuiConstants.notifications.type.INFO,
                     'title': 'Notification',
                     'content': 'Sample Message',
                     'linkConfig': {
                        'label': 'Click here for more information!',
                        'onClick': function () {
                           console.log('Simple link function!');
                        }
                     }
                  };

                  $scope.createNotification = function () {
                     vuiNotificationService.notify($scope.notificationOptions);
                  };
               }
            ]);
         </file>

         <file name="index.html">
            <h1>Notifications Service</h1>
            <hr/>

            <div class="row">
               <div class="col-md-6">
                  <h3>Example - Notification Service</h3>
               </div>
            </div>

            <div class="row" style="margin-top:20px" ng-controller="NotificationController">
               <div class="col-md-6">
                  <button class="btn btn-default" ng-click="createNotification()">
                     Toggle Notification
                  </button>
                  <p>
                  <h5 class="text-info">Options:</h5>
                     <textarea rows='10' vui-json-text ng-model='notificationOptions'></textarea>
                  </p>
               </div>
            </div>
         </file>
      </example>
 */

/**
  * @ngdoc object
  * @name vui.angular.notification.NotificationOptions
  * @module vui.angular.notification
  *
  * @description
  *    Configuration object for
  *    {@link vui.angular.notification.vuiNotificationService vuiNotificationService's}
  *    method notify.
  *
  * Example:
  *    ```js
  *    var config = {
  *                   'type': vuiConstants.notifications.type.ERROR
  *                   'title': 'Notification'
  *                   'content': 'An error has occured!'
  *                   'duration': 10000
  *                   'linkConfig': {
  *                       label: 'Click here for more information!',
  *                       onClick: function () {console.log('Simple link function!'); }
  *                    }
  *                  };
  *    ```
  *
  * @property {string} [type]
  *    Type: `string` **|** Optional **|** Default: `info`
  *
  *    Specifies the type of the notification.
  *
  *    Legal values:
  *
  *    - **`info`** : Info type notification
  *    - **`success`** : Success type notification
  *    - **`warning`** : Warning type notification
  *    - **`error`** : Error type notification
  *
  * @property {string} [title]
  *    Type: `string` **|** Optional **|** Default: `''`}
  *
  *    Title of the notification.
  *
  * @property {String} content
  *    Type: `String`
  *
  *    Content of the notification.
  *
  * @property {Action} [linkConfig]
  *    Type: `vui.angular.actions.Action` **|** Optional **|** Default: `{}`
  *
  *    An {@link vui.angular.actions:Action action} link
  *    that resides in the footer of the notification.
  *    Only the properties label and onClick are used.
  *
  */
.factory('vuiNotificationService', ['$timeout', 'vuiConstants', '$compile',
      '$rootScope', '$log', 'jsUtils', '$sce', 'vuiLocale',
      function ($timeout, vuiConstants, $compile, $rootScope, $log, jsUtils,
      $sce, vuiLocale) {

   var DEFAULT_DURATION = 10000;
   var scope = $rootScope.$new();
   var notificationContainerElement;
   scope.notifications = [];
   var container = document.body;
   var template = '<div class="vui-popup vui-notification bottom-right"' +
                     ' ng-repeat="notification in notifications"' +
                     ' ng-style="transform($index)">' +
                     '<header class="titlebar">' +
                        '<span ng-class="getNotificationTypeClass(notification.type)">' +
                        '</span>' +
                        '<span title="{{notification.title}}">{{notification.title}}</span>' +
                        '<span class="titlebar-right-icons">' +
                           '<span class="vui-icon18-dialog-close" ' +
                              'ng-click="removeNotification()"> </span>' +
                        '</span>' +
                     '</header>' +
                     '<main class="panel-content">' +
                        '<p ng-bind-html="notification.content"></p>' +
                     '</main>' +
                     '<footer class="panel-link-footer">' +
                        '<a ng-click="callLinkFunction(notification, $event)">' +
                           '{{notification.linkLabel}}' +
                        '</a>' +
                     '</footer>' +
                  '</div>';

   // Calls removeNotificationAndUpdate to remove a notification
   // from array and update timeout
   scope.removeNotification = function () {
      removeNotificationAndUpdate();
   };

   // Fires when user clicks on a linkMessage and calls
   // the notification link function
   scope.callLinkFunction = function (notification, event) {
      if (angular.isDefined(notification.onClick) &&
         angular.isFunction(notification.onClick)) {
         notification.onClick.call(undefined,
            notification, event);
      } else {
         $log.warn('No click function defined for notification ' +
            'link id: ' + notification.title);
      }

   };

   // Maps the appropriate css class icon to the type specified
   scope.getNotificationTypeClass = function (notificationType) {
      var iconClass;
      switch (notificationType) {
         case vuiConstants.notifications.type.INFO :
            iconClass = 'vui-icon-info';
            break;
         case vuiConstants.notifications.type.SUCCESS :
            iconClass = 'vui-icon-success';
            break;
         case vuiConstants.notifications.type.WARNING :
            iconClass = 'vui-icon-warning';
            break;
         case vuiConstants.notifications.type.ERROR :
            iconClass = 'vui-icon-critical';
            break;
         default :
            iconClass = 'vui-icon-info';
      }
      return iconClass;
   };

   // This function does a 2D translate on the notification elements
   // This creates the look of a stack of cards where every
   // new notification is slightly offset up by 5 pixels from the older one.
   scope.transform = function (index) {
      var translateValue = -5 * index;
      var translateString = 'translate(0px, ' + translateValue + 'px)';

      return {
         'transform': translateString,
         '-webkit-transform': translateString,
         '-ms-transform': translateString
      };
   };

   /** Private API */
   // Adds a notification to the notification array.
   // Adds a timeout to the first element in the array
   function makeNotification(notification) {

      if (!notificationContainerElement) {
         notificationContainerElement = $compile(template)(scope);
         angular.element(container).append(notificationContainerElement);
      }

      scope.notifications.push(notification);

      if (scope.notifications.length === 1 && notification.duration >= 0) {
         $timeout(removeNotificationAndUpdate, notification.duration);
      }
   }

   // Removes a notification from the notification array
   // Adds a timeout to the next element in the array.
   function removeNotificationAndUpdate() {
      // Pop head from queue
      scope.notifications.splice(scope.notifications.length - 1, 1);

      // Update timeout for the new head of queue for non-empty array
      if (scope.notifications.length !== 0 && scope.notifications[0].duration >= 0) {
         $timeout(removeNotificationAndUpdate, scope.notifications[0].duration);
      }
   }

   /** Public API */
   return {
      /**
        * @ngdoc method
        * @name notify
        * @methodOf vui.angular.notification.vuiNotificationService
        * @param {NotificationOptions} config
        *      {@link vui.angular.notification.NotificationOptions Notification}
        *      configuration object.
        */
      notify: function (config) {

         // Required params
         if (jsUtils.isUndefinedOrNull(config)) {
            $log.warn('Notification parameter (config) undefined.');
            return;
         }

         if (!jsUtils.hasProperty(config, 'content')) {
            $log.warn('Notification parameter (config) missing property content.');
            return;
         }

         var notification = {
            type: jsUtils.getProperty(config, 'type',
                     vuiConstants.notifications.type.INFO),
            title: jsUtils.getProperty(config, 'title',
                     vuiLocale.notification.defaultTitle),
            content: $sce.trustAsHtml(jsUtils.getProperty(config, 'content')),
            linkLabel: jsUtils.getProperty(config, 'linkConfig.label', ''),
            onClick: jsUtils.getProperty(config, 'linkConfig.onClick', null),
            duration: (config.duration ? config.duration : DEFAULT_DURATION)
         };
         makeNotification(notification);
      }
   };
}]);
;'use strict';
vui.angular.portlets

/**
 * @ngdoc object
 * @name vui.angular.portlets:PortletsOptions
 * @module vui.angular.portlets
 *
 * @description
 *    Configuration object for
 *    {@link vui.angular.portlets.directive:vuiPortlets vuiPortlets} directive.
 *
 *    Example:
 *    ```js
 *    var myPortlets = {
 *                         layout: vuiConstants.portlets.layout.TWO_COLUMNS,
 *                         portlets: [ ... ]
 *                     };
 *    ```
 *
 * @property {string} [layout]
 *    Type: `string` **|** Optional **|**
 *    Default:`vuiConstants.portlets.layout.TWO_COLUMNS`
 *
 *    Layout of the portlet container that will house portlets. If a layout is
 *    not specified, the portlet container will use a simple two column layout.
 *
 *    Legal values:
 *
 *    - **`vuiConstants.portlets.layout.TWO_COLUMNS`** : Use for two column layout.
 *    - **`vuiConstants.portlets.layout.THREE_COLUMNS`** : Use for three column layout.
 *    - **`vuiConstants.portlets.layout.FOUR_COLUMNS`** : Use for four column layout.
 *
 * @property {Array.<Object>} portlets
 *    Type: `Array.<Object>`
 *
 *    An array of {@link vui.angular.portlets:Portlet Portlet} objects.
 *
 *    Example:
 *    ```js
 *    portlets: [ portlet1, portlet2 ];
 *    ```
 *    `portlet1` and `portlet2` are
 *    {@link vui.angular.portlets:Portlet Portlet} objects.
 */

/**
 * @ngdoc object
 * @name vui.angular.portlets:Portlet
 * @module vui.angular.portlets
 *
 * @description
 *    Contains properties to define a portlet in
 *    {@link vui.angular.portlets.directive:vuiPortlets vuiPortlets} directive.
 *
 *    Example:
 *    ```js
 *    var tagPortlet = {
 *                         title: 'Tags',
 *                         contentUrl: 'tagportlet.html',
 *                         collapsed: false,
 *                         footerLinks: [ ... ]
 *                     };
 *    ```
 *
 * @property {string} title
 *    Type: `string`
 *
 *    Specifies the title of the portlet.
 * @property {string} contentUrl
 *    Type: `string`
 *
 *    The template URL of web content that will be loaded into and displayed
 *    within the current portlet.
 *
 *    **Note:** The portlet's content scope prototypically inherits from the
 *    scope on which {@link vui.angular.portlets.directive:vuiPortlets vui-portlets}
 *    directive is declared.
 * @property {boolean} [collapsed]
 *    Type: `boolean` **|** Optional **|** Default: `false`
 *
 *    Collapses the portlet if set to true.
 * @property {Array.<Object>} [footerLinks]
 *    Type: `Array.<Object>` **|** Optional
 *
 *    An array of {@link vui.angular.portlets:Link Link} objects.
 *
 *    Example:
 *    ```js
 *    footerLinks: [ link1, link2 ]
 *    ```
 *    `link1` and `link2` are {@link vui.angular.portlets:Link Link} objects.
 *
 */

/**
 * @ngdoc object
 * @name vui.angular.portlets:Link
 * @module vui.angular.portlets
 *
 * @description
 *    Configuration object for {@link vui.angular.portlets:Portlet footerLinks}
 *    in the {@link vui.angular.portlets:Portlet Portlet}.
 *
 *    Example:
 *    ```js
 *    var link1 = {
 *                   label: 'Refresh',
 *                   tooltipText: 'Refresh data',
 *                   onClick: clickFunction,
 *                   enabled: true
 *                };
 *    ```
 *
 * @property {string} label
 *    Type: `string`
 *
 *    The display text for the link.
 *    <script>$('.properties .label').removeClass('label');</script>
 * @property {string} [tooltipText]
 *    Type: `string` **|** Optional
 *
 *    The text that will be displayed in the link tooltip on hover. If content
 *    is not specified, {@link vui.angular.portlets:Link label} text will be
 *    used as tooltipText.
 * @property {Function} onClick
  *    Type: `Function`
 *
 *    A function that will be called when the link is clicked. The click event
 *    and the {@link vui.angular.portlets:Link Link} object will be passed as
 *    parameters to the function.
 *
 *    Example:
 *    ```js
 *    onClick: function (event, link) {
 *                ...
 *             };
 *    ```
 *    @property {boolean} enabled
 *    Type: `boolean` **|** Optional **|** Default: `true`
 *
 *    Disables the link when set to false.
 */

/**
 * @ngdoc directive
 * @name vui.angular.portlets.directive:vuiPortlets
 * @module vui.angular.portlets
 * @restrict A
 * @scope
 *
 * @description
 *    Directive to create portlets.
 *
 *    The portlet's content scope prototypically inherits from the
 *    scope on which {@link vui.angular.portlets.directive:vuiPortlets vui-portlets}
 *    directive is declared.
 *
 * @param {PortletsOptions} vuiPortlets
 *    Configuration object for portlets. It is of type
 *    {@link vui.angular.portlets:PortletsOptions PortletsOptions}.
 *
 * @example
      <example module="app">
         <file name="app.js">
            'use strict';
            var app = angular.module('app', ['vui.angular.portlets']);
            app.controller('PortletsCtrl', ['$scope', 'vuiConstants',
                  function ($scope, vuiConstants) {

               var clickFunction = function (event, link) {
                  $scope.linkClicked = link.label;
               };

               var portletSet = [{
                  title: 'Notes',
                  contentUrl: 'portletContent.html',
                  collapsed: false,
                  footerLinks: [
                     {
                        label: 'Edit...',
                        tooltipText: 'Edit content',
                        onClick: clickFunction
                     }, {
                        label: 'Clear',
                        tooltipText: 'Clear',
                        onClick: clickFunction
                     }, {
                        label: 'Delete',
                        tooltipText: 'Delete',
                        onClick: clickFunction,
                        enabled: false
                     }
                  ]
               }];

               // Options object for the vui-portlets
               $scope.portletsOptions = {
                  layout: vuiConstants.portlets.layout.TWO_COLUMNS,
                  portlets: portletSet
               };
            }]);
         </file>

         <file name="index.html">
            <link rel="stylesheet" href="css/vui-bootstrap.min.css"
                  type="text/css">
            <div ng-controller="PortletsCtrl">
               <div>
                  <!-- vui-portlets directive -->
                  <div vui-portlets='portletsOptions'>
                  </div>
               </div>
               <br/>
               <div>
                  <b> Last link clicked: </b> {{linkClicked}}
               </div>
            </div>
         </file>

         <file name="portletContent.html">
            <div contenteditable="true">Click to edit content inside this div.
            </div>
         </file>
      </example>
 */

.directive('vuiPortlets', ['$log', 'jsUtils', 'vuiConstants',
         function ($log, jsUtils, vuiConstants) {
   return {
      scope: true,
      restrict: 'A',
      replace: true,
      template: '<div class="vui-portlets-container"' +
                        ' ng-class="getLayout(portletsOptions.layout)">' +
                  '<div class="vui-portlet"' +
                        'ng-repeat="portlet in portletsOptions.portlets track by $index"' +
                        'ng-init="isCollapsed(portlet)"' +
                        'ng-class="{collapsed:portlet.collapsed}" >' +
                     '<div class="portlet-titlebar">' +
                        '<span ng-click="portlet.collapsed = ' +
                              '!portlet.collapsed" ng-class=' +
                              '"{true: \'right-arrow\',' +
                              ' false: \'down-arrow\'}' +
                              '[portlet.collapsed]">' +
                        '</span>' +
                        '<span class="titlebar-text"' +
                              'title="{{portlet.title}}">{{portlet.title}}' +
                        '</span>' +
                        '<!-- span class="vui-icon-dialog-maximize">' +
                        '</span -->' +
                     '</div>' +
                     '<!-- Portlet content must use iframe for sandboxed plugins -->' +
                     '<iframe ng-if="portlet.sandbox" ng-src="{{portlet.contentUrl}}"' +
                        ' class="sandbox-iframe portlet-content"></iframe>' +
                     '<div ng-if="!portlet.sandbox" class="portlet-content"' +
                           'ng-include="portlet.contentUrl">' +
                     '</div>' +
                     '<div class="portlet-footer">' +
                        '<a ng-repeat="link in portlet.footerLinks" ' +
                              'ng-if="hasLabel(link)" tabindex="0"' +
                              'title="{{link.tooltipText && ' +
                              'link.tooltipText.trim().length ?' +
                              ' link.tooltipText : link.label}}"' +
                              'ng-click="clickAction($event,link)"' +
                              'ng-class="{\'portlet-footer-link-disabled\': link.enabled === false}">' +
                           '{{link.label}}' +
                        '</a>' +
                     '</div>' +
                  '</div>' +
               '</div>',
      link: function (scope, element, attributes) {

         scope.portletsOptions = scope[attributes.vuiPortlets];

         //Returns options.layout if defined else will use two-columns as
         //default layout
         scope.getLayout = function (layout) {
            if (layout && layout.trim().length > 0) {
               return layout;
            } else {
               return vuiConstants.portlets.layout.TWO_COLUMNS;
            }
         };

         scope.isCollapsed = function (portlet) {
            portlet.collapsed =
               jsUtils.getProperty(portlet, 'collapsed', false);
         };

         //Returns true if link has label
         scope.hasLabel = function (link) {
            return (link.label && link.label.trim().length > 0);
         };

         scope.clickAction = function (event, link) {
            if (angular.isDefined(link.onClick) &&
                  angular.isFunction(link.onClick)) {
               link.onClick.call(undefined, event, link);
            } else {
               $log.warn('No onClick action defined for link: ' + link.label);
            }
         };
      }
   };

}]);
;'use strict';

vui.angular.progressBar

/**
 * @ngdoc object
 * @name vui.angular.progressBar:progressBarOptions
 * @module vui.angular.progressBar
 *
 * @description
 *    Configuration object for {@link vui.angular.progressBar.directive:vuiProgressBar vuiProgressBar} directive.
 *
 *    Example:
 *    ```js
 *    var progressBarOptions = {
 *                               percent: 0,
 *                               width: '200px',
 *                               height: '20px'
 *                            };
 *    ```
 *
 * @property {number} percent
 *    Type: `number`
 *
 *    The number corresponding to the percent complete.
 * @property {string} [width]
 *    Type: `string` **|** Optional **|** Default: `100px`
 *
 *    The width of the progressBar.
 * @property {string} [height]
 *    Type: `string` **|** Optional **|** Default: `20px`
 *
 *    The height of the progressBar in px.
 * @property {boolean} [showLabel]
 *    Type: `boolean` **|** Optional **|** Default: `true`
 *
 *    Specifies whether the progress percent ajacent to the progress bar is visible.
 */

/**
 * @ngdoc directive
 * @name vui.angular.progressBar.directive:vuiProgressBar
 * @module vui.angular.progressBar
 * @restrict A
 * @scope
 *
 * @description
 *    Directive to show live progress of a task or a process.
 *
 * @param {progressBarOptions} vuiProgressBar
 *    {@link vui.angular.progressBar:progressBarOptions progressBarOptions}
 *    Configuration object for the progressBar.
 *
 * @example
      <example module="app">

         <file name="app.js">
            'use strict';
            var app = angular.module('app', ['vui.angular']);
            app.controller('ProgressBarController', function ($scope, $interval) {
               $scope.progressBarOptions1 = {
                  percent: 0
               };

               $scope.progressBarOptions2 = {
                  percent: 0,
                  width: '200px',
                  height: '20px'
               };

               // Update progress
               $interval(function () {
                  $scope.progressBarOptions1.percent += 1;
                  $scope.progressBarOptions2.percent += 1;
               }, 200, 100);
            });
         </file>

         <file name="index.html">
            <link rel="stylesheet" href="css/vui-bootstrap.min.css"
                  type="text/css">
            <div ng-controller="ProgressBarController">
               <h1>Progress bar with default options</h1>
               <div vui-progress-bar='progressBarOptions1'></div>
               <h1>More options</h1>
               <div vui-progress-bar='progressBarOptions2'></div>
            </div>
         </file>

      </example>
 */

.directive('vuiProgressBar', ['jsUtils', function (jsUtils) {
   return {
      restrict: 'A',
      scope: {
         vuiProgressBar: '=vuiProgressBar'
      },
      template:
            '<div class="progress-container"  ng-style="getContainerStyles()">' +
               '<div class="progress" role="progressbar" aria-valuemin="0"' +
                     'aria-valuemax="100" aria-valuenow="{{vuiProgressBar.percent}}">' +
                  '<div class="progress-bar" ng-style="{\'width\': vuiProgressBar.percent + \'%\' }"></div>' +
               '</div>' +
               '<span class="progress-label" ng-if="vuiProgressBar.showLabel" ' +
                     'ng-style="{\'font-size\': labelFontSize }">{{vuiProgressBar.percent}}%</span>' +
            '</div>',
      controller: ['$scope', function ($scope) {
         $scope.vuiProgressBar.percent =  $scope.vuiProgressBar.percent ?
               $scope.vuiProgressBar.percent : 0;
         $scope.vuiProgressBar.showLabel =
               jsUtils.hasProperty($scope.vuiProgressBar, 'showLabel') ?
               $scope.vuiProgressBar.showLabel : true;

         var computeLabelFontSize = function () {
            // Automatically set font-size if a custom height is set and showLabel is true.
            if ($scope.vuiProgressBar.showLabel &&
                  $scope.vuiProgressBar.height) {
               var height = $scope.vuiProgressBar.height.replace('px', '');
               if (height < 10) {
                  $scope.labelFontSize = '8px';
               } else if (height < 13) {
                  $scope.labelFontSize = '10px';
               } else {
                  $scope.labelFontSize = '12px';
               }
            }
         };

         $scope.getContainerStyles = function () {
            computeLabelFontSize();
            return {
               height : $scope.vuiProgressBar.height,
               width : $scope.vuiProgressBar.width
            };
         };
      }]
   };
}]);
;'use strict';


(function () {
   /**
    * @name vui.angular.services.clipboardService
    * @module vui.angular.services
    *
    * @description
    *    Contains utility for working with the users clipboard.
    */
   vui.angular.services.service('clipboardService', ['$log', function ($log) {
      return {
         copyToClipboard: function (string) {
            var $el = angular.element('<textarea spellcheck="false" autocomplete="false" autocapitalize="false" style="line-height:0; font-size:0;"></textarea>');
            $el.val(string);
            angular.element('body').append($el);
            $el.select();
            try {
               var successful = document.execCommand('copy');
               if (!successful) {
                  $log.error('Copy operation returned false.');
               }
            } catch (err) {
               $log.error(err);
            } finally {
               $el.remove();
            }
         }
      };
   }]);

})();
;'use strict';

vui.angular.modal

/**
 * @ngdoc service
 * @name vui.angular.modal.vuiModalService
 * @module vui.angular.modal
 *
 * @description
 *    Service to create instances of pop-up allowing to hide, show and destroy them.
 *
 * # General usage
 *    The `vuiModalService` is a function which takes a single argument,
 *    a configuration object that is used to create a pop-up.
 *
 * @param {Object} config
 *    Configuration object containing the following properties:
 *
 *    - **scope** - Type: `$scope`<br/>
 *       Angular scope object to use for the pop-up.
 *       Its usually the scope of the controller or rootScope  or a custom scope.
 *       The scope of the pop-up prototypically inherits from the scope that
 *       is mentioned here.
 *
 *    - **configObjectName** – Type: `string`<br/>
 *       The name of the variable declared on the above mentioned scope
 *       that holds the options for the pop-up. The variable is of type
 *       {@link vui.angular.modal:ModalOptions ModalOptions}.
 *
 *    - **_vuiModalTemplate** - Type: `string`<br/>
 *       The template of the pop-up that will be compiled on the given
 *       scope and appended to body or backdrop element on show.
 *
 *    - **[showBackdrop]** - Type: `boolean` **|** Optional **|** Default: `false`<br/>
 *       Determines whether the pop-up should be modal.
 *
 *    - **[zIndex]** - Type: `number` **|** Optional **|** Default: `1050`<br/>
 *       The z-index of the pop-up or the pop-up modal layer(if the pop-up is modal).
 *
 *    - **[headerElementSelector]** - Type: `string` **|** Optional **|** Default: `null`<br/>
 *       The CSS selector used to find tbe header element of the pop-up.
 *       The selector is executed in the context of the pop-up i.e. the selector is
 *       relative to the root element of the pop-up.
 *       The headerElementSelector property is required for the pop-up maximizable
 *       and draggable features to work.
 *
 *    - **[destroyOnHide]** - Type: `boolean` **|** Optional **|** Default: `false`<br/>
 *       Whether to destroy the dialog when it's hidden.
 *
 *    - **[destroyOuterScope]** - Type: `boolean` **|** Optional **|** Default: `false`<br/>
 *       Whether to destroy the scope passed to this modal when the modal is destroyed.
 *
 * @returns {object}
 *    Returns a response object which has these methods:
 *
 *   - **show()** : `{method}` To display the pop-up.
 *   - **hide()** : `{method}` To hide the pop-up.
 *   - **destroy()** : `{method}` To destroy the pop-up.
 * @example
      <example module="app">

         <file name="app.js">
            'use strict';
            var appModule = angular.module('app', ['vui.angular.modal']);

            appModule.controller('modalServiceCtrl', ['$scope', 'vuiModalService',
                  function ($scope, vuiModalService) {

               $scope.statusText = 'Create a modal';

               var modalOptionsProto = {
                  show: false,
                  maximizable: true,
                  resizable: true,
                  resizeKeepAspectRatioKey: 'alt',
                  resizeMinWidth: 200,
                  resizeMaxWidth: 700,
                  resizeMinHeight: 300,
                  resizeMaxHeight: 800,
                  draggable: true
               };

               var modalConfigProto = {
                  scope: null,
                  configObjectName : 'modalOptions',
                  _vuiModalTemplate:
                        '<div style="\n' +
                        '      width: 400px;\n' +
                        '      height: 400px;\n' +
                        '      position: relative;\n' +
                        '      background-color: white;">\n' +
                        '  <div class="header" style="\n' +
                        '        background-color: gray;\n'+
                        '        width: 100%;\n' +
                        '        height: 50px;\n' +
                        '        font-weight: bold;\n' +
                        '        text-align: left;\n' +
                        '        line-height: 50px;\n' +
                        '        padding-left: 15px;\n' +
                        '        padding-right: 15px;">\n' +
                        '     HEADER\n' +
                        '  </div>\n' +
                        '  <button ng-click="modalOptions.show = false"\n' +
                        '        style="position: absolute;\n' +
                        '            right: 15px;\n' +
                        '            bottom: 15px;">\n' +
                        '     CLOSE\n' +
                        '  </button>\n' +
                        '</div>',
                  showBackdrop: true,
                  zIndex: 1337,
                  headerElementSelector: '> .header'
               };

               var modalConfig = null;
               var modalScope = null;
               var modalOptions = null;
               var modal = null;

               $scope.createModal = function () {
                  if (modal !== null) {
                      alert('Modal already exists');
                      return;
                  }

                  modalOptions = angular.copy(modalOptionsProto);

                  modalScope = $scope.$new(true, null);
                  modalScope.modalOptions = modalOptions;

                  modalConfig = angular.copy(modalConfigProto);
                  modalConfig.scope = modalScope;

                  modal = vuiModalService(modalConfig);

                  $scope.statusText = 'Modal created';
               };

               $scope.openModal = function () {
                  if (modal === null) {
                     alert('Create a modal first');
                     return;
                  }

                  modal.show();
               };

               $scope.destroyModal = function () {
                  if (modal === null) {
                     alert('Create a modal first');
                     return;
                  }

                  modal.destroy();
                  modal = null;
                  modalScope.$destroy();

                  $scope.statusText = 'Create a modal';
               };
            }]);
         </file>

         <file name="index.html">
            <div ng-controller="modalServiceCtrl">
               <div class="container">
                  <button ng-click="createModal()">Create Modal</button>
                  <button ng-click="openModal()">Open Modal</button>
                  <button ng-click="destroyModal()">Destroy Modal</button>
                  <br>
                  <br>
                  <br>
                  <span style="font-weight: bold; color: green;">{{statusText}}</span>
               </div>
            </div>
         </file>
      </example>
 */

/**
 * @ngdoc object
 * @name vui.angular.modal:ModalOptions
 * @module vui.angular.modal
 *
 * @description
 *    Options object for
 *    {@link vui.angular.modal.vuiModalService vuiModalService} service.
 *
 *    Example:
 *    ```js
 *    var modalOptions = {
 *                            show: true,
 *                            maximizable: true,
 *                            resizable: true,
 *                            resizeKeepAspectRatioKey: 'ctrl'
 *                            resizeMinWidth: 200,
 *                            resizeMaxWidth: 1000,
 *                            resizeMinHeight: 300,
 *                            resizeMaxHeight: 900,
 *                            draggable: true
 *                        };
 *    ```
 *
 * @property {boolean} [show]
 *    Type: `boolean` **|** Optional **|** Default: `false`
 *
 *    Shows the pop-up when set to true.
 *
 * @property {boolean} [maximizable]
 *    Type: `boolean` **|** Optional **|** Default: `false`
 *
 *    Determines whether the pop-up is maximizable/minimizable.
 *
 *    **Note**:
 *    The user interaction to maximize/minimize the pop-up is double-clicking on
 *    its header element. This property will not have any effect if the pop-up does
 *    not have a header element. For more info how the header element is determined
 *    please see documentation of
 *    {@link vui.angular.modal.vuiModalService vuiModalService}.
 *
 * @property {boolean} [resizable]
 *    Type: `boolean` **|** Optional **|** Default: `false`
 *
 *    Determines whether the pop-up is resizable.
 *
 *    **Note**:
 *    This property will not have any effect if the jQuery UI lib is not loaded.
 *
 * @property {string} [resizeKeepAspectRatioKey]
 *    Type: `string` **|** Optional **|** Default: `shift` **|** Values: `'ctrl'|'alt'|'shift'|''`
 *
 *    The keyboard key that when pressed during resizing of the pop-up will cause
 *    the resize to preserve the original (before the current resize operation start)
 *    aspect ratio of the pop-up. The value `''` means no key.
 *
 * @property {number} [resizeMinWidth]
 *    Type: `number` **|** Optional **|** Default: `null`
 *
 *    The minimum width of the pop-up during resize.
 *    This property will not have any effect if the pop-up is not resizable.
 *
 * @property {number} [resizeMaxWidth]
 *    Type: `number` **|** Optional **|** Default: `null`
 *
 *    The maximum width of the pop-up during resize.
 *    This property will not have any effect if the pop-up is not resizable.
 *
 * @property {number} [resizeMinHeight]
 *    Type: `number` **|** Optional **|** Default: `null`
 *
 *    The minimum height of the pop-up during resize.
 *    This property will not have any effect if the pop-up is not resizable.
 *
 * @property {number} [resizeMaxHeight]
 *    Type: `number` **|** Optional **|** Default: `null`
 *
 *    The maximum height of the pop-up during resize.
 *    This property will not have any effect if the pop-up is not resizable.
 *
 * @property {boolean} [draggable]
 *    Type: `boolean` **|** Optional **|** Default: `false`
 *
 *    Determines whether the pop-up is draggable.
 *
 *    **Note**:
 *    The drag interaction is performed on the header element of the pop-up.
 *    This property will not have any effect if the pop-up does not have a header
 *    element or jQuery UI lib is not loaded.
 *    For more info how the header element is determined please see documentation of
 *    {@link vui.angular.modal.vuiModalService vuiModalService}.
 */

.provider('vuiModalService', function () {

   var explicitDefaultConfig = null;
   var explicitDefaultOptions = null;

   this.setDefaultConfig = function (defaultConfig) {
      explicitDefaultConfig = $.extend(true, {}, defaultConfig);
   };

   this.setDefaultModalOptions = function (defaultOptions) {
      explicitDefaultOptions = $.extend(true, {}, defaultOptions);
   };

   this.$get = ['$window', '$rootScope', '$compile', 'jsUtils', 'vuiUtils', 'vuiConstants', 'vuiFocusTraceService', function ($window, $rootScope, $compile, jsUtils, vuiUtils, vuiConstants, vuiFocusTraceService) {
      var DEFAULT_CONFIG = {
         showBackdrop: false,
         zIndex: vuiConstants.internal.zIndex.DEFAULT,
         positionClass: '',
         headerElementSelector: '',
         destroyOnHide: false,
         destroyOuterScope: false,
         _outerScopeActual: null
      };

      var DEFAULT_OPTIONS = {
         show: false,
         maximizable: false,
         maximized: false,
         resizable: false,
         resizeKeepAspectRatioKey: 'shift', //Possible values are 'ctrl', 'shift', 'alt'
         resizeMinWidth: 0,
         resizeMaxWidth: Math.pow(2, 32),
         resizeMinHeight: 0,
         resizeMaxHeight: Math.pow(2, 32),
         draggable: false
      };

      var BACKDROP_CLASS = 'modal-backdrop';
      var BACKDROP_CENTER_CHILDREN_DISABLED_CLASS = 'center-children-disabled';
      var BACKDROP_HIDDEN_CLASS = 'hidden';
      var MODAL_CLASS = 'vui-modal-element';
      var MODAL_HIDDEN_CLASS = 'hidden';
      var MODAL_ADVANCED_FEATURES_ENABLED_CLASS = 'advanced-features-enabled';
      var MODAL_NO_BACKDROP_CLASS = 'no-backdrop';
      var MODAL_RESIZABLE_CONTAINMENT_HELPER_CLASS =
         'vui-modal-resizable-containment-helper';
      var MAXIMIZABLE_CLASS = 'maximizable';
      var MAXIMIZED_CLASS = 'maximized';
      var RESIZABLE_CLASS = 'resizable';
      var DRAGGABLE_CLASS = 'draggable';
      var RESIZE_KEEP_ASPECT_RATIO_KEYS = ['ctrl', 'alt', 'shift'];
      var NEXT_MODAL_INSTANCE_ID = 0;
      var bodyElement = angular.element($window.document.body);

      function ModalFactory(config) {

         var vuiModal = {};
         // The unique id of the current modal instance
         var instanceId = NEXT_MODAL_INSTANCE_ID++;

         var unbindShowWatch, unbindMaximizedWatch, unbindRecenterWatch;

         var modalElement, backdropElement, modalElementTitlebar,
            resizableContainmentHelperElement;

         var modalElementMaximizable, modalElementResizable, modalElementDraggable,
            modalElementAdvancedFeaturesEnabled, showBackdrop,
            modalElementCenteredUsingCss,
            centerModalElementAfterMinimize,
            resizedOrDraggedSinceLastCentering;

         var scope, modalScope, options, outerScope, destroyOnHide, destroyOuterScope;

         // Creates a namespace for the given event name so that event handlers are
         // easily removed, see http://www.w3schools.com/jquery/event_namespace.asp
         // and https://css-tricks.com/namespaced-events-jquery/.
         // Param is a string used to additionally narrow the event name
         function getNamespacedEventName(name, param) {
            var result = name +
               '.vuiModalService_modalInstance_' +
               instanceId;

            if (typeof param === 'string' && param.length !== 0) {
               result += '_' + param;
            }

            return result;
         }

         function setupMaximize() {
            if (jsUtils.isUndefinedOrNull(modalElementTitlebar) || !modalElementMaximizable) {
               return;
            }

            modalElementTitlebar.on('dblclick', function () {
               vuiUtils.safeApply(scope, function () {
                  options.maximized = !options.maximized;
               });
            });
         }

         function setupResize() {
            if (jsUtils.isUndefinedOrNull(modalElement) || !modalElementResizable) {
               return;
            }

            var containmentElement = null;
            if (showBackdrop) {
               containmentElement = backdropElement;
            } else {
               // If the modal is appended to the body it's not possible to contain it
               // to window, so create a fixed element that fills the window and
               // contain the modal to it if needed
               resizableContainmentHelperElement = $('<div></div>')
                  .addClass(MODAL_RESIZABLE_CONTAINMENT_HELPER_CLASS)
                  .appendTo(bodyElement);
               containmentElement = resizableContainmentHelperElement;
            }

            modalElement.resizable({
               containment: containmentElement,
               handles: 'se',
               distance: 0,
               delay: 0,
               minWidth: options.resizeMinWidth,
               maxWidth: options.resizeMaxWidth,
               minHeight: options.resizeMinHeight,
               maxHeight: options.resizeMaxHeight
            }).on({
               // Attach the events using 'on' instead of specifying the callbacks in
               // the options of the jQuery UI resizable as the order in which
               // events handlers are called is different between these cases
               'resizestart': resizableResizeStartHandler,
               'resize': resizableResizeHandler,
               'resizestop': resizableResizeStopHandler
            });
         }

         function setupDrag() {
            if (jsUtils.isUndefinedOrNull(modalElement) || !modalElementDraggable) {
               return;
            }

            modalElement.draggable({
               containment: (showBackdrop ? backdropElement : 'window'),
               handle: modalElementTitlebar,
               scroll: false
            }).on({
               'drag': draggableDragHandler
            });
         }

         function centerModalElement() {
            if (jsUtils.isUndefinedOrNull(modalElement) ||
               modalElementCenteredUsingCss) {
               return;
            }

            if (modalElementMaximizable && options.maximized) {
               centerModalElementAfterMinimize = true;
               return;
            }

            if (options.show === false) {
               return;
            }

            centerModalElementAfterMinimize = false;
            resizedOrDraggedSinceLastCentering = false;

            var layoutParentWidth;
            var layoutParentHeight;
            if (showBackdrop) {
               layoutParentWidth = backdropElement.width();
               layoutParentHeight = backdropElement.height();
            } else {
               layoutParentWidth = window.innerWidth;
               layoutParentHeight = window.innerHeight;
            }

            var modalElementLeft = parseInt((layoutParentWidth - modalElement.outerWidth()) / 2);
            var modalElementTop = parseInt((layoutParentHeight - modalElement.outerHeight()) / 2);
            modalElement.css('left', modalElementLeft + 'px');
            modalElement.css('top', modalElementTop + 'px');
         }

         function recenterModalElementIfNeeded() {
            if (!resizedOrDraggedSinceLastCentering) {
               centerModalElement();
            }
         }

         function updateModalElementMaximizedState() {
            if (jsUtils.isUndefinedOrNull(modalElement) || !modalElementMaximizable) {
               return;
            }

            modalElement.toggleClass(MAXIMIZED_CLASS, !!options.maximized);
            kendo.resize(modalElement);
            modalElement.trigger('modalResize');
         }

         function updateModalElementResizeEnabledState() {
            if (jsUtils.isUndefinedOrNull(modalElement) || !modalElementResizable) {
               return;
            }

            var disabled = modalElementMaximizable && options.maximized;
            modalElement.resizable('option', {disabled: disabled});
         }

         function updateModalElementDragEnabledState() {
            if (jsUtils.isUndefinedOrNull(modalElement) || !modalElementDraggable) {
               return;
            }

            var disabled = modalElementMaximizable && options.maximized;
            modalElement.draggable('option', {disabled: disabled});
         }

         function resizableResizeStartHandler() {
            // Attach a mousemove handler to the document so we can modify the
            // pressed key to preserve aspect ratio. By default jQuery UI resizable
            // preserves the aspect ratio if the shift key is pressed,
            // but we want to be able to change it.
            //
            // IMPORTANT!!!: The handler needs to be attached before the jQuery UI one
            $(document).on(
               getNamespacedEventName('mousemove'),
               resizableDocumentMouseMoveHandler
            );
         }

         function resizableResizeHandler(event) {
            // Prevent triggering of window resize event, but trigger a custom event in case
            // anything in the modal wants to know.
            kendo.resize(modalElement);
            modalElement.trigger('modalResize');
            event.stopImmediatePropagation();

            resizedOrDraggedSinceLastCentering = true;
         }

         function resizableResizeStopHandler() {
            // Remove the mousemove handler from the document
            $(document).off(getNamespacedEventName('mousemove'));
         }

         function resizableDocumentMouseMoveHandler(event) {
            var isResizeKeepAspectRatioKeyValid = ($.inArray(
               options.resizeKeepAspectRatioKey,
               RESIZE_KEEP_ASPECT_RATIO_KEYS
            ) >= 0);

            // If the resizeKeepAspectRatioKey is invalid, unset the
            // shift key as pressed
            if (!isResizeKeepAspectRatioKeyValid) {
               event.shiftKey = false;
               return;
            }

            var keyPropertyName = options.resizeKeepAspectRatioKey + 'Key';
            // If the resizeKeepAspectRatioKey is not pressed, unset the
            // shift key as pressed
            if (event[keyPropertyName] !== true) {
               event.shiftKey = false;
               return;
            }

            // If the resizeKeepAspectRatioKey is pressed, set the
            // shift key as pressed
            event.shiftKey = true;
         }

         function draggableDragHandler() {
            resizedOrDraggedSinceLastCentering = true;
         }

         function destroyModal(destroyModalScope) {
            if (destroyModalScope) {
               // when vuiModal.scope is destroyed this will
               // trigger the destroyModal() method
               $rootScope.$evalAsync(function () {
                  modalScope.$destroy();
               });
               return;
            }

            // Remove element
            if (modalElement) {
               modalElement.remove();
               modalElement = null;
            }

            // Remove backdrop
            if (backdropElement) {
               backdropElement.remove();
               backdropElement = null;
            }

            // Remove resizable containment helper element
            if (resizableContainmentHelperElement) {
               resizableContainmentHelperElement.remove();
               resizableContainmentHelperElement = null;
            }

            // Remove the document mousemove handler that was possibly set during resize
            $(document).off(getNamespacedEventName('mousemove'));
            // Remove the window resize handle that was possibly added
            $(window).off(getNamespacedEventName('resize'));

            // Remove the watches.
            unbindShowWatch();
            unbindMaximizedWatch();
            if (typeof unbindRecenterWatch === 'function') {
               unbindRecenterWatch();
            }

            // remove methods and properties of vuiModal
            vuiModal.$destroyed = true;
            vuiModal.scope = null;
            vuiModal.show = $.noop;
            vuiModal.hide = $.noop;
            vuiModal.destroy = $.noop;

            options.focusTarget = null;
         }

         if (!jsUtils.hasProperty(config, 'scope') ||
               !jsUtils.hasProperty(config, '_vuiModalTemplate') ||
               !jsUtils.hasProperty(config, 'configObjectName')) {
            throw new Error('Invalid config object');
         }

         scope = config.scope;
         options = scope[config.configObjectName];
         if (!options.focusTarget) {
            options.focusTarget = vuiFocusTraceService.popFocusableElement();
         }

         var defaultConfig = $.extend(true, { }, DEFAULT_CONFIG, explicitDefaultConfig);
         var defaultOptions = $.extend(true, { }, DEFAULT_OPTIONS, explicitDefaultOptions);

         $.extend(config, $.extend({}, defaultConfig, config));
         $.extend(options, $.extend({}, defaultOptions, options));

         outerScope = config._outerScopeActual || config.scope || null;
         destroyOnHide = config.destroyOnHide;
         destroyOuterScope = config.destroyOuterScope;

         modalElementMaximizable = (options.maximizable === true);
         modalElementResizable =
            !jsUtils.isUndefinedOrNull($.ui) &&
            (typeof $.ui.resizable === 'function') &&
            (options.resizable === true);
         modalElementDraggable =
            !jsUtils.isUndefinedOrNull($.ui) &&
            (typeof $.ui.draggable === 'function') &&
            (options.draggable === true) &&
            !jsUtils.isUndefinedOrEmpty(config.headerElementSelector);

         modalElementAdvancedFeaturesEnabled =
            modalElementMaximizable ||
            modalElementResizable ||
            modalElementDraggable;
         showBackdrop = (config.showBackdrop === true);
         modalElementCenteredUsingCss = !modalElementAdvancedFeaturesEnabled && showBackdrop;

         centerModalElementAfterMinimize = false;
         resizedOrDraggedSinceLastCentering = false;

         if (!modalElementCenteredUsingCss) {
            $(window).on(getNamespacedEventName('resize'), function () {
               vuiUtils.safeApply(scope, function () {
                  centerModalElement();
               });
            });
         }

         if (!modalElementCenteredUsingCss) {
            var isInPostDigestQueue = false;
            var isPostDigestRunning = false;

            unbindRecenterWatch = $rootScope.$watch(function () {
               if (isInPostDigestQueue || isPostDigestRunning) {
                  return;
               }

               isInPostDigestQueue = true;

               $rootScope.$$postDigest(function () {
                  isPostDigestRunning = true;

                  if (modalScope) {
                     recenterModalElementIfNeeded();
                  }

                  isPostDigestRunning = false;
                  isInPostDigestQueue = false;
               });
            });
         }

         // new child scope to perform destroy when we call vuiModal.destroy.
         modalScope = config.scope.$new();
         // Remove modal element and destroy scope when parent
         // scope is destroyed.
         modalScope.$on('$destroy', function () {
            destroyModal(false);

            if (destroyOuterScope) {
               $rootScope.$evalAsync(function () {
                  outerScope.$destroy();
               });
            }
         });

         vuiModal.scope = modalScope;
         vuiModal.$destroyed = false;

         vuiModal.show = function () {
            if (modalElement) {
               if (showBackdrop) {
                  backdropElement.removeClass(BACKDROP_HIDDEN_CLASS);
               } else {
                  modalElement.removeClass(MODAL_HIDDEN_CLASS);
               }
               options.show = true;
            } else {
               // Compile the directive template on the newly created child scope,
               // so that the directive would be a child of the child scope
               // and not the wrapper. This is required during destroy.
               modalElement = $compile(config._vuiModalTemplate)(modalScope);
               modalElement.addClass(MODAL_CLASS);
               modalElement = modalElement.addClass(config.positionClass);

               if (!jsUtils.isUndefinedOrEmpty(config.headerElementSelector)) {
                  modalElementTitlebar = $(config.headerElementSelector, modalElement);
                  if (modalElementTitlebar.length !== 1) {
                     throw new Error(
                        'Can\'t find header element with selector: "' +
                        config.headerElementSelector +
                        '"!'
                     );
                  }
               }

               modalElement.toggleClass(
                  MODAL_ADVANCED_FEATURES_ENABLED_CLASS,
                  modalElementAdvancedFeaturesEnabled
               );
               modalElement.toggleClass(MODAL_NO_BACKDROP_CLASS, !showBackdrop);
               modalElement.toggleClass(MAXIMIZABLE_CLASS, modalElementMaximizable);
               modalElement.toggleClass(RESIZABLE_CLASS, modalElementResizable);
               modalElement.toggleClass(DRAGGABLE_CLASS, modalElementDraggable);

               if (showBackdrop) {
                  backdropElement = $compile('<div></div>')(modalScope);
                  backdropElement.append(modalElement);
                  backdropElement.addClass(BACKDROP_CLASS);
                  backdropElement.toggleClass(
                     BACKDROP_CENTER_CHILDREN_DISABLED_CLASS,
                     modalElementAdvancedFeaturesEnabled
                  );
                  backdropElement.css('z-index', config.zIndex);
               } else {
                  modalElement.css('z-index', config.zIndex);
               }

               if (showBackdrop) {
                  bodyElement.append(backdropElement);
               } else {
                  bodyElement.append(modalElement);
               }

               options.show = true;

               setupMaximize();
               setupResize();
               setupDrag();

               centerModalElement();

               updateModalElementMaximizedState();
               updateModalElementResizeEnabledState();
               updateModalElementDragEnabledState();
            }
         };

         vuiModal.hide = function () {
            options.show = false;
            if (showBackdrop) {
               if (backdropElement) {
                  backdropElement.addClass(BACKDROP_HIDDEN_CLASS);
               }
            } else {
               if (modalElement) {
                  modalElement.addClass(MODAL_HIDDEN_CLASS);
               }
            }

            if (options.focusTarget) {
               options.focusTarget.focus();
            }
         };

         vuiModal.destroy = function () {
            destroyModal(true);
         };

         // set a watch on options.show
         unbindShowWatch = scope.$watch(
            config.configObjectName + '.show',
            function (newValue, oldValue) {
               if (newValue === false && oldValue === true) {
                  vuiModal.hide();

                  if (destroyOnHide) {
                     vuiModal.destroy();
                  }
               } else if (newValue === true && oldValue === false) {
                  vuiModal.show();
                  centerModalElement();
               }
            });

         // set a watch on options.maximize
         unbindMaximizedWatch = scope.$watch(
            config.configObjectName + '.maximized',
            function (newValue, oldValue) {
               if (jsUtils.isUndefinedOrNull(modalElement) || !modalElementMaximizable) {
                  return;
               }

               updateModalElementMaximizedState();
               updateModalElementResizeEnabledState();
               updateModalElementDragEnabledState();

               if (newValue === false &&
                  oldValue === true &&
                  centerModalElementAfterMinimize) {

                  centerModalElement();
               }
            });

         if (options.show) {
            vuiModal.show();
         }

         return vuiModal;
      }

      return ModalFactory;
   }];
});
;'use strict';


(function () {
   /**
    *
    * @description
    *    Keeps track of the invokers of the modal service
    *    to return focus to them when the modal is closed
    */
   vui.angular.services.factory('vuiFocusTraceService', ['$timeout', function ($timeout) {
      var FOCUS_ELEMENT_EXPIRATION_TIMEOUT = 5000;
      var _element = null;

      return {
         setFocusableElement: function (element) {
            if (!(element instanceof HTMLElement)) {
               throw new Error('Focusable element must be DOM element');
            }

            _element = element;
            $timeout(function () {
               _element = null;
            }, FOCUS_ELEMENT_EXPIRATION_TIMEOUT);
         },
         popFocusableElement: function () {
            var el = _element;
            _element = null;
            if (!el && document.activeElement) {
               return document.activeElement;
            }

            return el;
         }
      };
   }]);

})();
;'use strict';

vui.angular.services

/**
 * @name vui.angular.services.vuiLocale
 * @module vui.angular.services
 *
 * @description
 *    Contains locale strings.
 *    TODO : Returns the locale strings in the language detected.
 */
.factory('vuiLocale', [function () {

   // we can use $locale service in angular to detect locale
   //var locale = $locale.id;
   var localeStrings = {
      en : {
         'ok'     : 'OK',
         'back'   : 'Back',
         'next'   : 'Next',
         'finish' : 'Finish',
         'cancel' : 'Cancel',
         'filter' : 'Filter',
         'of'     : 'of',
         'item'   : 'item',
         'items'  : 'items',
         'actionsMenu' : {
            'empty'              : 'No Actions Available'
         },
         'datagrid' : {
            'emptyTable'         : 'No items to display',
            'zeroRecords'        : 'No matching records found',
            'colFilterInfo'      : 'Show items with value that',
            'colFilterBooleanInfo': 'Show items with value that is',
            'filter'             : 'Filter',
            'clear'              : 'Clear',
            'isTrue'             : 'True',
            'isFalse'            : 'False',
            'and'                : 'And',
            'or'                 : 'Or',
            // Filter operators
            'contains'           : 'Contains',
            'eq'                 : 'Is equal to',
            'neq'                : 'Is not equal to',
            'gte'                : 'Greater than or equal',
            'gt'                 : 'Greater than',
            'lte'                : 'Less than or equal',
            'lt'                 : 'Less than',

            // Page message
            'item'   : 'item',
            'items'  : 'items',

            'save'   : 'Apply changes',
            // Footer
            'copy' : 'Copy All'
         },
         'notification' : {
            'defaultTitle' : 'Notification'
         }
      },
      fr : {
         'ok'     : 'OKe',
         'back'   : 'Backe',
         'next'   : 'Nexte',
         'finish' : 'Finishe',
         'cancel' : 'Cancele',
         'filter' : 'Filtere',
         'of'     : 'ofe',
         'item'   : 'iteme',
         'items'  : 'itemse',
         'datagrid' : {
            'emptyTable'   : 'No data available in table1234',
            'zeroRecords'  : 'No matching records found1234'
         }
      }
   };

   return localeStrings.en;
}]);
;'use strict';

(function () {
   vui.angular.services.factory('vuiLocalizationService', [function () {
      var data = {};

      return {
         get: function (key) {
            return data[key];
         },
         set: function (key, value) {
            data[key] = value;
         }
      };
   }]);

})();
;'use strict';

vui.angular.services

/**
 * @name vui.angular.services.jsUtils
 * @module vui.angular.services
 *
 * @description
 *    Exposes the javascript utilities singleton `vui.Utils` as an angular service.
 *    TODO: link to `vui.Utils` documentation once hosted as independent project.
 */
.constant('jsUtils', vui.Utils)

/**
 * @name vui.angular.services.vuiUtils
 * @module vui.angular.services
 *
 * @description
 *    Various utility function for working with angularJS specific constructs, such as
 *    scope.
 */
.factory('vuiUtils', ['jsUtils', function (jsUtils) {

   /**
    * @name vui.angular.services.vuiUtils#jsUtils
    * @propertyOf vui.angular.services.vuiUtils
    *
    * @description
    *    Reference to the {@link vui.components.services.jsUtils `jsUtils`} service.
    */

   /**
    * @name vui.angular.services.vuiUtils#safeApply
    * @methodOf vui.angular.services.vuiUtils
    *
    * @description
    *    Passes the given function to 'scope.apply' only if angular is not already in a
    *    digest loop. Prevents $apply exceptions being raised. See [this explanation]
    * (http://www.yearofmoo.com/2012/10/more-angularjs-magic-to-supercharge-your-webapp.html#apply-digest-and-phase)
    *    for more details.
    *
    * @param {Object} scope
    *    `{Object}` The scope that should be updated by the given function.
    * @param {Function} function
    *    `{Function}` The function that should update the given scope.
    */
   function safeApply(scope, fn) {
      fn = fn || function () {};
      if (scope.$root.$$phase) {
         // Already in a digest loop, call directly
         fn();
      }
      else {
         scope.$apply(fn);
      }
   }

   /**
    * @name vui.angular.services.vuiUtils#updateScopeProperty
    * @methodOf vui.angular.services.vuiUtils
    *
    * @description
    *    Updates the given scope with the value for the given property. Only writes to
    *    property if it is already declared on scope. Calls $apply if necessary.
    *
    * @param {Object} scope
    *    `{Object}` The scope to update.
    * @param {string} property
    *    `{string}` The property to update on scope. Must already exist on scope else
    *    nothing happens.
    * @param {*} value
    *    The value to write for property on scope.
    * @param {boolean} [createIfMissing=false]
    *    `{boolean}` Set to true to write the property to the scope object even if the
    *    property is not declared on the scope object.
    */
   function updateScopeProperty(scope, property, value, createIfMissing) {
      if (!(jsUtils.hasProperty(scope, property)) && !createIfMissing) {
         return;
      }
      safeApply(scope, function () {
         jsUtils.setProperty(scope, property, value, createIfMissing);
      });
   }

   // Export API
   return {
      safeApply: safeApply,
      updateScopeProperty: updateScopeProperty
   };
}]);

;'use strict';

vui.angular.services

/**
 * @name vui.angular.services.vuiZoneService
 * @module vui.angular.services
 *
 * @description
 *    provide function for running functions inside Angular zone.
 *    scope.
 */
.factory('vuiZoneService', [function () {
   var angularZone = null;
   if (window.Zone && window.Zone.current && window.Zone.current.name === 'angular') {
      angularZone = window.Zone.current;
   }

   return {
      // Runs the method in the Angular zone (NG Zone)
      runInsideAngularZone: function (method) {
         if (angularZone && window.Zone.current.name !== 'angular') {
            angularZone.run(method);
         } else {
            method();
         }
      }
   };
}]);
;'use strict';

vui.angular.slider

/**
 * @ngdoc object
 * @name vui.angular.slider:SliderOptions
 * @module vui.angular.slider
 *
 * @description
 *    Configuration object for {@link vui.angular.slider.directive:vuiSlider vuiSlider} directive.
 *
 *    Example:
 *    ```js
 *    var sliderOptions = {
 *                               value: 0,
 *                               min: 0,
 *                               max: 10,
 *                               minLabel: '0 Gb',
 *                               maxLabel: '10 Gb',
 *                               enabled : true,
 *                               width: '250px',
 *                               dragHandleTitle: 'drag'
 *                         };
 *    ```
 *
 * @property {number} value
 *    Type: `number`
 *
 *    The underlying value of the Slider. If its not specified it defaults to the
 *    {@link vui.angular.slider:SliderOptions min} value. This attribute is both readable and writable.
 * @property {number} min
 *    Type: `number`**|** Optional **|** Default: `0`
 *
 *    The minimum value of the slider.
 * @property {number} max
 *    Type: `number`**|** Optional **|** Default: `10`
 *
 *    The maximum value of the slider.
 * @property {string} [minLabel]
 *    Type: `string` **|** Optional
 *
 *    The display text for the minimum value. It is placed to the left of the slider.
 *    If its not specified it defaults to the {@link vui.angular.slider:SliderOptions min} value.
 * @property {string} [maxLabel]
 *    Type: `string` **|** Optional
 *
 *    The display text for the maximum value. It is placed to the right of the slider.
 *    If its not specified it defaults to the {@link vui.angular.slider:SliderOptions max} value.
 * @property {string} [width]
 *    Type: `string` **|** Optional **|** Default: `220px`
 *
 *    The custom width of the slider. It includes the min and max labels.
 *
 *    Note: Minimum and maximum width of the slider without the labels is set to 80px and 300px respectively.
 * @property {boolean} [enabled]
 *    Type: `boolean` **|** Optional **|** Default: `true`
 *
 * @property {string} [dragHandleTitle]
 *    Type: `string` **|** Optional **|** Default: `drag`
 *    The title of the drag handle of the slider.
 *
 */

/**
 * @ngdoc directive
 * @name vui.angular.slider.directive:vuiSlider
 * @module vui.angular.slider
 * @restrict A
 * @scope
 *
 * @description
 *    Directive that is used to select a value in a range.
 *
 * @param {sliderOptions} vuislider
 *    {@link vui.angular.slider:SliderOptions SliderOptions}
 *    Configuration object for the slider.
 *
 *
 * @example
      <example module="app">

         <file name="app.js">
            'use strict';
            var app = angular.module('app', ['vui.angular']);
            app.controller('SliderController', function ($scope, $interval) {
               $scope.sliderOptions = {
                 value: 0,
                 min: 0,
                 max: 10,
                 minLabel: '0 Gb',
                 maxLabel: '10 Gb',
                 enabled : true,
                 width: '300px',
                 dragHandleTitle: 'drag'
               };
            });
         </file>

         <file name="index.html">
            <link rel="stylesheet" href="css/vui-bootstrap.min.css"
                  type="text/css">
            <div ng-controller="SliderController">
               <h1>Slider with default options</h1>
               <div vui-slider='sliderOptions'></div>
               <div>value : {{sliderOptions.value}}</div>
            </div>
         </file>

      </example>
 */
 .directive('vuiSlider', function () {
   return {
      restrict: 'A',
      scope: {
         vuiSlider: '=vuiSlider'
      },
      template:
         '<div class="vui-slider-container" ng-style="{width: vuiSlider.width}"' +
               'ng-class="{disabled: !vuiSlider.enabled}">' +
            '<div class="slider-label">{{vuiSlider.minLabel}}</div>' +
            '<input kendo-slider k-ng-model="vuiSlider.value" ng-model="vuiSlider.value"' +
                  ' k-min="vuiSlider.min"' +
                  ' k-max="vuiSlider.max" k-tick-placement="\'none\'" k-show-buttons="false"' +
                  ' class="vui-slider" k-ng-disabled="!vuiSlider.enabled" k-drag-handle-title="vuiSlider.dragHandleTitle" ></input>' +
            '<div class="slider-label">{{vuiSlider.maxLabel}}</div>' +
         '</div>',
      controller: ['$scope', function ($scope) {
         $scope.vuiSlider.min = $scope.vuiSlider.min ? $scope.vuiSlider.min : 0;
         $scope.vuiSlider.max = $scope.vuiSlider.max ? $scope.vuiSlider.max : 10;
         $scope.vuiSlider.minLabel = $scope.vuiSlider.minLabel ?
               $scope.vuiSlider.minLabel: $scope.vuiSlider.min;
         $scope.vuiSlider.maxLabel = $scope.vuiSlider.maxLabel ?
               $scope.vuiSlider.maxLabel: $scope.vuiSlider.max;
         $scope.vuiSlider.dragHandleTitle = !$scope.vuiSlider.dragHandleTitle ? 'drag' : $scope.vuiSlider.dragHandleTitle;
         $scope.vuiSlider.enabled = typeof $scope.vuiSlider.enabled === 'boolean' ? $scope.vuiSlider.enabled : true;
         $scope.vuiSlider.value = typeof $scope.vuiSlider.value === 'undefined' ? $scope.vuiSlider.min : $scope.vuiSlider.value;
      }],
      link: function (scope, element) {
         scope.$watch('vuiSlider.width', function () {
            var sliderWidget = element.find('[data-role="slider"]').getKendoSlider();
            if (sliderWidget) {
               sliderWidget.resize();
            }
         });
      }
   };
});
;'use strict';

vui.angular.splitter

/**
 * @ngdoc object
 * @name vui.angular.splitter:SplitterOptions
 * @module vui.angular.splitter
 *
 * @description
 *    Configuration object for
 *    {@link vui.angular.splitter.directive:vuiSplitter vuiSplitter} directive.
 *
 *    Example:
 *    ```js
 *    var splitterOptions = {
 *                               orientation: vuiConstants.splitter.orientation.VERTICAL,
 *                               pane: [
 *                                  { collapsible: true, size: '400px', max: '600px', min: '200px'},
 *                                  { collapsible: false }
 *                               ]
 *                         };
 *    ```
 *
 * @property {string} [orientation]
 *    Type: `string` **|** Optional **|** Default: `vuiConstants.splitter.orientation.HORIZONTAL`
 *
 *    The relative positioning of inner panes. The supported values are
 *    `vuiConstants.splitter.orientation.VERTICAL` and
 *    `vuiConstants.splitter.orientation.HORIZONTAL`.
 *
 * @property {Array.<PaneSpec>} pane
 *    Type: `Array.<PaneSpec>`
 *
 *    Array of {@link vui.angular.splitter:PaneSpec PaneSpec} objects
 *    specifying properties to be applied to each direct pane when loaded.
 */

/**
 * @ngdoc object
 * @name vui.angular.splitter:PaneSpec
 * @module vui.angular.splitter
 *
 * @description
 *    Contains properties to define a pane.
 *
 *    Example:
 *    ```js
 *    pane: [
 *       { collapsible: true, size: '400px', max: '600px', min: '200px'},
 *       { collapsible: false }
 *    ]
 *    ```
 *
 *    @property {boolean} [collapsible]
 *    Type: `boolean` **|** Optional **|** Default: `false`
 *
 *    Specifies whether a pane is collapsible or not collapsible.
 *
 *    @property {boolean} [resizable]
 *    Type: `boolean` **|** Optional **|** Default: `true`
 *
 *    Specifies whether a pane can be resized or not.
 *
 *    @property {number} [max]
 *    Type: `number` **|** Optional
 *
 *    Specifies the maximum size of a pane defined as pixels or as a percentage.
 *    The size of a resized pane cannot exceed the defined maximum size.
 *
 *    @property {number} [min]
 *    Type: `number` **|** Optional
 *
 *    Specifies the minimum size of a pane defined as pixels or as a percentage.
 *    The size of a resized pane cannot exceed the defined minimum size.
 *
 *    @property {number} [size]
 *    Type: `number` **|** Optional
 *
 *    Specifies the size of a pane defined as pixels or as a percentage.
 *    It is recommended that one pane is left without size in order to
 *    compensate for changes in the viewport size
 */

/**
 * @ngdoc directive
 * @name vui.angular.splitter.directive:vuiSplitter
 * @module vui.angular.splitter
 * @restrict A
 * @scope
 *
 * @description
 *    Directive that is used to create a split layout with collapsible and
 *    resizable inner panels.
 *
 * @param {splitterOptions} vuiSplitter
 *    {@link vui.angular.splitter:SplitterOptions SplitterOptions}
 *    Configuration object for the splitter.
 *
 * @example
      <example module="app">

         <file name="app.js">
            'use strict';
            var app = angular.module('app', ['vui.angular']);
            app.controller('SplitterController', ['$scope', 'vuiConstants',
                  function ($scope, vuiConstants) {
               $scope.splitterOptions = {
                  orientation: vuiConstants.splitter.orientation.HORIZONTAL,
                  panes: [
                     { collapsible: true, size: '200px', max: '300px', min: '100px'},
                     { collapsible: true }
                  ]
               };
            }]);
         </file>

         <file name="index.html">
            <link rel="stylesheet" href="css/vui-bootstrap.min.css"
                  type="text/css">
            <div ng-controller="SplitterController">
               <div vui-splitter="splitterOptions">
                     <!-- Left Pane -->
                     <div class="vui-panel">
                        <div class="panel-inner">
                           <div class="titlebar">
                              Left Pane
                           </div>
                           <div class="panel-content">
                             Panel content goes here.
                           </div>
                        </div>
                     </div>
                     <!-- Right Pane -->
                     <div class="vui-panel">
                        <div class="panel-inner">
                           <div class="titlebar">
                              Right Pane
                           </div>
                           <div class="panel-content">
                              Panel content goes here.
                           </div>
                        </div>
                     </div>
                  </div>
               </div>
            </div>
         </file>

      </example>
 */

.directive('vuiSplitter', ['vuiConstants', 'jsUtils',
      function (vuiConstants, jsUtils) {
   return {
      restrict: 'A',
      scope: {
         vuiSplitter: '=vuiSplitter'
      },
      transclude: true,
      template:
         '<div kendo-splitter k-orientation="vuiSplitter.orientation"' +
               'k-panes="vuiSplitter.panes" ng-transclude>' +
         '</div>',
      controller: ['$scope', function ($scope) {
         var initSplitterWatch = $scope.$watch('vuiSplitter', function (vuiSplitter) {
            if (!vuiSplitter) {
               return;
            }
            initSplitterWatch();

            var panes = vuiSplitter.panes;
            var pane;
            vuiSplitter.panes = [];

            vuiSplitter.orientation =
               jsUtils.getProperty(vuiSplitter, 'orientation',
                  vuiConstants.splitter.orientation.HORIZONTAL);

            if (Array.isArray(panes) && panes.length > 0) {
               for (var i = 0; i < panes.length; i++) {
                  pane = {};
                  for (var prop in panes[i]) {
                     pane[prop] = jsUtils.getProperty(panes[i], prop);
                  }
                  //Set default for collapsible and resizable
                  pane = setDefaultPaneProperties(pane);
                  vuiSplitter.panes.push(pane);
               }
            }
         });

         var setDefaultPaneProperties = function (pane) {
            if (pane.collapsible === undefined) {
               pane.collapsible = false;
            }
            if (pane.resizable === undefined) {
               pane.resizable = true;
            }
            return pane;
         };

      }]
   };
}]);
;'use strict';

vui.angular.stackView

.controller('VuiStackBlockController', ['$scope', '$element', 'jsUtils', function ($scope, $element, jsUtils) {

   var self = this;

   //---------------------------------------------------------------------------
   // Constants

   var ICON_LEFT_POSITION = 6;
   var ICON_LEFT_POSITION_WITH_CARET = 16;
   var LABEL_LEFT_PADDING = 16;
   var LABEL_LEFT_PADDING_WITH_ICON = 26;
   var LABEL_LEFT_PADDING_WITH_CARET_ICON = 36;
   var NESTING_INDENT = 16;
   var DEPTH_CLASS = 'depth';
   var STACK_BLOCK_PARENT_CLASS = 'stack-view-parent';
   var STACK_BLOCK_ROW_CLASS = 'stack-view-row';

   //-----------------------------------------------------------------------------------
   // Internal API

   /**
    * Handles keyboard expand and collapse commands.
    * @param $event
    */
   var handleKeypress = function ($event) {
      switch ($event.which) {
         case 13:
            // enter pressed
            setExpanded(!$scope.sb.expanded);
            break;
         case 32:
            // Space bar pressed
            // Prevent scrolling
            $event.preventDefault();
            setExpanded(!$scope.sb.expanded);
            break;
         case 39:
            // Right
            // Prevent scrolling
            $event.preventDefault();
            setExpanded(true);
            break;
         case 37:
            // Left
            // Prevent scrolling
            $event.preventDefault();
            setExpanded(false);
            break;
      }
   };

   /**
    * Expands or collapses this stack block, if it has children.
    * @param expand
    */
   var setExpanded = function (expand) {
      if (!hasChildren($scope.sb, $scope.childrenToTransclude)) {
         return;
      }
      $scope.sb.expanded = expand;
   };

   /**
    * @returns {{}}
    *    object containing classes suitable for this stack blocks status as a parent/
    *    child with or without an icon.
    */
   var getStackBlockCssClasses = function () {
      var cssClasses = {};
      // Set the caret direction.
      cssClasses.open = getExpanded();

      var rowClass = null;
      if (hasChildren($scope.sb, $scope.childrenToTransclude)) {
         rowClass = STACK_BLOCK_PARENT_CLASS;
      } else {
         rowClass = STACK_BLOCK_ROW_CLASS;
      }
      if ($scope.sb.iconClass) {
         rowClass += '-i';
      }
      cssClasses[rowClass] = true;
      return cssClasses;
   };

   /**
    * @returns {string}
    *    Returns the class appropriate for the label cell at this nesting level.
    */
   var getLabelCellClass = function () {
      var depth = getTreeDepth();
      if (depth === 0) {
         return null;
      }
      return DEPTH_CLASS + '-' + depth;
   };

   /**
    * @returns {{padding-left: string}}
    *    object containing a 'padding-left' style with the correct value for
    *    this blocks nesting level.
    */
   var createLabelCellStyles = function () {
      var normalLeftPadding = getLabelLeftPadding($scope.sb, $scope.childrenToTransclude);
      var depth = getTreeDepth();
      var indent = depth * NESTING_INDENT;
      return {
         'padding-left': normalLeftPadding + indent + 'px'
      };
   };

   /**
    * @returns {{left: string}}
    *    object containing a 'left' style with the correct indentation for this
    *    blocks nesting level.
    */
   var createIconSpanStyles = function () {
      var normalLeftPosition = getIconLeftPosition($scope.sb, $scope.childrenToTransclude);
      var depth = getTreeDepth();
      var indent = depth * NESTING_INDENT;
      return {
         'left': normalLeftPosition + indent + 'px'
      };
   };

   /**
    * @returns {boolean}
    *    true if the expanded content should be shown, else false.
    */
   var showExpandedContent = function () {
      if (jsUtils.isUndefinedOrNull($scope.contentExpanded) &&
            jsUtils.isEmpty($scope.contentExpandedToTransclude)) {
         return false;
      }
      return getExpanded();
   };

   /**
    * @returns {boolean}
    *    returns true if the children should be shown, else false.
    */
   var getExpanded = function () {
      if (!hasChildren($scope.sb, $scope.childrenToTransclude)) {
         return false;
      }
      return $scope.sb.expanded;
   };

   /**
    * @returns {number}
    *    number indicating the depth of the nesting stack block
    */
   var getTreeDepth = function () {
      var depth = $element.attr('data-tree-depth');
      if (!depth) {
         return 0;
      }
      return parseInt(depth);
   };

   /**
    * @param clone
    * @param sb
    * @returns {string}
    *    the icon CSS class for the stack block
    */
   var getIconClass = function (clone, sb) {
      if (jsUtils.hasProperty(sb, 'iconClass')) {
         // JS API takes precedence over transclusion.
         return sb.iconClass;
      }
      return clone.filter('span[vui-sb-label]:first').attr('vui-sb-icon-class');
   };

   /**
    * Adds the label in the clone to the template.
    * @param clone
    */
   var transcludeLabel = function (clone) {
      var labelSpan = clone.filter('span[vui-sb-label]:first');
      labelSpan.removeAttr('vui-sb-icon');
      var labelContainer = $element.children().eq(0).children().eq(0).children('p.stack-view-label');
      labelContainer.append(labelSpan);
   };

   /**
    * Adds the content div in the clone to the template.
    * @param clone
    */
   var transcludeContent = function (clone) {
      var content = clone.filter('div[vui-sb-content]:first');
      var contentContainer = $element.children().first().children().eq(1);
      contentContainer.append(content);
   };

   /**
    * Adds the content expanded div in the clone to the template.
    * @param clone
    * @returns {jQuery}
    *    the content added.
    */
   var transcludeContentExpanded = function (clone) {
      var content = clone.filter('div[vui-sb-content-expanded]:first');
      var contentContainer = $element.children().first().children().eq(2);
      contentContainer.append(content);
      return content;
   };

   /**
    * @param clone
    * @param $scope
    * @returns {Array}
    *    jQuery object containing the child stack blocks for the stack block.
    */
   var getChildren = function (clone) {
      // Children should be transcluded.
      var children = clone.filter('[vui-stack-block]');
      var treeDepth = getTreeDepth();
      var childTreeDepth = treeDepth + 1;
      // Add the tree depth attribute.
      children.attr('data-tree-depth', childTreeDepth);
      return children;
   };

   /**
    * Adds the given child stack blocks to this stack blocks DOM.
    * @param iElement
    * @param children
    */
   var transcludeChildren = function (iElement, children) {
      if (children.length === 0) {
         return;
      }
      var wrapper = getChildWrapper(iElement);
      wrapper.append(children);
   };

   self.handleKeypress = handleKeypress;
   self.setExpanded = setExpanded;
   self.getStackBlockCssClasses = getStackBlockCssClasses;
   self.getLabelCellClass = getLabelCellClass;
   self.createLabelCellStyles = createLabelCellStyles;
   self.createIconSpanStyles = createIconSpanStyles;
   self.showExpandedContent = showExpandedContent;
   self.getExpanded = getExpanded;
   self.getTreeDepth = getTreeDepth;
   self.getIconClass = getIconClass;
   self.transcludeLabel = transcludeLabel;
   self.transcludeContent = transcludeContent;
   self.transcludeContentExpanded = transcludeContentExpanded;
   self.getChildren = getChildren;
   self.transcludeChildren = transcludeChildren;

   //------------------------------------------------------------------------------------
   // Private.

   /**
    * @param sbState
    * @param childrenToTransclude
    * @returns {boolean}
    *    true if the stack block has any children, else false.
    */
   var hasChildren = function (sbState, childrenToTransclude) {
      return !jsUtils.isEmpty(sbState.children) ||
         !jsUtils.isEmpty(childrenToTransclude);
   };

   /**
    * @returns {number}
    *    The left position for the icon.
    */
   var getIconLeftPosition = function (sbState, childrenToTransclude) {
      if (hasChildren(sbState, childrenToTransclude)) {
         return ICON_LEFT_POSITION_WITH_CARET;
      } else {
         return ICON_LEFT_POSITION;
      }
   };

   /**
    * @returns {number}
    *    The left padding value for the label.
    */
   var getLabelLeftPadding = function (sbState, childrenToTransclude) {
      if (sbState.iconClass) {
         if (hasChildren(sbState, childrenToTransclude)) {
            return LABEL_LEFT_PADDING_WITH_CARET_ICON;
         } else {
            return LABEL_LEFT_PADDING_WITH_ICON;
         }
      } else {
         return LABEL_LEFT_PADDING;
      }
   };

   /**
    * @param iElement
    * @returns {jQuery}
    *    jQuery object containing the parent of the stack blocks children.
    */
   var getChildWrapper = function (iElement) {
      return iElement.children('div:last-child');
   };
}]);
;'use strict';

vui.angular.stackView

.controller('VuiStackViewController', ['$scope', '$element', 'jsUtils', function ($scope, $element, jsUtils) {

   var self = this;

   //-----------------------------------------------------------------------------------
   // Internal API

   /**
    * Handles up and down cursor key presses by focusing previous/next label cell.
    * @param $event
    */
   var handleKeypress = function ($event) {
      var direction = getDirection($event.which);
      if (direction === 0) {
         // key pressed was not up or down arrow.
         return;
      }

      var currentlyFocused = document.activeElement;

      if ((currentlyFocused !== $element.children()[0]) &&
         !isLabelCell(jQuery(currentlyFocused))) {
         return;
      }

      // Prevent scrolling
      $event.preventDefault();

      // If the stack view is focused, start at last focused cell.
      if (currentlyFocused === $element.children()[0]) {
         if (!$scope.lastFocused) {
            $scope.lastFocused = getLabelCell(getChildStackBlocks($element).first());
         }
         $scope.lastFocused.focus();
         return;
      }

      var targetLabelCell = [];
      if (direction < 0) {
         targetLabelCell =
            getPreviousStackBlockLabelCell(jQuery(currentlyFocused));
      } else {
         targetLabelCell =
            getNextStackBlockLabelCell(jQuery(currentlyFocused));
      }
      if (targetLabelCell.length === 1) {
         $scope.lastFocused = targetLabelCell.focus();
      }
   };

   /**
    * @param clone
    *    The content being transcluded.
    * @returns {jQuery}
    *    jQuery object containing the transcluded header content.
    */
   var getTranscludeHeader = function (clone) {
      return clone.filter('header:first');
   };

   /**
    * @param clone
    *    The content being transcluded.
    * @returns {jQuery}
    *    jQuery object containing the transcluded stack blocks.
    */
   var getTranscludeStackBlocks = function (clone) {
      return clone.filter('[vui-stack-block]').attr('vui-tree-depth', 0);
   };

   /**
    * If 'header' contains header content and the scope variable does not
    * (javascript API takes precedence over transclusion as spec'd in user docs),
    * removes this directive's header template button and replaces it with the
    * transcluded content.
    * @param header
    *    The transcluded content to use as a header.
    * @param iElement
    * @param svState
    *    Scope var specifying this stack view.
    */
   var transcludeHeader = function (header, iElement, svState) {
      if (!jsUtils.isUndefinedOrNull(svState.header) ||
         header.length !== 1) {
         return;
      }

      // Note the title is not being properly transcluded since we are just extracting
      // the text instead of appending the compiled, linked clone to the template.
      svState.header = {
         title: header.contents().filter(function () {
            // Filter out everything but text nodes.
            return this.nodeType === 3;
         }).text()
      };
      var buttonDom = header.children('button');
      buttonDom.addClass('edit').addClass('btn');
      var headerDom = iElement.find('.stack-view-header');
      headerDom.remove('button');
      headerDom.append(buttonDom);
   };

   /**
    * Appends the given stack blocks to the end of this stack view's children.
    * @param sbs
    * @param iElement
    */
   var transcludeStackBlocks = function (sbs, iElement) {
      iElement.find('.stack-view-table').append(sbs);
   };

   self.handleKeypress = handleKeypress;
   self.getTranscludeHeader = getTranscludeHeader;
   self.getTranscludeStackBlocks = getTranscludeStackBlocks;
   self.transcludeHeader = transcludeHeader;
   self.transcludeStackBlocks = transcludeStackBlocks;

   //------------------------------------------------------------------------------------
   // Private


   /**
    * @param keyCode
    * @returns {number}
    *    -1 if keyCode is 38 == up
    *    1 if keyCode is 40 == down
    *    else 0.
    */
   var getDirection = function (keyCode) {
      if (keyCode === 38) {
         // Up cursor key
         return -1;
      }
      if (keyCode === 40) {
         // Down cursor key
         return 1;
      }
      return 0;
   };

   /**
    * @param sb
    * @returns {jQuery}
    *    jQuery object holding the label cell of the given stack block.
    */
   var getLabelCell = function (sb) {
      return sb.children('div:first').children('div:first');
   };

   /**
    * @param sbOrSv
    *    jQuery object containing one stack block or one stack view.
    * @returns {jQuery | []}
    *    jQuery object holding the child stack blocks of the given stack block or
    *    stack view.
    */
   var getChildStackBlocks = function (sbOrSv) {
      if (!jsUtils.isUndefinedOrNull(sbOrSv.attr('vui-stack-block'))) {
         // Element is stack block
         return sbOrSv.children('div:last').children();
      }
      if (!jsUtils.isUndefinedOrNull(sbOrSv.attr('vui-stack-view'))) {
         // Element is stack view
         return sbOrSv.children().children().last().children();
      }
      // Unknown element
      return [];
   };

   /**
    * @param focusedElement
    * @returns {jQuery}
    *    jQuery object holding the label cell immediately prior to focusedElement,
    *    if there is one.
    *
    *    Here 'prior' means visually in the flattened, vertical dimension of the
    *    stack view, not in the tree structure.
    */
   var getPreviousStackBlockLabelCell = function (focusedElement) {
      var sb = getParentStackBlock(focusedElement);

      if (!isLabelCell(focusedElement)) {
         return getLabelCell(sb);
      }

      var previousSibling = getSiblingStackBlock(sb, true);

      if (previousSibling.length === 1) {
         var targetSb = previousSibling;
         var sbState = angular.element(previousSibling).isolateScope().sb;
         // while previousSibling is expanded, travel to last child.
         while (sbState.expanded) {
            var childSbs = getChildStackBlocks(targetSb);
            targetSb = childSbs.last();
            sbState = angular.element(targetSb).isolateScope().sb;
         }
         return getLabelCell(targetSb);
      }

      // If there is no previous sibling, then the previous label cell will
      // be the label cell of the parent stack block.
      var grandParentSb = getParentStackBlock(sb);
      return getLabelCell(grandParentSb);
   };

   /**
    * @param focusedElement
    * @returns {jQuery | []}
    *    jQuery object holding the label cell immediately after focusedElement,
    *    if there is one.
    *
    *    Here 'after' means visually in the flattened, vertical dimension of the
    *    stack view, not in the tree structure.
    */
   var getNextStackBlockLabelCell = function (focusedElement) {
      var sb = getParentStackBlock(focusedElement);

      var sbState = angular.element(sb).isolateScope().sb;
      if (sbState.expanded) {
         // Next label cell will be the first child of this stack block.
         return getLabelCell(getChildStackBlocks(sb).first());
      }

      var targetSb = [];

      // Next cell will be the sibling or ancestor cell immediately following sb.
      while (sb.length === 1) {
         targetSb = getSiblingStackBlock(sb);
         if (targetSb.length === 1) {
            return getLabelCell(targetSb);
         }
         // Try next level up.
         sb = getParentStackBlock(sb);
      }

      return targetSb;
   };

   /**
    * @param element
    * @returns {boolean}
    *    true if element is a stack block label cell, else false.
    */
   var isLabelCell = function (element) {
      // element is label cell if it has class stack-view-cell
      // and no previous sibling does.
      if (!element.hasClass('stack-view-cell')) {
         return false;
      }
      var prevSiblingCells = element.prevAll('.stack-view-cell');
      return prevSiblingCells.length === 0;
   };

   /**
    * @param sb
    * @param prev
    * @returns {jQuery | []}
    *    if prev is true, jQuery object containing the previous sibling, else
    *    the next sibling or empty if no such sibling.
    */
   var getSiblingStackBlock = function (sb, prev) {
      var parent = getStackBlockContainer(sb);
      var sbs = parent.children();
      var index = sbs.index(sb);
      if (index === -1) {
         return [];
      }
      var targetIndex = prev ? index - 1 : index + 1;
      if (targetIndex === -1) {
         return [];
      }
      return sbs.eq(targetIndex);
   };

   /**
    * @param element
    * @returns {jQuery}
    *    jQuery object containing the parent stack block of the given element, if
    *    element is a child of a stack block.
    */
   var getParentStackBlock = function (element) {
      // Note - can't use closest() here as would return element in case element
      // is stack block.
      return element.parents('div[vui-stack-block]').first();
   };

   /**
    * @param sb
    * @returns {jQuery}
    *    jQuery object holding the div that holds the given stack block.
    */
   var getStackBlockContainer = function (sb) {
      var grandParentSB = getParentStackBlock(sb);
      if (grandParentSB.length === 1) {
         // Parent is a stack block, go to div holding children
         return grandParentSB.children('div:last');
      }
      // Top level stack block, parent is the table element
      return sb.closest('div.stack-view-table');
   };

}]);
;'use strict';

vui.angular.stackView

/**
 * @ngdoc object
 * @name vui.angular.stackView:StackBlockItem
 * @module vui.angular.stackView
 *
 * @description
 *    Configuration object for
 *    {@link vui.angular.stackView.directive:vuiStackBlock vuiStackBlock}
 *    directive.
 *
 *    Contains properties to specify the content of a stack block.
 *
 *    Example:
 *    ```js
 *    var stackViewOptions = {
 *                             iconClass: 'foo-icon',
 *                             label: 'Foo Configuration',
 *                             content: 'Foo options',
 *                             contentExpanded: '<span>Enable foo <input type='checkbox'></span>',
 *                             children: [ sb1, sb2, ...],
 *                             expanded: false
 *                          };
 *    ```
 *
 * @property {string} [iconClass]
 *    Type: `string` **|** Optional
 *
 *    CSS class name of an icon to display for the node.
 *
 *    **Note**: Please refer to the vui-bootstrap documentation for a list of available
 *    icon classes. A custom class specifying a 16x16 icon can also be used.
 *
 * @property {string} [label]
 *    Type: `string` **|** Optional
 *
 *    The localized label for this stack block.
 *
 * @property {string} [content]
 *    Type: `string` **|** Optional
 *
 *    Valid HTML to render as the stack block content.
 *
 *    If `contentExpanded` is not specified the content given here is shown in both
 *    the collapsed and expanded states of the stack block.
 *
 *    **Note** that AngularJs will not compile or link this HTML to any scope. If you need
 *    this kind of functionality, use the transclusion API instead - see
 *    {@link vui.angular.stackView.directive:vuiStackBlock vuiStackBlock}.
 *
 * @property {string} [contentExpanded]
 *    Type: `string` **|** Optional
 *
 *    Valid HTML to render as the stack block content when in the expanded state.
 *
 *    If not specified, the `content` property gives the contents to show in the
 *    expanded sate.
 *
 *    **Note** that AngularJs will not compile or link this HTML to any scope. If you need
 *    this kind of functionality, use the transclusion API instead - see
 *    {@link vui.angular.stackView.directive:vuiStackBlock vuiStackBlock}.
 *
 * @property {StackBlockItem[]} [children]
 *    Type: `Array.<StackBlockItem>` **|** Optional
 *
 *    Optional array of child stack blocks.
 *
 *    Assigning a value of length greater than zero makes this stack block *expandable*.
 *
 *    When expanded, the stack block shows its child stack blocks underneath its content,
 *    indented slightly to the right.
 *
 *    A stack block may be expanded by clicking on its label, or by setting the
 *    `expanded` property to `true`.
 *
 * @property {boolean} expanded
 *    Type: `boolean`
 *
 *    Specifies whether the stack block is in the expanded or collapsed state.
 *
 *    Toggling this value causes the stack block to grow or shrink into its expanded or
 *    collapsed state, respectively. Setting this property has no effect if the stack block
 *    contains no children.
 */

/**
 * @ngdoc directive
 * @name vui.angular.stackView.directive:vuiStackBlock
 * @module vui.angular.stackView
 * @restrict A
 * @element div
 * @scope
 *
 * @description
 *    Directive to create a stack block container. For proper usage, this directive
 *    should be the child directive of the
 *    {@link vui.angular.stackView.directive:vuiStackView vui-stack-view} directive.
 *
 *    A stack block consists of an optional icon and a localized label, followed by some
 *    content, laid out in two horizontal columns.
 *
 *    The content of a stack block can be any arbitrary HTML, although see the
 *    {@link http://developercenter.vmware.com/web/standards/-/stack-view UE page}
 *    for usage guidelines.
 *
 *    Stack blocks are recursive structures that may include child stack blocks.
 *    These children are hidden until the stack block is clicked and enters its
 *    *expanded* state - a design  pattern called *progressive disclosure* that helps
 *    shield the user from non-essential detail. Clicking on the block a second time
 *    hides the children, returning the block to its *collapsed* state. The content
 *    of a stack block may be different in the collapsed and expanded states. Only
 *    stack blocks containing child stack blocks have an expanded state. Such stack
 *    blocks are visually indicated by a caret symbol.
 *
 *    Child stack blocks are shown below the stack block label/content pair, indented
 *    slightly to the right.
 *
 *    Stack blocks may be arbitrarily nested although again, see the
 *    {@link http://developercenter.vmware.com/web/standards/-/stack-view UE page}
 *    for usage guidelines.
 *
 *    The content of a stack block may be specified via the `vuiStackBlock` parameter
 *    (see below), via transclusion, or via a mixture of the two.
 *
 *    When specifying content via both the `vuiStackBlock` parameter and transclusion,
 *    the content of the `vuiStackBlock` parameter takes precedence. Any transcluded
 *    child stack blocks will appear below child stack blocks specified by the
 *    `vuiStackBlock` parameter.
 *
 *    # Permitted Transclusion Content:
 *
 *    In this order:
 *    1. a `<span vui-sb-label>` element containing the text for the stack block label.
 *    The span may optionally include a `vui-sb-icon-class` attribute whose value is the
 *    CSS class specifying the icon for the stack block.
 *    2. a `<div vui-sb-content>` element specifying the content of the stack block.
 *    3. an optional `<div vui-sb-content-expanded>` element specifying the content of the
 *    stack block in the expanded state. If this div is missing, the content is identical
 *    in the collapsed and expanded states.
 *    4. zero or more `<div vui-stack-block>` elements specifying the children of this
 *       stack block.
 *
 * @param {StackBlockItem} vuiStackBlock
 *    Configuration object for the stack block. It is of type
 *    {@link vui.angular.stackView:StackBlockItem StackBlockItem}.
 *
 * @example
 <example module="app">
 <file name="app.js">
 'use strict';
 var app = angular.module('app', ['vui.angular.stackView', 'vui.angular.progressBar', 'vui.angular.dropdown']);

 app.controller('StackViewController', function ($scope, $interval) {
      $scope.declarativeStackView = {
         //Uncomment this header to use this instead of the header defined in
         // the HTML view.

         //header: {
         //   title: 'Hello World from json'
         //},

         // Specify that the CPU block that is defined declaritively in the HTML view, should
         // start out expanded. Provide its label here to bind to.
         cpuBlock: {
            expanded: true,
            label: 'CPU Label bound to user scope'
         },

         cpuCoresBlock: {
            progressBarOptions: {
               percent: 0
            }
         },

         // Add some children into the storage block that is defined declaritively in HTML view.
         socketsBlock: {
            children: [{
               label: 'Datastore Clusters',
               iconClass: 'vui-icon-info',
               content: 'host.hardware.storage.datastoreClusters'
            }, {
               label: 'Datastores',
               iconClass: 'vui-icon-success',
               content: 'host.hardware.storage.datastores'
            }, {
               label: 'Physical Adapters',
               iconClass: 'vui-icon-warning',
               content: 'host.hardware.storage.physicalAdapters'
            }]
         },

         capacityBlock: {
            dropdownOptions: {
               data: ['pentium', 'amd', 'acme'],
               width: '300px',
               placeHolder: 'Choose a core'
            }
         },

         // Add an expanded content state for the processor block that is defined
         // declaritively in HTML view.
         processorBlock: {
            contentExpanded: '<input type="text" placeholder="host.hardware.cpu.processorType">'
         }
      };

      // Update progress bar in cpu cores stack block.
      $interval(function () {
         $scope.declarativeStackView.cpuCoresBlock.progressBarOptions.percent += 1;
      }, 200, 100);   });

 </file>

 <file name="index.html">
    <div ng-controller="StackViewController">
       <div vui-stack-view="declarativeStackView">
          <!-- Header -->
          <header>Host Hardware <button onclick="console.log('edit clicked')">Edit...</button></header>

          <!-- CPU stack block -->
          <div vui-stack-block="declarativeStackView.cpuBlock">
             <!-- Icon/Label -->
             <span vui-sb-label vui-sb-icon-class="vui-icon-info">{{declarativeStackView.cpuBlock.label}}</span>
             <!-- content collapsed-->
             <div vui-sb-content>host.hardware.cpu.cores</div>
             <!-- content expanded -->
             <div vui-sb-content-expanded>
               <span>host.hardware.cpu.cores  <input type="checkbox"></span>
             </div>
             <!-- children -->
             <div vui-stack-block>
                <!-- Label -->
                <span vui-sb-label>CPU Cores</span>
                <!-- content -->
                <div vui-sb-content><div vui-progress-bar='declarativeStackView.cpuCoresBlock.progressBarOptions'></div></div>
             </div>
             <div vui-stack-block="declarativeStackBlock.socketsBlock">
             <!-- Label -->
                <span vui-sb-label>Sockets</span>
                <!-- content -->
                <div vui-sb-content>host.hardware.cpu.sockets</div>
             </div>
             <div vui-stack-block>
             <!-- Label -->
                <span vui-sb-label>Core per socket</span>
                <!-- content -->
                <div vui-sb-content>host.hardware.cpu.corePerSocket</div>
             </div>
             <div vui-stack-block>
             <!-- Label -->
                <span vui-sb-label>Logical Processors</span>
                <!-- content -->
                <div vui-sb-content>host.hardware.cpu.logicalProcessors</div>
             </div>
             <div vui-stack-block>
             <!-- Label -->
                <span vui-sb-label>Hyperthreading</span>
                <!-- content -->
                <div vui-sb-content>host.hardware.cpu.hyperthreading</div>
             </div>
             <div vui-stack-block="declarativeStackView.processorBlock">
                <!-- Icon/Label -->
                <span vui-sb-label vui-sb-icon-class="vui-icon-success">Processor Type</span>
                <!-- content -->
                <div vui-sb-content>host.hardware.cpu.processorType</div>
                <!-- children -->
                <div vui-stack-block>
                   <!-- Label -->
                   <span vui-sb-label>Capacity</span>
                   <!-- content -->
                   <div vui-sb-content><div vui-dropdown="declarativeStackView.capacityBlock.dropdownOptions"></div></div>
                </div>
                <div vui-stack-block>
                   <!-- Label -->
                   <span vui-sb-label>Used</span>
                   <!-- content -->
                   <div vui-sb-content>host.hardware.virtualFlashResource.used</div>
                </div>
                <div vui-stack-block>
                   <!-- Icon/Label -->
                   <span vui-sb-label vui-sb-icon-class="vui-icon-success">Free</span>
                   <!-- content -->
                   <div vui-sb-content>host.hardware.virtualFlashResource.free</div>
                </div>
             </div>
          </div>
       </div>
    </div>
 </file>
 </example>
 */
.directive('vuiStackBlock', ['$compile', 'jsUtils', '$sce', '$timeout', function ($compile, jsUtils, $sce, $timeout) {
   return {
      restrict: 'A',
      scope: {
         sb: '=?vuiStackBlock'
      },
      transclude: true,
      controller: 'VuiStackBlockController',
      controllerAs: 'ctrl',
      template:
         '<div ng-class="ctrl.getStackBlockCssClasses()">' +
            '<div class="stack-view-cell" ng-keydown="ctrl.handleKeypress($event)" tabindex="-1" ng-click="ctrl.setExpanded(!sb.expanded)" ng-class="ctrl.getLabelCellClass()" ng-style="ctrl.createLabelCellStyles()">' +
               '<span class="vui-icon" ng-class="sb.iconClass" ng-if="sb.iconClass" ng-style="ctrl.createIconSpanStyles()"></span>' +
               '<p style="margin-bottom: 0px" class="stack-view-label">{{sb.label}}</p>' +
            '</div>' +
            '<div class="stack-view-cell" ng-show="!ctrl.showExpandedContent()" style="width: 100%">' +
               '<div ng-if="content" ng-bind-html="content"></div>' +
            '</div>' +
            '<div class="stack-view-cell" ng-show="ctrl.showExpandedContent()">' +
               '<div ng-if="contentExpanded" ng-bind-html="contentExpanded"></div>' +
            '</div>' +
         '</div>' +
         '<div ng-show="ctrl.getExpanded()">' +
            '<div ng-repeat="child in sb.children" vui-stack-block="child" data-tree-depth="{{ctrl.getTreeDepth() + 1}}"></div>' +
         '</div>',
      compile: function (tElement) {

         // Creating this closure for the link function in the compile function allows
         // the directive to become recursive in the template. Without this angular
         // throws a stack overflow.
         var contents = tElement.contents().remove();
         var compiledContents;

         return function ($scope, iElement, iAttrs, ctrl, transclude) {
            var kendoResizeTimeoutPromise = null;

            // --------------------------------------------------------------------------
            // Set up scope

            // We'll populate this later in the transclusion function.
            $scope.childrenToTransclude = [];

            if (jsUtils.isUndefinedOrNull($scope.sb)) {
               $scope.sb = {};
            }
            if (jsUtils.isUndefinedOrNull($scope.sb.expanded)) {
               $scope.sb.expanded = false;
            }

            $scope.$watch('sb.content', function (newVal) {
               if (!jsUtils.isEmpty(newVal)) {
                  $scope.content = $sce.trustAsHtml(newVal);
               }
            });

            $scope.$watch('sb.contentExpanded', function (newVal) {
               if (!jsUtils.isEmpty(newVal)) {
                  $scope.contentExpanded = $sce.trustAsHtml(newVal);
               }
            });

            $scope.$watch('sb.expanded', function () {
               // ng-show & ng-hide are executed in $$postDigest so we are going
               // to resize on next frame. Resize all widgets in the app because
               // expanding/collapsing the stack-block might add/remove scrollers
               if (kendoResizeTimeoutPromise !== null) {
                  return;
               }

               kendoResizeTimeoutPromise = $timeout(function () {
                  kendoResizeTimeoutPromise = null;
                  kendo.resize($(document));
               }, 0, false);
            });

            $scope.$on('$destroy', function () {
               if (kendoResizeTimeoutPromise === null) {
                  return;
               }

               $timeout.cancel(kendoResizeTimeoutPromise);
               kendoResizeTimeoutPromise = null;

            });

            //---------------------------------------------------------------------------
            // Compilation

            if (!compiledContents) {
               // Not compiled yet, create linking function now.
               compiledContents = $compile(contents);
            }

            // Link the directive, adding the content to the top of the container.
            compiledContents($scope, function (clone) {
               iElement.prepend(clone);
            });

            //---------------------------------------------------------------------------
            // Transclusion

            transclude(function (clone) {
               // Note the icon is not really transcluded - so changes to the icon
               // in users controller will not be reflected.
               $scope.sb.iconClass = ctrl.getIconClass(clone, $scope.sb);
               if (!$scope.sb.label) {
                  ctrl.transcludeLabel(clone);
               }
               if (!$scope.content) {
                  ctrl.transcludeContent(clone);
               }
               if (!$scope.contentExpanded) {
                  $scope.contentExpandedToTransclude = ctrl.transcludeContentExpanded(clone);
               }
               $scope.childrenToTransclude = ctrl.getChildren(clone, $scope);
               ctrl.transcludeChildren(iElement, $scope.childrenToTransclude);
            });

         };
      }
   };
}]);
;'use strict';

vui.angular.stackView

/**
 * @ngdoc object
 * @name vui.angular.stackView:StackViewOptions
 * @module vui.angular.stackView
 *
 * @description
 *    Configuration object for
 *    {@link vui.angular.stackView.directive:vuiStackView vuiStackView}
 *    directive.
 *    Contains properties to specify a stack view and populate it with
 *    child stack blocks.
 *
 *    Example:
 *    ```js
 *    var stackViewOptions = {
 *                             header: {
 *                                title: 'Hardware Config',
 *                                button: action
 *                             },
 *                             blocks: [ sb1, sb2, sb3, ... ]
 *                          };
 *    ```
 *
 * @property {object} [header]
 *    Type: `object` **|** Optional
 *
 *    Specifies an optional header for the stack view.
 *
 *    The object should contain an optional 'title' property giving the localized title
 *    of the stack view and an optional 'action' property of type
 *    {@link vui.angular.actions:Action Action} specifying a button to take some user
 *    defined action.
 *
 *    Example:
 *    ```js
 *    header: {
 *       title: 'Hardware Config',
 *       button: action
 *    };
 *    ```
 *    `action` is an {@link vui.angular.actions:Action Action} object.
 *
 * @property {StackBlockItem[]} [blocks]
 *    Type: `Array.<StackBlockItem>` **|** Optional
 *
 *    An array containing {@link vui.angular.stackView:StackBlockItem StackBlockItem}
 *    objects. Each item specifies a stack block in the stack view.
 *
 *    Example:
 *    ```js
 *    blocks: [ sb1, sb2, ... ];
 *    ```
 *    `sb1` and `sb2` are {@link vui.angular.stackView:StackBlockItem StackBlockItem}
 *    objects.
 */

/**
 * @ngdoc directive
 * @name vui.angular.stackView.directive:vuiStackView
 * @module vui.angular.stackView
 * @restrict A
 * @element div
 * @scope
 *
 * @description
 *    Directive to create a stack view container.
 *
 *    A stack view is a container for displaying label/content pairs called stack blocks
 *    in a vertical stack. Each stack block is contained within a child
 *    {@link vui.angular.stackView.directive:vuiStackBlock vui-stack-block} directive.
 *
 *    The content of the stack view may be specified via the *vuiStackView* parameter
 *    (see below), via transclusion, or via a mixture of the two.
 *
 *    When specifying content via both the `vuiStackView` parameter and transclusion,
 *    the content of the `vuiStackView` parameter takes precedence. Any transcluded
 *    stack blocks will appear below stack blocks specified by the `vuiStackView` parameter.
 *
 *    # Permitted Transclusion Content:
 *
 *    In this order:
 *    1. an optional `<header>` element to specify the header, containing
 *      1. an optional title as a string for the stack view
 *      2. an optional `<button>` element for taking some user action.
 *    2. zero or more `<div vui-stack-block>` elements.
 *
 * @param {StackViewOptions} vuiStackView
 *    Configuration object for the stack view. It is of type
 *    {@link vui.angular.stackView:StackViewOptions StackViewOptions}.
 *
 * @example
 <example module="app">
 <file name="app.js">
 'use strict';
 var app = angular.module('app', ['vui.angular.stackView']);

 app.controller('StackViewController', function ($scope) {
      $scope.declarativeStackView = {
         //Uncomment this header to use this instead of the header defined in
         // the HTML view.

         //header: {
         //   title: 'Hello World from json'
         //},

         // Specify that the CPU block that is defined declaritively in the HTML view, should
         // start out expanded.
         cpuBlock: {
            expanded: true
         },

         // Add some children into the storage block that is defined declaritively in HTML view.
         storageBlock: {
            children: [{
               label: 'Datastore Clusters',
               iconClass: 'vui-icon-info',
               content: 'host.hardware.storage.datastoreClusters'
            }, {
               label: 'Datastores',
               iconClass: 'vui-icon-success',
               content: 'host.hardware.storage.datastores'
            }, {
               label: 'Physical Adapters',
               iconClass: 'vui-icon-warning',
               content: 'host.hardware.storage.physicalAdapters'
            }]
         },

         // Add an expanded content state for the processor block that is defined
         // declaritively in HTML view.
         processorBlock: {
            contentExpanded: '<input type="text" placeholder="host.hardware.cpu.processorType">'
         }
      };
      // These will be the first blocks, the ones defined in the HTML view will appear
      // after these.
      $scope.declarativeStackView.blocks = [
         {
            label: 'Memory',
            iconClass: 'vui-icon-info',
            content: 'host.hardware.memory.total' + 'MB'
         }, {
            label: 'Virtual Flash Resource',
            iconClass: 'vui-icon-success',
            content: 'host.hardware.virtualFlashResource.used'
         }, {
            label: 'Networking',
            iconClass: 'vui-icon-warning',
            content: 'host.hardware.networking.hostlabel'
         }
      ];
   });
 </file>

 <file name="index.html">
    <div ng-controller="StackViewController">
       <div vui-stack-view="declarativeStackView">
          <!-- Header -->
          <header>Host Hardware <button onclick="console.log('edit clicked')">Edit...</button></header>

          <!-- Model stack block -->
          <div vui-stack-block>
             <!-- Label -->
             <span vui-sb-label>Model</span>
             <!-- content -->
             <div vui-sb-content>host.hardware.model</div>
          </div>

          <!-- Storage stack block -->
          <div vui-stack-block="declarativeStackView.storageBlock">
             <!-- Label -->
             <span vui-sb-label>Storage</span>
             <!-- content collapsed -->
             <div vui-sb-content>host.hardware.storage.datastoreClusters</div>
             <!-- content expanded -->
             <div vui-sb-content-expanded>
                <p>host.hardware.storage.datastoreClusters</p>
                <br/>
                <p>Some further long-winded explanation about what storage is in the context of a cluster, formatted by a paragraph DOM element.</p>
             </div>
          </div>

          <!-- Manufacturer stack block -->
          <div vui-stack-block>
             <!-- Label -->
             <span vui-sb-label>Manufacturer</span>
             <!-- content -->
             <div vui-sb-content>host.hardware.manufacturer</div>
          </div>

          <!-- CPU stack block -->
          <div vui-stack-block="declarativeStackView.cpuBlock">
             <!-- Icon -->
             <span vui-sb-label vui-sb-icon-class="vui-icon-info">CPU</span>
             <!-- content collapsed-->
             <div vui-sb-content>host.hardware.cpu.cores</div>
             <!-- content expanded -->
             <div vui-sb-content-expanded>
               <span>host.hardware.cpu.cores  <input type="checkbox"></span>
             </div>
             <!-- children -->
             <div vui-stack-block>
                <!-- Label -->
                <span vui-sb-label>CPU Cores</span>
                <!-- content -->
                <div vui-sb-content>host.hardware.cpu.cores</div>
             </div>
             <div vui-stack-block>
             <!-- Label -->
                <span vui-sb-label>Sockets</span>
                <!-- content -->
                <div vui-sb-content>host.hardware.cpu.sockets</div>
             </div>
             <div vui-stack-block>
             <!-- Label -->
                <span vui-sb-label>Core per socket</span>
                <!-- content -->
                <div vui-sb-content>host.hardware.cpu.corePerSocket</div>
             </div>
             <div vui-stack-block>
             <!-- Label -->
                <span vui-sb-label>Logical Processors</span>
                <!-- content -->
                <div vui-sb-content>host.hardware.cpu.logicalProcessors</div>
             </div>
             <div vui-stack-block>
             <!-- Label -->
                <span vui-sb-label>Hyperthreading</span>
                <!-- content -->
                <div vui-sb-content>host.hardware.cpu.hyperthreading</div>
             </div>
             <div vui-stack-block="declarativeStackView.processorBlock">
                <!-- Icon/Label -->
                <span vui-sb-label vui-sb-icon-class="vui-icon-success">Processor Type</span>
                <!-- content -->
                <div vui-sb-content>host.hardware.cpu.processorType</div>
                <!-- children -->
                <div vui-stack-block>
                   <!-- Label -->
                   <span vui-sb-label>Capacity</span>
                   <!-- content -->
                   <div vui-sb-content>host.hardware.virtualFlashResource.capacity</div>
                </div>
                <div vui-stack-block>
                   <!-- Label -->
                   <span vui-sb-label>Used</span>
                   <!-- content -->
                   <div vui-sb-content>host.hardware.virtualFlashResource.used</div>
                </div>
                <div vui-stack-block>
                   <!-- Icon -->
                   <span vui-sb-label vui-sb-icon-class="vui-icon-success">Free</span>
                   <!-- content -->
                   <div vui-sb-content>host.hardware.virtualFlashResource.free</div>
                </div>
             </div>
          </div>
       </div>
    </div>
 </file>
 </example>
 */
.directive('vuiStackView', ['jsUtils', function (jsUtils) {
   return {
      restrict: 'A',
      scope: {
         sv: '=?vuiStackView'
      },
      transclude: true,
      controller: 'VuiStackViewController',
      controllerAs: 'ctrl',
      template:
         '<div class="vui-stack-view" tabindex="0" ng-keydown="ctrl.handleKeypress($event)">' +
            '<div class="stack-view-header" ng-show="sv.header">' +
               '<span class="title">{{sv.header.title}}</span>' +
                // TODO mcritch: replace with vui-action-button once factored out of vui-action-bar
               '<button class="edit btn" ng-if="sv.header.action" ng-click="sv.header.action.onClick($event, sv.header.action)">{{sv.header.action.label}}</button>' +
            '</div>' +
            '<div class="stack-view-table">' +
               '<div ng-repeat="block in sv.blocks" vui-stack-block="block"></div>' +
            '</div>' +
         '</div>',
      link: function ($scope, iElement, iAttrs, ctrl, transclude) {

         //------------------------------------------------------------------------------
         // Private state.

         // stack blocks to transclude from user's template.
         var childrenToTransclude = [];
         // header to transclude from user's template
         var headerToTransclude = [];

         //------------------------------------------------------------------------------
         // Normalize scope

         if (jsUtils.isUndefinedOrNull($scope.sv)) {
            $scope.sv = {};
         }

         //-----------------------------------------------------------------------------
         // Transclusion

         transclude(function (clone) {
            headerToTransclude = ctrl.getTranscludeHeader(clone);
            childrenToTransclude = ctrl.getTranscludeStackBlocks(clone);
         });

         ctrl.transcludeHeader(headerToTransclude, iElement, $scope.sv);
         ctrl.transcludeStackBlocks(childrenToTransclude, iElement);
      }
   };
}]);
;'use strict';
vui.angular.tabs

/**
 * @ngdoc object
 * @name vui.angular.tabs:TabOptions
 * @module vui.angular.tabs
 *
 * @description
 *    Configuration object for
 *    {@link vui.angular.tabs.directive:vuiTabs vuiTabs} directive.
 *
 *    Example:
 *    ```js
 *    var tabOptions = {
 *                         tabType: vuiConstants.tabs.type.PRIMARY,
 *                         tabs: [ ... ]
 *                     };
 *    ```
 *
 * @property {string} tabType
 *    Type: `string`
 *
 *    Defines the presentation style of the tabs.
 *
 *    Legal values:
 *
 *    - **`vuiConstants.tabs.type.PRIMARY`** : Use for a top-level tab style.
 *    - **`vuiConstants.tabs.type.SECONDARY`** : Use for a tab bar style.
 *    - **`vuiConstants.tabs.type.TERTIARY`** : Use for table of contents
 *       (TOC) style.
 *
 * @property {Array.<Object>} tabs
 *    Type: `Array.<Object>`
 *
 *    An array of {@link vui.angular.tabs.TabOptions:Tab Tab} objects.
 *
 *    Example:
 *    ```js
 *    tabs: [ summaryTab, monitorTab ]
 *    ```
 *    `summaryTab` and `monitorTab` are
 *    {@link vui.angular.tabs.TabOptions:Tab Tab} objects.
 *
 * @property {number} [selectedTabIndex]
 *    Type: `number` **|** Optional **|** Default: `0`
 *
 *    Indicates the zero-based index of the tab that is selected.
 *    If no index is specified, the first tab will be selected.
 */

/**
 * @ngdoc object
 * @name vui.angular.tabs.TabOptions:Tab
 * @module vui.angular.tabs
 *
 * @description
 *     Contains properties to define a tab in
 *    {@link vui.angular.tabs.directive:vuiTabs vuiTabs} directive.
 *
 *    Example:
 *    ```js
 *    var summaryTab = {
 *                         label: 'Summary',
 *                         tooltipText: 'View summary',
 *                         contentUrl: 'summaryPage.html',
 *                         onClick: clickFunction
 *                     };
 *    ```
 *
 * @property {string} label
 *    Type: `string`
 *
 *    The display text for the tab.
 *    <script>$('.properties .label').removeClass('label');</script>
 *
 * @property {string} [iconClass]
 *    Type: `string` **|** Optional
 *
 *    CSS class name of an icon to display for the tab.
 *
 *    **Note:** Please refer to the vui-bootstrap documentation for a list of
 *    available icon classes. A custom class specifying a 16x16 icon can also
 *    be used.
 *
 * @property {string} [tooltipText]
 *    Type: `string` **|** Optional
 *
 *    The text that will be displayed in the tab tooltip on hover.
 *    If content is not specified,
 *    {@link vui.angular.tabs.TabOptions:Tab label} text will be used as
 *    tooltipText.
 * @property {string} contentUrl
 *    Type: `string`
 *
 *    The template URL of web content that will be loaded into and displayed
 *    within the current tab's content area.
 *
 *    **Note:** The tab's content scope prototypically inherits from the
 *    scope on which {@link vui.angular.tabs.directive:vuiTabs vui-tabs}
 *    directive is declared.
 * @property {Function} onClick
 *    Type: `Function`
 *
 *    A function that will be called when the link is clicked. The click event
 *    and the{@link vui.angular.tabs.TabOptions:Tab Tab} object will be passed
 *    as parameters to the function.
 *
 *    Example:
 *    ```js
 *    onClick: function (event, tab) {
 *               ...
 *             };
 *    ```
 */

/**
 * @ngdoc directive
 * @name vui.angular.tabs.directive:vuiTabs
 * @module vui.angular.tabs
 * @restrict A
 *
 * @description
 *    Directive to create primary, secondary or tertiary type tabs.
 *
 *    The tab's content scope prototypically inherits from the
 *    scope on which {@link vui.angular.tabs.directive:vuiTabs vui-tabs}
 *    directive is declared.
 *
 * @param {TabOptions} vuiTabs
 *    {@link vui.angular.tabs:TabOptions TabOptions} Configuration object for
 *    tabs.
 *
 * @example
      <example module="app">
         <file name="app.js">
         'use strict';
         var app = angular.module('app', ['vui.angular.tabs']);

         app.controller('TabCtrl', ['$scope', 'vuiConstants',
               function ($scope, vuiConstants) {

            var pClickFunction = function (event, tab) {
               $scope.primaryTabClicked = tab.label + ' clicked';
            };

            $scope.primaryTabOptions = {
               tabs: [{
                  label: 'Getting Started',
                  iconClass: 'vui-icon-home',
                  tooltipText: 'Tab tooltip1',
                  onClick: pClickFunction,
                  contentUrl: 'tab1Content.html'
               }, {
                  label: 'Summary',
                  iconClass: 'vui-icon-info',
                  tooltipText: 'Tab tooltip2',
                  onClick: pClickFunction,
                  contentUrl: 'tab2Content.html'
               }, {
                  label: 'Monitor',
                  tooltipText: 'Tab tooltip3',
                  onClick: pClickFunction,
                  contentUrl: 'tab1Content.html'
               }, {
                  label: 'Manage',
                  iconClass: 'vui-icon-success',
                  tooltipText: 'Tab tooltip4',
                  onClick: pClickFunction,
                  contentUrl: 'tab2Content.html'
               }, {
                  label: 'Related Objects',
                  tooltipText: 'Tab tooltip5',
                  onClick: pClickFunction,
                  contentUrl: 'tab1Content.html'
               }],
               tabType: vuiConstants.tabs.type.PRIMARY,
               selectedTabIndex: 3
            };

            var sClickFunction = function (event, tab) {
               $scope.secondaryTabClicked = tab.label + ' clicked';
            };

            $scope.secondaryTabOptions = {
               tabs: [{
                  label: 'Settings',
                  tooltipText: 'Tab tooltip1',
                  onClick: sClickFunction,
                  contentUrl: 'tab1Content.html'
               }, {
                  label: 'Alarm Definitions',
                  tooltipText: 'Tab tooltip2',
                  onClick: sClickFunction,
                  contentUrl: 'tab2Content.html'
               }, {
                  label: 'Tags',
                  tooltipText: 'Tab tooltip3',
                  onClick: sClickFunction,
                  contentUrl: 'tab1Content.html'
               }, {
                  label: 'Permissions',
                  tooltipText: 'Tab tooltip4',
                  onClick: sClickFunction,
                  contentUrl: 'tab2Content.html'
               }, {
                  label: 'Schedule Tasks',
                  tooltipText: 'Tab tooltip5',
                  onClick: sClickFunction,
                  contentUrl: 'tab1Content.html'
               }],
               tabType: vuiConstants.tabs.type.SECONDARY
            };

            var tClickFunction = function (event, tab) {
               $scope.tertiaryTabClicked = tab.label + ' clicked';
            };

            $scope.tertiaryTabOptions = {
               tabs: [{
                  label: 'Tab 1',
                  tooltipText: 'Tab tooltip1',
                  onClick: tClickFunction,
                  contentUrl: 'tab1Content.html'
               }, {
                  label: 'Tab 2',
                  tooltipText: 'Tab tooltip2',
                  onClick: tClickFunction,
                  contentUrl: 'tab2Content.html'
               }, {
                  label: 'Tab 3',
                  tooltipText: 'Tab tooltip3',
                  onClick: tClickFunction,
                  contentUrl: 'tab1Content.html',
                  click: tClickFunction
               }, {
                  label: 'Tab 4',
                  tooltipText: 'Tab tooltip4',
                  onClick: tClickFunction,
                  contentUrl: 'tab2Content.html'
               }, {
                  label: 'Tab 5',
                  tooltipText: 'Tab tooltip5',
                  onClick: tClickFunction,
                  contentUrl: 'tab1Content.html'
               }],
               tabType: vuiConstants.tabs.type.TERTIARY
            };

         }]);
         </file>

         <file name="index.html">
         <div ng-controller="TabCtrl">
         <h3 class="text-info">Primary Tabs</h3>
            <div vui-tabs="primaryTabOptions"></div>
            <hr/>
            <h3 class="text-info">Secondary Tabs</h3>
            <div vui-tabs="secondaryTabOptions"></div>
            <hr/>
            <h3 class="text-info">Tertiary Tabs</h3>
            <div vui-tabs="tertiaryTabOptions"></div>
         </div>
         </file>

         <file name="tab1Content.html">
         <div><p>Lorem ipsum dolor sit amet, consectetur adipiscing elit. Fusce
               gravida felis leo, eget feugiat elit vestibulum at. Nam quis
               facilisis mauris. Phasellus vitae augue diam. Donec suscipit
               massa a massa bibendum, quis vulputate justo commodo. In
               tristique risus ac dui dapibus, nec aliquam ipsum aliquet.
               Aliquam at luctus nunc. Nam scelerisque, quam vel scelerisque
               accumsan, risus turpis euismod quam, in feugiat mi diam eget enim.
               </p></div>
         </file>

         <file name="tab2Content.html">
         <div><p>Ut non viverra odio. Aliquam mauris sapien, vehicula non quam
               rutrum, vestibulum vestibulum purus. Nam blandit libero quis est
               vehicula ullamcorper. Vestibulum ullamcorper vel sem ut posuere.
               Nunc ac tellus ligula. Duis auctor convallis diam, ut ultrices
               diam rutrum vitae. Pellentesque viverra adipiscing eros, at
               elementum nisi malesuada sed. Sed eu convallis tellus. Vivamus
               quis convallis tortor, a pellentesquenisi.
               </p></div>
         </file>

      </example>

 */
.directive('vuiTabs', ['$log', 'jsUtils', '$compile', 'vuiConstants', '$timeout',
         function ($log, jsUtils, $compile, vuiConstants, $timeout) {
   return {
      scope: true,
      restrict: 'A',
      replace: true,
      template: '<div class="vui-tabs-container"></div>',
      link: function (scope, element, attributes) {
         var template;
         scope.tabOptions = scope[attributes.vuiTabs];
         var getTabsTemplateStr = function (tabType) {
            var templateStr = '';
            if (tabType === vuiConstants.tabs.type.PRIMARY) {
               templateStr =
               '<div class="vui-primary-tabs">' +
                  '<ul class="nav nav-tabs" ng-if="!tabOptions.hideWhenSingleTab || tabOptions.tabs.length > 1" role="tablist" >' +
                     '<li ng-repeat="tab in tabOptions.tabs"' +
                           'ng-class="{active: tabOptions.selectedTabIndex === $index}" ' +
                           'ng-click="clickAction($event,tab)" ' +
                           'ng-keydown="onKeyDown($event)" ' +
                           'role="tab" tabindex="0" aria-selected="{{tabOptions.selectedTabIndex === $index}}" aria-label="{{getTooltipText(tab)}}">' +
                        '<a title="{{getTooltipText(tab)}}"><span ng-class="tab.iconClass"></span>' +
                              '{{tab.label}}</a>' +
                     '</li>' +
                  '</ul>' +
                  '<div ng-class="{active:tabOptions.selectedTabIndex === $index, \'no-tabs\': tabOptions.hideWhenSingleTab === true && tabOptions.tabs.length == 1}"' +
                        'ng-repeat="tab in tabOptions.tabs"' +
                        'class="vui-tab-content">' +
                  '<div ng-include="tab.contentUrl"></div></div>' +
               '</div>';
            }
            else if (tabType === vuiConstants.tabs.type.SECONDARY) {
               templateStr =
               '<div class="vui-secondary-tabs">' +
                  '<div class="btn-group" ng-if="!tabOptions.hideWhenSingleTab || tabOptions.tabs.length > 1"  ' +
                     'role="tablist" ' +
                     'ng-class="{pills: tabOptions.tabStyle === \'' + vuiConstants.tabs.style.PILLS + '\'}">' +
                     '<button ng-repeat="tab in tabOptions.tabs"' +
                           'ng-class="{active: tabOptions.selectedTabIndex === $index, \'btn-sm\': tabOptions.tabStyle === \'' + vuiConstants.tabs.style.PILLS + '\'}"' +
                           'title="{{getTooltipText(tab)}}"' +
                           'ng-click="clickAction($event,tab)" ' +
                           'ng-keydown="onKeyDown($event)" ' +
                           'role="tab" aria-selected="{{tabOptions.selectedTabIndex === $index}}" ' +
                           'class="btn btn-default">' +
                        '<span class="btn-content">{{tab.label}}</span>' +
                     '</button>' +
                  '</div>' +
                  '<div ng-class="{active:tabOptions.selectedTabIndex === $index}"' +
                        'ng-repeat="tab in tabOptions.tabs"' +
                        'class="vui-tab-content">' +
                  '<div ng-include="tab.contentUrl"></div></div>' +
               '</div>';
            }
            else if (tabType === vuiConstants.tabs.type.TERTIARY) {
               templateStr =
               '<div class="vui-tertiary-tabs">' +
                  '<div class="vui-toc-tabs" ng-if="!tabOptions.hideWhenSingleTab || tabOptions.tabs.length > 1">' +
                     '<ul role="tablist">' +
                        '<li ng-repeat="tab in tabOptions.tabs"' +
                            'ng-class="{active: tabOptions.selectedTabIndex === $index}" ' +
                            'ng-click="clickAction($event,tab)" ' +
                            'ng-keydown="onKeyDown($event)" ' +
                            'role="tab" tabindex="0" aria-selected="{{tabOptions.selectedTabIndex === $index}}" aria-label="{{::getTooltipText(tab)}}">' +
                           '<a title="{{getTooltipText(tab)}}"' +
                                 'ng-click="clickAction($event,tab)">' +
                                 '{{tab.label}}</a>' +
                        '</li>' +
                     '</ul>' +
                  '</div>' +
                  '<div ng-class="{active:tabOptions.selectedTabIndex === $index}"' +
                        'ng-repeat="tab in tabOptions.tabs" class="vui-tab-content">' +
                     '<h4>{{tab.label}}</h4>' +
                     '<div ng-include="tab.contentUrl"></div>' +
                  '</div>' +
               '</div>';
            }
            return templateStr;
         };

         template = $compile(getTabsTemplateStr(scope.tabOptions.tabType))(scope);
         element.append(template);

         scope.setSelectedTab = function (tabIndex) {
            jsUtils.setProperty(scope.tabOptions, 'selectedTabIndex',
                  tabIndex);
         };

         scope.getSelectedTab = function () {
            return jsUtils.getProperty(scope.tabOptions,
                  'selectedTabIndex');
         };

         scope.getTooltipText = function (tab) {
            return tab.tooltipText && tab.tooltipText.trim().length ?
                  tab.tooltipText : tab.label;
         };

         scope.clickAction = function (event, tab) {
            for (var i = 0; i < scope.tabOptions.tabs.length; i++) {
               if (scope.tabOptions.tabs[i] === tab) {
                  scope.setSelectedTab(i);
               }
            }
            if (angular.isDefined(tab.onClick) && angular.isFunction(tab.onClick)) {
               tab.onClick.call(undefined, event, tab);
            }
         };

         // Arrow keys tabs navigation
         scope.onKeyDown = function (event) {
            var keyCode = event.keyCode, next;
            if (keyCode === vuiConstants.internal.keys.ENTER || keyCode === vuiConstants.internal.keys.SPACE) {
               $timeout(function () {
                  event.target.click();
               });
               return;
            }

            if (keyCode === vuiConstants.internal.keys.LEFT || keyCode === vuiConstants.internal.keys.UP) {
               next = event.target.previousElementSibling;
            } else if (keyCode === vuiConstants.internal.keys.RIGHT || keyCode === vuiConstants.internal.keys.DOWN) {
               next = event.target.nextElementSibling;
            }

            if (next) {
               next.focus();
            }
         };

         // set first tab as selected if no selectedTabIndex
         function init() {
            if (scope.getSelectedTab() === undefined) {
               var defaults = { selectedTabIndex : 0 };
               angular.extend(scope.tabOptions, defaults, scope.tabOptions);
            } else if (scope.getSelectedTab() < 0 || scope.getSelectedTab() > (scope.tabOptions.tabs.length - 1)) {
               $log.warn('Selected tab index does not exist');
            }
         }

         init();
      }
   };
}]);
;'use strict';

vui.angular.treeView

/**
 * @ngdoc object
 * @name vui.angular.treeView:TreeViewOptions
 * @module vui.angular.treeView
 *
 * @description
 *    Configuration object for
 *    {@link vui.angular.treeView.directive:vuiTreeView vuiTreeView}
 *    directive.
 *    Contains properties to specify a tree view and populate it with
 *    data.
 *
 *    Example:
 *    ```js
 *    var treeViewOptions = {
 *                             data: [ ... ],
 *                             onClick: function (dataItem) {
 *                                console.log(dataItem.label + ' was clicked');
 *                             }
 *                          };
 *    ```
 *
 * @property {TreeItem[]} data
 *    Type: `Array.<TreeItem>`
 *
 *    An array containing {@link vui.angular.treeView:TreeItem TreeItem} objects. Each
 *    item specifies a node in the tree.
 *
 *    In order to update the nodes in the tree, assign a new value to this property.
 *
 *    Example:
 *    ```js
 *    data: [ treeItem1, treeItem2 ];
 *    ```
 *    `treeItem1` and `treeItem2` are {@link vui.angular.treeView:TreeItem TreeItem}
 *    objects.
 *
 * @property {Function} [onClick]
 *    Type: `Function` **|** Optional
 *
 *    Function called whenever a node in the tree is clicked.
 *
 *    The {@link vui.angular.treeView:TreeItem TreeItem} specifying the clicked node
 *    is passed to the function as a parameter.
 *
 *    Example:
 *    ```js
 *    onClick: function (treeItem) {
 *                console.log('Clicked on ' + treeItem.label);
 *             };
 *    ```
 *
 */

/**
 * @ngdoc object
 * @name vui.angular.treeView:TreeItem
 * @module vui.angular.treeView
 *
 * @description
 *    Configuration object for an individual node in a
 *    {@link vui.angular.treeView.directive:vuiTreeView vuiTreeView}
 *    directive.
 *
 *    Example:
 *    ```js
 *    var treeItem1 = {
 *                       label: 'Furniture',
 *                       iconClass: 'furniture-icon',
 *                       items: [
 *                                 {
 *                                    label: 'Chair',
 *                                    iconClass: 'chair-icon'
 *                                 },
 *                                 {
 *                                    label: 'Table',
 *                                    iconClass: 'table-icon'
 *                                 }
 *                              ],
 *                       expanded: false
 *                    };
 *    ```
 *
 *    **Note:** Apart from the following documented properties, items may also contain
 *    any number of additional meta-data properties that will not affect the behaviour
 *    of the directive.
 *
 * @property {string} [label]
 *    Type: `string` **|** Optional
 *
 *    Localized text to display for the node.
 *    <script>$('.properties .label').removeClass('label');</script>
 *
 * @property {string} [iconClass]
 *    Type: `string` **|** Optional
 *
 *    CSS class name of an icon to display for the node.
 *
 *    **Note:** Please refer to the vui-bootstrap documentation for a list of
 *    available icon classes. A custom class specifying a 16x16 icon can also
 *    be used.
 *
 * @property {TreeItem[]} [items]
 *    Type: `Array.<TreeItem>` **|** Optional **|** Default: `[]`
 *
 *    Array of child nodes for this node, also of type
 *    {@link vui.angular.treeView:TreeItem TreeItem}.
 *
 *    If this property is not specified it will be initialized to an empty array.
 *
 * @property {boolean} [expanded]
 *    Type: `boolean` **|** Optional **|** Default: `false`
 *
 *    Specifies whether this node is expanded or collapsed.
 *
 *    If this property is not specified it will be initialized to false.
 */

/**
 * @ngdoc directive
 * @name vui.angular.treeView.directive:vuiTreeView
 * @module vui.angular.treeView
 * @restrict A
 * @element div
 * @scope
 *
 * @description
 *    Directive to create a Tree View navigation control.
 *
 * @param {TreeViewOptions} vuiTreeView
 *    Configuration object for the Tree View. It is of type
 *    {@link vui.angular.treeView:TreeViewOptions TreeViewOptions}.
 *
 * @example
 <example module="app">
 <file name="app.js">
 'use strict';
 var app = angular.module('app', ['vui.angular.treeView']);

 app.controller('TreeViewController', function ($scope) {
   $scope.treeViewOptions = {
      data: [
         { label: 'Furniture', iconClass: 'vui-icon-info', items: [
            { label: 'Tables & Chairs', iconClass: 'vui-icon-success'},
            { label: 'Sofas', iconClass: 'vui-icon-success'},
            { label: 'Occasional Furniture', iconClass: 'vui-icon-success'}
         ] },
         { label: 'Decor', iconClass: 'vui-icon-info', items: [
            { label: 'Bed Linen', iconClass: 'vui-icon-success'},
            { label: 'Curtains & Blinds', iconClass: 'vui-icon-success'},
            { label: 'Carpets', iconClass: 'vui-icon-success'}
         ] }
      ],
      onClick: function (dataItem) {
         console.log(dataItem.label + ' was clicked');
      }
   };
});
 </file>

 <file name="index.html">
    <div ng-controller="TreeViewController">
       <div vui-tree-view="treeViewOptions">
    </div>
 </file>

 </example>
 */
.directive('vuiTreeView', ['jsUtils', 'vuiUtils',
   function (jsUtils, vuiUtils) {
      return {
         scope: {
            treeViewOptions: '=vuiTreeView'
         },
         restrict: 'A',
         replace: true,
         link: function (scope, element) {

            // --------------------------------------------------------------------------
            // Directive State.

            // Reference to the kendo UI tree view widget.
            var kendoTreeView;

            // Mapping from uids of data items in the kendo widget data provider back to
            // corresponding user scope data items.
            var scopeDataByKendoDataUid = {};

            // Schema required by Kendo data source for correct data rendering.
            var dataSourceSchema = {
               model: {
                  fields : {
                     label : {
                        // Prevents kendo from showing 'undefined' when no label present.
                        parse: function (item) {
                           if (jsUtils.isUndefinedOrNull(item)) {
                              return '';
                           }
                           return item;
                        }
                     }
                  },
                  hasChildren: function (item) {
                     var items = jsUtils.getProperty(item, 'items', null);
                     if (!items) {
                        items = [];
                     }
                     return items.length > 0;
                  },
                  children: 'items'
               }
            };

            // --------------------------------------------------------------------------
            // Util Functions.

            // Updates the 'expanded' property of the scope data item corresponding to
            // the expanded or collapsed node.
            var onExpandChange = function (event) {
               var kendoDataItem = kendoTreeView.dataItem(event.node);
               var scopeDataItem = scopeDataByKendoDataUid[kendoDataItem.uid];
               if (!scopeDataItem) {
                  console.error('couldnt find data item ' + kendoDataItem.uid);
                  return;
               }
               vuiUtils.safeApply(scope, function () {
                  jsUtils.setProperty(scopeDataItem, 'expanded', kendoDataItem.expanded, true);
               });
            };

            function buildNodeDataObject(clickedItem) {
               var kendoDataItem = kendoTreeView.dataItem(clickedItem);

               return {
                  label: kendoDataItem.label,
                  iconClass: kendoDataItem.iconClass,
                  expanded: kendoDataItem.expanded,
                  metadata: kendoDataItem.metadata
               };
            }

            // jQuery click function. If the element clicked was within a tree node, we call
            // the user's onClick function, passing it the clicked node.
            // Note we prefer this to the kendo 'select' event because the latter is not called
            // when the same node is clicked repeatedly.
            var onClick = function (event, selectedNode) {
               if (!jsUtils.hasProperty(scope.treeViewOptions, 'onClick')) {
                  // User has not specified a click function.
                  return;
               }

               var clickedItem = selectedNode;
               if (!clickedItem) {
                  // CSS class k-in is permanently on every span representing a node in the tree.
                  clickedItem = jQuery(event.target).closest('span.k-in');
                  if (clickedItem.length !== 1) {
                     // Click did not occur within a node.
                     return;
                  }
                  clickedItem = clickedItem[0];
               }

               vuiUtils.safeApply(scope, function () {
                  scope.treeViewOptions.onClick(buildNodeDataObject(clickedItem));
               });
            };

            // Creates the kendo tree view data source from the scope data and the
            // mapping to go from kendo back to scope data.
            var createKendoDataSource = function (sourceData) {
               var kendoDataSource = new kendo.data.HierarchicalDataSource({
                  data: sourceData,
                  schema: dataSourceSchema
               });
               // Next line fills all of the 'items' properties' (no lazy loading).
               kendoDataSource.read();
               var kendoData = kendoDataSource.data();
               scopeDataByKendoDataUid = {};
               createDataMap(kendoData, sourceData);
               return kendoDataSource;
            };

            // Adds map entries to scopeDataByKendoDataUid. fromData is kendo data.
            // toData is scope data.
            // TODO mcritch: generalize this technique and move to a utility that all
            // kendo wrappers can benefit from.
            var createDataMap = function (fromData, toData) {
               for (var i = 0; i < fromData.length; i++) {
                  var fromItem = fromData[i];
                  var toItem = toData[i];
                  scopeDataByKendoDataUid[fromItem.uid] = toItem;
                  if (jsUtils.getProperty(fromItem, 'hasChildren', false)) {
                     // Recursive case. Next line makes sure 'items' is populated.
                     fromItem.children.read();
                     var fromChildren = fromItem.children.data();
                     createDataMap(fromChildren, toItem.items);
                  }
               }
            };

            var createKendoOptions = function () {
               return {
                  dataTextField: 'label',
                  dataSpriteCssClassField: 'iconClass',
                  expand: onExpandChange,
                  collapse: onExpandChange,
                  loadOnDemand: false,
                  dataSource: createKendoDataSource(scope.treeViewOptions.data)
               };
            };

            // --------------------------------------------------------------------------
            // Directive Initialization.

            (function init() {
               jsUtils.validateObject(scope.treeViewOptions, [['data', 'object']]);

               var options = createKendoOptions();

               // When the data object changes, update the widget. Note this is not a
               // deep watch for performance reasons (but could be).
               scope.$watch('treeViewOptions.data', function () {
                  // For some reason kendoDataSource.data(scope.treeViewOptions.data); does not work as advertised
                  // in the API.
                  // Why have they provided this setDataSource API peculiar to this widget?
                  kendoTreeView.setDataSource(createKendoDataSource(scope.treeViewOptions.data));
               });

               // Register the jQuery click event listener
               element.click(onClick);
               element.keypress(function (event) {
                  if (event.which !== 13) {
                     // Enter key not pressed
                     return;
                  }
                  var selectedNode = kendoTreeView.select();
                  onClick(event, selectedNode);
               });
               element.on('$destroy', function () {
                  element.off('click', 'keypress');
               });

               // Initialize and save the kendo tree view.
               kendoTreeView = element.kendoTreeView(options).data('kendoTreeView');
            })();

         }
      };
   }
]);
;'use strict';
vui.angular.validationBanner

/**
 * @ngdoc object
 * @name vui.angular.validationBanner:ValidationBannerOptions
 * @module vui.angular.validationBanner
 *
 * @description
 *    Configuration object for validation banner in
 *    {@link vui.angular.dialog.directive:vuiDialog vuiDialog} directive and
 *    {@link vui.angular.wizard.vuiWizardService vuiWizardService}.
 *    Contains properties to define messages in the validation banner.
 *
 * @property {Array.<Object>} messages
 *    Type: `Array.<Object>`
 *
 *    An array of {@link vui.angular.validationBanner:Message Message} objects.
 *    If this array is empty the validation banner will not be displayed.
 *    The array is cleared when the user closes the validation banner.
 *
 *    Example:
 *    ```js
 *    messages: [ errorMessage, warningMessage]
 *    ```
 *    `errorMessage` and `warningMessage` are
 *    {@link vui.angular.validationBanner:Message Message} objects.
 * @property {boolean} [showAllMessages]
 *    Type: `boolean` **|** Optional **|** Default: `false`
 *
 *    Shows all messages when set to true. If no value is set,
 *    only the first message will be displayed.
 */

/**
 * @ngdoc object
 * @name vui.angular.validationBanner:Message
 * @module vui.angular.validationBanner
 *
 * @description
 *    Contains properties to define a message in validation banner.
 *
 *    Example:
 *    ```js
 *    var errorMessage = {
 *                           type: vuiConstants.validationBanner.ERROR,
 *                           text: 'You must select a target folder.'
 *                       };
 *    ```
 *
 * @property {string} type
 *    Type: `string`
 *
 *    Defines the contextual type of the message.
 *
 *    Legal values:
 *
 *    - **`vuiConstants.validationBanner.type.ERROR`** : Use to indicate an
 *       error or a problem that has occurred.
 *    - **`vuiConstants.validationBanner.type.WARNING`** : Use to indicate a
 *       condition that might cause a problem in the future.
 *
 * @property {string} text
 *    Type: `string`
 *
 *    The message text that is displayed in validation banner.
 */

 /**
 * @xngdoc directive
 * @name vui.angular.validationBanner.directive:vuiValidationBanner
 * @module vui.angular.validationBanner
 * @restrict A
 * @scope
 *
 * @description
 *    A simple validationBanner directive that displays error/warning messages.
 *    Displayed when messages array is non-empty. Hidden when empty.
 *
 * @param {ValidationBannerOptions} vuiValidationBanner
 *    {@link vui.angular.validationBanner:ValidationBannerOptions ValidationBannerOptions}
 *    Configuration object for the validation banner.
 */

.directive('vuiValidationBanner', ['vuiLocalizationService', function (vuiLocalizationService) {
   return {
      template:
         '<div class="vui-validation-banner">' +
            '<div class="alert alert-danger" ' +
               'role="alert" aria-live="assertive" aria-atomic="true" ng-if="validationBannerVisible()">' +
               '<div class="alert-items">' +
                  '<div class="alert-item static" ng-repeat="msg in validationBanner.messages">' +
                     '<div class="alert-icon-wrapper">' +
                        '<clr-icon class="alert-icon" shape="exclamation-circle" aria-hidden="true"></clr-icon>' +
                     '</div>' +
                     '<span class="alert-text" ng-bind-html="msg.text"></span>' +
                  '</div>' +
               '</div>' +
               '<button type="button" class="close" ng-if="isCloseable(validationBanner.isCloseable)"' +
               ' aria-label="{{closeButtonText}}" ng-click="clearValidationBanner()">' +
                  '<clr-icon aria-hiden="true" shape="close"></clr-icon>' +
               '</button>' +
            '</div>' +
         '</div>',
      scope : {
         validationBanner: '=vuiValidationBanner'
      },
      restrict: 'A',
      controller: ['$scope', '$element', 'vuiConstants', '$sce', function ($scope, $element, vuiConstants, $sce) {
         $scope.closeButtonText = vuiLocalizationService.get('alertCloseText') || 'Close';
         var validationBanner = { 'showAllMessages' : false, messages : [] };
         angular.extend(validationBanner, $scope.validationBanner);
         $scope.validationBanner = validationBanner;

         // Watch on the message array and mark messages to be html safe
         $scope.$watch('validationBanner.messages', function (newVal) {
            if (newVal.length > 0) {
               newVal.forEach(function (msg) {
                  msg.text = $sce.trustAsHtml(msg.text);
               });
            }
         });

         $scope.validationBannerVisible = function () {
            return $scope.validationBanner.messages.length > 0 ? true : false;
         };

         $scope.isCloseable = function (isCloseable) {
            if (isCloseable === undefined) {
               return true;
            }

            return isCloseable;
         };

         $scope.getValidationMessageClass = function (msgType) {
            var iconClass;
            switch (msgType) {
               case vuiConstants.validationBanner.type.ERROR :
                  iconClass = 'vui-icon-critical';
                  break;
               case vuiConstants.validationBanner.type.WARNING :
                  iconClass = 'vui-icon-warning';
                  break;
               default :
                  iconClass = '';
            }
            return iconClass;
         };

         $scope.getValidationBannerClass = function () {
            return $scope.validationBanner.showAllMessages ?
                  'validation-show-all' : 'validation-show-one';
         };
         // Show text "show all" if message is very long
         // or there are more than one message
         $scope.showAllToggle = function () {
            return $scope.validationBanner.messages.length > 1 ||
                  ($element.find('.validation-message')[0] &&
                  $element.find('.validation-message')[0].clientHeight > 16);
         };
         $scope.ifShowAllMessagesEnabled = function () {
            return $scope.showAllToggle() &&
                 !$scope.validationBanner.showAllMessages;
         };
         $scope.ifHideMessagesEnabled = function () {
            return $scope.showAllToggle() &&
                  $scope.validationBanner.showAllMessages;
         };
         $scope.toggleMessages = function (boolFlag) {
            $scope.validationBanner.showAllMessages = boolFlag;
         };
         $scope.clearValidationBanner = function () {
            $scope.validationBanner = {
               'showAllMessages' : false,
               messages : []
            };
         };
      }]
   };
}]);
;(function () {

   function VuiTreeAccessibilityServiceFactory(zoneService, vuiConstants, $timeout) {
      return {
         create: function (container, $scope) {
            return new VuiTreeAccessibilityService(zoneService, container, $scope, vuiConstants, $timeout);
         }
      };
   }

   var MoveDirection = {
      Back: 0,
      Forward: 1
   };

   function VuiTreeAccessibilityService(zoneService, container, $scope, vuiConstants, $timeout) {
      this.ARIA_ACTIVEDESCENDANT = 'aria-activedescendant';
      this.ARIA_ID = 'aria-activedescendant-id-' + Date.now();
      this.STATE_FOCUSED = 'state-focused';
      this.VIEW_FOCUSED = 'view-focused';
      this.zoneService = zoneService;
      this.init(container, $scope);
      this.vuiContants = vuiConstants;
      this.$timeout = $timeout;
   }

   VuiTreeAccessibilityService.prototype.init = function (container, $scope) {
      this.view = jQuery(container).attr('role') === 'tree' ?
            jQuery(container).get(0) : jQuery(container).find('[role=tree]').get(0);

      if (!this.view) {
         return;
      }

      var _this = this;
      var keydownHandler = function (e) { _this.handleKeyDown(e); };
      var focusHandler = function () {
         _this.handleFocus();
      };

      var blurHandler = function () {
         _this.$timeout(function () {
            if (!_this.view.contains(document.activeElement)) {
               _this.view.classList.remove(_this.VIEW_FOCUSED);
               _this.unfocusElement(_this.currentFocusedElement);
               _this.currentFocusedElement = null;
            }
         }, 0);
      };

      jQuery(this.view)
         .on('keydown', keydownHandler)
         .on('focus', focusHandler)
         .on('blur', blurHandler);

      $scope.$on('$destroy', function () {
         jQuery(_this.view)
            .off('keydown', keydownHandler)
            .off('focus', focusHandler)
            .off('blur', blurHandler);
      });
   };
   VuiTreeAccessibilityService.prototype.handleFocus = function () {
      this.view.classList.add(this.VIEW_FOCUSED);

      var selected = this.getSelectedElement();
      if (selected) {
         this.focusElement(selected);
         return;
      }

      var firstElement = this.getFirstElement();
      if (firstElement) {
         this.focusElement(firstElement);
      }
   };

   VuiTreeAccessibilityService.prototype.handleKeyDown = function (event) {
      switch (event.keyCode) {
         case this.vuiContants.internal.keys.DOWN:
         case this.vuiContants.internal.keys.UP:
            this.focus(event);
            break;
         case this.vuiContants.internal.keys.LEFT:
            this.collapse();
            break;
         case this.vuiContants.internal.keys.RIGHT:
            this.expand();
            break;
         case this.vuiContants.internal.keys.ENTER:
         case this.vuiContants.internal.keys.SPACE:
            this.select();
            event.preventDefault();
            break;
      }
   };

   VuiTreeAccessibilityService.prototype.focus = function (event) {
      if (!this.currentFocusedElement) {
         // the handleFocus should be executed before this call is made
         return;
      }

      var direction = this.getDirection(event);
      var listItems = jQuery(this.view).find('[role=treeitem]')
         .filter(function (i, val) {
            // filter out the nodes in collapsed groups
            return !jQuery(val).is('.closed [role=treeitem]');
         });

      if (!listItems.length) {
         return;
      }

      var indexToFocus = direction === MoveDirection.Forward ? 0 : listItems.length - 1;
      if (this.currentFocusedElement) {
         var currentIndex = listItems.get().indexOf(this.currentFocusedElement);
         indexToFocus = direction === MoveDirection.Forward ? currentIndex + 1 : currentIndex - 1;

         if (indexToFocus < 0) {
            indexToFocus = listItems.length - 1;
         }

         if (indexToFocus >= listItems.length) {
            indexToFocus = 0;
         }
      }

      // if we are navigating with tab / tab + shift
      // we should stp the event to keep the focus in the tree
      // but only if we are not on the last/first element
      if (indexToFocus >= 0 && indexToFocus < listItems.length) {
         event.preventDefault();
         event.stopPropagation();
      }

      var elementToFocus = listItems.get(indexToFocus);
      if (elementToFocus) {
         this.focusElement(elementToFocus);
      }
   };

   VuiTreeAccessibilityService.prototype.getDirection = function (event) {
      if (event.keyCode === jQuery.ui.keyCode.UP) {
         return MoveDirection.Back;
      }

      return MoveDirection.Forward;
   };

   VuiTreeAccessibilityService.prototype.collapse = function () {
      // check if we are on a collapsible section title
      // and collapse / expand it
      if (!this.currentFocusedElement) {
         return;
      }

      var _this = this;
      if (this.currentFocusedElement.getAttribute('aria-expanded') === 'true') {
         this.currentFocusedElement.setAttribute('aria-expanded', 'false');
         this.zoneService.runInsideAngularZone(function () {
            jQuery(_this.currentFocusedElement).click();
         });
      }

   };

   VuiTreeAccessibilityService.prototype.expand = function () {
      // check if we are on a collapsible section title
      // and collapse / expand it
      if (!this.currentFocusedElement) {
         return;
      }

      var _this = this;
      if (this.currentFocusedElement.getAttribute('aria-expanded') === 'false') {
         this.currentFocusedElement.setAttribute('aria-expanded', 'true');
         this.zoneService.runInsideAngularZone(function () {
            jQuery(_this.currentFocusedElement).click();
         });
      }
   };

   VuiTreeAccessibilityService.prototype.select = function () {
      // deselect current element
      // select the new one and trigger click
      if (!this.currentFocusedElement) {
         return;
      }

      var _this = this;
      this.zoneService.runInsideAngularZone(function () {
         jQuery(_this.currentFocusedElement).click();

      });

   };

   VuiTreeAccessibilityService.prototype.getSelectedElement = function () {
      var selected =
         jQuery(this.view).find('[aria-selected=true]');
      if (!selected.length) {
         return null;
      }

      return selected.get(0);
   };

   VuiTreeAccessibilityService.prototype.getFirstElement = function () {

      return jQuery(this.view).find('[role=teeitem]').get(0) || null;
   };

   VuiTreeAccessibilityService.prototype.focusElement = function (el) {
      // unfocus previously focused element
      this.unfocusElement(this.currentFocusedElement);
      // mark el as current
      this.currentFocusedElement = el;

      // if the element does not have its own id
      // default id is assigned
      if (!this.currentFocusedElement.id) {
         this.currentFocusedElement.id = this.ARIA_ID;
      }
      // set aria-activedescendant of the tree
      // with the id of the element
      this.view.setAttribute(this.ARIA_ACTIVEDESCENDANT, this.currentFocusedElement.id);
      el.classList.add(this.STATE_FOCUSED);
      el.scrollIntoView();
   };

   VuiTreeAccessibilityService.prototype.unfocusElement = function (el) {
      if (!el) {
         return;
      }
      // if the tree view has an aria-activedescendant attribute
      // with value the id of the current element - remove it
      if (this.view.getAttribute(this.ARIA_ACTIVEDESCENDANT) === el.id) {
         this.view.setAttribute(this.ARIA_ACTIVEDESCENDANT, '');
      }

      if (el.id === this.ARIA_ID) {
         // if the element has default id it should be moved
         // in order to be used by the next element
         el.id = '';
      }

      if (el.classList.contains(this.STATE_FOCUSED)) {
         el.classList.remove(this.STATE_FOCUSED);
      }
   };

   VuiTreeAccessibilityService.prototype.getTreeItem = function (eventTarget) {
      if ($(eventTarget).attr('role') === 'treeitem') {
         return eventTarget;
      }

      var parents = $(eventTarget).parents('[role=treeitem]');
      if (parents.length) {
         return parents.get(0);
      }

      throw new Error('Clicked element is nor a treeitem nor a child of treeitem');
   };

   vui.angular.wizard
      .factory('vuiTreeAccessibilityServiceFactory',
         ['vuiZoneService', 'vuiConstants', '$timeout', VuiTreeAccessibilityServiceFactory]);
})();
;'use strict';
vui.angular.wizard

/**
 * @xngdoc object
 * @name vui.angular.wizard.VuiWizard
 * @module vui.angular.wizard
 *
 * @description
 *    Configuration object for vui-wizard. Contains properties to specify the wizard
 *    and populate with pages.
 *
 * @property {Array.<Object>} pages
 *    `{Array.<Object>}` Array of {@link vui.angular.wizard.Page Page} objects.
 *    objects defining the properties of each page in the wizard. The pages in this array
 *    should appear in display order.
 * @property {string} title
 *    `{string}` Title of the wizard.
 * @property {string} iconClass
 *    `{string}` Name of the class that contains the wizard icon.
 * @property {string} closeIconClass
 *    `{string}` Name of the class that contains the wizard close icon.
 * @property {boolean} show
 *    `{boolean}` Hides/Shows the wizard.
 * @property {Function} [onFinish]
 *    `{Function}` A function that is called when finish button is clicked. Its expected
 *    to return true/false or a promise. If true the wizard is closed, else its not.
 * @property {Function} [onClose]
 *    `{Function}` A function that is called when the cancel button is clicked. It allows
 *    a custom onClose to be written but expects that function to
 *    return a promise and if the promise is rejected then it will close the wizard but not the browser/app
 * @property {Page} currentPage
 *    {{@link vui.angular.wizard.Page Page}} contains the current page displayed by the
 *    wizard. The onCommit function can be defined on this inside the pageController if
 *    required.
 * @property {Array.<Object>} The css classes of the current wizard
 *    `{Array.<Object>}` CSS class or an array CSS classes to be attached to to the
 *    current wizard
 *
 */

 /**
 * @xngdoc object
 * @name vui.angular.wizard.Pages
 * @module vui.angular.wizard
 *
 * @description
 *    Configuration object for a page in the wizard.
 *
 * @property {string} title
 *    `{string}` Title of the page.
 * @property {string} description
 *    `{string}` Description of what the page is meant for.
 * @property {string} contentUrl
 *    `{string}` Template URL of the page.
 * @property {string} state
 *    `{string}` Name of the class that contains the wizard icon.
 * @property {boolean} hideToc
 *    `{boolean}` Option to display or hide the table of contents.
 * @property {boolean} hideHeader
 *    `{boolean}` Option to display or hide the header.
 * @property {Function} [onCommit]
 *    `{Function}` A function that is to be called when the on Next button if clicked.
 *    Its expected to return true/false or a promise. If true the wizard shows the next
 *    page, else its stays on the same. This function is where you could change the page
 *    state of the current page.
 * @property {string} cssClass
 *    `{string}` A css class that will be added to the page
 */

/**
 * @xngdoc directive
 * @name vui.angular.datagrid.wizard:vuiWizard
 * @module vui.angular.wizard
 * @restrict A
 * @scope
 *
 * @description
 *    Directive to create a wizard.
 *
 * @param {vuiWizard} vuiWizard
 *    {@link vui.angular.wizard.vuiWizard vuiWizard} Configuration object for the wizard.
 *
 * @example
      <example module="app">

         <file name="app.js">
            'use strict';
            var app = angular.module('app', ['vui.angular.wizard']);

            app.controller('WizardCtrl', ['$scope', 'vuiConstants',
                  function ($scope, vuiConstants) {
               var pageStates = vuiConstants.wizardPageStates;
               $scope.wizardOptions = {};
               $scope.wizardOptions.show = false;
               $scope.wizardOptions.title = 'Bake a cake wizard';
               $scope.wizardOptions.iconClass = 'icon-create';
               $scope.wizardOptions.closeIconClass = 'icon-close';
               $scope.wizardOptions = {
                  show: false,
                  title: 'Bake a cake wizard',
                  iconClass: 'icon-create',
                  pages: [{
                     title: 'Page1',
                     contentUrl: 'page1.html',
                     state: pageStates.INCOMPLETE,
                     description: 'first page'
                  }, {
                     title: 'Skipped page',
                     contentUrl: 'skippedPage.html',
                     state: pageStates.SKIPPED,
                     description: 'skipped page - to describe dynamic pages',
                  }, {
                     title: 'page2',
                     contentUrl: 'page2.html',
                     state: pageStates.INCOMPLETE,
                     description: 'page 2 will set validation data to show warning',
                  }, {
                     title: 'page3',
                     contentUrl: 'page3.html',
                     state: pageStates.INCOMPLETE,
                     description: 'last page'
                  }]
               };
               $scope.wizardOptions.data = {
                  flavor : null,
                  shape : null
               }
               $scope.openWizard = function () {
                  $scope.wizardOptions.show = true;
               };
            }]);

            app.controller('page1Ctrl', ['$scope', 'vuiConstants', function ($scope, vuiConstants) {
               // Get data from
               $scope.flavor = $scope.vuiWizard.data.flavor;
               $scope.shape = $scope.vuiWizard.data.shape;

               // Function to be called when Next button is pressed.
               $scope.vuiWizard.currentPage.onCommit = function() {
                  if($scope.flavor && $scope.shape){
                     $scope.vuiWizard.currentPage.state =
                           vuiConstants.wizardPageStates.completed;
                     // Storing the name and age onto the wizard.data object
                     // for persisting.
                     $scope.vuiWizard.data.flavor = $scope.flavor;
                     $scope.vuiWizard.data.shape = $scope.shape;
                     return true;
                  } else {
                     $scope.vuiWizard.validationBanner.messages  = [{
                        text: 'Fill up the fields to proceed',
                        type: vuiConstants.validationBanner.ERROR
                     }];
                     return false;
                  }
               };
            }]);

            app.controller('page2Ctrl', ['$scope', 'vuiConstants',
                  function ($scope, vuiConstants) {
               $scope.vuiWizard.validationBanner.messages = [{
                  text: 'The cake contains eggs/gluten and may cause allergy',
                  type: vuiConstants.validationBanner.WARNING
               }];
               $scope.vuiWizard.currentPage.onCommit = function() {
                  $scope.vuiWizard.currentPage.state =
                        vuiConstants.wizardPageStates.completed;
                  return true;
               };
            }]);

            app.controller('page3Ctrl', ['$scope', 'vuiConstants',
                  function ($scope, vuiConstants) {
               $scope.vuiWizard.currentPage.onCommit = function() {
                  $scope.vuiWizard.currentPage.state =
                        vuiConstants.wizardPageStates.completed;
                  return true;
               };
            }]);
         </file>

         <file name="index.html">
            <div ng-controller="WizardCtrl">
               <button ng-click="openWizard()">Open Wizard</button>
               <!-- vui-wizard Directive -->
               <div vui-wizard="wizardOptions"></div>
            </div>
         </file>

         <file name="page1.html">
            <div ng-controller="page1Ctrl">
               <p>First page of the wizard.</p>
               <p> Please fillup the form below to proceed.</p>
               <form>
                  <label> Enter the flavor you want: </label>
                  <input type="text" ng-model="flavor"></input><br><br>
                  <label> Enter the shape circle/square: </label>
                  <input type="text" ng-model="shape"></input>
               </form>
            </div>
         </file>

         <file name="skippedPage.html">
            <div>Hidden or skipped page</div>
         </file>

         <file name="page2.html">
            <div ng-controller="page2Ctrl">Page 2 </div>
         </file>

         <file name="page3.html">
            <div ng-controller="page3Ctrl">
               <p>Ready to complete page. Usaully contains a summary of
               the info entered in the previous pages </p>
               <ul>
                  <li>Flavor : {{vuiWizard.data.flavor}}</li>
                  <li>Age : {{vuiWizard.data.shape}}</li>
               </ul>
            </div>
         </file>

    </example>
 */

.directive('vuiWizard', [ '$document', '$templateCache', 'vuiConstants',
   'vuiZoneService', function ($document, $templateCache, vuiConstants, vuiZoneService) {
   return {
      controller: 'VuiWizardController',
      scope: true,
      replace: true,
      link: function (scope, elem, attrs, wizardCtrl) {

         var defaultLoadingTemplate = '' +
            '<div class="progress-bordered progress-centered">' +
               '<div class="progress progress-striped active">' +
                  '<div class="progress-bar" style="width: 100%" ' +
                     'role="alert"' +
                     'aria-label="{{vuiWizard.loadingAriaMessage ? vuiWizard.loadingAriaMessage : vuiWizard.loadingMessage}}">' +
                  '</div>' +
               '</div>' +
               '<div class="progress-text" ng-if="vuiWizard.loadingMessage" ' +
                  'aria-hidden="true">{{vuiWizard.loadingMessage}}' +
               '</div>' +
            '</div>';

         $templateCache.put('vuiWizardDefaultLoadingTemplate.html', defaultLoadingTemplate);

         scope.vuiWizard.loadingTemplate = scope.vuiWizard.loadingTemplate || 'vuiWizardDefaultLoadingTemplate.html';

         var wizard = scope.vuiWizard;
         var wizardManager = scope.wizardManager;
         var wizardPages = scope.vuiWizardPages;
         var pageStateConst = vuiConstants.wizard.pageState;

        /**
         * Recursive function that finds if the pages from given index to 0
         * are either COMPLETED or SKIPPED.
         *
         * @param pageIndex number Index of the wizard Page form which we need to perform
         * the check.
         *
         * @returns boolean True if all pages have states as skipped/complete.
         */
         var isPageComplete = function (pageIndex) {
            if (pageIndex === 0) {
               return (wizardPages[pageIndex].state === pageStateConst.COMPLETED ||
                     wizardPages[pageIndex].state === pageStateConst.SKIPPED);
            }
            return (wizardPages[pageIndex].state === pageStateConst.COMPLETED ||
                  wizardPages[pageIndex].state === pageStateConst.SKIPPED) &&
                  isPageComplete(pageIndex - 1);
         };

         var checkFinishState = function () {
            var finishFlag = false;
            var lastPageIndex = wizardPages.length - 1;
            while (wizardPages[lastPageIndex].state ===
                  pageStateConst.SKIPPED) {
               lastPageIndex--;
            }

            // Enable Finish when
            // 1) All Pages are completed.
            // 2) Wizard is a single page.
            // 3) All Pages are completed except the last page is incomplete
            // and user is in the lastPage.
            if (lastPageIndex === 0 ||  isPageComplete(lastPageIndex - 1)) {
               if (wizardPages[lastPageIndex].state ===  pageStateConst.COMPLETED ||
                     wizard.currentPage === wizardPages[lastPageIndex]) {
                  finishFlag = true;
               }
            }

            wizardManager.enableFinishFlag = finishFlag;
         };

         var isComboboxElement = function (element) {
            return element && element.getAttribute && element.getAttribute('role') === 'combobox';
         };

         var isDropdownElement = function (element) {
            return element && element.getAttribute && element.getAttribute('role') === 'listbox';
         };

         var escKeydownHandler = function (event) {
            var focusedElement = event.currentTarget.activeElement;
            if (isComboboxElement(focusedElement) || isDropdownElement(focusedElement)) {
               return;
            }

            // Avoid propagation of any keydown event since some other dialog might be
            // listening for it and trigger undesired functionality as closing the other
            // dialog when the current one on is on focus.
            event.stopImmediatePropagation();
            scope.$apply(function () {
               wizardManager.closeWizard();
            });
         };

         var keydownHandler = function (event) {
            var keyCode = event.which || event.keyCode;

            if (keyCode === vuiConstants.internal.keys.ESC && scope.vuiWizard.closeOnEsc) {
               escKeydownHandler(event);
            }
         };

         var bodyKeydownHandler = function (event) {
            //Run keydown handler in angular zone
            vuiZoneService.runInsideAngularZone(keydownHandler.bind(this, event));
         };

         var addBodyKeydownHandler = function () {
            $document.on('keydown', bodyKeydownHandler);
         };

         var removeBodyKeydownHandler = function () {
            $document.off('keydown', bodyKeydownHandler);
         };

         scope.$watch('vuiWizard.show', function () {
            if (scope.vuiWizard.show) {
               checkFinishState();
               // Initialize wizard
               wizardCtrl.initialize();
               addBodyKeydownHandler();
            } else {
               wizardManager.closeWizard();
               removeBodyKeydownHandler();
            }
         });
         // Watch on pages array and currentPage to recompute finish button state.
         scope.$watch('vuiWizardPages', function () {
            checkFinishState();
         }, true);

         scope.$watch('vuiWizard.currentPage', function () {
            if (scope.vuiWizard.currentPage.hasOwnProperty('decisionOptions')) {
               // Builds updated wizard path
               wizardManager.generateWizardPages();
               wizardPages = scope.vuiWizardPages;
            }
            checkFinishState();
         }, true);

         scope.showTocExplicitly = function () {
            scope.vuiWizard.showTocExplicitly = !scope.vuiWizard.showTocExplicitly;

            if (scope.vuiWizard.showTocExplicitly) {
               // when toc is displayed attach document focusin handler to
               // auto-hide it user clicks somewhere
               addFocusInHandler();
            } else {
               removeFocusInHandler();
            }
         };

         var addFocusInHandler = function () {
            $document.on('focusin', focusInHandler);
         };

         var removeFocusInHandler = function () {
            $document.off('focusin', focusInHandler);
         };

         var focusInHandler = function (event) {
            var tocEl = elem[0].querySelector('.wizard-steps-list');
            if (!tocEl.contains(event.target)) {
               scope.vuiWizard.showTocExplicitly = false;
               removeFocusInHandler();
            }
         };

         // Clean up
         scope.$on('$destroy', function () {
            removeBodyKeydownHandler();
            removeFocusInHandler();
         });

      },
      template:
            '<div class="vui-wizard default-text" vui-focus-trap ng-class="vuiWizard.cssClass" role="dialog" aria-labelledby="vuiWIzardTitle-{{$id}}" tabindex="0" aria-modal="true">' +
               '<div class="wizard-modal-titlebar">' +
                  '<span class="titlebar-left-icons">' +
                     '<span ng-if="vuiWizard.iconClass" class="vui-icon" ' +
                           'ng-class="vuiWizard.iconClass"' +
                           'aria-label="{{vuiWizard.iconClassAriaLabel}}"' +
                           'ng-click="showTocExplicitly()">' +
                     '</span>' +
                  '</span>' +
                  '<h3 id="vuiWIzardTitle-{{$id}}" class="titlebar-text">' +
                     '{{vuiWizard.title}}</h3>' +
                  '<span class="titlebar-right-icons">' +
                     '<span ng-if="vuiWizard.closeIconClass" class="vui-icon" ' +
                        'ng-class="vuiWizard.closeIconClass" ' +
                        'aria-label="{{vuiWizard.closeIconClassAriaLabel}}"' +
                        'ng-click="vuiWizard.show = false;">' +
                     '</span>' +
                  '</span>' +
               '</div>' +
               '<div class="wizard-modal-body" ng-if="vuiWizard.show">' +
                  '<vui-wizard-toc ng-if="!vuiWizard.hideToc"></vui-wizard-toc>' +
                  '<div class="wizard-content-container  {{vuiWizardPages[wizardManager.currentIdx].cssClass}}" role="main"' +
                   ' ng-class="{\'wizard-content-loading\': vuiWizard.loading}" tabindex="0">' +
                     '<div vui-validation-banner="vuiWizard.validationBanner"></div>' +
                        '<div class="wizard-content-header-area" ng-if="!vuiWizard.hideHeader">' +
                           '<div ng-if="!vuiWizardPages[wizardManager.currentIdx].headerUrl">' +
                              '<h4 class="wizard-content-title"' +
                                    'ng-model="vuiWizardPages[wizardManager.currentIdx].title">' +
                                    '{{vuiWizardPages[wizardManager.currentIdx].title}}' +
                              '</h4>' +
                              '<div class="wizard-content-instructional-text"' +
                                    'ng-model="vuiWizardPages[wizardManager.currentIdx].description">' +
                                    '{{vuiWizardPages[wizardManager.currentIdx].description}}' +
                              '</div>' +
                           '</div>' +
                           '<div ng-if="vuiWizardPages[wizardManager.currentIdx].headerUrl" ' +
                                 'ng-include="vuiWizardPages[wizardManager.currentIdx].headerUrl">' +
                           '</div>' +
                        '</div>' +
                        '<div class="wizard-content"' +
                              'ng-include="vuiWizardPages[wizardManager.currentIdx].contentUrl">' +
                        '</div>' +
                  '</div>' +
                  '<div aria-live="polite" aria-atomic="true">' +
                     '<div class="wizard-loading-indicator" ng-if="vuiWizard.loading" ng-include="vuiWizard.loadingTemplate"></div>' +
                  '</div>' +
               '</div>' +
               '<div class="wizard-modal-footer">' +
                  '<vui-wizard-footer></vui-wizard-footer>' +
               '</div>' +
         '</div>'
   };
}]);
;'use strict';
vui.angular.wizard

/**
 * @name vui.angular.wizard.VuiWizardController
 * @module vui.angular.wizard
 *
 * @description
 *    Controller for {@link vui.angular.wizard.directive:vuiWizard vui-wizard} directive.
 */

.controller('VuiWizardController', ['$scope', '$element', '$q', '$log', 'vuiConstants',
         function ($scope, $element, $q, $log, vuiConstants) {
   var vuiWizardOptions = $scope[$element.attr('vui-wizard')];
   $scope.vuiWizard = vuiWizardOptions;

   var self = this;
   var wizard = {};
   var wizardManager = {};

   /**
   * == Wizard page states ==
   * INCOMPLETE - Indicates that the page is ready for user input, but
   * still incomplete. This is the default value for the page state.
   * COMPLETE - Indicates that the page is complete.
   * The page is considered complete when its output data
   * is ready to be used (by user input or default values).
   * INVALID - Indicates that the page is invalid; i.e contains
   * validation errors.
   * DISABLED - Indicates that the page can not be activated by the
   * user.
   * SKIPPED - Indicates that the page is excluded from the pages
   * considered by the wizard in all operations.
   * Setting the page state to this value is equivalent to
   * removing it from the wizard pages list.
   */
   var pageStateConst = vuiConstants.wizard.pageState;

   //=============== Private methods ==================

   /**
    *
    * @returns an array of wizard pages.
    */
   var getWizardPages = function (vuiWizardOptions) {
      return getPages(vuiWizardOptions.pages, vuiWizardOptions.flows, []);
   };

   /**
    * Generates array of wizard pages.
    * The path required by wizard.
    *
    * @param pages The default pages
    *
    * @param flows Required to fetch flow
    *
    * @returns array of wizard pages.
    */
   var getPages = function (pages, flows, wizardPages) {
      var decisionPage, flowId, flow;
      for (var  i = 0; i < pages.length; i++) {
         wizardPages.push(pages[i]);
         // Decision Page
         if (pages[i].hasOwnProperty('decisionOptions')) {
            decisionPage = pages[i].decisionOptions;
            if (decisionPage.hasOwnProperty('selectedFlowId')) {
               flowId = decisionPage.selectedFlowId;
            } else {
               flowId = decisionPage.flowIds[0];
            }
            // Fetch required flow object
            flow = fetchFlow(flows, flowId);
            if (flow.hasOwnProperty('pages') && flow.pages.length > 0) {
               getPages(flow.pages, flows, wizardPages);
            } else {
               $log.warn('Flow with Id ' + flowId + ' has no page to insert in the wizard path.');
            }
         }
      }
      return wizardPages;
   };

   // Returns the flow from flows pool with respective flowId
   var fetchFlow = function (flows, flowId) {
      var flow = $.grep(flows, function (e) {
         return (e.id === flowId);
      });
      return flow[0];
   };

   // Creates the path of pages for wizard to follow
   $scope.vuiWizardPages = getWizardPages(vuiWizardOptions);

   var updateWizardVars = function (pageIndex) {
      wizardManager.currentIdx = pageIndex;
      wizard.currentPage = $scope.vuiWizardPages[pageIndex];
      wizardManager.clearValidationBanner();
   };

   // Builds updated wizard path
   // Called from watch on decision page
   // Calls getWizardPages function
   // Validates selectedFlowId
   var generateWizardPages = function () {
      var flowId = wizard.currentPage.decisionOptions.selectedFlowId;
      if (wizard.currentPage.decisionOptions.flowIds.indexOf(flowId) > -1) {
         $scope.vuiWizardPages = getWizardPages(vuiWizardOptions);
      } else {
         // Warning if flowId is not present in the array of flowIds
         $log.warn('Flow with Id ' + flowId + ' is not accessible from this wizard page.');
      }
   };

   //================ Methods exposed via wizard =========

   // Reset the wizard on close
   var resetWizard = function () {
      // scope variable to share info
      $scope.wizardManager.currentIdx = 0;
      $scope.vuiWizard.validationBanner = {
         showAllMessages : false,
         messages : []
      };
      $scope.vuiWizard.currentPage = {};
      wizardManager = $scope.wizardManager;
      wizard = $scope.vuiWizard;
   };

   // Holds the onCommit function of the current page
   var onCommit = function () {
      var pageCommitFunction = wizard.currentPage.onCommit;
      if (pageCommitFunction && typeof pageCommitFunction === 'function') {
         return pageCommitFunction();
      } else {
         return true;
      }
   };

   /**
    * Calls onFinish function of the wizard if available.
    *
    * @returns boolean that onFinish returns else returns true.
    */
   var onFinish = function () {
      if (wizard.onFinish && typeof wizard.onFinish === 'function') {
         return wizard.onFinish();
      } else {
         return true;
      }
   };

   /**
    * Calls onCommit function of currentPage if available
    * and moves to next navigable  page.
    */
   var onNextClick = function () {
      var totalSteps = $scope.vuiWizardPages.length;
      if ($scope.isOnCommitInProgress) {
         return;
      }

      $scope.isOnCommitInProgress = true;

      function successCallback(result) {
         $scope.isOnCommitInProgress = false;
         if (!result) {
            return;
         }
         if (wizardManager.currentIdx === totalSteps - 1) {
            onFinishClick();
            return;
         }
         wizardManager.currentIdx++;
         while ($scope.vuiWizardPages[wizardManager.currentIdx].state ===
         pageStateConst.SKIPPED &&
         $scope.vuiWizardPages[wizardManager.currentIdx].state !==
         pageStateConst.DISABLED &&
         totalSteps > wizardManager.currentIdx) {
            wizardManager.currentIdx++;
         }
         wizard.currentPage = $scope.vuiWizardPages[wizardManager.currentIdx];
         wizardManager.clearValidationBanner();
      }

      function failureCallback() {
         $scope.isOnCommitInProgress = false;
      }

      try {
         $q.when(onCommit()).then(successCallback, failureCallback);
      } catch (e) {
         failureCallback();
         throw e;
      }
   };

   /**
    * Handles jumping between pages when user clicks on a page in TOC.
    *
    * @param pageIndex The index of the page clicked.
    *
    */
   var moveToPage = function (pageIndex) {

      // Do nothing when user clicks on the same page or a disabled page.
      if (wizardManager.currentIdx === pageIndex ||
            $scope.vuiWizardPages[pageIndex].state === pageStateConst.DISABLED) {
         return;
      }

      // Call  onCommit of currentPage when jumping to a page
      // with index < currentIdx.
      if (pageIndex < wizardManager.currentIdx) {
         updateWizardVars(pageIndex);
         return;
      } else {
         $q.when(onCommit()).then(function (result) {
            if (!result) {
               return;
            }

            // if user has clicked to some page in the TOC that might become disabled
            // after the onCommit() function (which can disable the subsequent pages),
            // we have to get the index of the first non-skipped and non-disabled
            // page before the page we are navigating to.
            var wizardPages = $scope.vuiWizardPages;
            var currentPageIndex = wizardManager.currentIdx;
            var nextPageIndex = -1;
            for (var i = pageIndex; i >= currentPageIndex; i--) {
               var isPageEnabled = wizardPages[i].state !== pageStateConst.SKIPPED &&
                     wizardPages[i].state !== pageStateConst.DISABLED;
               if (isPageEnabled) {
                  nextPageIndex = i;
                  break;
               }
            }

            if (nextPageIndex < 0 || nextPageIndex >= wizardPages.length) {
               nextPageIndex = pageIndex;
            }

            updateWizardVars(nextPageIndex);
         });
      }
   };

   var clearValidationBanner = function () {
      wizard.validationBanner.messages = [];
      wizard.validationBanner.showAllMessages = false;
   };

   /**
    * Calls onCommit, if onCommit returns true calls onFinish and
    * then finally calls closeWizard if onFinish returns true.
    */
   var onFinishClick = function () {
      $q.when(onCommit()).then(function (result) {
         if (!result) {
            return;
         } else {
            $q.when(onFinish()).then(function (result) {
               if (!result) {
                  return;
               }
               closeWizard();
            });
         }
      });
   };
   var closeWizard = function () {
      // Call onClose if available. Dont wait on the return value.
      if (wizard.onClose && typeof wizard.onClose === 'function') {
         wizard.onClose();
      }
      wizard.show = false;
      resetWizard();
   };

   self.initialize = function () {
      var totalSteps = -1;
      resetWizard();
      totalSteps = wizard.pages.length;
      // To handle the case where first page is a skipped page
      while ($scope.vuiWizardPages[wizardManager.currentIdx].state ===
            pageStateConst.SKIPPED &&
            totalSteps > wizardManager.currentIdx) {
         wizardManager.currentIdx++;
      }
      //Refreshing the list of pages.
      updateWizardVars(0);
      $scope.vuiWizardPages = getWizardPages(vuiWizardOptions);
      wizard.currentPage = $scope.vuiWizardPages[wizardManager.currentIdx];
   };

   $scope.wizardManager = {
      disableFinishFlag : true,
      onNextClick : onNextClick,
      moveToPage : moveToPage,
      onFinishClick : onFinishClick,
      closeWizard : closeWizard,
      clearValidationBanner : clearValidationBanner,
      resetWizard : resetWizard,
      generateWizardPages : generateWizardPages
   };
}]);
;'use strict';
vui.angular.wizard

/**
 * This Directive loads the wizard footer buttons.
 * Disables or enables the buttons based on the current step.
 */
.directive('vuiWizardFooter', ['vuiConstants', '$q', function (vuiConstants, $q) {
   return {
      template:
         '<button class="btn btn-link" ng-click="onClose()" aria-label="{{vuiLocale.cancel}}">{{vuiLocale.cancel}}</button>' +
         '<button class="btn" ng-click="onBackClick()"' +
               'ng-disabled="disableBack()" aria-label="{{vuiLocale.back}}">{{vuiLocale.back}}</button>' +
         '<button class="btn btn-primary" ng-click="onNextClick()" aria-label="{{vuiLocale.next}}" ' +
               'ng-disabled="disableNext()" ng-if="!includeFinish()" data-type="next">{{vuiLocale.next}}</button>' +
         '<button class="btn btn-primary" ng-click="onFinish()" aria-label="{{vuiLocale.finish}}" data-type="finish" ' +
               'ng-disabled="!finishReady()" ng-if="includeFinish()">{{vuiLocale.finish}}</button>' +
         '<span class="wizard-resize-icon-span"></span>',
      scope: true,
      restrict: 'E',
      controller: ['$scope', 'vuiLocale', function ($scope, vuiLocale) {
         $scope.vuiLocale = vuiLocale;
         var wizard = $scope.vuiWizard;
         var wizardManager = $scope.wizardManager;
         var pageStateConst = vuiConstants.wizard.pageState;
         var wizardPages;

         var updateWizardPages = function () {
            wizardPages = $scope.vuiWizardPages;
         };

         $scope.disableBack = function () {
            updateWizardPages();
            if (wizard.loading) {
               return true;
            }

            var firstNonSkippedPageIndex = 0;
            while (wizardPages && firstNonSkippedPageIndex < wizardPages.length &&
                  wizardPages[firstNonSkippedPageIndex].state === pageStateConst.SKIPPED) {
               firstNonSkippedPageIndex++;
            }
            return wizardManager.currentIdx === firstNonSkippedPageIndex;
         };

         $scope.disableNext = function () {
            updateWizardPages();
            if (wizard.loading) {
               return true;
            }

            var disableNextFn = wizardPages[wizardManager.currentIdx] && wizardPages[wizardManager.currentIdx].disableNext;
            if (angular.isFunction(disableNextFn) && disableNextFn()) {
               return true;
            }

            var lastNonSkippedPageIndex = wizardPages.length - 1;
            while (wizardPages && lastNonSkippedPageIndex >= 0 &&
                  wizardPages[lastNonSkippedPageIndex].state === pageStateConst.SKIPPED) {
               lastNonSkippedPageIndex--;
            }

            return wizardManager.currentIdx === lastNonSkippedPageIndex;
         };

         $scope.finishReady = function () {
            updateWizardPages();
            var ready = true;
            var currentPage = wizardPages[wizardManager.currentIdx];
            if (currentPage.finishReady) {
               ready = currentPage.finishReady();
            } else {
               ready = true;
            }
            return ready;
         };

         $scope.includeFinish = function () {
            return wizardManager.enableFinishFlag;
         };

         $scope.onBackClick = function () {
            updateWizardPages();
            wizardManager.currentIdx--;
            while (wizardPages[wizardManager.currentIdx].state ===
               pageStateConst.SKIPPED && wizardManager.currentIdx !== 0) {
               wizardManager.currentIdx--;
            }
            wizard.currentPage = wizardPages[wizardManager.currentIdx];
            wizardManager.clearValidationBanner();
         };
         $scope.onNextClick = function () {
            wizardManager.onNextClick();
         };
         $scope.onClose = function () {
            // Call on close callback if provided by user, otherwise go back to start state
            if (wizard.onClose && angular.isFunction(wizard.onClose)) {
               $q.when(wizard.onClose()).then(function (result) {
                  if (!result) {
                     return;
                  } else {
                     wizard.show = false;
                  }
               });
            } else {
               wizard.show = false;
            }
         };
         $scope.onFinish = function () {
            wizardManager.onFinishClick();
         };
      }]
   };
}]);
;'use strict';
vui.angular.wizard

/**
 * @ngdoc object
 * @name vui.angular.wizard:WizardOptions
 * @module vui.angular.wizard
 *
 * @description
 *    **Extends** {@link vui.angular.modal:ModalOptions ModalOptions}.
 *
 *    Configuration object for {@link vui.angular.wizard.vuiWizardService vuiWizardService}.
 *    Contain properties to specify the wizard and populate it with pages.
 *
 *    Example:
 *    ```js
 *    var wizardOptions = {
 *                            title: 'Bake a cake',
 *                            iconClass: 'icon-create',
 *                            show: false,
 *                            pages: [ ... ]
 *                        };
 *    ```
 *
 * @property {string} title
 *    Type: `string`
 *
 *    Specifies the title of the wizard.
 * @property {string} [iconClass]
 *    Type: `string` **|** Optional
 *
 *    The CSS class name representing the icon displayed in the wizard title bar.
 *
 *    **Note:** Please refer to the vui-bootstrap documentation for a list of
 *    available icon classes. A custom class specifying a 16x16 icon can also
 *    be used.
 * @property {Array.<Object>} pages
 *    Type: `Array.<Object>`
 *
 *    Array of {@link vui.angular.wizard:Page Page} objects defining the
 *    properties of each page in the wizard. The pages are displayed in the
 *    same order in which they are specified in the array.
 *
 *    Example:
 *    ```js
 *    pages: [ page1, page2 ]
 *    ```
 *    `page1` and `page2` are {@link vui.angular.wizard:Page Page}
 *    objects.
 *
 *    **Note:** If you want to preserve data during navigation between pages, save
 *    the current page data on the object outer to the page scope, i.e. `$scope.vuiWizard.data`.
 *    Check the sample code {@link vui.angular.wizard.vuiWizardService here} in app.js.
 *
 * @property {Array.<Object>} flows
 *    Type: `Array.<Object>`
 *
 *    Array of {@link vui.angular.wizard:Flow Flow} objects defining the
 *    properties of each flow in the wizard. These flow objects are used to build different wizard paths.
 *
 *    Example:
 *    ```js
 *    flows: [ flow1, flow2 ]
 *    ```
 *    `flow1` and `flow2` are {@link vui.angular.wizard:Flow Flow}
 *    objects.
 *
 *
 * @property {Function} [onFinish]
 *    Type: `Function` **|** Optional
 *
 *    A function that will be called when Finish button is clicked.
 *    It should return true/false or a promise (resolving to true/false).
 *    If the return value is true the wizard is closed, else its not.
 *
 *    If {@link vui.angular.wizard:Page onCommit} function for the current page is defined,
 *    onFinish function will be called only when the onCommit function returns true.
 *
 *    Example:
 *    ```js
 *    onFinish: function () {
 *                 ...
 *              };
 *    ```
 * @property {Function} [onClose]
 *    Type: `Function` **|** Optional
 *
 *    A function that will be called when the wizard closes i.e.,
 *    when Cancel button is clicked or when Finish button is clicked and
 *    the wizard closes after{@link vui.angular.wizard:WizardOptions onFinish}
 *    function returns true.
 *
 *    Example:
 *    ```js
 *    onClose: function () {
 *                 ...
 *             };
 *    ```
 * @property {Object} currentPage
 *    Type: `Object`
 *
 *    Read-only property that specifies object of current page displayed by the
 *    wizard. This property can be used to dynamically set the
 *    {@link vui.angular.wizard:Page onCommit} function of the page.
 *
 *    Example:
 *    ```js
 *    $scope.vuiWizard.currentPage.onCommit = function () {
 *          $scope.vuiWizard.currentPage.state =
 *                vuiConstants.wizard.pageState.COMPLETED;
 *          return true;
 *       };
 *    ```
 *    **Note:** Inside the page scope you can access the configObject source using `$scope.vuiWizard`.
 *
 * @property {Object} [validationBanner]
 *    Type: `Object` **|** Optional
 *
 *    This is used to pop-up the validationBanner with errors/warnings when required. Here are the
 *    {@link vui.angular.validationBanner:ValidationBannerOptions options}.
 *
 * @property {boolean} [closeOnEsc]
 *    Type: `boolean` **|** Optional **|** Default: `false`
 *
 *    Determines whether to hide the wizard when the escape key is pressed.
 *
 * @property {boolean} [loading]
 *
 *    Whether to show the indeterminate loading indicator.
 *
 * @property {String} [loadingTemplate]
 *
 *    The template to render when loadingMessage is set.
 *
 * @property {String} [loadingMessage]
 *
 *    Property that determines whether a loading message should be shown, and what
 *    the localized content of the loading message is.
 *
 *    A loading message can be used in the loading template like so:
 *    ```html
 *       <div class='my-loading-indicator'>
 *          {{vuiWizard.loadingMessage}}
 *       </div>
 *    ```
 *
 * @property {String} [loadingAriaMessage]
 *    Property that determines the loading message which should be read by the
 *    screen readers when the loading is in progress.
 */

 /**
 * @ngdoc object
 * @name vui.angular.wizard:Page
 * @module vui.angular.wizard
 *
 * @description
 *    Contains properties to define a page in the wizard.
 *
 *    Example:
 *    ```js
 *    var page1 = {
 *                   title: 'Page1',
 *                   description: 'Sample page',
 *                   contentUrl: 'page1.html',
 *                   state: vuiConstants.wizard.pageState.INCOMPLETE
 *                }
 *    ```
 *
 * @property {string} title
 *    Type: `string`
 *
 *    Specifies the title of the page.
 * @property {string} description
 *    Type: `string`
 *
 *    Description of what the page is meant for.
 * @property {string} contentUrl
 *    Type: `string`
 *
 *    The template URL of web content that will be loaded into and displayed
 *    within this page.
 *
 *   **Note:** The page content scope prototypically inherits from the
 *    scope thats mentioned in the config object during wizard creation.
 *
 * @property {string} state
 *    Type: `string`
 *
 *    Defines the state of the wizard page.
 *
 *    Legal values:
 *
 *    - **`vuiConstants.wizard.pageState.COMPLETED`** : Use to indicate that
 *       the page is complete. The page is considered complete when its output
 *       data is ready to be used (by user input or default values).
 *    - **`vuiConstants.wizard.pageState.INCOMPLETE`** : Use to indicate that
 *       the page is incomplete and ready for user input.
 *    - **`vuiConstants.wizard.pageState.DISABLED`** : Use to indicate that
 *       the page can not be visited by the user. The page title will be displayed
 *       in the table of contents with a disbled state.
 *    - **`vuiConstants.wizard.pageState.SKIPPED`** : Use to indicate that the
 *       page is excluded from the pages considered by the wizard in all
 *       operations. The page will not be displayed.
 *
 *   **Note:** The state of first page  should be either **`COMPLETED`**
 *   or **`INCOMPLETE`**, else the page will not be displayed.
 *
 * @property {object} [decisionOptions]

 *    Type: `Object<decisionOptions>` **|** Optional
 *
 *    Contains {@link vui.angular.wizard:DecisionOptions properties} used by the page to trigger flows, based on the user input.
 *
 *    This would then update the wizard path.
 *
 * @property {Function} [onCommit]
 *    Type: `Function` **|** Optional
 *
 *    A function that will be called when Next button is clicked. It should
 *    return true/false or a promise (resolving to true/false).
 *    If the return value is true the wizard shows the next page, else it stays
 *    on the same page. This function can be used to change the page state of the
 *    current page or set validation banner messages if required.
 *
 *    This function is also called when an user clicks on any successive page
 *    on the TOC. If the return value is true the wizard navigates to the page that
 *    was clicked, else it stays on the same page.
 *
 *    Example:
 *    ```js
 *    onCommit: function () {
 *                 ...
 *              };
 *    ```
 *
 * @property {Function} [disableNext]
 *    Type: `Function` **|** Optional
 *
 *    A function that will be called in order to determine if the Next button in the footer should be disabled.
 *    It should return true/false.
 *    If the return value is true the wizard will disable the Next button
 *
 *    Example:
 *    ```js
 *    disableNext: function () {
 *                 return true/false based on some page logic
 *              };
 *    ```
 *
 *
 * @property {Function} [finishReady]
 *    Type: `Function` **|** Optional
 *
 *    A function that will be called in order to determine if the Finish button on the last page of the wizard
 *    should be disabled.
 *
 *    This function will only matter if defined on the last page of a wizard
 *
 *    It should return true/false.
 *    If the return value is true the wizard will disable the Finish button
 *
 *    Example:
 *    ```js
 *    finishReady: function () {
 *                    return true/false based on some page logic
 *                 };
 *    ```
 */

/**
 * @ngdoc object
 * @name vui.angular.wizard:DecisionOptions
 * @module vui.angular.wizard
 *
 * @description
 *    Contains properties to define decision options for page in the wizard.
 *
 *    Example:
 *    ```js
 *    decisionOptions: {
 *                  selectedFlowId: ‘flowId1’,
 *                  flowIds: [ 'flowId1', 'flowId2' ]
 *              }
 *    ```
 *
 * @property {string} [selectedFlowId]
 *    Type: `string` **|** Optional **|** Default: `flowIds[0]`
 *
 *    Set this property to the id of the flow that needs to be selected.
 *    This means that the pages contained in the specified flow will be inserted into the
 *    wizard path after this page.
 *
 *    If not specified, by default the first flowId in the array of flowIds will be used as selectedFlowId.
 *
 *    It should be one of the identifier specified in the array of flowIds.
 *
 *   **Note:** Must be updated, according to user input on the page. This would update the wizard path.
 *
 * @property {Array.<string>} flowIds
 *    Type: `Array.<string>`
 *
 *    Maintains an array of identifiers of {@link vui.angular.wizard:Flow FLow} objects,
 *    which signify the flows allowed to be triggered from this page.
 *
 */

/**
 * @ngdoc object
 * @name vui.angular.wizard:Flow
 * @module vui.angular.wizard
 *
 * @description
 *    Contains properties to define a flow in the wizard.
 *
 *    Example:
 *    ```js
 *    var flow1 = {
 *                   id: flowId1,
 *                   pages: [{
 *                          title: page4,
 *                          description: This is first page of flow with FlowId 1,
 *                          state: INCOMPLETE,
 *                          contentUrl: page4.html
 *                    }, {
 *                         title: page5,
 *                         description: This is second page of flow with FlowId 1,
 *                         state: INCOMPLETE,
 *                         contentUrl: page5.html
 *                }]
 *    ```
 *
 * @property {string} id
 *    Type: `string`
 *
 *    A unique identifier for the flow.
 *
 * @property {Array.<Object>} pages
 *    Type: `Array.<Object>`
 *
 *    Array of {@link vui.angular.wizard:Page Page} objects defining the
 *    properties of each page in this flow. The pages are displayed in the
 *    same order in which they are specified in the array.
 *
 *    Example:
 *    ```js
 *    pages: [ page1, page2 ]
 *    ```
 *    `page1` and `page2` are {@link vui.angular.wizard:Page Page}
 *    objects.
 *
 */

/**
 * @ngdoc service
 * @name vui.angular.wizard.vuiWizardService
 * @module vui.angular.wizard
 *
 * @description
 *    Service to create multiple instances of the wizard. The service allows
 *    you to create, show, hide and destroy the wizard.
 *
 * # General usage
 *    The `vuiWizardService` is a function which takes a single argument, a
 *    configuration object that is used to create a wizard.
 *
 * @param {Object} config
 *    Object containing the scope and the configObjectName.
 *
 *    - **scope** – `{$scope}` – Angular scope object of the wizard container.
 *       It can be the scope of the controller or rootScope or a custom
 *       scope.
 *
 *       The scope of each page in the wizard prototypically inherits from the
 *    scope thats mentioned here.
 *    - **configObjectName** – `{string}` – The name of the scope variable
 *       declared on the above mentioned scope. The variable is of type
 *       {@link vui.angular.wizard:WizardOptions WizardOptions}.
 *
 *    Example:
 *    ```js
 *    var config = {
 *                     scope: $scope,
 *                     configObjectName: 'wizardOptions'
 *                 };
 *    ```
 *
 * @returns {Object}
 *    Returns a response object which has these methods:
 *
 *    - **show()** – `{Function}` To display the wizard.
 *    - **hide()** – `{Function}` To hide the wizard.
 *    - **destroy()** – `{Function}` To destroy the wizard.
 *
 * @example
      <example module="app">

         <file name="app.js">
            'use strict';
            var appModule = angular.module('app', ['vui.angular.wizard']);

            appModule.controller('WizardCtrl', ['$scope', 'vuiConstants', 'vuiWizardService',
                  function ($scope, vuiConstants, vuiWizardService) {

               // Get pagestate from vuiConstants service.
               var pageStateConst = vuiConstants.wizard.pageState;

               // Create a scope object to define wizard options.
               $scope.wizardOptions = {
                  'show' : false,
                  'title' : 'Create a Virtual Machine',
                  'iconClass' : 'vui-icon-home',
                  'closeOnEsc': true
               };

               // Wizard pages
               $scope.wizardOptions.pages = [{
                     title: 'Configure Virtual Machine ',
                     contentUrl: 'page1.html',
                     state: pageStateConst.INCOMPLETE,
                     description: 'Configure a new Virtual Machine'
                  }, {
                     title: 'Select OS',
                     contentUrl: 'page2.html',
                     state: pageStateConst.INCOMPLETE,
                     description: 'Select the Operating System'
                  }, {
                     title: 'Ready to complete',
                     contentUrl: 'page3.html',
                     state: pageStateConst.INCOMPLETE,
                     description: 'Review your selections before finishing the wizard.'
                  }
               ];

               $scope.statusText = 'Wizard not yet created';

               // Some global data that needs to be persisted across pages
               $scope.wizardOptions.data = {completed : false, os: 'Windows8' };

               var resetWizardConfig = function () {
                  // Reset page states
                  $scope.wizardOptions.pages[0].state = pageStateConst.INCOMPLETE;
                  $scope.wizardOptions.pages[1].state = pageStateConst.INCOMPLETE;
                  $scope.wizardOptions.pages[2].state = pageStateConst.INCOMPLETE;
                  // Set global data to default
                  $scope.wizardOptions.data = {completed : false, os: 'Windows8' };
               };

               $scope.wizardOptions.onClose = function () {
                  $scope.statusText = 'Wizard closed';
                  resetWizardConfig();
                  return true;
               };

               // Mark pages 2 and 3 as complete when page 1 is filled up with right values.
               $scope.$watch('wizardOptions.data.completed', function (value) {
                  if (value === true) {
                     $scope.wizardOptions.pages[1].state = pageStateConst.COMPLETED;
                  }
               });

               // Config object for the wizard service
               var config = {
                  scope: $scope,
                  configObjectName:  'wizardOptions'
               };

               // To hold the object we get back form the vuiWizard service.
               var wizard1 =  null;

               $scope.createWizard = function () {
                  if (wizard1 === null) {
                     wizard1 = vuiWizardService(config);
                     $scope.statusText = 'Wizard created';
                  } else {
                     alert('Wizard already exists');
                  }
               };

               $scope.openWizard = function () {
                  if (wizard1 === null) {
                     alert('Create a wizard first.');
                     return;
                  }
                  wizard1.show();
                  $scope.statusText = 'Wizard opened';
               };

               $scope.destroyWizard = function () {
                  if (!wizard1) {
                     return;
                  }
                  wizard1.destroy();
                  wizard1 = null;
                  resetWizardConfig();
                  $scope.statusText = 'Wizard destroyed';
               };

            }]);

            appModule.controller('page1Ctrl', ['$scope', 'vuiConstants',
                  function ($scope, vuiConstants) {
               // Retrieve values form the global data object when page is initialized
               $scope.name = $scope.vuiWizard.data.name;
               $scope.ram = $scope.vuiWizard.data.ram;
               // Function to be called when Next button is pressed.
               // vuiWizard is a reference to the configObject declared
               // in outer scope in this case - 'wizardOptions'.
               $scope.vuiWizard.currentPage.onCommit = function () {
                  if ($scope.name && $scope.ram) {
                     $scope.vuiWizard.currentPage.state =
                           vuiConstants.wizard.pageState.COMPLETED;
                     // Mark the global var to true. So that other pages can be marked as complete.
                     $scope.vuiWizard.data.completed = true;
                     // Storing the name and age onto the wizard.data object for persisting.
                     $scope.vuiWizard.data.name = $scope.name;
                     $scope.vuiWizard.data.ram = $scope.ram;
                     return true;
                  } else {
                     $scope.vuiWizard.validationBanner.messages  = [{
                        text: 'Fill up the fields to proceed',
                        type: vuiConstants.validationBanner.type.ERROR
                     }];
                     return false;
                  }
               };
            }]);

            appModule.controller('page2Ctrl', ['$scope', function ($scope) {
               $scope.operatingSystems = ['Windows8', 'Linux64', 'Windows7'];
               $scope.os = $scope.vuiWizard.data.os || $scope.operatingSystems[0];
               $scope.vuiWizard.currentPage.onCommit = function () {
                  $scope.vuiWizard.data.os = $scope.os;
                  return true;
               };
            }]);

         </file>

         <file name="index.html">
            <div ng-controller="WizardCtrl">
               <h3>Using wizard as a service</h3>
               <p>
                  This is a sample create VM wizard. The first page has fields that are
                  required to be filled. Once you click next and all the required fields
                  in Page 1 are filled, Page2 and Page3 are set to COMPLETED state as thet
                  have default values pre-filled.
               </p>
               <div class="container">
                  <button ng-click="createWizard()">Create Wizard</button>
                  <button ng-click="openWizard()">Open Wizard</button>
                  <button ng-click="destroyWizard()">Destroy Wizard</button>
                  <br><br><b style="color:green;">{{statusText}}</b>
               </div>
            </div>
         </file>

         <file name="page1.html">
            <div ng-controller="page1Ctrl">
               <form>
                  <div class="form-group form-group-horizontal">
                     <div class="label-col col-xs-3">
                        <label for="name">Name:</label>
                     </div>
                     <div class="input-col col-xs-5">
                        <input type="text" id="name" ng-model="name"
                              placeholder="vm1">
                     </div>
                  </div>
                  <div class="form-group form-group-horizontal">
                     <div class="label-col col-xs-3">
                        <label for="ram">RAM:</label>
                     </div>
                     <div class="input-col col-xs-5">
                        <input type="text" id="ram" ng-model="ram"
                              placeholder="8">
                     </div>
                     <div class="col-xs-2">
                        GB
                     </div>
                  </div>
               </form>
            </div>
         </file>

         <file name="page2.html">
            <div ng-controller="page2Ctrl">
               <div ng-controller="page2Ctrl">
                  <form>
                     <div class="row">
                        <div class="col-md-4">
                           <label class="control-label" for="os">
                              Operating Systems:
                           </label>
                        </div>
                        <div class="icol-md-4">
                           <select id="os" ng-model="os"
                                 ng-options="os as os for os in operatingSystems">
                           </select>
                        </div>
                     </div>
                  </form>
               </div>
            </div>
         </file>

         <file name="page3.html">
            <p>
               This page contains a summary of the data that the user entered in the wizard.
            </p>
            <ul>
               <li>Name : {{vuiWizard.data.name}}</li>
               <li>RAM : {{vuiWizard.data.ram}}</li>
               <li>OS : {{vuiWizard.data.os}}</li>
            </ul>
         </file>

      </example>
 */

.provider('vuiWizardService', function () {

   var explicitDefaultConfig = null;
   var explicitDefaultOptions = null;

   this.setDefaultConfig = function (defaultConfig) {
      explicitDefaultConfig = $.extend(true, {}, defaultConfig);
   };

   this.setDefaultWizardOptions = function (defaultOptions) {
      explicitDefaultOptions = $.extend(true, {}, defaultOptions);
   };

   this.$get = ['$window', '$rootScope', '$compile', 'vuiModalService', 'vuiConstants', function ($window, $rootScope, $compile, vuiModalService, vuiConstants) {

      var DEFAULT_CONFIG = {
         showBackdrop: true,
         zIndex: vuiConstants.internal.zIndex.WIZARD
      };

      // Currently there aren't any default options for the wizard. Leave this to have
      // parity with modal and dialog and also to future-proof ourselves
      var DEFAULT_OPTIONS = {
         closeOnEsc: false,
         submitOnEnter: true
      };

      function WizardFactory(config) {
         var wizardDirectiveTemplate;

         var defaultConfig = $.extend(true, { }, DEFAULT_CONFIG, explicitDefaultConfig);
         var defaultOptions = $.extend(true, { }, DEFAULT_OPTIONS, explicitDefaultOptions);

         var options = config.scope[config.configObjectName];
         $.extend(config, $.extend({}, defaultConfig, config));
         $.extend(options, $.extend({}, defaultOptions, options));

         // The wizard directive template
         wizardDirectiveTemplate =
               '<div vui-wizard="' + config.configObjectName + '"></div>';

         config._vuiModalTemplate = wizardDirectiveTemplate;
         config.headerElementSelector = '> .wizard-modal-titlebar';
         config._outerScopeActual = null;

         return vuiModalService(config);
      }

      return WizardFactory;
   }];
});
;'use strict';
vui.angular.wizard

   .directive('vuiWizardToc', ['vuiConstants', 'vuiTreeAccessibilityServiceFactory',
              function (vuiConstants, accessibilityServiceFactory) {

      var iconBase64 = 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAAAXNSR0IArs4c6QAABHppVFh0WE1MOmNvbS5hZG9iZS54bXAAAAAAADx4OnhtcG1ldGEgeG1sbnM6eD0iYWRvYmU6bnM6bWV0YS8iIHg6eG1wdGs9IlhNUCBDb3JlIDUuNC4wIj4KICAgPHJkZjpSREYgeG1sbnM6cmRmPSJodHRwOi8vd3d3LnczLm9yZy8xOTk5LzAyLzIyLXJkZi1zeW50YXgtbnMjIj4KICAgICAgPHJkZjpEZXNjcmlwdGlvbiByZGY6YWJvdXQ9IiIKICAgICAgICAgICAgeG1sbnM6eG1wTU09Imh0dHA6Ly9ucy5hZG9iZS5jb20veGFwLzEuMC9tbS8iCiAgICAgICAgICAgIHhtbG5zOnN0UmVmPSJodHRwOi8vbnMuYWRvYmUuY29tL3hhcC8xLjAvc1R5cGUvUmVzb3VyY2VSZWYjIgogICAgICAgICAgICB4bWxuczp0aWZmPSJodHRwOi8vbnMuYWRvYmUuY29tL3RpZmYvMS4wLyIKICAgICAgICAgICAgeG1sbnM6eG1wPSJodHRwOi8vbnMuYWRvYmUuY29tL3hhcC8xLjAvIj4KICAgICAgICAgPHhtcE1NOkRlcml2ZWRGcm9tIHJkZjpwYXJzZVR5cGU9IlJlc291cmNlIj4KICAgICAgICAgICAgPHN0UmVmOmluc3RhbmNlSUQ+eG1wLmlpZDo0QURGRTI5NjEzMjA2ODExODIyQUJGMzNERjE1NzQyRTwvc3RSZWY6aW5zdGFuY2VJRD4KICAgICAgICAgICAgPHN0UmVmOmRvY3VtZW50SUQ+eG1wLmRpZDowMTgwMTE3NDA3MjA2ODExOEMxNEM3QTM5NUE3RTRFMTwvc3RSZWY6ZG9jdW1lbnRJRD4KICAgICAgICAgPC94bXBNTTpEZXJpdmVkRnJvbT4KICAgICAgICAgPHhtcE1NOkRvY3VtZW50SUQ+eG1wLmRpZDozRUI3RDAwODQ4RTIxMUU0QTQ2NDg2QzZFRkFFODM2NDwveG1wTU06RG9jdW1lbnRJRD4KICAgICAgICAgPHhtcE1NOkluc3RhbmNlSUQ+eG1wLmlpZDozRUI3RDAwNzQ4RTIxMUU0QTQ2NDg2QzZFRkFFODM2NDwveG1wTU06SW5zdGFuY2VJRD4KICAgICAgICAgPHhtcE1NOk9yaWdpbmFsRG9jdW1lbnRJRD54bXAuZGlkOjAxODAxMTc0MDcyMDY4MTE4QzE0QzdBMzk1QTdFNEUxPC94bXBNTTpPcmlnaW5hbERvY3VtZW50SUQ+CiAgICAgICAgIDx0aWZmOk9yaWVudGF0aW9uPjE8L3RpZmY6T3JpZW50YXRpb24+CiAgICAgICAgIDx4bXA6Q3JlYXRvclRvb2w+QWRvYmUgUGhvdG9zaG9wIENTNiAoTWFjaW50b3NoKTwveG1wOkNyZWF0b3JUb29sPgogICAgICA8L3JkZjpEZXNjcmlwdGlvbj4KICAgPC9yZGY6UkRGPgo8L3g6eG1wbWV0YT4KOz7cZwAAANtJREFUOBFjYBhowEiKA2zWqP5nYGZgYGT5337Y904VSC8TsQaANHOLcTKw8DAyfHvzqxKmjygXgDSzijAx/AfCVxc/MlzLfwnXB2fYrlXN/fX9zySQySdj7sPFQZqZ+RmBWhkY3lxD1QxSC/fC2yefJrEKszCAMNivQEkQzcTNwPDvzz+smkEGwG0CcbQmiv8XUuAF28b4GyjFCnL0f4b3D76gOBukFgZQDAAJggzhl+YGBjYzw18g/Pj0K07NIPUYBsAM4RHjZvjyCr9mkFqcAOQSnJKjEtQNAQB/8VEEGU05ZQAAAABJRU5ErkJggg==';

      return {
         template:
         '<div class="wizard-panel-TOC"' +
            ' ng-class="{\'show\': vuiWizard.showTocExplicitly}"' +
            ' role="navigation"' +
            ' tabindex="0"' +
            ' aria-label="{{vuiWizard.title}}">' +
            '<ul class="wizard-steps-list" role="menubar">' +
               '<li ng-repeat="page in vuiWizardPages" ng-class="getStepClass(page.state, $index)" ' +
                  'ng-if="criteriaMatch(page.state)" ' +
                  'ng-click="moveTo($index)" ' +
                  'role="menuitem" ' +
                  'aria-selected="{{getAriaActive($index) ? \'true\' : \'false\'}}"' +
                  'aria-disabled="{{isDisabled(page.state)}}"' +
                  'tabindex="-1">' +
                  '<span class="vui-inline-icon-container">' +
                     '<ng-container ng-if="page.state===\'completed\'">' +
                        '<img ng-src="{{icon}}" alt="Completed"/>' +
                     '</ng-container>' +
                  '</span>' +
            '<a>{{page.title}}</a>' +
               '</li>' +
            '</ul>' +
         '</div>',
         scope : true,
         restrict: 'E',
         controller: ['$scope', '$element', function ($scope, $element) {
            var pageStateConst = vuiConstants.wizard.pageState;
            var wizardManager = $scope.wizardManager;

            setTimeout(function () {
               accessibilityServiceFactory.create($element, $scope);
            }, 0);

            $scope.icon = iconBase64;

            $scope.criteriaMatch = function (pageState) {
               return pageState !== pageStateConst.SKIPPED;
            };
            $scope.getStepClass = function (pageState, pageIndex) {
               var stepClass = (pageIndex === wizardManager.currentIdx) ?
                  'wizard-steps-current ' : '';
               if (pageState === pageStateConst.DISABLED) {
                  stepClass += 'wizard-steps-not-available';
               } else if (pageState === pageStateConst.COMPLETED) {
                  stepClass += 'wizard-steps-completed';
               } else if (pageState === pageStateConst.INCOMPLETE) {
                  stepClass += 'wizard-steps-incomplete';
               }
               return stepClass;
            };

            $scope.isDisabled = function (pageState) {
               if (pageState === pageStateConst.DISABLED) {
                  return true;
               }
               return false;
            };

            $scope.getAriaActive = function (pageIndex) {
               return pageIndex === wizardManager.currentIdx;
            };

            $scope.moveTo = function (pageIndex) {
               $scope.vuiWizard.showTocExplicitly = false;
               wizardManager.moveToPage(pageIndex);
            };

         }]
      };
   }]);
