/* **********************************************************
 * Copyright 2021 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.cis.util;

import static java.util.concurrent.TimeUnit.MILLISECONDS;

import java.util.Objects;
import java.util.function.Supplier;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vapi.cis.authn.json.JsonSignatureVerificationProcessor;

/**
 * Represents a cache for a value that can be refreshed. The refresh can occur only after the
 * minimum timeout between refreshes expires
 */
public class RefreshableCache<T> {
    private static final Logger logger = LoggerFactory
    .getLogger(JsonSignatureVerificationProcessor.class);

    private final Supplier<T> supplier;
    private final long refreshTimeoutNs;
    private final Object lock = new Object();
    private volatile Holder cache;

    /**
     * @param supplier The supplier will be called whenever the cache is refreshed
     * @param refreshTimeoutMs The minimum period of time in milliseconds between refreshes. That
     *        is, when {@link RefreshableCache#refresh()} is called but the refreshTimeoutMs is not
     *        expired the cache will not be refreshed
     */
    public RefreshableCache(Supplier<T> supplier, long refreshTimeoutMs) {
        if (refreshTimeoutMs < 0) {
            throw new IllegalArgumentException("refreshTimeoutMs cannot be negative");
        }
        Objects.requireNonNull(supplier, "supplier cannot be null");
        this.supplier = supplier;
        this.refreshTimeoutNs = MILLISECONDS.toNanos(refreshTimeoutMs);
        cache = new Holder(supplier.get());
    }

    /**
     * @return The cached value
     */
    public T get() {
        return cache.getValue();
    }

    /**
     * Refreshes the cache only if the refresh timeout has expired
     */
    public void refresh() {
        if (cache.refreshTimeoutExpired()) {
            synchronized (lock) {
                // Check if it is already refreshed
                if (!cache.refreshTimeoutExpired()) {
                    logger.trace("Using newly provisioned cache");
                    return;
                }
                logger.trace("Refreshing cache");
                cache = new Holder(supplier.get());
                return;
            }
        } else {
            logger.trace("Not refreshing cache. Timeout has not expired");
        }
    }

    private class Holder {
        private final T value;
        private final long createdAt;

        Holder(T value) {
            this.value = value;
            this.createdAt = System.nanoTime();
        }

        boolean refreshTimeoutExpired() {
            long now = System.nanoTime();
            boolean canRefresh = (now - createdAt > refreshTimeoutNs);
            return canRefresh;
        }

        T getValue() {
            return value;
        }
    }
}