/* **********************************************************
 * Copyright (c) 2013, 2017-2019 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.bindings;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.Serializable;

import com.vmware.vapi.bindings.type.BinaryType;
import com.vmware.vapi.bindings.type.BooleanType;
import com.vmware.vapi.bindings.type.DoubleType;
import com.vmware.vapi.bindings.type.IntegerType;
import com.vmware.vapi.bindings.type.SecretType;
import com.vmware.vapi.bindings.type.StringType;
import com.vmware.vapi.bindings.type.Type;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.StructValue;
import com.vmware.vapi.internal.bindings.BindingsUtil;
import com.vmware.vapi.internal.bindings.TypeConverter;
import com.vmware.vapi.internal.bindings.TypeConverterImpl;
import com.vmware.vapi.internal.util.Validate;

/**
 * {@link DynamicStructure} implementation backed by a {@link StructValue}.
 */
public final class DynamicStructureImpl implements DynamicStructure, Serializable {
    private static final long serialVersionUID = 1L;
    private transient TypeConverter converter;
    private final StructValue strValue;

    /**
     * Constructor. default TypeConverter will be used.
     *
     * @param strValue the runtime struct value to be converted; must not be
     *        <code>null</code>
     */
    public DynamicStructureImpl(StructValue strValue) {
        this(strValue, new TypeConverterImpl());
    }

    /**
     * Constructor.
     *
     * @param strValue the runtime struct value to be converted; must not be
     *        <code>null</code>
     * @param converter for conversion from/to native types; must not be
     *        <code>null</code>
     */
    public DynamicStructureImpl(StructValue strValue, TypeConverter converter) {
        Validate.notNull(strValue);
        Validate.notNull(converter);
        this.strValue = strValue;
        this.converter = converter;
    }

    @Override
    public StructValue _getDataValue() {
        return strValue;
    }

    @Override
    public <T> T getField(String fieldName, Class<T> targetClass) {
        DataValue fieldValue = strValue.getField(fieldName);
        Type targetType = findTargetType(targetClass);

        switch (fieldValue.getType()) {
            case BLOB:
                validateTargetClass(byte[].class, targetClass);
                break;
            case BOOLEAN:
                validateTargetClass(Boolean.class, targetClass);
                break;
            case DOUBLE:
                validateTargetClass(Double.class, targetClass);
                break;
            case INTEGER:
                validateTargetClass(Long.class, targetClass);
                break;
            case SECRET:
                validateTargetClass(char[].class, targetClass);
                break;
            case STRING:
                validateTargetClass(String.class, targetClass);
                break;
            case STRUCTURE:
                if (!Structure.class.isAssignableFrom(targetClass)) {
                    throw new RuntimeException("Inappropriate target class");
                }
                break;
        }

        return converter.<T>convertToJava(fieldValue, targetType);
    }

    private void validateTargetClass(Class<?> expected, Class<?> actual) {
        if (expected != actual) {
            throw new RuntimeException("Inappropriate target class");
        }
    }

    @Override
    public <T> void setField(String fieldName, T newValue, Class<T> valueClass) {
        Type targetType = findTargetType(valueClass);
        DataValue newDataValue = converter.convertToVapi(newValue, targetType);
        strValue.setField(fieldName, newDataValue);
    }

    @SuppressWarnings("unchecked")
    private Type findTargetType(Class<?> valueClass) {
        if (valueClass == byte[].class) {
            return new BinaryType();
        } else if (valueClass == Boolean.class) {
            return new BooleanType();
        } else if (valueClass == Double.class) {
            return new DoubleType();
        } else if (valueClass == Long.class) {
            return new IntegerType();
        } else if (valueClass == char[].class) {
            return new SecretType();
        } else if (StaticStructure.class.isAssignableFrom(valueClass)) {
            return BindingsUtil.extractBindingType((Class<StaticStructure>) valueClass);
        } else if (valueClass == String.class) {
            return new StringType();
        } else {
            throw new RuntimeException("Unsupported static type class: " +
                    valueClass);
        }
    }

    @Override
    public boolean _hasTypeNameOf(Class<? extends Structure> clazz) {
        return BindingsUtil.hasTypeNameOf(strValue, clazz);
    }

    @Override
    public <T extends Structure> T _convertTo(Class<T> clazz) {
        return BindingsUtil.convertTo(this, clazz, converter.reusableThis());
    }

    @Override
    public String _getCanonicalName() {
        return strValue.getName();
    }

    @Override
    public boolean equals(Object other) {
        if (!DynamicStructure.class.isInstance(other)) {
            // Dynamic structure can be equal only to other dynamic structure.
            return false;
        }

        return strValue.equals(((DynamicStructure)other)._getDataValue());
    }

    @Override
    public int hashCode() {
        return strValue.hashCode();
    }

    private void readObject(ObjectInputStream in) throws IOException, ClassNotFoundException {
        in.defaultReadObject();
        converter = new TypeConverterImpl();
    }

    @Override
    public String toString() {
        return strValue.toString();
    }
}