/* **********************************************************
 * Copyright 2012, 2019 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.bindings;

import java.util.Collections;

import com.vmware.vapi.core.ApiProvider;
import com.vmware.vapi.internal.util.Validate;
import com.vmware.vapi.protocol.ProtocolConnection;
import com.vmware.vapi.protocol.local.LocalConnection;
import com.vmware.vapi.protocol.local.LocalProtocol;
import com.vmware.vapi.provider.ApiInterface;
import com.vmware.vapi.provider.local.LocalProvider;

/**
 * Factory for stubs which wrap local (in-process) service implementations.
 */
public final class LocalStubFactory {

    /**
     * Calls {@link #createStub(ApiInterface, Class, StubConfigurationBase)}
     * with no stub configuration.
     */
    public <T extends Service> T createStub(ApiInterface skeleton,
                                            Class<T> stubIface) {
        return createStub(skeleton, stubIface, null);
    }

    /**
     * Creates a stub for the specified service implementation.
     *
     * @param skeleton service implementation; must not be <code>null</code>
     * @param stubIface stub interface; must not be <code>null</code>
     * @param config stub configuration; can be <code>null</code>
     * @return stub for the service
     */
    public <T extends Service> T createStub(ApiInterface skeleton,
                                            Class<T> stubIface,
                                            StubConfigurationBase config) {
        ApiProvider provider = new LocalProvider("LocalProvider for "
                + skeleton.getIdentifier().toString(),
                Collections.singletonList(skeleton));

        return createStub(skeleton, stubIface, config, provider);
    }

    public <T extends Service> T createStub(ApiInterface skeleton,
                                            Class<T> stubIface,
                                            StubConfigurationBase config,
                                            ApiProvider provider) {
        Validate.notNull(skeleton);
        Validate.notNull(stubIface);

        ProtocolConnection c = new LocalConnection(new LocalProtocol(provider));

        StubFactory stubFactory = new StubFactory(c.getApiProvider());

        return stubFactory.createStub(stubIface, config);
    }
}
