/* **********************************************************
 * Copyright 2013, 2019 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/

/*
 * LocMessageFactory.java --
 *
 *      Factory for LocalizableMessage instances
 */
package com.vmware.vapi.bindings;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import com.vmware.vapi.internal.util.Validate;
import com.vmware.vapi.l10n.LocalizationException;
import com.vmware.vapi.l10n.MessageArgConverter;
import com.vmware.vapi.l10n.TemplateFormatter;
import com.vmware.vapi.std.LocalizableMessage;

/**
 * Factory for {@link LocalizableMessage} instances.
 *
 * <p>Message data (keys, templates) are loaded from {@link ResourceBundle}.
 */
public class LocalizableMessageFactory {
    private ResourceBundle msgBundle;
    private TemplateFormatter formatter;
    private MessageArgConverter msgArgConverter;

    /**
     * Constructor.
     *
     * @param msgBundle resource bundle for messages; can not be
     *                  <code>null</code>
     * @param formatter message template formatter to be used for default
     *                  message generation for the created
     *                  <code>LocalizableMessage</code>s; may be null
     */
    public LocalizableMessageFactory(ResourceBundle msgBundle,
                                     TemplateFormatter formatter) {
        Validate.notNull(msgBundle);

        this.msgBundle = msgBundle;
        this.formatter = formatter;
        this.msgArgConverter = new MessageArgConverter();
    }

    /**
     * Creates a <code>LocalizableMessage</code> for the specified ID and
     * arguments.
     *
     * <p>In case message template can not be found for the specified
     * <code>msgId</code> a default 'unknown' message is returned.
     *
     * <p>If <code>TemplateFormatter</code> is set for this factory
     * {@link LocalizableMessage#getDefaultMessage()} will be set to
     * localized string for the locale of the <code>ResourceBundle</code> for
     * this factory. Otherwise default message will be the raw message template.
     *
     * @param msgId  identifier for the message
     * @param args   arguments to substitute in the message template
     * @return fully constructed <code>LocalizableMessage</code> isntance
     */
    public LocalizableMessage getLocalizableMessage(String msgId,
                                                    String... args) {
        Validate.notNull(msgId);

        try {
            String msgTemplate = msgBundle.getString(msgId);
            List<String> argsList = Arrays.asList(args);

            String defaultMessage = buildDefaultMessage(msgTemplate,
                                                        argsList,
                                                        msgBundle.getLocale());

            return new LocalizableMessage(msgId, defaultMessage, argsList);
        } catch (MissingResourceException ex) {
            return buildUnavailableMessage(msgId, args);
        } catch (ClassCastException ex) {
            return buildUnavailableMessage(msgId, args);
        }
    }

    /**
     * Creates a <code>LocalizableMessage</code> for the specified ID and
     * typed arguments.
     *
     * <p>Supported types are
     * <ul>
     * <li><code>java.lang.Long</code>
     * <li><code>java.lang.Double</code>
     * <li><code>java.lang.String</code>
     * <li><code>java.util.Calendar</code>
     * </ul>
     *
     * @throws LocalizationException if argument of unsupported type is
     *         presented
     *
     * @see #getLocalizableMessage(String, String...)
     */
    public LocalizableMessage getLocalizableMessage(String msgId,
                                                    Object... args) {
        return getLocalizableMessage(msgId,
                                     msgArgConverter.convertToStrings(args));

    }

    LocalizableMessage buildUnavailableMessage(String msgId, String... args) {
        String msgTemplate = "Unknown message ID %s requested with parameters %s";
        return new LocalizableMessage(
                "vapi.message.unknown",
                String.format(msgTemplate, msgId, Arrays.toString(args)),
                Collections.<String>emptyList());
    }

    String buildDefaultMessage(String msgTemplate,
                               List<String> argsList,
                               Locale locale) {
        if (formatter == null) {
            return msgTemplate;
        }

        if (locale == null) {
            locale = Locale.ENGLISH;
        }

        return formatter.format(msgTemplate, argsList, locale);
    }
}
