/* **********************************************************
 * Copyright (c) 2011-2013, 2017, 2019 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.bindings;

import java.lang.reflect.Constructor;

import com.vmware.vapi.core.ApiProvider;
import com.vmware.vapi.internal.ClassLoaderUtil;
import com.vmware.vapi.internal.bindings.Stub;
import com.vmware.vapi.internal.bindings.TypeConverter;
import com.vmware.vapi.internal.util.Validate;

/**
 * Factory for client-side vAPI stubs.
 */
public final class StubFactory implements StubCreator {
    private static final String STUB_SUFFIX = "Stub";

    private final ApiProvider apiProvider;
    private final TypeConverter typeConverter;

    /**
     * Constructor.
     *
     * @param apiProvider to be used for invocations by this stub
     * @see #StubFactory(ApiProvider, TypeConverter)
     */
    public StubFactory(ApiProvider apiProvider) {
        this(apiProvider, null);
    }

    /**
     * Constructor.
     *
     * @param apiProvider to be used for invocations by this stub
     * @param typeConverter for conversion from/to native types; if {@code null}
     *        default one will be used
     */
    // TODO: rest native: exposing the .internal TypeConverter here is not good,
    //       probably better to create a separate .internal RestStubFactory
    public StubFactory(ApiProvider apiProvider, TypeConverter typeConverter) {
        Validate.notNull(apiProvider);

        this.apiProvider = apiProvider;
        this.typeConverter = typeConverter;
    }

    @Override
    public <T extends Service> T createStub(Class<T> vapiIface) {
        return createStub(vapiIface, null);
    }

    @Override
    public <T extends Service> T createStub(Class<T> vapiIface,
                                            StubConfigurationBase config) {
        if (vapiIface == null) {
            throw new IllegalArgumentException("vapiIface is required");
        }

        if (!vapiIface.isInterface()) {
            throw new IllegalArgumentException(
                    "vapiIface must represent Java interface");
        }
        if (config == null) {
            config = new StubConfiguration();
        }

        String implClassName = resolveStubClassName(vapiIface);
        try {
            ClassLoader cl = ClassLoaderUtil.getServiceClassLoader();
            Class<?> implClass = cl.loadClass(implClassName);

            if (!vapiIface.isAssignableFrom(implClass)) {
                throw new RuntimeException(
                        "Found stub implementation class which doesn't " +
                        "implement requested interface: " +
                        vapiIface.getName());
            }

            if (!Stub.class.isAssignableFrom(implClass)) {
                throw new RuntimeException(
                        "Found stub implementation class which doesn't extend " +
                        Stub.class.getName());
            }
            return vapiIface.cast(instantiateStubClass(config, implClass));
        } catch (RuntimeException ex) {
            throw ex;
        } catch (Exception ex) {
            // TODO: better error reporting
            throw new RuntimeException(ex);
        }
    }

    private Object instantiateStubClass(StubConfigurationBase config,
                                        Class<?> implClass) throws Exception {
        if (typeConverter == null) {
            Constructor<?> ctor = implClass.getConstructor(ApiProvider.class,
                                                           StubConfigurationBase.class);
            ctor.setAccessible(true);
            return ctor.newInstance(apiProvider, config);
        } else {
            Constructor<?> ctor = implClass.getConstructor(ApiProvider.class,
                                                           TypeConverter.class,
                                                           StubConfigurationBase.class);
            ctor.setAccessible(true);
            return ctor.newInstance(apiProvider, typeConverter, config);
        }
    }

    /**
     * Determines expected class for a client-side stub implementation
     * of given vAPI interface.
     *
     * @param vapiIface <code>Class</code> representing a vAPI interface
     * @return <code>String</code> representing the stub class name
     */
    private static String resolveStubClassName(Class<?> vapiIface) {
        // TODO: just a naming convention
        // use properties/XML or other metadata to keep the mapping?
        return vapiIface.getName() + STUB_SUFFIX;
    }
}
