/* **********************************************************
 * Copyright 2012-2014 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.bindings.client;

import com.vmware.vapi.bindings.UnresolvedError;
import com.vmware.vapi.client.exception.ClientException;
import com.vmware.vapi.std.Progress;
import com.vmware.vapi.std.errors.Error;

/**
 * This callback is used to receive the status (progress, result or error) of an
 * asynchronous operation invocation.
 *
 * <p>
 * Multiple progress updates can arrive for an operation. Note that progress
 * updates might be delivered after the result. For example consider client with
 * an application thread pool - the thread that delivers a progress update might
 * be scheduled before the thread that delivers the result.
 *
 * <p>
 * <i>Thread-safety:</i> The implementation must be thread-safe. The methods of
 * the callback might be invoked on different threads (for example consider
 * client with an I/O reactor thread pool and/or an application thread pool).
 * </p>
 *
 * @param <T> type of the operation result; for <code>void</code> type, use
 *            <code>java.lang.Void</code>
 */
public abstract class AsyncCallback<T> {

    /**
     * Reports progress for the operation invocation.
     *
     * @param progress  progress status of the operation invocation
     */
    public abstract void onProgress(Progress progress);

    /**
     * Completes the invocation with a result object.
     *
     * @param result  operation result
     */
    public abstract void onResult(T result);

    /**
     * Completes the invocation with an error. The error can be a legal/allowed
     * vAPI error (e.g. descendant of {@link Error}) coming from the provider;
     * {@link UnresolvedError} if the client bindings do not recognize the vAPI
     * error from the provider; {@link ClientException} if the client-side
     * runtime reports an error (e.g. network failure) or some other unchecked
     * exception like {@link IllegalArgumentException} if an internal problem
     * occurs in the client-side runtime.
     *
     * @param error operation error
     */
    public abstract void onError(RuntimeException error);
}
