/* **********************************************************
 * Copyright (c) 2013, 2017, 2019 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.bindings.type;

import java.util.Collections;
import java.util.List;

import com.vmware.vapi.CoreException;
import com.vmware.vapi.data.StructValue;
import com.vmware.vapi.internal.bindings.HasFieldsOfValidator;
import com.vmware.vapi.internal.bindings.TypeConverter;
import com.vmware.vapi.internal.data.ConstraintValidator;
import com.vmware.vapi.internal.data.TypeConverterDrivenConstraintValidator;
import com.vmware.vapi.internal.util.Validate;

/**
 * Description of a "dynamic structure" type in static language bindings.
 */
public class DynamicStructType implements Type {

    private final List<ConstraintValidator> validators;

    /**
     * Constructor.
     */
    public DynamicStructType() {
        this(Collections.<ConstraintValidator>emptyList());
    }

    /**
     * @param validators validators for constraints defined for in this
     *                   {@link DynamicStructType}. must not be <code>null</code>.
     */
    public DynamicStructType(List<ConstraintValidator> validators) {
        Validate.notNull(validators);
        this.validators = validators;
    }

    @Override
    public void accept(TypeVisitor visitor) {
        visitor.visit(this);
    }

    /**
     * Enforces all static (defined in the IDL) or dynamic constrains defined
     * over this structure.
     *
     * @param structValue struct value to validate; cannot be null
     * @param converter preferred converter to be used for validation purposes;
     *        may be {@code null} in which case default one will be used
     * @throws CoreException if the constraint is not satisfied
     */
    public void validate(StructValue structValue, TypeConverter converter) {
        for (ConstraintValidator validator : validators) {
            if (validator instanceof TypeConverterDrivenConstraintValidator && converter != null) {
                ((TypeConverterDrivenConstraintValidator) validator).validate(structValue, converter);
            } else {
                validator.validate(structValue);
            }
        }
    }
}
