/* **********************************************************
 * Copyright 2012-2013 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.bindings.type;

/**
 * Visitor of types. If you want to add a new operation that works on all
 * available types, implement this interface. To apply the operation on a type,
 * clients need to call the {@link Type#accept(TypeVisitor)} method passing the
 * visitor.
 */
public interface TypeVisitor {

    /**
     * Invoked by {@link VoidType#accept(TypeVisitor)}.
     *
     * @param type  void type
     */
    void visit(VoidType type);

    /**
     * Invoked by {@link IntegerType#accept(TypeVisitor)}.
     *
     * @param type  integer type
     */
    void visit(IntegerType type);

    /**
     * Invoked by {@link DoubleType#accept(TypeVisitor)}.
     *
     * @param type  double type
     */
    void visit(DoubleType type);

    /**
     * Invoked by {@link StringType#accept(TypeVisitor)}.
     *
     * @param type  string type
     */
    void visit(StringType type);

    /**
     * Invoked by {@link BooleanType#accept(TypeVisitor)}.
     *
     * @param type  boolean type
     */
    void visit(BooleanType type);

    /**
     * Invoked by {@link BinaryType#accept(TypeVisitor)}.
     *
     * @param type  binary type
     */
    void visit(BinaryType type);

    /**
     * Invoked by {@link DateTimeType#accept(TypeVisitor)}.
     *
     * @param type  datetime type
     */
    void visit(DateTimeType type);

    /**
     * Invoked by {@link UriType#accept(TypeVisitor)}.
     *
     * @param type  URI type
     */
    void visit(UriType type);

    /**
     * Invoked by {@link OptionalType#accept(TypeVisitor)}.
     *
     * @param type  optional type
     */
    void visit(OptionalType type);

    /**
     * Invoked by {@link ListType#accept(TypeVisitor)}.
     *
     * @param type  list type
     */
    void visit(ListType type);

    /**
     * Invoked by {@link StructType#accept(TypeVisitor)}.
     *
     * @param type  structure type
     */
    void visit(StructType type);

    /**
     * Invoked by {@link OpaqueType#accept(TypeVisitor)}.
     *
     * @param type  opaque type
     */
    void visit(OpaqueType type);

    /**
     * Invoked by {@link SecretType#accept(TypeVisitor)}.
     *
     * @param type  secret type
     */
    void visit(SecretType type);

    /**
     * Invoked by {@link TypeReference#accept(TypeVisitor)}.
     *
     * @param type  reference type
     */
    void visit(TypeReference<? extends Type> type);

    /**
     * Invoked by {@link EnumType#accept(TypeVisitor)}.
     *
     * @param type  enumeration type
     */
    void visit(EnumType type);

    /**
     * Invoked by {@link ErrorType#accept(TypeVisitor)}.
     *
     * @param type  error type
     */
    void visit(ErrorType type);

    /**
     * Invoked by {@link IdType#accept(TypeVisitor)}.
     *
     * @param idType  ID type
     */
    void visit(IdType idType);

    /**
     * Invoked by {@link SetType#accept(TypeVisitor)}.
     *
     * @param setType set type
     */
    void visit(SetType setType);

    /**
     * Invoked by {@link MapType#accept(TypeVisitor)}.
     *
     * @param mapType map type
     */
    void visit(MapType mapType);

    /**
     * Invoked by {@link DynamicStructType#accept(TypeVisitor)}.
     *
     * @param type  dynamic structure type
     */
    void visit(DynamicStructType type);

    /**
     * Invoked by {@link AnyErrorType#accept(TypeVisitor)}.
     *
     * @param type  any-error type
     */
    void visit(AnyErrorType type);

}
