/* ************************************************************************
 * Copyright (c) 2020 VMware, Inc. All rights reserved. VMware Confidential
 * ************************************************************************/
package com.vmware.vapi.client.exception;

/**
 * Exception which is reported when HTTP response is received with status
 * <pre>
 * 526 Invalid SSL Certificate
 * </pre>
 * <p>
 * This might happen when the client uses vanilla HTTP to communicate with an
 * intermediary which forwards the request over TLS/SSL to the target server.
 * <p>
 * The 526 response then signals that the intermediary failed to establish trust
 * with the target server. The response might contain the PEM encoded
 * certificate presented by the target server during the SSL handshake with the
 * intermediary (the SSL client). This certificate can be used to establish
 * subsequent successful connection if is considered trusted (Trust On First Use
 * scenarios).
 *
 * <p>
 * An example of the described scenario is the Envoy Sidecar used on the vCSA.
 */
public class InvalidSslCertificateException extends TransportProtocolException {

    private static final long serialVersionUID = 1L;

    /** {@code 526 Invalid SSL Certificate} */
    public static final int STATUS_CODE = 526;

    private final String content;

    public InvalidSslCertificateException(String content) {
        super("HTTP response with status code 526 (enable debug logging for details)");
        this.content = content;
    }

    /**
     * The body content of the 526 HTTP error as a <tt>UTF-8</tt>
     * {@code String}. Can be null if there was no body or the body could not be
     * read.
     * <p>
     * This might be the PEM-encoded certificate presented by the target server.
     *
     * @return the HTTP response body content
     */
    public String getContent() {
        return content;
    }
}
