/*
 * **********************************************************************
 * Copyright 2014-2015 VMware, Inc. All rights reserved. VMware Confidential
 * **********************************************************************
 */

package com.vmware.vapi.config;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vapi.core.ApiProvider;
import com.vmware.vapi.core.ApiProviderStub;
import com.vmware.vapi.core.InterfaceIdentifier;
import com.vmware.vapi.internal.provider.introspection.ApiIntrospectionStubImpl;
import com.vmware.vapi.protocol.ProtocolConnection;
import com.vmware.vapi.provider.introspection.ApiIntrospectionStub;
import com.vmware.vapi.provider.local.LocalProvider;

/**
 * Helper class containing methods to create maps of API providers.
 *
 * These methods are intended to facilitate configuring an {@link ApiAggregator}
 * in a spring config file.
 */
public class InterfacesHelper {
    static final Logger logger =
        LoggerFactory.getLogger(InterfacesHelper.class);

    /**
     * Return a map from service id to the local provider for all the services
     * handled by the local provider.
     *
     * This method is intended to facilitate configuring an
     * {@link ApiAggregator} in a spring config file.
     */
    public Map<String, ApiProvider> buildInterfacesMap(
            LocalProvider localProvider) {
        logger.debug("Adding local interfaces.");

        Map<String, ApiProvider> result = new HashMap<String, ApiProvider>();
        Set<InterfaceIdentifier> localInterfaces =
            localProvider.getInterfaceIdentifiers(null);
        for (InterfaceIdentifier interfaceIdentifier : localInterfaces) {
            result.put(interfaceIdentifier.toString(), localProvider);
        }
        return result;
    }

    /**
     * Return a map from service id to the API provider for all the services
     * handled by the API provider.
     *
     * This method is intended to facilitate configuring an
     * {@link ApiAggregator} in a spring config
     * file.
     * @param apiProvider the API provider exposing introspection services
     * @return Map from service id to API provider for all services handled by the provider
     */
    public Map<String, ApiProvider> buildInterfacesMap(ApiProvider apiProvider) {
        Map<String, ApiProvider> result = new HashMap<String, ApiProvider>();
        ApiIntrospectionStub introspectionService = new ApiIntrospectionStubImpl(apiProvider);
        Set<InterfaceIdentifier> interfaces = introspectionService.getInterfaceIdentifiers();
        for (InterfaceIdentifier interfaceIdentifier : interfaces) {
            result.put(interfaceIdentifier.toString(), apiProvider);
        }
        return result;
    }

    /**
     * Return a map from service id to the provider for all the services handled by the provider of
     * the protocol connection.
     *
     * This method is intended to facilitate configuring an {@link ApiAggregator} in a spring config
     * file.
     */
    public Map<String, ApiProvider> buildInterfacesMap(
            ProtocolConnection protocolConnection) {
        ApiProvider provider = protocolConnection.getApiProvider();
        ApiProviderStub stub = protocolConnection.getApiProviderStub();
        Map<String, ApiProvider> result = new HashMap<String, ApiProvider>();
        for (InterfaceIdentifier id : stub.getInterfaceIdentifiers()) {
            result.put(id.getName(), provider);
        }
        return result;
    }
}
