/* **********************************************************
 * Copyright 2012-2013 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.core;

import com.vmware.vapi.data.DataValue;

/**
 * This handle is used to complete an operation invocation (with an output value
 * or an error) or report progress. The invocation will block until either
 * {@link #setResult(Object)} or {@link #setError(RuntimeException)} marks the
 * operation as complete.
 *
 * <p>
 * Completing the operation more than once is forbidden. Reporting progress
 * after the operation is complete is forbidden for the service implementation
 * (i.e. the server side). However on the client side progress updates might be
 * delivered after the result. For example consider client with an application
 * thread pool - the thread that delivers a progress update might be scheduled
 * before the thread that delivers the result.
 *
 * <p>
 * <i>Thread-safety:</i> The implementation must be thread-safe. The methods of
 * the callback might be invoked on different threads (for example consider
 * client with an I/O reactor thread pool and/or an application thread pool).
 * </p>
 *
 * @param <T> type of the operation output
 */
public abstract class AsyncHandle<T> {

    /**
     * Reports progress for the operation invocation. May be invoked multiple
     * times.
     *
     * @param progress  progress status of the operation invocation
     * @throws IllegalStateException if the operation is already completed
     */
    public abstract void updateProgress(DataValue progress);

    /**
     * Completes the invocation with an output value. Must not be called after
     * the operation is already completed (with either an output or an error).
     *
     * @param result  operation output
     * @throws IllegalStateException if the operation is already completed
     */
    public abstract void setResult(T result);

    /**
     * Completes the invocation with an error. Must not be called after the
     * operation is already completed (with either an output or an error).
     *
     * @param error  operation error
     * @throws IllegalStateException if the operation is already completed
     */
    public abstract void setError(RuntimeException error);

}
