/* **********************************************************
 * Copyright (c) 2011-2013, 2017-2020, 2022 VMware, Inc. All rights reserved. -- VMware Confidential
 * ********************************************************* */

package com.vmware.vapi.core;

import java.io.Serializable;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;

import com.vmware.vapi.internal.util.Validate;
import com.vmware.vapi.protocol.HttpConfiguration;

/**
 * The <code>ExecutionContext</code> class provides information about the
 * execution of an API method invocation.
 * <p>
 * It consists of
 * <ol>
 * <li>out-of-band context data provided by client and passed along with the
 * invocation/request</li>
 * <li>authentication/security context and data</li>
 * <li>runtime details, specific to the request being processed</li>
 * </ol>
 */
public class ExecutionContext implements Serializable {
    private static final long serialVersionUID = 1L;

    private final transient SecurityContext securityContext;
    private final transient ApplicationData applicationData;
    private final transient RuntimeData runtimeData;

    /**
     * Empty context
     */
    public static final ExecutionContext EMPTY = new ExecutionContext();

    /**
     * Default constructor. Convenience one for creating an empty
     * <code>ExecutionContext</code>.
     * <p>
     * Use {@link ExecutionContext#EMPTY} when distinct instance is not
     * required.
     */
    public ExecutionContext() {
        this(null, null, null);
    }

    /**
     * Do not use this constructor to recreate {@link ExecutionContext}
     * instances with modified parameters. Leverage the {@link Builder} class
     * instead.
     *
     * @param securityContext may be {@code null}
     */
    public ExecutionContext(SecurityContext securityContext) {
        this(null, securityContext, null);
    }

    /**
     * Do not use this constructor to recreate {@link ExecutionContext}
     * instances with modified parameters. Leverage the {@link Builder} class
     * instead.
     *
     * @param wireData may be {@code null}
     * @param securityContext may be {@code null}
     */
    public ExecutionContext(ApplicationData wireData,
                            SecurityContext securityContext) {
        this(wireData, securityContext, null);
    }

    /**
     * Do not use this constructor to recreate {@link ExecutionContext}
     * instances with modified parameters. Leverage the {@link Builder} class
     * instead.
     *
     * @param wireData may be {@code null}
     * @param securityContext may be {@code null}
     * @param runtimeData may be {@code null}
     */
    public ExecutionContext(ApplicationData wireData,
                            SecurityContext securityContext,
                            RuntimeData runtimeData) {
        this.applicationData = wireData;
        this.securityContext = securityContext;
        this.runtimeData = runtimeData;
    }

    /**
     * Returns a new {@link ExecutionContext} instance with replaced application
     * data
     *
     * @param appData {@link ApplicationData} to be used in the new instance
     * @return new {@link ExecutionContext} instance with updated
     *         {@link ApplicationData}
     */
    public ExecutionContext withApplicationData(ApplicationData appData) {
        return new ExecutionContext(appData, securityContext, runtimeData);
    }

    /**
     * Returns a new {@link ExecutionContext} instance with replaced security
     * context
     *
     * @param secCtx {@link SecurityContext} to be used in the new instance
     * @return new {@link ExecutionContext} instance with updated
     *         {@link SecurityContext}
     */
    public ExecutionContext withSecurityContext(SecurityContext secCtx) {
        return new ExecutionContext(applicationData, secCtx, runtimeData);
    }

    /**
     * @return the application data; may be {@code null}
     */
    public ApplicationData retrieveApplicationData() {
        return applicationData;
    }

    /**
     * @return the security context; may be {@code null}
     */
    public SecurityContext retrieveSecurityContext() {
        return securityContext;
    }

    /**
     * @return the runtime data; may be {@code null}
     */
    public RuntimeData retrieveRuntimeData() {
        return runtimeData;
    }

    /**
     * Immutable class representing additional application specific data
     * associated with the request for method execution represented by this
     * <code>ExecutionContext</code>.
     * <p>
     * The application data format is key-value pairs of <code>String</code>s.
     *
     * <p>
     * This application data is provided by the client initiating the execution,
     * it is then transported as is over the wire and is available for the
     * provider-side service implementations on the server. This extra data is
     * completely opaque for the infrastructure, in other words it is a contract
     * between the client and the service implementation only.
     */
    public static class ApplicationData {
        /**
         * {@code ApplicationData} property key for user agent.
         */
        public static final String USER_AGENT_KEY = "$userAgent";

        /**
         * {@code ApplicationData} timezone preference property key.
         * <p>
         * Optional Timezone ID from the IANA database to be used for rendering
         * date and time values. When missing, UTC will be used.
         */
        public static final String TIMEZONE_KEY = "timezone";

        /**
         * {@code ApplicationData} translation language preference property key.
         * <p>
         * Optional parameter following the format of the HTTP 1.1
         * Accept-Language header (RFC 7231 section 5.3.5) used for selecting
         * localization bundle.
         * <p>
         * When missing "en" will be used.
         */
        public static final String ACCEPT_LANGUAGE_KEY = "accept-language";

        /**
         * {@code ApplicationData} formatting locale property key.
         * <p>
         * Optional parameter following the format of the HTTP 1.1
         * Accept-Language header (RFC 7231 section 5.3.5) used for overriding
         * the value format locale. When missing, implementations will fall back
         * to the {@code accept-language} parameter.
         *
         * <p>
         * If both parameters are missing, locale neutral formatting should be
         * applied i.e. short date to use "yyyy-MM-dd".
         */
        public static final String FORMAT_LOCALE_KEY = "format-locale";

        private final Map<String, String> wireData;

        /**
         * Constructs an empty instance.
         */
        public ApplicationData() {
            wireData = Collections.emptyMap();
        }

        /**
         * Constructs an instance with the specified data as a source.
         *
         * @param data key-value entries
         * @throws NullPointerException if <code>data</code> is
         *         <code>null</code>
         */
        public ApplicationData(Map<String, String> data) {
            wireData = new HashMap<>(data.size(), 1f);
            wireData.putAll(data);
        }

        /**
         * Constructs an instance with a single property only.
         *
         * @param key the key of the property
         * @param value the value of the property
         */
        public ApplicationData(String key, String value) {
            wireData = Collections.singletonMap(key, value);
        }

        private ApplicationData(ApplicationData source,
                                Map<String, String> data) {
            wireData = new HashMap<>(source.wireData.size() + data.size(), 1f);
            wireData.putAll(source.wireData);
            wireData.putAll(data);
        }

        private ApplicationData(ApplicationData source,
                                String key,
                                String value) {
            wireData = new HashMap<>(source.wireData.size() + 1, 1f);
            wireData.putAll(source.wireData);
            wireData.put(key, value);
        }

        /**
         * @param key of the property
         * @return value for the specified <code>key</code> or <code>null</code>
         *         if it is unavailable
         */
        public String getProperty(String key) {
            return wireData.get(key);
        }

        /**
         * @return an immutable {@code Map} representing all the properties;
         *         cannot be {@code null}
         */
        public Map<String, String> getAllProperties() {
            return Collections.unmodifiableMap(wireData);
        }

        /**
         * Merges the specified {@code source} and {@code data} into a newly
         * created {@code ApplicationData} instance. Any conflicts are resolved
         * in favor of the specified {@code data}.
         *
         * @param source the source of data; may be {@code null}
         * @param data a collection of properties to merge with the
         *        {@code source}
         * @return a new instance of {@code ApplicationData} combining the
         *         specified {@code source} and {@code data}
         * @throws NullPointerException if <code>data</code> is
         *         <code>null</code>
         */
        public static ApplicationData merge(ApplicationData source,
                                            Map<String, String> data) {
            if (source == null) {
                return new ApplicationData(data);
            }
            return new ApplicationData(source, data);
        }

        /**
         * Merges the specified {@code source} and property into a newly created
         * {@code ApplicationData} instance. The specified {@code value} will
         * override the previous one, if a property with the specified
         * {@code key} is already present in the {@code source}.
         *
         * @param source the source of data; may be {@code null}
         * @param key the key of the property to merge with the {@code source}
         * @param value the value of the property to merge with the
         *        {@code source}
         * @return a new instance of {@code ApplicationData} combining the
         *         specified {@code source} and property
         */
        public static ApplicationData merge(ApplicationData source,
                                            String key,
                                            String value) {
            if (source == null || source.wireData.size() == 0) {
                return new ApplicationData(key, value);
            }
            return new ApplicationData(source, key, value);
        }
    }

    /**
     * Implementations of this interface will provide all needed data for
     * authentication for the given invocation.
     */
    public interface SecurityContext {

        public static final String AUTHENTICATION_SCHEME_ID = "authn_scheme_id";
        public static final String AUTHENTICATION_DATA_ID = "authn_data_id";

        /**
         * @return the security context property under the specified key
         */
        Object getProperty(String key);

        /**
         * @return an unmodifiable <code>Map</code> representing all the
         *         properties
         */
        Map<String, Object> getAllProperties();
    }

    /**
     * This class contains data and settings related to the execution of the
     * invocation by the vAPI runtime, protocol and transport. Such examples
     * might be HTTP and/or TCP request execution settings, headers, SSL
     * certificates, etc.
     *
     * <p>
     * Note that some of the data items might be relevant and present only for
     * particular types of protocol and transport implementations, or particular
     * side of the API invocation (client or server).
     * </p>
     *
     * <p>
     * This class is immutable.
     * </p>
     */
    public static class RuntimeData {
        private final Integer readTimeout;
        private final HttpResponseAccessor responseAccessor;
        private final Map<String, Object> keyValueEntries;

        private RuntimeData(Integer readTimeout,
                            HttpResponseAccessor responseAccessor,
                            Map<String, Object> keyValueEntries) {
            this.readTimeout = readTimeout;
            this.responseAccessor = responseAccessor;
            this.keyValueEntries = copyMap(keyValueEntries);
        }

        /*
         * A memory-optimized shallow copy
         */
        private static Map<String, Object> copyMap(Map<String, Object> map) {
            if (map == null || map.size() == 0) {
                return null;
            }
            if (map.size() == 1) {
                Entry<String, Object> entry = map.entrySet().iterator().next();
                return Collections.singletonMap(entry.getKey(), entry.getValue());
            }
            Map<String, Object> entries = new HashMap<>(map.size(), 1);
            entries.putAll(map);
            return entries;
        }

        /**
         * Returns client-side read timeout as specified in
         * {@link Builder#setReadTimeout(Integer)}.
         *
         * @return the number of milliseconds the client will wait for a
         *         response from the server;
         */
        public Integer getReadTimeout() {
            return readTimeout;
        }

        /**
         * Returns client-side raw REST response accessor as specified in
         * {@link Builder#setResponseAccessor(HttpResponseAccessor)}.
         *
         * @return the response accessor instance or {@code null}
         */
        public HttpResponseAccessor getResponseAccessor() {
            return responseAccessor;
        }

        /**
         * Returns the value of the data entry with key <code>key</code>.
         *
         * @param key the entry key. Must not be <code>null</code>
         * @return the value of the entry with key <code>key</code> or <code>null</code> if it is
         *         unavailable
         * @throws IllegalArgumentException if the parameter is <code>null</code>
         */
        public Object getValue(String key) {
            Validate.notNull(key);
            if (keyValueEntries == null) {
                return null;
            }
            return keyValueEntries.get(key);
        }
    }

    public static class Builder {
        private Map<String, String> appData;
        private SecurityContext securityContext;
        private Integer readTimeout;
        private HttpResponseAccessor responseAccessor;
        private Map<String, Object> runtimeKeyValueEntries;

        public Builder() {
        }

        private Builder(ExecutionContext ec) {
            Objects.requireNonNull(ec);
            setApplicationData(ec.retrieveApplicationData());
            this.securityContext = ec.retrieveSecurityContext();
            setRuntimeData(ec.retrieveRuntimeData());
        }

        /**
         * Initializes the builder with existing {@link ExecutionContext} data.
         * This allows to update one or more settings.
         *
         * @param ec instance to be copied; If {@code null} blank fields will be
         *        used as starting point
         * @return builder instance with mutable fields
         */
        public static Builder from(ExecutionContext ec) {
            return ec != null ? new Builder(ec) : new Builder();
        }

        /**
         * @param applicationData can be {@code null}
         * @return this instance
         */
        public Builder setApplicationData(ApplicationData applicationData) {
            if (applicationData != null) {
                appData = new HashMap<>(applicationData.getAllProperties());
            } else {
                appData = null;
            }
            return this;
        }

        private void setRuntimeData(RuntimeData runtimeData) {
            if (runtimeData == null) {
                return;
            }
            this.readTimeout = runtimeData.getReadTimeout();
            this.responseAccessor = runtimeData.getResponseAccessor();
            if (runtimeData.keyValueEntries != null) {
                this.runtimeKeyValueEntries = new HashMap<>(runtimeData.keyValueEntries);
            }
        }

        /**
         * Add setting to {@link ApplicationData}. If {@link ApplicationData} is {@code null} new
         * instance will be created. If the {@code key} exists it will be replaced.
         * <p>
         * <i>Thread-safety:</i> This method is not thread-safe.
         * @param key data key
         * @param value data value
         * @return the {@link Builder} instance
         */
        public Builder mergeApplicationData(String key, String value) {
            if (appData == null) {
                appData = new HashMap<>();
            }
            appData.put(key, value);
            return this;
        }

        /**
         * Add multiple entries to {@link ApplicationData}. If {@link ApplicationData} is
         * {@code null} new instance will be created. Existing keys will be replaced (or values for
         * existing keys)
         * <p>
         * <i>Thread-safety:</i> This method is not thread-safe.
         * @param data values
         * @return the {@link Builder} instance
         */
        public Builder mergeApplicationData(Map<String, String> data) {
            if (appData == null) {
                appData = new HashMap<>();
            }
            appData.putAll(data);
            return this;
        }

        /**
         * @param securityContext can be {@code null}
         * @return this instance
         */
        public Builder setSecurityContext(SecurityContext securityContext) {
            this.securityContext = securityContext;
            return this;
        }

        /**
         * Sets the amount of time in milliseconds the client will wait for a
         * response from the server before timing out.
         * <p>
         * For the HTTP 1.1 transport protocol this value maps to the
         * {@code SO_TIMEOUT}.
         * <p>
         * If not {@code null}, this property overrides the one read from
         * {@link HttpConfiguration#getSoTimeout()}.
         * <p>
         * The default value is {@code null}.
         * <p>
         * A value of zero is interpreted as an infinite timeout, {@code null}
         * is interpreted as unspecified at this level (concrete invocation) and
         * client-level configuration applies.
         *
         * @param readTimeout non-negative value or {@code null}
         * @return this instance
         */
        public Builder setReadTimeout(Integer readTimeout) {
            assertNonNegative(readTimeout);
            this.readTimeout = readTimeout;
            return this;
        }

        /**
         * Sets a {@link HttpResponseAccessor} instance which can access and
         * capture information from the raw REST response.
         *
         * <p>
         * This accessor is only applicable for requests executed from a client
         * that uses REST transport.
         *
         * @param responseAccessor which will be provided with the raw REST
         *        response
         * @return this instance
         */
        public Builder setResponseAccessor(HttpResponseAccessor responseAccessor) {
            this.responseAccessor = responseAccessor;
            return this;
        }

        /**
         * Store additional data in the API invocation execution context in the form of a key-value
         * entry. The entry is added to the {@link RuntimeData}. The value can be accessed
         * throughout the invocation by calling {@link RuntimeData#getValue(key)}. Existing keys
         * will be replaced.
         * <p>
         * This key-value entry will not be serialized on the wire.
         * <p>
         * <i>Thread-safety:</i> This method is not thread-safe.
         *
         * @param key the entry key
         * @param value the entry value
         * @return this instance
         */
        public Builder setRuntimeKeyValue(String key, Object value) {
            if (runtimeKeyValueEntries == null) {
                runtimeKeyValueEntries = new HashMap<>();
            }
            runtimeKeyValueEntries.put(key, value);
            return this;
        }

        private static void assertNonNegative(Integer value) {
            if (value != null && value < 0) {
                throw new IllegalArgumentException("Non-negative value or null expected");
            }
        }

        public ExecutionContext build() {
            return new ExecutionContext(getApplicationData(),
                                        securityContext,
                                        getRuntimeData());
        }

        private ApplicationData getApplicationData() {
            return appData != null ? new ApplicationData(appData) : null;
        }

        private RuntimeData getRuntimeData() {
            RuntimeData rData = new RuntimeData(readTimeout,
                                                responseAccessor,
                                                runtimeKeyValueEntries);
            return rData;
        }
    }
}
