/* **********************************************************
 * Copyright (c) 2011, 2019 VMware, Inc. All rights reserved. -- VMware Confidential
 * **********************************************************/

/*
 * InterfaceDefinition.java --
 *
 *      Detailed information about a vAPI interface.
 */

package com.vmware.vapi.core;

import java.util.Set;

import com.vmware.vapi.internal.util.Validate;

/**
 * The <code>InterfaceDefinition</code> class contains detailed
 * information about a vAPI interface. This should contain all the
 * information required to address an interface in the vAPI runtime.
 */
public class InterfaceDefinition {
    private InterfaceIdentifier id;
    private Set<MethodIdentifier> methodIds;

    /**
     * Constructor.
     *
     * @param id           interface identifier
     * @param methodIds    set of identifiers for methods in the interface
     *
     * @throws IllegalArgumentException if {@code id} or {@code methodIds} is
     *         {@code null} or {@code methodIds} contains {@code null} element
     */
    public InterfaceDefinition(InterfaceIdentifier id,
                               Set<MethodIdentifier> methodIds) {
        Validate.notNull(id, "Interface identifier is required.");
        Validate.notNull(methodIds, "'methodIds' is required");
        Validate.noNullElements(methodIds,
                "'methodIds' must not contain null elements");

        this.id = id;
        this.methodIds = methodIds;
    }

    /**
     * Returns the identifier of this interface.
     *
     * @return  interface identifier of this interface
     */
    public InterfaceIdentifier getIdentifier() {
        return id;
    }

    /**
     * Returns the set of identifiers for methods in this interface.
     *
     * @return  set of method identifiers for this interface
     */
    public Set<MethodIdentifier> getMethodIdentifiers() {
        return methodIds;
    }

    @Override
    public String toString() {
        return "Interface id: " + id + "\n " +
                " methods: " + methodIds;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (!(obj instanceof InterfaceDefinition)) {
            return false;
        }
        InterfaceDefinition other = (InterfaceDefinition) obj;
        return getIdentifier().equals(other.getIdentifier()) &&
                getMethodIdentifiers().equals(other.getMethodIdentifiers());
    }

    @Override
    public int hashCode() {
        int hash = 1;
        hash = hash * 31 + getIdentifier().hashCode();
        hash = hash * 31 + getMethodIdentifiers().hashCode();
        return hash;
    }

}