/* **********************************************************
 * Copyright (c) 2011-2013, 2019-2020 VMware, Inc. All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.core;

import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.ErrorValue;

/**
 * The <code>MethodResult</code> class contains the result of a method
 * call.  It contains either the output of the method invocation or
 * an error reported by the method invocation. These are mutually
 * exclusive.
 */
public class MethodResult {
    public static final MethodResult EMPTY = new MethodResult();

    private DataValue output;
    private ErrorValue error;
    private Consumer<AsyncHandle<MethodResult>> next;

    private MethodResult() { }

    /**
     * Constructor. Creates an instance that is associated with
     * {@code DataValue} return value or {@code ErrorValue} error
     * result of a method invocation.
     * <p>Exactly one of {@code output} and {@code error} must be {@code null}
     *
     * <p>Next handle is set to {@code null}
     *
     * @param error  the error to associate with this result
     * @param output  the return value to associate with this result
     * @throws IllegalArgumentException if {@code output} and {@code error} are
     *         both {@code null}, or are both not {@code null}
     */
    public MethodResult(DataValue output, ErrorValue error) {
        this(output, error, null);
    }

    /**
     * Constructor. Creates an instance that is associated with
     * {@code DataValue} return value or {@code ErrorValue} error
     * result of a method invocation.
     * <p>Exactly one of {@code output} and {@code error} must be {@code null}
     *
     * <p>Next handle allows supplying an {@code AsyncHandle} responsible
     * for handling the following {@code MethodResult}. Can be {@code null}
     *
     * @param error  the error to associate with this result
     * @param output  the return value to associate with this result
     * @param next  the next handle consumer
     * @throws IllegalArgumentException if {@code output} and {@code error} are
     *         both {@code null}, or are both not {@code null}
     */
    public MethodResult(DataValue output,
                        ErrorValue error,
                        Consumer<AsyncHandle<MethodResult>> next) {
        if (output != null && error == null) {
            setOutput(output);
        } else if (output == null && error != null) {
            setError(error);
        } else {
             throw new IllegalArgumentException(
                     "Exactly one of 'output' and 'error' must be null");
        }
        this.next = next;
    }

    /**
     * Returns the output of the method call.
     *
     * @return  data value returned by the method invocation
     */
    public DataValue getOutput() {
        return output;
    }

    /**
     * Returns the error for this <code>MethodResult</code>.
     *
     * @return the error or null if there is none
     */
    public ErrorValue getError() {
        return error;
    }

    /**
     * Returns the handle to receive the next {@link AsyncHandle} or
     * {@code null}.
     * <p>
     * If the recipient wishes to terminate the connection they ought to pass a
     * {@code null} {@link AsyncHandle} to the publisher.
     *
     * @return handle to receive the next {@link AsyncHandle}
     */
    public Consumer<AsyncHandle<MethodResult>> getNext() {
        return next;
    }

    /**
     * Returns <code>true</code> if the invoked method completed
     * successfully, i.e. it hasn't reported an error, but returned
     * a result.
     *
     * @return  <code>true</code> if the method completed successfully
     */
    public boolean success() {
        return getError() == null;
    }

    /**
     * Sets the output of the method call.
     *
     * @param output  data value returned by the method invocation
     */
    private void setOutput(DataValue output) {
        this.output = output;
    }

    /**
     * Sets the error for this <code>MethodResult</code>.
     *
     * @param error  the error to set
     */
    private void setError(ErrorValue error) {
        this.error = error;
    }

    /**
     * Sets the {@code Consumer} for the next frame of the result
     */
    public void setNext(Consumer<AsyncHandle<MethodResult>> next) {
        this.next = next;
    }

    /** {@inheritDoc} */
    @Override
    public String toString() {
        return "Output: " + output + "\nError: " + error;
    }

    /**
     * Static factory method for creating {@code MethodResult} instance
     * for normal return value (as opposed to error)
     *
     * @param output  the method return value for created result
     * @return  new {@code MethodResult} instance
     * @see MethodResult#MethodResult(DataValue, ErrorValue)
     */
    public static MethodResult newResult(DataValue output) {
        return new MethodResult(output, null);
    }

    /**
     * Static factory method for creating {@code MethodResult} instance
     * for error reported by method invocation
     *
     * @param error  the error reported by method
     * @return  new {@code MethodResult} instance
     * @see MethodResult#MethodResult(DataValue, ErrorValue)
     */
    public static MethodResult newErrorResult(ErrorValue error) {
        return new MethodResult(null, error);
    }

}